/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.SQLType;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
/**
 * This class efficiently computes metadata for a given EClass. It caches the
 * results to avoid repetitive calculations.
 */
public class ClassMetadata {
	protected EAttribute[] singleAttributes;
	protected EAttribute[] manyAttributes;
	protected EClass[] rootSuperclasses;
	protected EClass[] nonRootSuperclasses;
	protected EClass[] allSuperclasses;
	protected EClass[] allSubclasses;
	protected EReference[] containReferences;
	protected EReference[] notContainReferences;
	protected EReference[] manyReferences;
	protected EReference[] classTableReferences;
	protected EReference containerReference;
	protected Map classesToData;
	protected DBMap dbMap;
	protected EClass eClass;
	protected RDBHelper helper;
	protected Table table;
	protected Column primaryKey;
	protected DBMap.ClassData classData;
	protected List classTableColumns;
	protected List referenceClassTableColumns;
	protected int[] classTableSQLTypes;
	protected int[] manyAttributeSQLTypes;
	protected EFactory[] typeFactories;
	protected EFactory classFactory;

	/**
	 * Constructor.
	 */
	public ClassMetadata(EClass eClass, DBMap dbMap, RDBHelper helper) {
		super();
		this.eClass = eClass;
		this.dbMap = dbMap;
		this.helper = helper;
	}

	/**
	 * Return the class table.
	 */
	public Table getTable() {
		if (table == null)
			table = computeTable();

		return table;
	}

	protected Table computeTable() {
		if (classData == null)
			classData = computeClassData();

		return classData.getTable();
	}

	protected DBMap.ClassData computeClassData() {
		return (DBMap.ClassData) dbMap.getDBRepresentation(eClass);
	}

	/**
	 * Return the Column that is the primary key for the class table.
	 */
	public Column getPrimaryKey() {
		if (primaryKey == null)
			primaryKey = computePrimaryKey();

		return primaryKey;
	}

	protected Column computePrimaryKey() {
		if (table == null)
			getTable();

		return helper.getPrimaryKey(table);
	}

	/**
	 * Compute both the class table attributes and the many attributes for the
	 * given class at the same time.
	 */
	protected void processAttributes() {
		List attributes = eClass.getEAttributes();
		List single = new ArrayList();
		List many = new ArrayList();

		for (int i = 0, l = attributes.size(); i < l; i++) {
			EAttribute attribute = (EAttribute) attributes.get(i);

			if (!attribute.isTransient()) {
				if (attribute.isMany())
					many.add(attribute);
				else
					single.add(attribute);
			}
		}

		singleAttributes = (EAttribute[]) single.toArray(new EAttribute[single.size()]);
		manyAttributes = (EAttribute[]) many.toArray(new EAttribute[many.size()]);
	}

	/**
	 * Returns the local, non-transient attributes for the given class that have
	 * single values.
	 */
	public EAttribute[] getSingleAttributes() {
		if (singleAttributes == null)
			processAttributes();

		return singleAttributes;
	}

	/**
	 * Returns the local, non-transient attributes for the given class that have
	 * multiple values.
	 */
	public EAttribute[] getManyAttributes() {
		if (manyAttributes == null)
			processAttributes();

		return manyAttributes;
	}

	/**
	 * Returns the factory for each attribute type for the single-valued
	 * attributes of the class, in the same order as the singleAttributes array.
	 */
	public EFactory[] getTypeFactories() {
		if (typeFactories == null)
			typeFactories = computeTypeFactories();

		return typeFactories;
	}

	protected EFactory[] computeTypeFactories() {
		if (singleAttributes == null)
			processAttributes();

		EFactory[] factories = new EFactory[singleAttributes.length];

		for (int i = 0; i < singleAttributes.length; i++) {
			EDataType datatype = singleAttributes[i].getEAttributeType();
			EPackage pkg = (EPackage) datatype.eContainer();
			EFactory factory = pkg.getEFactoryInstance();
			factories[i] = factory;
		}

		return factories;
	}

	/**
	 * Return a factory that can be used to instantiate the class.
	 */
	public EFactory getClassFactory() {
		if (classFactory == null)
			classFactory = computeClassFactory();

		return classFactory;
	}

	protected EFactory computeClassFactory() {
		EPackage pkg = (EPackage) eClass.eContainer();
		return pkg.getEFactoryInstance();
	}

	/**
	 * Gets the class table columns.
	 */
	public List getClassTableColumns() {
		if (classTableColumns == null)
			classTableColumns = computeClassTableColumns();

		return classTableColumns;
	}

	/**
	 * Computes the column for each single-valued attribute in the class table.
	 */
	protected List computeClassTableColumns() {
		List columns = new ArrayList();

		if (singleAttributes == null)
			processAttributes();

		for (int i = 0; i < singleAttributes.length; i++) {
			DBMap.AttributeData data;
			data = (DBMap.AttributeData) dbMap.getDBRepresentation(singleAttributes[i]);
			columns.add(data.getValueColumn());
		}

		return columns;
	}

	/**
	 * Return the columns in the class table for references, including order
	 * columns, if appropriate.
	 */
	public List getReferenceClassTableColumns() {
		if (referenceClassTableColumns == null)
			processClassTableReferences();

		return referenceClassTableColumns;
	}

	/**
	 * Returns a list of the references that are stored in the class table.
	 */
	public EReference[] getClassTableReferences() {
		if (classTableReferences == null)
			processClassTableReferences();

		return classTableReferences;
	}

	/**
	 * Compute both the class table columns for references and the class table
	 * references themselves.
	 */
	protected void processClassTableReferences() {
		Table classTable = getTable();
		List columns = classTable.getColumns();
		List references = new ArrayList();
		referenceClassTableColumns = new ArrayList();

		for (int i = 0, l = columns.size(); i < l; i++) {
			Column column = (Column) columns.get(i);
			EReference reference = dbMap.getReference(column);

			if (reference != null) {
				references.add(reference);
				referenceClassTableColumns.add(column);

				if (reference.isMany()) {
					DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(reference);
					referenceClassTableColumns.add(data.getOrderColumn());
				}
			}
		}

		classTableReferences = (EReference[]) references.toArray(new EReference[references.size()]);
	}

	/**
	 * Gets the SQL types for the attribute columns in the class table, in the
	 * same order the attributes appear in the singleAttributes array.
	 */
	public int[] getClassTableSQLTypes() {
		if (classTableSQLTypes == null)
			classTableSQLTypes = computeClassTableSQLTypes();

		return classTableSQLTypes;
	}

	protected int[] computeClassTableSQLTypes() {
		if (classTableColumns == null)
			classTableColumns = computeClassTableColumns();

		int[] types = new int[classTableColumns.size()];

		for (int i = 0, l = classTableColumns.size(); i < l; i++) {
			SQLType type = ((Column) classTableColumns.get(i)).getType();
			types[i] = type.getSqlType();
		}

		return types;
	}

	/**
	 * Returns true if an id needs to be computed; false otherwise.
	 */
	public boolean isComputeId() {
		if (classData == null)
			classData = computeClassData();

		return classData.isComputeId();
	}

	/**
	 * Sets the root classes, non-root classes, and superclasses for the given
	 * class, if they are not set.
	 */
	protected void processSuperclasses() {
		if (allSuperclasses == null)
			allSuperclasses = computeAllSuperclasses();

		List roots = new ArrayList();
		List other = new ArrayList();

		for (int i = 0; i < allSuperclasses.length; i++) {
			EClass superclass = allSuperclasses[i];

			if (superclass.getESuperTypes().isEmpty())
				roots.add(superclass);
			else
				other.add(superclass);
		}

		rootSuperclasses = (EClass[]) roots.toArray(new EClass[roots.size()]);
		nonRootSuperclasses = (EClass[]) other.toArray(new EClass[other.size()]);
	}

	protected EClass[] computeAllSuperclasses() {
		List superclasses = eClass.getEAllSuperTypes();
		return (EClass[]) superclasses.toArray(new EClass[superclasses.size()]);
	}

	/**
	 * Returns the root classes for the given class. The root classes are the
	 * superclasses that do not have superclasses themselves. If the given class
	 * has no superclasses, an empty array is returned.
	 */
	public EClass[] getRootSuperclasses() {
		if (rootSuperclasses == null)
			processSuperclasses();

		return rootSuperclasses;
	}

	/**
	 * Returns the superclasses for the given class that are not root classes.
	 * Returns an empty array if the class has no superclasses.
	 */
	public EClass[] getNonRootSuperclasses() {
		if (nonRootSuperclasses == null)
			processSuperclasses();

		return nonRootSuperclasses;
	}

	/**
	 * Gets all of the superclasses for this class.
	 */
	public EClass[] getAllSuperclasses() {
		if (allSuperclasses == null)
			allSuperclasses = computeAllSuperclasses();

		return allSuperclasses;
	}

	/**
	 * Return the SQL types for the many-valued attributes for the class, in the
	 * same order as the many attributes array.
	 */
	public int[] getManyAttributeSQLTypes() {
		if (manyAttributeSQLTypes == null)
			manyAttributeSQLTypes = computeManyAttributeSQLTypes();

		return manyAttributeSQLTypes;
	}

	protected int[] computeManyAttributeSQLTypes() {
		if (manyAttributes == null)
			processAttributes();

		int[] types = new int[manyAttributes.length];

		for (int i = 0; i < manyAttributes.length; i++) {
			EAttribute attrib = manyAttributes[i];
			DBMap.AttributeData attributeData;
			attributeData = (DBMap.AttributeData) dbMap.getDBRepresentation(attrib);
			Column value = attributeData.getValueColumn();
			types[i] = value.getType().getSqlType();
		}

		return types;
	}

	/**
	 * Returns all subclasses of the given class; if the class does not have
	 * subclasses, an empty array is returned.
	 */
	public EClass[] getAllSubclasses() {
		if (allSubclasses == null) {
			List subclasses = dbMap.getAllSubclasses(eClass);
			allSubclasses = (EClass[]) subclasses.toArray(new EClass[subclasses.size()]);
		}

		return allSubclasses;
	}

	/**
	 * Returns the local and inherited, non-transient, non-container references
	 * for the given class. It returns references in bi-directional
	 * relationships, unlike the getAddReferences method, which ignores certain
	 * references.
	 */
	public EReference[] getReferences() {
		if (notContainReferences == null)
			processReferences();

		return notContainReferences;
	}

	/**
	 * Returns the local and inherited containment references for the given
	 * class.
	 */
	public EReference[] getContainmentReferences() {
		if (containReferences == null)
			processReferences();

		return containReferences;
	}

	/**
	 * Returns bi-directional many reference where neither end is transient.
	 */
	public EReference[] getManyReferences() {
		if (manyReferences == null)
			processReferences();

		return manyReferences;
	}

	/**
	 * Returns the single container reference for the class.
	 */
	public EReference getContainerReference() {
		if (manyReferences == null)
			processReferences();

		return containerReference;
	}

	/**
	 * Compute the containment references and the other references for the given
	 * class.
	 */
	protected void processReferences() {
		List notContain = new ArrayList();
		List contain = new ArrayList();
		List many = new ArrayList();
		List references = eClass.getEAllReferences();

		for (int i = 0, l = references.size(); i < l; i++) {
			EReference reference = (EReference) references.get(i);

			if (reference.isContainer())
				containerReference = reference;

			if (reference.isTransient() || reference.isContainer())
				continue;

			if (reference.isContainment())
				contain.add(reference);
			else
				notContain.add(reference);

			EReference opposite = (EReference) reference.getEOpposite();

			if (reference.isMany() && (opposite == null || opposite.isMany()))
				many.add(reference);
			else if (reference.isMany() && !reference.isUnique())
				many.add(reference);
		}

		containReferences = (EReference[]) contain.toArray(new EReference[contain.size()]);
		notContainReferences = (EReference[]) notContain.toArray(new EReference[notContain.size()]);
		manyReferences = (EReference[]) many.toArray(new EReference[many.size()]);
	}
	/**
	 * @return Returns the dbMap.
	 */
	public DBMap getDbMap() {
		return dbMap;
	}
} // ClassMetaData
