/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
/**
 * Create a query for a class table for the given class that returns all rows
 * for the given ids. The columns are ordered so the id column is first, the
 * proxy uri column is second, and the attribute columns are in order according
 * to the getClassTableColumns() method of ClassMetadata.
 */
public class ClassQuery extends QueryStatement {
	protected ClassMetadata metadata;
	protected int[] ids;
	protected boolean getAllObjects;
	protected boolean includeContainment;
	public ClassQuery(DatabaseType dbType, EClass eClass, ClassMetadata metadata, int[] ids, DBMap map, boolean getAllObjects, boolean includeContainment) {
		super(dbType, map, eClass);
		this.metadata = metadata;
		this.ids = ids;
		this.getAllObjects = getAllObjects;
		this.includeContainment = includeContainment;
	}

	protected void createSelectFrom() {
		statement.append("SELECT ");
		statement.append(addQuotes(getIdColumnName()));
		statement.append(", ");
		statement.append(addQuotes(DBMapBuilder.IS_PROXY_COLUMN_NAME));
		List attributeColumns = metadata.getClassTableColumns();

		for (int i = 0, l = attributeColumns.size(); i < l; i++) {
			Column column = (Column) attributeColumns.get(i);
			statement.append(", ");
			statement.append(addQuotes(column.getName()));
		}

		if (getAllObjects) {
			List columns = metadata.getReferenceClassTableColumns();

			for (int i = 0, l = columns.size(); i < l; i++) {
				Column column = (Column) columns.get(i);
				EReference reference = dbMap.getReference(column);

				if (reference != null && reference.isContainment() && !includeContainment) {
					if (reference.isMany())
						++i;

					continue;
				}

				statement.append(", ");
				statement.append(addQuotes(column.getName()));
			}
		}

		statement.append(" FROM ");
		statement.append(addQuotes(table.getName()));
	}

	protected String getIdColumnName() {
		return metadata.getPrimaryKey().getName();
	}

	protected void createWhereClause(int begin) {
		statement.append(" WHERE ");
		statement.append(addQuotes(getIdColumnName()));
		statement.append(" IN (");

		for (int i = begin; i < ids.length; i++) {
			String id = "" + ids[i];

			if (statement.length() + id.length() > dbType.getMaximumSQLLength()) {
				initial = i;
				break;
			}

			if (i != begin)
				statement.append(", ");

			statement.append(ids[i]);
		}

		statement.append(")");
	}
} // ClassQuery
