/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.DBResource;
import org.eclipse.hyades.resources.database.internal.InternalDatabase;
import org.eclipse.hyades.resources.database.internal.PagingList;
/**
 * This adapter listens for changes to objects in a DBResource and updates the
 * database accordingly.
 */
public class DBResourceContentAdapter extends EContentAdapter implements Adapter {
	protected DBResource resource;
	protected boolean isLoading;
	protected Map classesToMetadata;
	protected Collection notLoadedClasses;

	public void setTarget(Notifier target) {
		if (resource == null && target instanceof DBResource) {
			resource = (DBResource) target;
		} else if (target instanceof EObject) {
			ignorePagingListsContent((EObject) target);
			return;
		}

		super.setTarget(target);
	}

	/**
	 * Get the not loaded classes from the default load options of the
	 * DBResource.
	 */
	protected void setNotLoadedClasses() {
		Map options = resource.getDefaultLoadOptions();
		notLoadedClasses = (Collection) options.get(DBResource.OPTION_NOT_LOADED_CLASSES);

		if (notLoadedClasses == null)
			notLoadedClasses = Collections.EMPTY_LIST;
	}

	/**
	 * Add this adapter to each object in the object contents, ignoring the
	 * contents of paging lists.
	 */
	protected void ignorePagingListsContent(EObject object) {
		List contents = getContents(object);

		for (int i = 0, l = contents.size(); i < l; i++) {
			Notifier notifier = (Notifier) contents.get(i);
			notifier.eAdapters().add(this);
		}
	}

	protected List getContents(EObject object) {
		List contents = new ArrayList();
		List containments = object.eClass().getEAllContainments();

		for (int i = 0, l = containments.size(); i < l; i++) {
			EReference containment = (EReference) containments.get(i);

			if (!object.eIsSet(containment))
				continue;

			if (!containment.isMany())
				contents.add(object.eGet(containment, false));
			else if (!inNotLoadedClasses(containment.getEReferenceType())) {
				contents.addAll((List) object.eGet(containment, false));
			}
		}

		return contents;
	}

	protected boolean inNotLoadedClasses(EClass eClass) {
		if (notLoadedClasses == null)
			setNotLoadedClasses();

		if (notLoadedClasses.isEmpty())
			return false;

		if (notLoadedClasses.contains(eClass))
			return true;

		ClassMetadata metadata = getMetadata(eClass);
		EClass[] superclasses = metadata.getAllSuperclasses();

		for (int i = 0; i < superclasses.length; i++)
			if (notLoadedClasses.contains(superclasses[i]))
				return true;

		return false;
	}

	protected ClassMetadata getMetadata(EClass eClass) {
		if (classesToMetadata == null)
			classesToMetadata = new HashMap();

		ClassMetadata metadata = (ClassMetadata) classesToMetadata.get(eClass);

		if (metadata == null) {
			metadata = new ClassMetadata(eClass, null, new RDBHelper());
			classesToMetadata.put(eClass, metadata);
		}

		return metadata;
	}

	public void notifyChanged(Notification notification) {
		super.notifyChanged(notification);

		if (isLoading)
			return;

		int type = notification.getEventType();

		switch (type) {
			case Notification.ADD :
				processAdd(notification);
				break;
			case Notification.SET :
				processSet(notification);
				break;
			default :
		// Do nothing
		}
	}

	protected void processAdd(Notification notification) {
		if (notification.getFeature() == null)
			return;

		EObject owner = (EObject) notification.getNotifier();
		EStructuralFeature feature = (EStructuralFeature) notification.getFeature();
		if(feature.isTransient())
			return;
		if (owner.eGet(feature) instanceof PagingList)
			updateForPagingListAdd(notification, owner, (EReference) feature);
		else if (feature.isMany() && resource.isInDatabase()) {
			updateDatabase(notification, owner);
		}
	}

	protected void updateForPagingListAdd(Notification notification, EObject owner, EReference reference) {
		if (!resource.isInDatabase()) {
			saveResource();

			if (reference.isContainment())
				return;
		}

		updateDatabase(notification, owner);
	}

	protected void updateDatabase(Notification notification, EObject owner) {
		try {
			synchronized (resource.getDatabase()) {
				((InternalDatabase) resource.getDatabase()).update(notification);
			}
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
	}

	protected void saveResource() {
		try {
			synchronized (resource.getDatabase()) {
				resource.save(null);
			}
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
	}

	protected void processSet(Notification notification) {
		if (notification.getFeature() == null)
			return;

		if (!resource.isInDatabase())
			return;

		EObject owner = (EObject) notification.getNotifier();
		EStructuralFeature feature = (EStructuralFeature) notification.getFeature();
		if(feature.isTransient())
			return;

		updateDatabase(notification, owner);
	}

	public void setIsLoading(boolean value) {
		isLoading = value;
	}
} // DBResourceContentAdapter
