/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

//==============================================================================
// BCIEngProbe.cpp
//------------------------------------------------------------------------------
// BCI Engine for the probe kit
//==============================================================================
#pragma warning(disable:4786)

#include "CommonDef.h"
#include "ModuleJ.h"				// Instrumentation module for Java
#include "JVMInsSet.h"				// JVM Instruction set
#include "ExtRefJ_StatMethod.h"		// Java external ref. implemented as static
#include "JMemStream.h"				// Java memory stream
#include "BCIEngProbe.h"			// BCI Engine for Probe Kit

#include <iostream>
#include <fstream>


// Strings for the argument names
// These have to be in the same order as the ARG_BITS enum 

static const char *argNames[] = {
	"returnedObject",
	"exceptionObject",
	"className",
	"methodName",
	"methodSig",
	"thisObject",
	"args",
	"isFinally",
	NULL
};

unsigned int
CProbeRef::ComputeArgBits(CSTR i_szArgList)
{
	unsigned int result = 0;
	CSTR p = i_szArgList;
	
	while (*p) {
		int i = 0;
		unsigned int bit = 1;
		bool found = false;
		for ( ; argNames[i] != NULL; i++, bit <<= 1) {
			if (strncmp(p, argNames[i], strlen(argNames[i])) == 0) {
				result |= bit;
				p += strlen(argNames[i]);
				CBCIEngProbeException::Assert((!*p || *p == ','), "Bad argument name list");
				if (*p) p++;
				found = true;
				break;
			}
		}
		CBCIEngProbeException::Assert(found, "Unknown argument name in argument list");
	}
	return result;
}

//------------------------------------------------------------------------------
//
// CProbeRef::Instrument
//
// This is the base class functionality for all CProbeRef types.
// They all do the same thing: set the context fields that are specific
// to the ProbeRef, call PushArguments, and insert the invokestatic instruction
// that will call this ProbeRef's fragment's function.
//

void
CProbeRef::Instrument(CProbeInsertionContext& ctxt)
{
	ctxt.probeRefType = m_enuType;
	ctxt.argBits = m_argBits;
	PushArguments(ctxt);
	CInstruction* invokeStaticInstruction = m_pextref->CreateInstruction();
	ctxt.Insert(invokeStaticInstruction);
}

// emit_ldc_for_string: helper function used by PushArguments
static void emit_ldc_for_string(CProbeInsertionContext& ctxt,
								CCPUtf8Info& utf8Name)
{
	unsigned const_pool_temp = ctxt.pConstPool->Find(&utf8Name);
	// CCPStringInfo doesn't have a constructor that takes an ordinary string.
	// If it did none of the above would be necessary - we could pass i_pmtd->getName()
	// or whatever that function is called.
	// TODO: optimize, don't add strings to the constant pool if they're already there.
	CCPStringInfo* pccpstrName =  new CCPStringInfo(const_pool_temp);
	const_pool_temp = ctxt.pConstPool->Add(pccpstrName);
	ctxt.Insert(CInsSetJ::Create_ldc(const_pool_temp));
}

static unsigned get_class_ref(CProbeInsertionContext& ctxt, const char* classname)
{
	// TODO: optimize, don't add class refs if they're already there.
	unsigned class_utf8num = ctxt.pConstPool->Add(new CCPUtf8Info(classname));
	return ctxt.pConstPool->Add(new CCPClassInfo(class_utf8num));
}

static unsigned get_name_and_type_ref(CProbeInsertionContext& ctxt, const char* methodName, const char* signature)
{
	// TODO: optimize, don't add method name refs if they're already there
	unsigned method_ref = ctxt.pConstPool->Add(new CCPUtf8Info(methodName));
	unsigned type_ref = ctxt.pConstPool->Add(new CCPUtf8Info(signature));
	return ctxt.pConstPool->Add(new CCPNameAndTypeInfo(method_ref, type_ref));
}

static unsigned get_method_ref(CProbeInsertionContext& ctxt, 
							   unsigned classRef, 
							   const char* methodName, 
							   const char* signature)
{
	// TODO: optimize, don't add if it's already there
	unsigned nameAndTypeRef = get_name_and_type_ref(ctxt, methodName, signature);
	return ctxt.pConstPool->Add(new CCPMethodrefInfo(classRef, nameAndTypeRef));
}

//-----------------------------------------------------------------------------------------
// Scan a Java function signature and return the next argument type as a char.
// Also advance the pointer argument to the next char after the argument type string.
// Returns the null char (zero) and does not advance at the end of the argument type list.
//
// If the pFulltype parameter is not null, it means the caller wants to get the full
// text string of the next type, if it's not a primitive.
// If the next argument *is* a primitive or there isn't one, *pFulltype gets NULL.
// Otherwise, it gets a new char[] big enough to hold the type string.
//
// The fulltype string it gets doesn't include the 'L' prefix or the ';' suffix,
// just the type name. (The string for array types does include the open bracket
// and the L and trailing semicolon. This is what Java wants in the constant pool.)
//
// The caller must eventually delete[] the array or it will leak.

static char get_next_java_arg_type(const char** pSig, char** pFulltype = NULL)
{
	const char* pScan = *pSig;
	char result = *pScan;

	if (result == '(') {
		// Skip the open paren
		pScan++;
		result = *pScan;
		// and fall through to process the new result char
	}

	const char* start = pScan;

	if (result == ')') {
		// end of args - stop returning types, leave *pSig on the close paren
		*pSig = pScan;
		if (pFulltype) *pFulltype = NULL;
		return '\0';
	}

	// At this point, "result" is the first char of the type.
	// It's either a primitive or an L or an open bracket.
	// Note that *pScan is the same as "result" at this point.

	// Scan past any open brackets
	while (*pScan == '[') pScan++;

	// Scan past the type name (if any) to the semicolon
	if (*pScan == 'L') {
		while (*pScan != ';') pScan++;
	}

	// When we fall out, pScan points to one of three things:
	// 1. The primitive type char which is the whole argument type.
	// 2. The primitive type char following one or more open brackets.
	// 3. The semicolon following an L type name (possibly after some open brackets).

	// Advance past the type char or semicolon
	pScan++;

	// if you wanted the full type, you get it now
	if (pFulltype) {
		int count = pScan - start;
		if (count != 1) {
			// type was not a single letter - not a primitive
			if (result == 'L') {
				// discount the leading 'L' and trailing ';'
				count -= 2;
				++start;
			}
			char* fullstr = new char[count+1];
			strncpy(fullstr, start, count);
			fullstr[count] = '\0';
			*pFulltype = fullstr;
		}
		else {
			// primitive type - fill in NULL
			*pFulltype = NULL;
		}
	}

	// Fill in *pSig with the new scan position and return
	*pSig = pScan;
	return result;
}

// Return the Java return type derived from the signature.
static char
get_java_return_type(const char* sig)
{
	while (*sig && *sig != ')') sig++;
	CBCIEngProbeException::Assert(*sig != 0, "Bad signature passed to get_java_return_type");
	++sig;
	return *sig;
}

// Return the number of arguments in the signature.
static int
count_java_args(const char* sig)
{
	int argcount = 0;
	while (get_next_java_arg_type(&sig) != '\0') {
		argcount++;
	}
	return argcount;
}

static inline bool
java_type_is_category_2(char type)
{
	return (type == 'D' || type == 'J');
}

static inline bool
java_type_is_primitive(char type)
{
	return (type != 'L' && type != '[');
}

//------------------------------------------------------------------------
//
// BOXING AND UNBOXING
//
// The box_data_table array holds information about how to "box"
// a Java value.
//	typechar is the char for the Java type this row describes.
//	typesize is the number of stack positions that value uses, aka "category" (1 or 2)
//	pLoad_creator is a pointer to a function that takes a local variable slot number and returns the
//		appropriate "load" instruction for the type -- iload for integers, dload for doubles, etc.
//	class_name is the name of the Java box class for this type (like Integer for int)
//	init_sig is the signature of the <init> method for the box class.
//
// The last row of the table starts with an typechar of zero, as a signal that it's
// the end of the table. Really, though, all searches should stop when they hit
// the row they're looking for, and nobody should fall off the end.

static struct box_data {
	char typechar;
	char typesize;
	CInstruction* (*pLoad_creator)(int arg_slot);
	const char* class_name;
	const char* init_sig;
	const char* unbox_method;
	const char* unbox_sig;
} box_data_table[] = {
	{ 'B', 1, CInsSetJ::Create_iload, "java/lang/Byte",		 "(B)V", "byteValue", "()B" },
	{ 'C', 1, CInsSetJ::Create_iload, "java/lang/Character", "(C)V", "charValue", "()C" },
	{ 'D', 2, CInsSetJ::Create_dload, "java/lang/Double",	 "(D)V", "doubleValue", "()D" },
	{ 'F', 1, CInsSetJ::Create_fload, "java/lang/Float",	 "(F)V", "floatValue", "()F" },
	{ 'I', 1, CInsSetJ::Create_iload, "java/lang/Integer",	 "(I)V", "intValue", "()I" },
	{ 'J', 2, CInsSetJ::Create_lload, "java/lang/Long",		 "(J)V", "longValue", "()J" },
	{ 'S', 1, CInsSetJ::Create_iload, "java/lang/Short",	 "(S)V", "shortValue", "()S" },
	{ 'Z', 1, CInsSetJ::Create_iload, "java/lang/Boolean",	 "(Z)V", "booleanValue", "()Z" },
	{ '\0', 0, NULL, NULL, NULL }
};

// box_stacked_value
//
// Takes a char telling what type the top of the stack has,
// and a bunch of parameters that let us emit instructions
// and access the constant pool. Returns the number of
// stack slots used to achieve the result.
// 
// This function emits instructions that make a boxed version of 
// the value at the top of the stack. By "boxed" I mean "placed
// inside an object of the appropriate type."
//
// The boolean "keep_original" controls whether the original value
// is preserved at the top of the stack, and we box a COPY of that value.
//
// Initially, the value we want to box is on top of the stack.
// When these instructions are finished, there is a new thing on top 
// of the stack: a reference to an object of the right type 
// (e.g. Integer for int) that holds that value. 
// If keep_original is true, the original value is
// still on the stack, under the reference to the box object.
//
// This is used for converting a primitive return value into
// an object for passing to the exit fragment. The original return
// value starts on the top of the stack, and keep_original is true:
// when the exit fragment is finished, the original return value 
// is STILL on top of the stack.
//
// This is also used from CallsiteStoreArgs, which moves values
// from the stack into an Object array. In that case keep_original
// is false, and the values are actually consumed off the stack.
//
// The instructions you emit are different for Category 1 and
// Category 2 values. (Long and Double are Category 2 values.)
//
// For Category 1 values emit this:
//	dup (conditional on keep_original)
//	new class_name
//	dup_x1
//	swap
//	invokespecial <init> (I)V (or whatever signature)
//
// For Category 2 values (double and long) emit this:
//	dup2 (conditional on keep_original)
//	new class_name
//	dup_x2
//	dup_x2
//	pop
//	invokespecial <init> (D)V / (J)V
//
// Note: the combination "dup_x2 / pop" is used to achieve "swap"
// where the top value is category 1 and the next value is category 2.

static void
box_stacked_value(CProbeInsertionContext& ctxt,
				  char type,
				  bool keep_on_stack)
{
	const box_data* pBoxData = NULL;

	for (pBoxData = box_data_table; pBoxData->typechar != '\0'; pBoxData++) {
		if (type == pBoxData->typechar)
			break;
	}
	// could assert that pBoxData isn't null, but there shouldn't be any doubt

	unsigned int class_ref = get_class_ref(ctxt, pBoxData->class_name);
	unsigned int meth_ref = get_method_ref(ctxt, class_ref, "<init>", pBoxData->init_sig);

	if (pBoxData->typesize == 2) {
		// Category 2 value on the stack (double or long)
		// See top of function for the instructions we implement and why

		// Initial stack is "v1 v2 -- (that is, two parts of a cat2 value)
		if (keep_on_stack) {
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup2));
		}

		// now stack (not counting dup'd value if any) is v1 v2 --
		ctxt.Insert(CInsSetJ::Create_new(class_ref));
		// now stack is v1 v2 obj --
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup_x2));
		// now stack is obj v1 v2 obj --
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup_x2));
		// now stack is obj obj v1 v2 obj --
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_pop));
		// now stack is obj obj v1 v2 --
		ctxt.Insert(CInsSetJ::Create_invokespecial(meth_ref));
		// now stack is obj --
	}
	else {
		// Category 1 value on the stack
		// Initial stack is "v --" (that is, the single category 1 value)
		if (keep_on_stack) {
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup));
		}
		// now stack (not counting dup'd value if any) is v --
		ctxt.Insert(CInsSetJ::Create_new(class_ref));
		// now stack is v obj --
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup_x1));
		// now stack is obj v obj --
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_swap));
		// now stack is obj obj v --
		ctxt.Insert(CInsSetJ::Create_invokespecial(meth_ref));
		// now stack is obj --
	}
}

// box_local_value
//
// This function does the same as box_stacked_value,
// except that the value starts out as a local variable.
// Leaves the boxed object reference on the stack.
//
// Returns a bool: true if the local was a "category 2"
// (and therefore consumed two slots in the local variable table).

static void
box_local_value(CProbeInsertionContext& ctxt, char local_type, int local_num)
{
	// Look up the details in box_data_table.
	const box_data* pBoxData;
	for (pBoxData = box_data_table; pBoxData->typechar != '\0'; pBoxData++) {
		if (local_type == pBoxData->typechar)
			break;
	}
	// Could assert that we found it, but failure Should Never Happen

	unsigned class_ref = get_class_ref(ctxt, pBoxData->class_name);
	unsigned meth_ref = get_method_ref(ctxt, class_ref, "<init>", pBoxData->init_sig);
	ctxt.Insert(CInsSetJ::Create_new(class_ref));
	ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup));
	ctxt.Insert((*(pBoxData->pLoad_creator))(local_num));
	ctxt.Insert(CInsSetJ::Create_invokespecial(meth_ref));
}

// unbox_stacked_value
//
// Takes a char telling what type the top of the stack should be,
// and a bunch of parameters that let us emit instructions
// and access the constant pool. Returns the number of
// stack slots used to achieve the result.
// 
// This function emits instructions that extract the value from
// a boxed version of the value at the top of the stack. 
// By "boxed" I mean "placed inside an object of the appropriate type."
//
// The top value on the stack expected to be an Object;
// the first thing this function emits is a checkcast instruction
// to convert it to the proper type, like Integer for int.
//
// Then we emit an "invokevirtual" instruction to get the
// value out of the box. The function being called and its 
// signature are always different.
// For an Integer, for example, we emit this:
//	checkcast java/lang/Integer
//	invokevirtual intValue(Ljava/lang/Integer)I
//
// But for a Long we emit this:
//	checkcast java/lang/Long
//	invokevirtual longValue(Ljava/lang/Long)J

static int
unbox_stacked_value(CProbeInsertionContext& ctxt, char type)
{
	const box_data* pBoxData = NULL;

	for (pBoxData = box_data_table; pBoxData->typechar != '\0'; pBoxData++) {
		if (type == pBoxData->typechar)
			break;
	}
	// could assert that pBoxData isn't null, but there shouldn't be any doubt

	unsigned int class_ref = get_class_ref(ctxt, pBoxData->class_name);
	unsigned int meth_ref = get_method_ref(ctxt, class_ref, pBoxData->unbox_method, pBoxData->unbox_sig);

	ctxt.Insert(CInsSetJ::Create_checkcast(class_ref));
	ctxt.Insert(CInsSetJ::Create_invokevirtual(meth_ref));

	// Return the added stack depth: zero for Category 1, one for Category 2.
	return (pBoxData->typesize == 2 ? 1 : 0);
}

//------------------------------------------------------------------------
//
// Notes on doing insertion to pass the argument list
//
// If any fragment that applies to the current method wants the
// argument list as a parameter, we do insertion on entry to capture
// the arguments in an array of Objects. We store that array in
// a new local variable, and pass it to those fragments that want it.
//
// (If the argument list is only needed once, this is slightly suboptimal:
// we could create it in-place without allocating a local. But the current
// approach is more optimal in other cases, and simpler to maintain.)
//
// The inserted instruction sequence starts with creating the object array:
//		iconst (argcount)
//		anewarray ("java/lang/Object")
//
// Now the object array reference is at the top of the stack.
// It will stay there for the whole remainder of the sequence,
// until it's finally stored in a new local variable.
//
// Then, for a reference-type argument, we arrange the stack
// for the aastore instruction, which wants the array object, the
// slot number, and the new value on the stack in that order.
// The slot number is "arrayslot" and the new value is obtained
// from the argument/local array using aload from "argslot."
// The insertion looks like this:
//		dup
//		iconst (arrayslot)
//		aload (argslot)
//		aastore
//
// For a primitive-type argument, we have to "box" the primitive first.
// That is, we have to wrap an "int" into an Integer object on the heap.
// So we dup the array object, push the slot number (in preparation
// for the ultimate aastore instruction), and then push the
// primitive value and box it, leaving the box object on the stack
// for the aastore instruction to see. It looks like this 
// (this example is for an integer):
//		dup
//		iconst (arrayslot)
//		// Next 4 insns emitted by box_local_value(argslot, argtype)
//		new (java/lang/Integer)
//		dup
//		iload (argslot)
//		invokespecial java/lang/Integer <init> (I)V
//		// end of box_local_value()
//		aastore
//

static void
emit_args_list(CProbeInsertionContext& ctxt)
{
	bool any_boxing = false;
	bool any_long_boxing = false;

	int argcount = count_java_args(ctxt.methodSignature);

	// Emit instructions to create a new array of type Object[argcount]
	unsigned classRef = get_class_ref(ctxt, "java/lang/Object");
	ctxt.Insert(CInsSetJ::Create_push_constant(argcount));
	ctxt.Insert(CInsSetJ::Create_anewarray(classRef));

	// Re-scan the signature
	const char* p = ctxt.methodSignature;

	// The argslot variable is the aload parameter for the
	// current argument. It starts at zero for static methods,
	// or one for instance methods (because zero is "this").
	// Advances by "argsize" each time through the loop.
	// Argsize is usually 1, but it is 2 for doubles and longs.
	int argslot = (ctxt.hasThis ? 1 : 0);
	int argsize;

	// The argtype variable holds the type of the current argument.
	char argtype;

	// The arrayslot variable holds the slot number
	// into which each argument will be placed. Increments by one each time.
	int arrayslot = 0;

	for (argtype = get_next_java_arg_type(&p);
		 argtype != '\0'; 
		 arrayslot++, argslot += argsize, argtype = get_next_java_arg_type(&p)) 
	{
		// Dup the array object and push the index to store into.
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup));
		ctxt.Insert(CInsSetJ::Create_push_constant(arrayslot));

		// Now push the value to store.
		// In the 'a' case (object or array), use aload.
		// Otherwise use box_local_value to create a temporary.
		if (java_type_is_primitive(argtype)) {
			// Primitive type. Box it.
			box_local_value(ctxt, argtype, argslot);

			// Track knowledge we need for 
			// advancing to the next arg in the table
			any_boxing = true;
			if (java_type_is_category_2(argtype)) {
				// Category 2
				any_long_boxing = true;
				argsize = 2;
			}
			else {
				argsize = 1;
			}
		}
		else {
			// Non-primitive type
			ctxt.Insert(CInsSetJ::Create_aload(argslot));
			argsize = 1;
		}

		// Emit "aastore" now that the right args are on the stack
		ctxt.Insert(CInsSetJ::Create_simple(JVMI_aastore));
	}
}




// This function returns "true" if a "this" argument is accessible to
// the current proberef as applied to the current method. 
// ("Current" means "the one described in the CProbeInsertionContext.")
//
// Just because the context says the method hasThis, that doesn't
// mean the probe can access "this."
//
// In addition, an exception-exit handler for a constructor
// can't access "this." The reason: we put a "finally"
// wrapper around the whole original function, and that means
// we're wrapping some code where "this" is uninitialized.
// TODO: perform liveness analysis to identify when "this"
// becomes initialized, and create two exception-exit wrappers:
// one for before that time (passing null as this) and
// one for after that time (passing the real this value).
//
// Result: entry probes, beforecall probes, and exception exit probes
// can't access "this" in constructors.
//

static bool
proberef_can_access_this(CProbeInsertionContext& ctxt)
{
	if (ctxt.hasThis)
	{
		if ((ctxt.probeRefType == CProbeRef::PROBE_ONENTRY ||
			 ctxt.probeRefType == CProbeRef::PROBE_BEFORECALL ||
			 ctxt.isExceptionExit) &&
			strcmp(ctxt.methodName, "<init>") == 0) 
		{
			return false;
		}
		else
		{
			return true;
		}
	}
	else
	{
		return false;
	}
}

// PushArguments
// This function emits instructions that push the arguments indicated by m_argBits.
// Some auxiliary parameters are used to pass extra info, like isFinally.
//
// The string-type parameters are pushed using the helper function emit_ldc_for_string,
// which puts the string in the constant pool and emits "ldc" for that string.
//
// Returns the added stack depth - the maximum depth of stack used to
// compute and pass the arguments to the probe function.
//
// ---- HACK ALERT ---- HACK ALERT ---- HACK ALERT ---
//
// The BCI for the returnedObject and exceptionObject parameter types is
// tricky. First of all, it relies on those being the first two
// parameters in the canonical ordering, so the BCI for passing
// those parameters occurs first, before the BCI for passing
// other parameters. Second, they are entangled: the BCI for 
// exceptionObject is slightly different if the fragment also 
// asked for returnedObject.
//
// Here's the story:
//
// returnedObject occurs at return instructions. Since returnedObject is
// the first parameter type in the canonical ordering, the value being returned
// is at the top of the stack. We can use a simple "dup" instruction
// to create the parameter to the fragment function. (We then need to
// "box" the value into an object, if it's a primitive type.) This only
// works because returnedObject is the first parameter in the canonical ordering.
//
// exceptionObject is significant at the tops of catch and finally clauses.
// In this case, again, the thrown object is at the top of the stack.
// If the fragment doesn't also want the returnedObject, a simple "dup"
// instruction creates the parameter we need. If the fragment does want
// the returnedObject, then the BCI will already have pushed a "null"
// for that, and now the thrown object is two items down on the stack.
// In that case we emit "swap / dup_x1" to create the stack pattern
// we need, with the exception object, a null, and the exception object
// on the stack in that order.
//
// The insertion done here for returnedObject and exceptionObject
// always leaves the relevant object on the stack - that is, they start
// by duplicating the object.
//

void
CProbeRef::PushArguments(CProbeInsertionContext& ctxt)
{
	// push the arguments, one after another, in the canonical order,
	// based on the information in the Probe Insertion Context.

	for (unsigned int i = 1; i < ARG_BITS_LAST; i <<= 1) {
		if (ctxt.argBits & i) {
			switch (i) {
				case ARG_BITS_CLASSNAME: {
					CCPUtf8Info utf8Name(ctxt.className);
					emit_ldc_for_string(ctxt, utf8Name);
					break;
				}
				case ARG_BITS_METHODNAME: {
					CCPUtf8Info utf8Name(ctxt.methodName);
					emit_ldc_for_string(ctxt, utf8Name);
					break;
				}
				case ARG_BITS_METHODSIG: {
					CCPUtf8Info utf8Name(ctxt.methodSignature);
					emit_ldc_for_string(ctxt, utf8Name);
					break;
				}
				case ARG_BITS_THISOBJ: {
					if (proberef_can_access_this(ctxt)) {
						ctxt.Insert(CInsSetJ::Create_aload(ctxt.localVariableForThis));
					}
					else {
						// No accessible "this" object - pass NULL
						ctxt.Insert(CInsSetJ::Create_simple(JVMI_aconst_null));
					}
					break;
				}
				case ARG_BITS_ARGSLIST: {
					// The caller must already have created the local containing the Object[] array.
					// Function insertion callers did this with emit_args_list;
					// callsite insertion callers did it with another function.
					// assert(ctxt.localVariableForArgs != -1);
					ctxt.Insert(CInsSetJ::Create_aload(ctxt.localVariableForArgs));
					break;
				}
				case ARG_BITS_RETURNEDOBJ: {
					// ---- HACK ALERT ---- HACK ALERT ---- HACK ALERT ---
					// See comments at the top of this function. returnedObject must
					// be the first parameter type in the canonical order.

					CBCIEngProbeException::Assert(
						ctxt.probeRefType == PROBE_ONEXIT || ctxt.probeRefType == PROBE_AFTERCALL,
						"Invalid use of returnedObject: not in exit probe");

					if (ctxt.isExceptionExit) {
						// The function is exiting by exception, not a return instruction.
						// Pass null as the returnedObject data item.
						ctxt.Insert(CInsSetJ::Create_simple(JVMI_aconst_null));
					}
					else {
						char ret_type = get_java_return_type(ctxt.methodSignature);
						if (ret_type == 'V') {
							// function is void: pass null
							ctxt.Insert(CInsSetJ::Create_simple(JVMI_aconst_null));
						}
						else if (java_type_is_primitive(ret_type)) {
							// box_stacked_value takes a bool telling whether to leave
							// the value on the stack or not. In this case, yes.
							box_stacked_value(ctxt, ret_type, true);
						}
						else {
							// returns an object
							ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup));
						}
					}
					break;
				}
				case ARG_BITS_EXOBJ: {
					// ---- HACK ALERT ---- HACK ALERT ---- HACK ALERT ---
					// See comments at the top of this function. exceptionObject must
					// be the second parameter type in the canonical order.
					bool bExitProbe = ctxt.probeRefType == PROBE_ONCATCH 
						           || ctxt.probeRefType == PROBE_ONEXIT
								   || ctxt.probeRefType == PROBE_AFTERCALL;

					CBCIEngProbeException::Assert(bExitProbe,
						"Invalid use of exceptionObject: not in catch or exit probe");

					if ((ctxt.probeRefType == PROBE_ONEXIT || ctxt.probeRefType == PROBE_AFTERCALL) && 
						!ctxt.isExceptionExit) 
					{
						// The insertion point is a "return" instruction, not an exception handler.
						// Pass null.
						ctxt.Insert(CInsSetJ::Create_simple(JVMI_aconst_null));
					}
					else 
					{
						// The insertion point is an exception point: catch, finally, or exception exit.
						// If this fragment also wanted the return value,
						// then the ARG_BITS_RETURNEDOBJ case above has already pushed a null.
						// In that case, we have to emit swap / dup_x1 to get a copy of the exception 
						// object onto the stack in the right place. 
						// If this fragment did not want the return value, just emit "dup."
						if (ctxt.argBits & ARG_BITS_RETURNEDOBJ) {
							CBCIEngProbeException::Assert(ctxt.probeRefType != PROBE_ONCATCH,
								"Internal error: a catch probe wanted the return object?!");
							ctxt.Insert(CInsSetJ::Create_simple(JVMI_swap));
							ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup_x1));
						}
						else {
							ctxt.Insert(CInsSetJ::Create_simple(JVMI_dup));
						}
					}
					break;
				}
				case ARG_BITS_ISFINALLY: {
					CBCIEngProbeException::Assert(ctxt.probeRefType == PROBE_ONCATCH, 
						"Invalid use of isFinally: not in catch probe");

					if (ctxt.isFinally) {
						ctxt.Insert(CInsSetJ::Create_simple(JVMI_iconst_1));
					}
					else {
						ctxt.Insert(CInsSetJ::Create_simple(JVMI_iconst_0));
					}
					break;
				}
			}
		}
	}
}

//==============================================================================
// Helper function that performs a quick consistency check: the number of '1' bits 
// in the argBits should be the same as the number of arguments in the signature.
// Also, the signature should end with )V since no probes return values yet.
// TODO: change this when we implement InvocationObject, where the entry probe
// returns a value.

static void
verify_argbits_and_sig(unsigned i_argBits, CSTR i_szSig)
{
	int bitcount = 0;
	for (unsigned int bit = 1; bit < CProbeRef::ARG_BITS_LAST; bit <<= 1) {
		if (i_argBits & bit) bitcount++;
	}

	const char *p = i_szSig;

	if (*p != '(') {
		throw CBCIEngProbeException("Malformed probe method signature (1)");
	}

	int argcount = count_java_args(p);

	// Scan 'p' to the closing paren
	while (*p && *p != ')') p++;
	if (!*p) {
		throw CBCIEngProbeException("Malformed probe method signature (2)");
	}

	// p points to the closing paren. Check that the return type is void.
	if (get_java_return_type(i_szSig) != 'V') {
		throw CBCIEngProbeException("Malformed probe method signature (3)");
	}

	if (argcount != bitcount) {
		throw CBCIEngProbeException("Probe signature and arg list size mismatch");
	}

	// else return without throwing an exception
}

//==============================================================================
//
// CALLSITE INSERTION HELPER FUNCTIONS
//
// The callsite insertion logic is a slightly different animal compared to
// function insertion. It works by instrumenting (duh) the call site instead of 
// the head and tail of the function you want to find out about.
//
// If there are any callsite probes in the mix, the BCI engine has to scan
// every function that gets loaded to see if it makes any calls any of the functions
// you want to apply callsite probes to. If so, it has to do callsite
// insertion around those calls: entry insertion before the call, and exit
// insertion after.
//
// You'd use callsite insertion if, say, you wanted to trace calls to the
// Jdbc.query() method, but you can't do BCI on the class Jdbc because it
// loads in a way that your BCI engine doesn't have access to. (Many app servers
// provide a "class load hook" to give a chance for BCI, but only "application
// classes" get passed through that hook. System classes do not.)
//
// We haven't decided whether we have to emit
// a call to the probe's exit fragment in the case where the query() call
// returns by exception. TODO: decide this and implement if necessary.
//
// A callsite probe "entry" fragment can access the same data that the other
// kind can, but the byte codes are different. The BCI occurs immediately
// before the "invokevirtual" / static / special / interface instruction.
//
// To build the Object array that holds the argument list, we have to pop the 
// arguments off the stack one by one and store them in it. Then we 
// can pass the Object array to the entry probe. Then, to make the
// original call, we we push the arguments back on the stack.
//
// If the callsite probe "exit" fragment wants the argument list, well, we've
// got it sitting right there. 
//
// If the entry fragment wants the "this" argument, we have to create the
// Object array to store all the arguments from the stack in order to get down
// to the "this" argument. We can probably use "dup" carefully to keep the
// "this" argument on the stack and still pass it to the entry fragment.
// We might even dup it twice before the call, if the exit fragment
// wants it too. If we implement exception-exit, we'll have to store it
// in a local variable.
//
// A note about creating a new local variable: the JVM standard says 
// we don't need to add an entry to the local variable table - that's
// optional and used for debuggers anyway. But the lifetime of the value in
// the local covers the whole function, so if we store an object reference
// into one, we should null it out when we're finished with it.
//
// TODO: figure out a way to pass "this" to exit insertion when the
// function being called is a constructor. Today we can't do it because
// we can't assign "this" into a local variable until it's been initialized.
// The way to do it is to pop the arguments off the stack, then dup the "this"
// value on the stack, then put the args back on and call the constructor.
// When the constructor returns, the "this" object on the stack will
// have been been initialized and we can store it into a local for later
// passing to the exit insertion.

// CallsiteStoreArgs
//
// Create an array of Objects and put it into a local.
// Assign values into it by popping them off the stack. The types of the
// values on the stack are given by a signature string.
// The LAST argument in the signature string is the TOP item on the stack.
// 

static void
CallsiteStoreArgs(CProbeInsertionContext& ctxt)
{
	// Count arguments from the signature
	int argcount = count_java_args(ctxt.methodSignature);

	// Create the object array and put it in a local
	unsigned classRef = get_class_ref(ctxt, "java/lang/Object");
	ctxt.Insert(CInsSetJ::Create_push_constant(argcount));
	ctxt.Insert(CInsSetJ::Create_anewarray(classRef));
	ctxt.Insert(CInsSetJ::Create_astore(ctxt.localVariableForArgs));

	if (argcount != 0) {
		// Walk the arguments list backwards,
		// and emit code to copy the values from the stack
		// into the array.

		// First, copy the arg type chars into a local array
		const char* p = ctxt.methodSignature;
		char* arg_type_chars = (char*)alloca(argcount);
		char* q = arg_type_chars;
		char next;

		while ((next = get_next_java_arg_type(&p)) != '\0') {
			*q++ = next;
		}

		// Walk the arg type char array backwards.
		// Those stack elements which are primitive types must be
		// boxed first, by calling box_stacked_value.
		//
		// Since the value we want to store into the array is
		// already on top of the stack, we have to push the
		// array reference and index behind it, using a swap
		// instruction after pushing each one. 
		//
		// After boxing, the type at the top of the stack
		// is of type "reference." Then we emit this sequence:
		//	aload i_localForCallsiteArgs
		//	swap
		//	iconst arg_number
		//	swap
		//	aastore

		for (int i = argcount-1; i >= 0; i--) {
			// If necessary, box the value at the top of the stack
			if (java_type_is_primitive(arg_type_chars[i])) {
				box_stacked_value(ctxt, arg_type_chars[i], false);
			}
			// Store the value at the top of the stack into obj_array[i]
			ctxt.Insert(CInsSetJ::Create_aload(ctxt.localVariableForArgs));
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_swap));
			ctxt.Insert(CInsSetJ::Create_push_constant(i));
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_swap));
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_aastore));
		}
	}
	else {
		// Arg count is zero.
		// We have created a valid, zero-length array of Object
		// and stored a reference to it in the appropriate local variable.
		int dummy = 0; // for coverage purposes
	}

	// Finished. All arguments (if any) are stored into the Object array.
}

// CallsiteStoreThis
//
// Store the "this" argument for the called method into the local variable
// slot given as i_localForCallsiteThis. If the called method is static,
// or the called method is a constructor, then function should not be called.
// See TODO elsewhere about storing "this" *after* the call to a constructor.
//

static void
CallsiteStoreThis(CProbeInsertionContext& ctxt)
{
	ctxt.Insert(CInsSetJ::Create_astore(ctxt.localVariableForThis));
}

// CallsiteReloadArgsAndThis
//
// Restore the "this" pointer and arguments from the
// given local variables.
//
// Primitive values which were boxed need to be unboxed.
//
// If i_localForCallsiteThis is -1, it means there is no "this" parameter.
// If i_localForCallsiteArgs is -1, it means there are no args to restore.
//
// Getting each arg out of the object array looks like this:
//
//	aload (local)
//	push slot_number
//	aaload			(get object ref out of array)
//	checkcast		(cast it to its proper type)
//
// If the value was originally a primitive, the checkcast leaves it
// as the box type -- Integer for int. THen it needs to be unboxed;
// see unbox_stacked_value for how that's done.
//

static void
CallsiteReloadArgsAndThis(CProbeInsertionContext& ctxt)
{
	if (ctxt.localVariableForThis != -1) {
		ctxt.Insert(CInsSetJ::Create_aload(ctxt.localVariableForThis));
	}

	if (ctxt.localVariableForArgs != -1) {
		const char* p = ctxt.methodSignature;
		char* fulltype;
		char next;
		int i = 0;
		while ((next = get_next_java_arg_type(&p, &fulltype)) != '\0') {
			ctxt.Insert(CInsSetJ::Create_aload(ctxt.localVariableForArgs));
			ctxt.Insert(CInsSetJ::Create_push_constant(i));
			ctxt.Insert(CInsSetJ::Create_simple(JVMI_aaload));
			
			if (java_type_is_primitive(next)) {
				unbox_stacked_value(ctxt, next);
			}
			else {
				// Emit checkcast to convert to the proper type
				unsigned int cpInd = get_class_ref(ctxt, fulltype);
				ctxt.Insert(CInsSetJ::Create_checkcast(cpInd));
				delete[] fulltype;
			}
			++i;
		}
	}

	// We do not adjust the function's max stack depth because this
	// function only runs after CallsiteStoreArgs, and we don't add any
	// net stack usage over that function.
	// TODO: remove this comment when we compute stack usage for real.
}

//==============================================================================
CProbeRef::CProbeRef(CProbe* parent, ProbeRef_enu i_enuType, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
{
	m_enuType = i_enuType;
	m_pextref = new CExtRefJ_StatMethod(i_szClass, i_szMethod, i_szSig);
	m_argBits = ComputeArgBits(i_szArgList);
	m_parent = parent;
	verify_argbits_and_sig(m_argBits, i_szSig);	// throws an exception on error
}

CProbeRef::~CProbeRef()
{
	delete m_pextref;
}

//==============================================================================

CProbeRefOnEntry::CProbeRefOnEntry(CProbe* parent, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
:CProbeRef(parent, CProbeRef::PROBE_ONENTRY, i_szClass, i_szMethod, i_szSig, i_szArgList)
{
}

//==============================================================================

CProbeRefOnExit::CProbeRefOnExit(CProbe* parent, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
:CProbeRef(parent, CProbeRef::PROBE_ONEXIT, i_szClass, i_szMethod, i_szSig, i_szArgList)
{
}

//==============================================================================

CProbeRefOnCatch::CProbeRefOnCatch(CProbe* parent, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
:CProbeRef(parent, CProbeRef::PROBE_ONCATCH, i_szClass, i_szMethod, i_szSig, i_szArgList)
{
}

//==============================================================================

CProbeRefBeforeCall::CProbeRefBeforeCall(CProbe* parent, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
:CProbeRef(parent, CProbeRef::PROBE_BEFORECALL, i_szClass, i_szMethod, i_szSig, i_szArgList)
{
}

//==============================================================================

CProbeRefAfterCall::CProbeRefAfterCall(CProbe* parent, CSTR i_szClass, CSTR i_szMethod, CSTR i_szSig, CSTR i_szArgList)
:CProbeRef(parent, CProbeRef::PROBE_AFTERCALL, i_szClass, i_szMethod, i_szSig, i_szArgList)
{
}

//==============================================================================
// Implementations for functions of class CFilterRule

// Constructor
CFilterRule::CFilterRule(CSTR i_szPackageName, 
					  CSTR i_szClassName, 
					  CSTR i_szMethodName, 
					  CSTR i_szMethodSig, 
					  CFilterRule::action_t i_action)
{
	m_strPackageName = strdup(i_szPackageName);
	m_strClassName = strdup(i_szClassName);
	m_strMethodName = strdup(i_szMethodName);
	m_strMethodSig = strdup(i_szMethodSig);
	m_action = i_action;
}

// Copy constructor
CFilterRule::CFilterRule(const CFilterRule& other)
{
	m_strPackageName = strdup(other.m_strPackageName);
	m_strClassName = strdup(other.m_strClassName);
	m_strMethodName = strdup(other.m_strMethodName);
	m_strMethodSig = strdup(other.m_strMethodSig);
	m_action = other.m_action;
}

// Assignment operator
CFilterRule&
CFilterRule::operator=(const CFilterRule& other)
{
	m_strPackageName = strdup(other.m_strPackageName);
	m_strClassName = strdup(other.m_strClassName);
	m_strMethodName = strdup(other.m_strMethodName);
	m_strMethodSig = strdup(other.m_strMethodSig);
	m_action = other.m_action;
	return *this;
}

// Destructor
CFilterRule::~CFilterRule()
{
	free(m_strPackageName);
	free(m_strClassName);
	free(m_strMethodName);
	free(m_strMethodSig);
}

//------------------------------------------------------------------------------
bool 
CFilterRule::WildcardStringMatch(const char* pattern, const char* candidate)
{
	// Special case: don't-care on the candidate side
	if (strcmp(candidate, "*") == 0) {
		return true;
	}
	else if (strcmp(pattern, "*") == 0) {
		// star pattern matches anything
		return true;
	}
	else if (strchr(pattern, '*') == NULL) {
		// no wildcards, just compare strings
		return (strcmp(pattern, candidate) == 0);
	}

	// Else do the wild(card) thing. We know pattern has a star in it.
	//
	// We divide the pattern into substrings delimited by stars.
	// For each substring, we scan ahead in the candidate 
	// for the first match. If we don't find one,
	// the whole wildcard match fails. If we do find one, 
	// we "consume" the candidate up to the end of the match.
	// The next iteration will start matching at that point.
	//
	// The first iteration is different: if the pattern doesn't start 
	// with a star, the initial substring of the pattern must appear 
	// at the head of the candidate.
	//
	// The last iteration is different, too. If the pattern does not
	// end with a star, we explicitly check the END of the candidate 
	// against the last non-star substring. This way we get
	// the right behavior even if the candidate has more than one
	// instance of that last substring. EXAMPLE: the pattern is "a*c" and
	// the candidate is "abcabc" ... if the pattern 'c' matched the first
	// 'c' in the candidate, we'd think we fell off the end of the pattern
	// before consuming the whole candidate, and fail. But this candidate
	// DOES match the pattern, so we have to treat the last non-star
	// substring specially if the pattern doesn't end with a star.

	const char* pattern_pos;
	const char* candidate_pos;
	const char* star_pos = strchr(pattern, '*');

	// With no initial star, the heads of the strings have to match.
	if (star_pos != pattern) {
		// Star is not at start of pattern.
		// Start of candidate must match substring
		if (strncmp(candidate, pattern, (star_pos - pattern)) != 0) {
			// no leading star, heads don't match, FAIL.
			return false;
		}
		else {
			// Start the loop below at the end of this match in candidate.
			candidate_pos = candidate + (star_pos - pattern);
		}
	}
	else {
		// Pattern has an initial star. Start the loop below at the start of candidate.
		candidate_pos = candidate;
	}

	// Set pattern_pos past the star (either an initial star or after an initial non-star substring)
	pattern_pos = star_pos + 1;

	// Now we start a loop. 
	// At the top of each iteration, pattern_pos is the start of a non-star substring
	// in the pattern, following a star.
	// Also, at the top of each iteration, candidate_pos is the first position of 
	// the candidate that we haven't consumed yet. 
	//
	// See comments above for the algorithm, including notes on the last non-star
	// substring when the pattern doesn't end with a star.

	while (*pattern_pos && *candidate_pos) {
		// Find next star in pattern
		star_pos = strchr(pattern_pos, '*');
		if (star_pos == NULL) {
			// No remaining stars, compare tail of pattern against tail of candidate
			int tail_len = strlen(pattern_pos);
			if (strlen(candidate_pos) < tail_len) {
				// remainder of candidate isn't long enough, so there's no match
				return false;
			}
			else {
				const char* candidate_tail = candidate + strlen(candidate) - tail_len;
				if (strcmp(pattern_pos, candidate_tail) == 0) {
					// no remaining stars, tails match, MATCH
					return true;
				}
				else {
					// no remaining stars, tails don't match. NO MATCH
					return false;
				}
			}
		}

		// OK, there is a star out there at star_pos.
		// Find the non-star substring of pattern in candidate
		int sub_len = star_pos - pattern_pos;
		const char* match_pos;
		{
			// Look for the substring of pattern_pos (sub_len long) in candidate_pos..
			// Set match_pos to the start of the match in the candidate.
			match_pos = candidate_pos;
			int candidate_remainder_len = strlen(candidate_pos);
			if (sub_len > candidate_remainder_len) {
				// Not enough candidate left to match
				return false;
			}
			const char* last_match_start = candidate_pos + (candidate_remainder_len - sub_len);
			while (match_pos <= last_match_start && strncmp(match_pos, pattern_pos, sub_len) != 0) {
				++match_pos;
			}
			if (match_pos > last_match_start) {
				// Fell out of the loop with NO MATCH.
				return false;
			}
		}

		// Found this non-star substring in candidate.
		// Advance candidate_pos past the match, 
		// and advance pattern_pos past the substring and stars,
		// and loop again.
		candidate_pos = match_pos + sub_len;
		pattern_pos = pattern_pos + sub_len;
		while (*pattern_pos == '*') {
			pattern_pos++;
		}
	}

	// We fall out of this loop when we've consumed one or the other string.
	// The loop would have returned if we'd consumed the whole pattern and it
	// didn't end with a star, so if the pattern is consumed we know it DID
	// end with a star.
	if (!*pattern_pos) {
		// Bottomed out, star at end of pattern, MATCH
		return true;
	}
	else {
		// We know we're at the end of candidate (otherwise we'd have looped again).
		// This means we consumed the whole candidate without consuming the last non-star
		// part of the pattern. NO MATCH.
		return false;
	}
}

//------------------------------------------------------------------------------
bool 
CFilterRule::Match(const char* i_strPackage, 
	   const char* i_strClass, 
	   const char* i_strMethod, 
	   const char* i_strMethodSig)
{
	// Match package name first
	if (!WildcardStringMatch(m_strPackageName, i_strPackage)) {
		return false;
	}
	// Match class name next
	if (!WildcardStringMatch(m_strClassName, i_strClass)) {
		return false;
	}
	if (!WildcardStringMatch(m_strMethodName, i_strMethod)) {
		// pkg/class match but method doesn't
		return false;
	}
	if (!WildcardStringMatch(m_strMethodSig, i_strMethodSig)) {
		// pkg/class/method match, but signature doesn't
		return false;
	}
	// complete match: package, class, method, and signature
	return true;
}

//==============================================================================
//------------------------------------------------------------------------------
CProbe::CProbe()
{
}

void
CProbe::AddFilterRule(CSTR i_szPackageName, 
					  CSTR i_szClassName, 
					  CSTR i_szMethodName, 
					  CSTR i_szMethodSig, 
					  CFilterRule::action_t i_action)
{
	CFilterRule fr(i_szPackageName, i_szClassName, i_szMethodName, i_szMethodSig, i_action);
	m_ruleList.push_back(fr);
}

//------------------------------------------------------------------------------
CProbe::~CProbe()
{
	// Delete the CProbeRefs for this probe
	for (CProbeRefList::iterator ref_iter = m_probereflst.begin();
		 ref_iter != m_probereflst.end();
		 ref_iter++) 
	{
		delete (*ref_iter);
	}
}

//------------------------------------------------------------------------------
void
CProbe::AddProbeRef(CProbeRef::ProbeRef_enu i_enuType, 
		            CSTR i_szClass, CSTR i_szMethod, CSTR i_szMethodSig, 
					CSTR i_szRefType, CSTR i_szArgList)
{
	CProbeRef* pproberef = NULL;
	switch(i_enuType)
	{
		case CProbeRef::PROBE_ONENTRY:
			pproberef = new CProbeRefOnEntry(this, i_szClass, i_szMethod, i_szMethodSig, i_szArgList);
			break;
		case CProbeRef::PROBE_ONEXIT:
			pproberef = new CProbeRefOnExit(this, i_szClass, i_szMethod, i_szMethodSig, i_szArgList);
			break;
		case CProbeRef::PROBE_ONCATCH:
			pproberef = new CProbeRefOnCatch(this, i_szClass, i_szMethod, i_szMethodSig, i_szArgList);
			break;
		case CProbeRef::PROBE_BEFORECALL:
			pproberef = new CProbeRefBeforeCall(this, i_szClass, i_szMethod, i_szMethodSig, i_szArgList);
			break;
		case CProbeRef::PROBE_AFTERCALL:
			pproberef = new CProbeRefAfterCall(this, i_szClass, i_szMethod, i_szMethodSig, i_szArgList);
			break;
	}
	m_probereflst.push_back(pproberef);
}

//------------------------------------------------------------------------------
CProbeRefList&	
CProbe::GetProbeRefList()
{
	return m_probereflst;
}

//------------------------------------------------------------------------------
// IsCallsiteProbe
// Return true if this probe is a callsite probe.
//
// TODO: Currently we only support fragments which have ONLY
// before/after call, or those which ONLY have entry/exit/catch.
// You shouldn't mix them.
//
// This limitation means we can answer the question by looking
// at the first ProbeRef in this probe's list.
//

bool
CProbe::IsCallsiteProbe()
{
	if (m_probereflst.size() == 0) 
		return false;

	CProbeRef* firstref = m_probereflst[0];
	if (firstref->GetType() == CProbeRef::PROBE_BEFORECALL ||
		firstref->GetType() == CProbeRef::PROBE_AFTERCALL) 
	{
		return true;
	}
	else {
		return false;
	}
}

//------------------------------------------------------------------------------
// Match function: the arguments are the package, class, method, and signature
// strings of a candidate function. Process the rules stored in
// this CProbe in order. If a rule's pattern matches the candidate, return
// that rule's action: true for include, false for exclude.
//
bool		
CProbe::Match(CSTR i_szPkgAndClass, CSTR i_szMethod, CSTR i_szSig)
{
	// Split out the package from the class
	const char* szClass;
	char* pkg = strdup(i_szPkgAndClass);
	
	// First, convert dots to slashes
	int len = strlen(pkg);
	for (int i = 0; i < len; i++) {
		if (pkg[i] == '.') 
			pkg[i] = '/'; 
	}

	// Now find the last slash - everything before it is the package name
	char* lastslash = strrchr(pkg, '/');
	if (lastslash != NULL) {
		*lastslash = '\0';
		szClass = lastslash + 1;
	}
	else {
		// No slashes, so package name is the empty string
		// and class name is the whole input string
		*pkg = '\0';
		szClass = i_szPkgAndClass;
	}

	// If there's no match of any rule, result will be "true"
	// because this is how the existing profiling tools' filter system works
	bool result = true;

	CFilterRuleList::iterator iter;
	for (iter = m_ruleList.begin(); iter != m_ruleList.end(); iter++) {
		if ((*iter).Match(pkg, szClass, i_szMethod, i_szSig)) {
			CFilterRule& r = (*iter);
			if (r.m_action == CFilterRule::ACTION_INCLUDE) {
				result = true;
				break;
			}
			else {
				// Matched, and the action is "exclude."
				result = false;
				break;
			}
		}
	}

	free(pkg);
	return result;
}

//==============================================================================

//------------------------------------------------------------------------------
CProbeList::CProbeList()
{
}

//------------------------------------------------------------------------------
void	
CProbeList::AddProbe(CProbe* i_pprobe)
{
	push_back(i_pprobe);
}

//==============================================================================
// CBCIEngProbe implementation

//------------------------------------------------------------------------------
// Constructor
CBCIEngProbe::CBCIEngProbe()
{
	m_ipFinally = (IP_t)-1;
	m_ipCatchAll = (IP_t)-1;
}

//------------------------------------------------------------------------------
// Destructor
CBCIEngProbe::~CBCIEngProbe()
{
	// Delete the probes in m_probelst.
	for (CProbeList::iterator probeIter = m_probelst.begin();
		probeIter != m_probelst.end();
		probeIter++) 
	{
		delete (*probeIter);
	}
}

//------------------------------------------------------------------------------
// Instrument class
// In:
//	i_pInClass	- pointer to the beginning of Java class
//	i_cbInClass	- size of the input class
// Out:
//	o_ppOutClass - address of pointer to the instrumented class
//	o_pcbOutClass - address of size of the instrumented class in bytes
//
// Returns:
//	true if instrumented otherwise false
//  ToDo: just throw an exception ?
//
// If you want to stub out instrumentation, replace
// this function with these lines:
//	*o_pcbOutClass = i_cbInClass;
//	*o_ppOutClass = m_pfnMalloc(*o_pcbOutClass);
//	memcpy(*o_ppOutClass, i_pInClass, *o_pcbOutClass);
//

bool	
CBCIEngProbe::Instrument(void* i_pInClass, size_t i_cbInClass,
			 		     void** o_ppOutClass, size_t* o_pcbOutClass)
{
	bool bResult = true;
	CJMemStream		InStream;					// Memory stream
	CJStream		JStreamIn(&InStream);		// Java input stream
	CModuleJ*		pModuleJ = new CModuleJ;
	CJClassFile* pClass = new CJClassFile;      // Will be deleted in the module destructor
	InStream.Open(i_pInClass, i_cbInClass);
	pClass->Read(JStreamIn);
	pModuleJ->Open(pClass, true);
	pModuleJ->SetAccessFlags(pClass->GetAccessFlags());
	pModuleJ->Parse();
	if(NULL != m_pfnCallback 
		&& 0 != (m_wCBFlags & CBCIEngProbe::CALLBACK_MODULE)
		&& !m_pfnCallback(pModuleJ->GetName(), strlen(pModuleJ->GetName()), CBCIEngProbe::CALLBACK_MODULE))
	{
		*o_pcbOutClass = i_cbInClass;
		*o_ppOutClass = i_pInClass;
		return false;
	}

	// Dump the module before instrumentation

	// If we were asked to, then dump the module before and after instrumentation
	const char *dump_output_file;
	if ((dump_output_file = getenv("PROBEKIT_DUMP_FILE")) != NULL) {
		std::ofstream ofs(dump_output_file, std::ios::app);
		if (ofs.is_open()) {
			ofs << "=========== DUMP BEFORE INSTRUMENTATION" << std::endl;
			pModuleJ->Dump(ofs);
			ofs.close();
		}
	}

	Instrument(pModuleJ);
	pModuleJ->Emit();

	if (dump_output_file != NULL) {
		std::ofstream ofs(dump_output_file, std::ios::app);
		if (ofs.is_open()) {
			ofs << "=========== DUMP AFTER INSTRUMENTATION" << std::endl;
			pModuleJ->Dump(ofs);
			ofs.close();
		}
	}

	*o_pcbOutClass = pClass->GetSize();
	*o_ppOutClass = m_pfnMalloc(*o_pcbOutClass);

	CJMemStream		OutStream;					// Memory stream
	OutStream.Open(*o_ppOutClass, *o_pcbOutClass);
	CJStream		JStreamOut(&OutStream);		// Java output stream
	pClass->Write(JStreamOut);
	delete pModuleJ;
	return bResult;
}


//------------------------------------------------------------------------------
const	CProbeList& 
CBCIEngProbe::GetProbes() const
{
	return m_probelst;
}

//------------------------------------------------------------------------------
void	
CBCIEngProbe::AddProbe(CProbe* i_pprobe)
{
	m_probelst.AddProbe(i_pprobe);
}

//------------------------------------------------------------------------------
CProbe* 
CBCIEngProbe::CreateProbe() const
{
	CProbe* pProbe = new CProbe();
	return pProbe;
}

//- Protected ------------------------------------------------------------------
//------------------------------------------------------------------------------
// Instrument
// In: 
//	i_pmod	- pointer to the module to instrument
// Out:
//	-
// Returns:
//
// Throws:
//	CBCIEngProbeException
//
void
CBCIEngProbe::Instrument(CModule* i_pmod)
{
	int	 nProbes = 0;

	// Reset the list of selected probe refs.
	m_probelstMod.clear();		

	if(i_pmod->IsInstrumented())
	{
		throw CBCIEngProbeException("Module is already instrumented");
	}
	// Mark module as instrumented
	CModuleJ* pmodj= (CModuleJ*)i_pmod;
	pmodj->AddStringAttrib("Instrumented", "BCIEngProbe");

	// Walk the list of probes. Add each one to m_probelstMod.
	// Also, add the external references for the probes to the module.
	// TODO: if you want to prune the set of probes to those which can apply to this class,
	// implement that pruning here.

	for(CProbeList::iterator itrProbes = m_probelst.begin(); itrProbes != m_probelst.end(); itrProbes++)
	{
		nProbes++;
		m_probelstMod.push_back(*itrProbes);
		CProbeRefList& proberefs = (*itrProbes)->GetProbeRefList();
		for(CProbeRefList::iterator itrRefs = proberefs.begin(); itrRefs != proberefs.end(); itrRefs++)
		{
			pmodj->AddExtRef(*(*itrRefs)->GetExtRef());
		}
	}

	// InstrumentMethod will go through the probes in m_probelstMod, the
	// list of probes for this module. Since the loop above populated that
	// list with all the probes, it might seem useless to have two lists.
	// This is a vestige of an optimization which only added those probes
	// that apply to this module, and we might reinstate that optimization,
	// and that's the reason for this redundancy.

	if(nProbes > 0)
	{
		// There are some probes to use, so instrument all methods.
		CMethods* pmethods = i_pmod->GetMethods();
		for(CMethods::iterator itrMeth = pmethods->begin(); itrMeth != pmethods->end(); itrMeth++)
		{
			InstrumentMethod(*itrMeth); 
		}
	}
	else 
	{
		// No probes in the probe list? Throw an exception.
		// TODO: if the set of probes was pruned to "only those that might apply to this class"
		// then don't throw an exception here, just exit (with the i_pmod module unmodified).
		throw CBCIEngProbeException(CBCIEngException::REASON_Unknown);
	}
}

//------------------------------------------------------------------------
// BlockInsertionStash helper class
//
// Use this class to hold on to new CInsBlocks you create that you
// are going to want to insert into a method, but that you don't
// want to insert right now because you don't want them to be seen
// by other instrumentation passes.
//
// It's used by callsite insertion for the "aftercall" part.
// The invoke instruction is at the end of an insertion block,
// and the "aftercall" insertion is performed in a new CInsBlock
// which has to follow the original. But if we link it in to the method
// too soon, it'll be seen by other parts of the insertion engine,
// and we don't want that.
//
// The Add method takes an old block pointer and a new block pointer.
// When you call PerformBlockInsertions, each new block will be 
// inserted AFTER the corresponding old block.
//
// If the "old block" pointer is NULL, it means you want to insert
// the new block at the top of the function, before the first original block.
//
 
class BlockInsertionStash : private vector<pair<CInsBlock*, CInsBlock* > >
{
public:
	// Record the fact that we want to insert newBlock after oldBlock
	void Add(CInsBlock* oldBlock, CInsBlock* newBlock) {
		push_back(pair<CInsBlock*, CInsBlock*>(oldBlock, newBlock));
	}

	// Perform the insertions recorded in this stash
	void PerformBlockInsertions(CMethod* pMeth)
	{
		for (BlockInsertionStash::iterator stashIter = begin();
			 stashIter != end();
			 stashIter++)
		{
			CInsBlock* oldBlock = (*stashIter).first;
			CInsBlock* newBlock = (*stashIter).second;

			if (oldBlock == NULL) {
				// Insert the new block at the top of the method
				CInsBlocks* methBlocks = pMeth->GetInsBlocks();
				methBlocks->insert(methBlocks->begin(), newBlock);
			}
			else {
				// Scan the method looking for the old block
				CInsBlocks* methBlocks = pMeth->GetInsBlocks();
				for (CInsBlocks::iterator methBlockIter = methBlocks->begin();
					 methBlockIter != methBlocks->end();
					 methBlockIter++)
				{
					CInsBlock* methBlock = (*methBlockIter);
					if (methBlock == oldBlock) {
						CInsBlocks::iterator tempMethBlockIter = methBlockIter;
						tempMethBlockIter++;
						methBlocks->insert(tempMethBlockIter, newBlock);
						break;	// don't need to keep looking
					}
				}
			}
		}
	}
};

//------------------------------------------------------------------------------
// HandleCallsiteInsertion
//
// This function does everything needed for callsite insertion.
//
// It takes a large number of parameters from its only caller,
// CBCIEngProbe::InstrumentMethod. It was split out of that function
// because it's so long and reasonably self-contained.
//

void
CBCIEngProbe::HandleCallsiteInsertion(CProbeInsertionContext& ctxt,
						CProbeRefList& beforeCallRefs,
						CProbeRefList& afterCallRefs,
						BlockInsertionStash& block_insertion_stash)
{
	// walk every instruction of every block. At each invoke-type
	// instruction, see if it's a call to a function that matches
	// a callsite probe target. If so, perform callsite insertion on it.

	// These variables hold the local variable numbers for storing callsite args
	// and callsite "this" parameters.We only need to allocate one per function 
	// we do insertion on, no matter how many call sites get instrumented.
	int localVariableForCallsiteArgs = -1;
	int localVariableForCallsiteThis = -1;

	CInsBlocks* pinsb = ctxt.pMethodJ->GetInsBlocks();
	for (CInsBlocks::iterator itrBlk = pinsb->begin(); 
		 itrBlk != pinsb->end(); 
		 itrBlk++)
	{
		CInstructions* pins = (*itrBlk)->GetInstructions();
		for (CInstructions::iterator itrIns = pins->begin();
			 itrIns != pins->end();
			 itrIns++)
		{
			CInstruction* pi = *itrIns;
			if(pi->GetSemTag() == SEM_CALL) {
				// Call site: invokespecial, invokevirtual, invokestatic, invokeinterface
				// Get the package, class, method, and signature of the method being called
				CInstruction_InvokeJ* pInvokeInstr = (CInstruction_InvokeJ*)pi;
				unsigned cpIndex = pInvokeInstr->GetCpIndex();
				CCPUtf8Info* class8 = ctxt.pConstPool->GetMethodClass(cpIndex);
				string className = (string)(*class8);
				CCPUtf8Info* meth8 = ctxt.pConstPool->GetMethodName(cpIndex);
				string methodName = (string)(*meth8);
				CCPUtf8Info* sig8 = ctxt.pConstPool->GetMethodType(cpIndex);
				string methodSig = (string)(*sig8);

				// See if any callsite probes apply to this invoke instruction.
				// We have to check the proberefs on both the BEFORECALL and AFTERCALL lists.
				// And while we're there, collect information about whether ANY of them want
				// the argument list and/or the "this" pointer.

				CProbeRefList beforeCalls;
				CProbeRefList afterCalls;
				bool any_callsite_probes_on_this_invoke = false;
				bool any_callsite_wants_this = false;
				bool any_callsite_wants_args = false;

				CProbeRefList::iterator itrRef;
				for (itrRef = beforeCallRefs.begin();
					 itrRef != beforeCallRefs.end();
					 itrRef++) 
				{
					CProbeRef* pRef = (*itrRef);
					CProbe* parent = pRef->GetParentProbe();
					if (parent->Match(className.c_str(), methodName.c_str(), methodSig.c_str())) {
						any_callsite_probes_on_this_invoke = true;
						beforeCalls.push_back(pRef);
						if (pRef->GetArgBits() & CProbeRef::ARG_BITS_THISOBJ) {
							any_callsite_wants_this = true;
						}
						if (pRef->GetArgBits() & CProbeRef::ARG_BITS_ARGSLIST) {
							any_callsite_wants_args = true;
						}
					}
				}

				// Same loop as above, for AFTERCALL.
				for (itrRef = afterCallRefs.begin();
					 itrRef != afterCallRefs.end();
					 itrRef++)
				{
					CProbeRef* pRef = (*itrRef);
					CProbe* parent = pRef->GetParentProbe();
					if (parent->Match(className.c_str(), methodName.c_str(), methodSig.c_str())) {
						any_callsite_probes_on_this_invoke = true;
						afterCalls.push_back(pRef);
						if (pRef->GetArgBits() & CProbeRef::ARG_BITS_THISOBJ) {
							any_callsite_wants_this = true;
						}
						if (pRef->GetArgBits() & CProbeRef::ARG_BITS_ARGSLIST) {
							any_callsite_wants_args = true;
						}
					}
				}

				if (any_callsite_probes_on_this_invoke) {
					CProbeInsertionContext callsite_ctxt;
					callsite_ctxt.className = className.c_str();
					callsite_ctxt.pMethodJ = ctxt.pMethodJ;
					callsite_ctxt.methodName = methodName.c_str();
					callsite_ctxt.methodSignature = methodSig.c_str();
					callsite_ctxt.pConstPool = ctxt.pConstPool;
					// (callsite_ctxt.interfaceNames is always empty)

					callsite_ctxt.localVariableForThis = -1;
					callsite_ctxt.localVariableForArgs = -1;
					callsite_ctxt.itrIns = &itrIns;	// see caution at CProbeInsertionContext.itrIns decl
					callsite_ctxt.pIns = pins;

					// Set "hasThis" - means just one thing, that the method
					// actually has a "this" pointer. Doesn't mean a
					// fragment can access it. Constructors, for instance
					// have "this" but entry fragments can't access it -
					// it's uninitialized at that point.

					callsite_ctxt.hasThis = (pi->GetOpCode() != JVMI_invokestatic);

					CCodeAttribute* pcode = ctxt.pMethodJ->GetCodeAttribute();

					// We have to store args now if anybody wants them,
					// or if anybody wants "this" (because it's buried under the args)
					if (any_callsite_wants_args || any_callsite_wants_this) {
						// Allocate a local in the calling function to hold the args.
						// We only have to do this once per function we do insertion on,
						// no matter how many call sites get instrumentation.
						if (localVariableForCallsiteArgs == -1) {
							localVariableForCallsiteArgs = pcode->GetMaxLocals();
							pcode->SetMaxLocals(pcode->GetMaxLocals() + 1);
						}
						callsite_ctxt.localVariableForArgs = localVariableForCallsiteArgs;
						CallsiteStoreArgs(callsite_ctxt);
					}
					if (any_callsite_wants_this) {
						if (localVariableForCallsiteThis == -1) {
							localVariableForCallsiteThis = pcode->GetMaxLocals();
							pcode->SetMaxLocals(pcode->GetMaxLocals() + 1);
						}
						callsite_ctxt.localVariableForThis = localVariableForCallsiteThis;
						CallsiteStoreThis(callsite_ctxt);
					}

					// Now insert all the call-before probe calls
					
					for (itrRef = beforeCalls.begin();
						 itrRef != beforeCalls.end();
						 itrRef++)
					{
						CProbeRef* pRef = (*itrRef);
						pRef->Instrument(callsite_ctxt);
					}

					// Now restore the stack so we can run the original invoke instruction
					CallsiteReloadArgsAndThis(callsite_ctxt);

					// That's the end of beforecall logic. Now do aftercall logic.

					if (afterCalls.size() != 0) {
						// We create a new insertion block, with label "-1" meaning "no label."
						// We will link it in after the one that we just did insertion on -
						// that is, the one that ended with the invoke instruction.
						//
						// This is predicated on the idea that the invoke instruction
						// was the end of the block it's in. If we change the code that slices
						// functions into insertion blocks, we'll have to change this too.

						// Assert that the "invoke" instruction is the last instruction in its block.
						CInstructions::iterator itrInsTemp = itrIns;
						itrInsTemp++;
						CBCIEngProbeException::Assert(itrInsTemp == pins->end(), 
							"Invoke instruction is not at the end of its insertion block");

						CInsBlock* pNewInsBlock = new CInsBlock((unsigned)-1);
						callsite_ctxt.pIns = pNewInsBlock->GetInstructions();
						CInstructions::iterator newInsBlockIter = callsite_ctxt.pIns->begin();
						callsite_ctxt.itrIns = &newInsBlockIter;
						callsite_ctxt.isExceptionExit = false;

						for (itrRef = afterCalls.begin();
							itrRef != afterCalls.end();
							itrRef++)
						{
							CProbeRef* pRef = (*itrRef);
							pRef->Instrument(callsite_ctxt);
						}

						// Record the fact that I want to insert this new block after the current block

						// Build exception exit interceptor for the callsite
						// TODO: factor out
						// Add new exception exit block to the end of the code
						// Set labels
						IP_t ipStart = pi->GetIP();
						IP_t ipEnd = pi->GetIP();
						IP_t ipCatchAll = callsite_ctxt.pMethodJ->CreateUniqueLabel();
						IP_t ipNext = pi->GetIP() + pi->GetSize();
						CLabels* plabels = ctxt.pMethodJ->GetLabels();
						plabels->insert(LabelsEntry_t(ipNext, ipNext));
						pNewInsBlock->AddInstruction(CInsSetJ::Create_goto(ipNext));
						// Create a new block for exception handler and save it in the stash
						CInsBlock* pblkExceptionExit = new CInsBlock(ipCatchAll);
						CMethodException* pmtdex = new CMethodExceptionJ(0, ipStart, ipEnd, pblkExceptionExit);
						callsite_ctxt.pMethodJ->AddException(pmtdex, CMtdExTable::TOP);

						// Note: we add blocks to the stash in the inverse order that we
						// want them emitted in. This is a defect in the stash that
						// will be fixed soon. 
						// TODO: reorder these and remove this comment when it's fixed.
						block_insertion_stash.Add(*itrBlk, pblkExceptionExit);
						block_insertion_stash.Add(*itrBlk, pNewInsBlock);

						// Prepare the insertion context
						callsite_ctxt.pIns = pblkExceptionExit->GetInstructions();
						CInstructions::iterator itrInsCatchAll = callsite_ctxt.pIns->begin();
						callsite_ctxt.itrIns = &itrInsCatchAll;
						callsite_ctxt.isExceptionExit = true;
						for (itrRef = afterCalls.begin(); itrRef != afterCalls.end(); itrRef++)
						{
							CProbeRef* pRef = (*itrRef);
							pRef->Instrument(callsite_ctxt);
							callsite_ctxt.Insert(CInsSetJ::Create_simple(JVMI_athrow));
						}

					}
				}
			}
		}
	}
}

//------------------------------------------------------------------------------
// InstrumentMethod
// In:
//	i_pmeth - pointer to the method (CMethod*)
// Out:
//	-
// Throws:
//
// How this works.
// First we populate the array of probe references.
// Index in the array represents type of probe reference (entry, exit etc.)
// Each element of the array contains a list of probe references.
// Thus more than one probe method can be assigned to every probe type.
// Note that the instrumentation itself is defined in the appropriate
// specialization of the CProbeRef class.
//
void
CBCIEngProbe::InstrumentMethod(CMethod* i_pmeth)
{
	// Functions with no code attribute don't get any farther.
	// They are placeholders for interfaces and native functions..
	CMethodJ* pmethj = (CMethodJ*)i_pmeth;
	CCodeAttribute* pcode = pmethj->GetCodeAttribute();
	if(NULL == pcode)
	{
		return;
	}

	// Perform the callback to say we're about to instrument this method, and let the callback tell us not to.
	if (NULL != m_pfnCallback &&
		(m_wCBFlags & CBCIEngProbe::CALLBACK_METHOD) &&
		!m_pfnCallback(i_pmeth->GetName(), strlen(i_pmeth->GetName()), CBCIEngProbe::CALLBACK_METHOD))
	{
		return;
	}

	CInstruction* pi;
	CInsBlocks::iterator itrBlk;

	// Create the insertion context for this method.
	// Most of the insertion context stays the same for the whole method.
	// Some parts change, like isFinally and the instruction block to insert into
	CProbeInsertionContext ctxt;

	CModuleJ* pmodj = (CModuleJ*)(i_pmeth->GetModule());
	ctxt.className = pmodj->GetName();

	ctxt.pMethodJ = pmethj;
	ctxt.methodName = i_pmeth->GetName();
	ctxt.methodSignature = pmethj->GetSignature();
	ctxt.hasThis = i_pmeth->GetHasThis();

	// Fill in ctxt.interfaceNames
	ctxt.interfaceNames = pmethj->GetModule()->GetInterfaces();

	// During function instrumentation, the local variable for "this" is always number zero.
	// During callsite instrumentation it'll be something else.
	ctxt.localVariableForThis = 0;

	CJClassFile& jclass = pmodj->GetClass();
	ctxt.pConstPool = jclass.GetConstPool();

	// Initialize localVariableForArgs to -1.
	// But if anybody actually wants it, we'll change it.
	ctxt.localVariableForArgs = -1;
	bool any_fn_probes_want_args_list = false;

	bool any_proberefs_match = false;
	bool any_callsite_probes = false;

	// This flag goes true if we encounter something that is going to
	// need function entry insertion.
	bool any_entry_insertion_needed = false;
	// or function exit insertion
	bool any_exit_insertion_needed = false;
	// Populate the list of probes that apply to this method.
	// All callsite probes apply to this method.
	// Non-callsite (function) probes only apply if they Match() the class/name/signature
	// of this class, or of a (direct) interface implemented by this class.
	// TODO: factor this loop into another function to make this one simpler.

	CProbeRefList   probeRefs[CProbeRef::PROBE_LAST];

	for(CProbeList::iterator itrProbe = m_probelstMod.begin(); itrProbe != m_probelstMod.end(); itrProbe++)
	{
		CProbe* pProbe = (*itrProbe);
		bool match = false;

		if (pProbe->IsCallsiteProbe()) {
			match = true;
		}
		else if (pProbe->Match(ctxt.className, ctxt.methodName, ctxt.methodSignature)) {
			match = true;
		}
		else {
			// check against direct interface names (plus this method name and signature)
			for (vector<string>::iterator interface_iter = ctxt.interfaceNames.begin();
				 interface_iter != ctxt.interfaceNames.end();
				 interface_iter++)
			{
				string& interface_name = (*interface_iter);
				if (pProbe->Match(interface_name.c_str(), ctxt.methodName, ctxt.methodSignature)) {
					match = true;
					break;
				}
			}
		}

		if (match) {
			CProbeRefList& proberefs = pProbe->GetProbeRefList();
			for(CProbeRefList::iterator itrRef = proberefs.begin(); itrRef != proberefs.end(); itrRef++)
			{
				CProbeRef* pRef = (*itrRef);
				int nType = pRef->GetType();
				CBCIEngProbeException::Assert(nType >= 0 && nType < CProbeRef::PROBE_LAST,
					"Unknown probe reference type (not entry, catch, exit, etc.)");
				probeRefs[nType].push_back(pRef);
				any_proberefs_match = true;
				if (!pProbe->IsCallsiteProbe() &&
					(pRef->GetArgBits() & CProbeRef::ARG_BITS_ARGSLIST)) 
				{
					any_fn_probes_want_args_list = true;
				}
				if (nType == CProbeRef::PROBE_BEFORECALL || nType == CProbeRef::PROBE_AFTERCALL) {
					any_callsite_probes = true;
				}
				if (nType == CProbeRef::PROBE_ONENTRY) {
					any_entry_insertion_needed = true;
				}
				if (nType == CProbeRef::PROBE_ONEXIT) {
					any_exit_insertion_needed = true;
				}
			}
		}
	}

	if (any_proberefs_match == false) {
		return;
	}

	// If any of the probes that apply to this method wants the argument list, 
	// allocate the local variable number for it now.
	if (any_fn_probes_want_args_list) {
		CCodeAttribute* pcode = pmethj->GetCodeAttribute(); 
		ctxt.localVariableForArgs = pcode->GetMaxLocals();
		pcode->SetMaxLocals(pcode->GetMaxLocals() + 1);

		// Flag the fact that we have some entry insertion to do
		any_entry_insertion_needed = true;
	}

	// Declare the block insertion stash. This is a memory bucket that stores
	// new CInsBlocks that you are going to want to insert into this function later,
	// after all our instrumentation is complete.

	BlockInsertionStash block_insertion_stash;

	// Now it's time to instrument the code

	CInsBlocks* pinsb = pmethj->GetInsBlocks();

	//==================================================================

	// If any of the matching probes are callsite probes,
	// call the callsite handler function now.
	if (any_callsite_probes) {
		HandleCallsiteInsertion(ctxt,
							probeRefs[CProbeRef::PROBE_BEFORECALL],
							probeRefs[CProbeRef::PROBE_AFTERCALL],
							block_insertion_stash);
	}

	// Now, do catch insertion on blocks which are exception handlers.
	//
	// For each block,
	//   For each exception handler,
	//		If this block is a handler,
	//		  For each catch-type probe fragment
	//			Call "Insert" on this fragment.
	//
	// TODO: optimize. If there's more than one catch/finally in the
	// function, we're causing bloat. Instead, insert a JSR to a
	// common "top of catch/finally" clause. Place the isFinally
	// flag value in a local or something, so it can be passed to
	// the fragment.
	//
	// TODO: factor this into a separate function to make this one shorter.

	CProbeRefList::iterator itrRef;
	CExTable& extable = pcode->GetExTable();
	for(itrBlk = pinsb->begin(); itrBlk != pinsb->end(); itrBlk++)
	{
		// TODO: optimize so we don't loop through ex table on each insn
		for(CExTable::iterator itrEx = extable.begin(); itrEx != extable.end(); itrEx++)
		{
			if(itrEx->GetHandlerPC() == (*itrBlk)->GetLabel())
			{	// Exception Handler
				bool isFinally = (itrEx->GetCatchtype() == 0);
				for (itrRef = probeRefs[CProbeRef::PROBE_ONCATCH].begin();
					 itrRef != probeRefs[CProbeRef::PROBE_ONCATCH].end();
					 itrRef++)
				{
					ctxt.pIns = (*itrBlk)->GetInstructions();
					CInstructions::iterator itrIns = ctxt.pIns->begin();
					ctxt.itrIns = &itrIns;
					ctxt.isFinally = isFinally;
					(*itrRef)->Instrument(ctxt);
				}
			}
		}
	}

	// TODO: factor exit wrapping and return instruction
	// instrumentation into a separate function.

	// Create a finally wrapper for all return instructions,
	// and one grand "catch all" exception handler to call exit
	// fragments upon exception exit.
	if(any_exit_insertion_needed)
	{
		CreateExitWrap(ctxt, probeRefs[CProbeRef::PROBE_ONEXIT]);
		// Walk every instruction of every block. Before each "return"
		// instruction, insert a JSR to the finally block created
		// by CreateExitWrap. Its location was stored in m_ipFinally.
		for(itrBlk = pinsb->begin(); itrBlk != pinsb->end(); itrBlk++)
		{
			CInstructions* pins = (*itrBlk)->GetInstructions();
			for (CInstructions::iterator itrIns = pins->begin();
				 itrIns != pins->end();
				 itrIns++)
			{
				pi = *itrIns;
				if(pi->GetSemTag() == SEM_RET)
				{
					pins->insert(itrIns, CInsSetJ::Create_jsr(m_ipFinally));
				}
			}
		}
	}

	// Perform top-of-method insertion, if any_entry_insertion_needed.
	// Right now two situations can require entry insertion:
	//		1. A PROBE_ONENTRY that applies to this method,
	//	or	2. Any function probes want the args list
	//
	// We create a new insertion block, with label "-1" meaning "no label"
	// and add it to the insertion stash, saying we want it inserted
	// before the first block of the method.
	//
	// TODO: factor this into a separate function.
	// TODO: this logic can move anywhere in the function thanks to the stash.

	if (any_entry_insertion_needed) {
		CInsBlock* pNewInsBlock = new CInsBlock((unsigned)-1);
		CInstructions* pinsList = pNewInsBlock->GetInstructions();
		CInstructions::iterator itrIns = pinsList->begin();

		ctxt.pIns = pinsList;
		ctxt.itrIns = &itrIns;
		
		if (any_fn_probes_want_args_list) {
			// Insert instructions to allocate and populate the Object[] array for args.
			// We do this just once and re-use it for every probe fragment that wants it.
			emit_args_list(ctxt);
			ctxt.Insert(CInsSetJ::Create_astore(ctxt.localVariableForArgs));
		}

		for (itrRef = probeRefs[CProbeRef::PROBE_ONENTRY].begin();
			 itrRef != probeRefs[CProbeRef::PROBE_ONENTRY].end();
			 itrRef++)
		{
			(*itrRef)->Instrument(ctxt);
		}

		// Record that we want the new block inserted at the top of the function.
		block_insertion_stash.Add(NULL, pNewInsBlock);
	}

	// Process the block insertion stash now that we're finished.
	block_insertion_stash.PerformBlockInsertions(pmethj);
}

//------------------------------------------------------------------------------
// CreateExitWrap
// In:
//	ctxt - CProbeInsertionContext - the context for the insertion
// Out:
//	-
// Throws:
//
// This function creates the ordinary exit and exception exit wrappers for a function. 
// This adds two blocks to the function: a "catch any" exception handler block for
// exception exit, and a block which serves as a "finally" block.
//
// The "catch any" exception handler calls the "exit" fragment of every probe 
// that applies to this function. The argument list passed to the fragments
// can include the exception object that caused this function to exit.
// The last thing the "catch any" handler does is rethrow the exception that got us there.
//
// The "finally" block contains a call to the "exit" fragment of every probe 
// that applies to this function. It makes available the return value (if there
// is one) as a possible argument to each fragment function.
// Then it returns to the place it was called from - which will be
// a JSR instruction inserted just before every "return" instruction
// in the original function.
//
void	
CBCIEngProbe::CreateExitWrap(CProbeInsertionContext& ctxt, 
                             CProbeRefList& exit_probe_refs)
{
	CCodeAttribute* pcode = ctxt.pMethodJ->GetCodeAttribute(); 
	CExTable& extab = pcode->GetExTable();
	CInsBlocks* pinsblks = ctxt.pMethodJ->GetInsBlocks();
	CInsBlock* pinsblkLast = *(pinsblks->end() - 1);
	CInstructions* pLastBlockInstructions = pinsblkLast->GetInstructions();
	CInstruction* pLastInstruction;

	// Set pLastInstruction to the last instruction in the function.
	// Then synthesize a new placeholder, and set pLastInstruction to it.
	// This placeholder exists as the target instruction for the
	// end of the try/finally region that covers the entire function.
	CInstructions::reverse_iterator ritrIns;
	ritrIns = pLastBlockInstructions->rbegin();
	pLastInstruction = *ritrIns;

	// Compute labels
	IP_t ipStart = (*pinsblks->begin())->GetLabel();
	IP_t ipEnd = pLastInstruction->GetIP();
	m_ipCatchAll = ctxt.pMethodJ->CreateUniqueLabel();
	m_ipFinally = ctxt.pMethodJ->CreateUniqueLabel();


	// Create new blocks for the finally clause
	// Catch All block
	CInsBlock* pinsblkCatchAll = new CInsBlock(m_ipCatchAll);
	CInstructions* pinsCatchAll = pinsblkCatchAll->GetInstructions();
	// Finally block
	CInsBlock* pinsblkFinally = new CInsBlock(m_ipFinally);
	CInstructions* pinsFinally = pinsblkFinally->GetInstructions();

	// Create the exception descriptor and add it to the exception table
	CMethodException* pmtdex = new CMethodExceptionJ(0, ipStart, ipEnd, pinsblkCatchAll);
	ctxt.pMethodJ->AddException(pmtdex);

	// Generate code
	// CatchAll block
	//
	// The catchall block starts execution with the exception object
	// on the top of the stack. We call Instrument
	// for each ONEXIT fragment, and those which want to pass the exception
	// object to the fragment will dup it and pass it appropriately.
	//
	// Last, we emit "athrow" to rethrow this exception object.

	// This iterator local variable is used twice in this function
	// for different loops.
	CInstructions::iterator itrIns;
	CProbeRefList::iterator itrRef;

	// Insert exit probes here
	for (itrRef = exit_probe_refs.begin();
		 itrRef != exit_probe_refs.end();
		 itrRef++)
	{
		ctxt.pIns = pinsCatchAll;
		itrIns = pinsCatchAll->end();
		ctxt.itrIns = &itrIns;
		ctxt.isExceptionExit = true;
		ctxt.isFinally = false;
		(*itrRef)->Instrument(ctxt);
	}
	pinsCatchAll->push_back(CInsSetJ::Create_simple(JVMI_athrow));

	// Finally block
	//
	// The finally block starts execution with the return address at the
	// top of the stack, and after that is the return value, if any.
	//
	// First we store the return address in a new local variable - that leaves
	// the return value at the top of the stack. PushArguments for a probe
	// fragment which wants the return object will correctly dup that value
	// and leave the original on the stack. The last thing we do is emit
	// a return through the new local.

	u2 u2Ret = pcode->GetMaxLocals();
	pcode->SetMaxLocals(u2Ret + 1);
	CInstruction* pi = CInsSetJ::Create_astore(u2Ret);
	pinsFinally->push_back(pi);

	// Insert exit probes here
	for (itrRef = exit_probe_refs.begin();
		 itrRef != exit_probe_refs.end();
		 itrRef++)
	{
		ctxt.pIns = pinsFinally;
		itrIns = pinsFinally->end();
		ctxt.itrIns = &itrIns;
		ctxt.isExceptionExit = false;
		ctxt.isFinally = false;
		(*itrRef)->Instrument(ctxt);
	}
	pinsFinally->push_back(CInsSetJ::Create_ret(u2Ret));

	// Add catch and finally blocks to the method
	pinsblks->push_back(pinsblkCatchAll);
	pinsblks->push_back(pinsblkFinally);
}

//==============================================================================
// CBCIEngProbeException
// Probe engine exception
// ToDo: figure out how to hand out other exceptions without exposing 
//       the library internals
//
CBCIEngProbeException::CBCIEngProbeException(unsigned i_uReason)
:CBCIEngException(i_uReason)
{
	m_szReason[0] = '\0';
}

CBCIEngProbeException::CBCIEngProbeException(const char* i_szReason)
:CBCIEngException(REASON_Unknown)
{
	strncpy(m_szReason, i_szReason, 128);
	m_szReason[127] = '\0';
}

//= End of BCIENgProbe.cpp =====================================================
