/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

//==============================================================================
// ProbeInstrumenter.cpp
//
// This is the "main" that drives the BCIEngProbe libary to instrument
// class files on disk.
// 
// USAGE: ProbeInstrumenter engine_script class_file [ class_file ... ]
//
// Instruments the named class files. The output file name for each
// is the original name plus ".bci"
//

#pragma warning(disable:4786)
#include <string>

#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>
#include "CommonDef.h"
#include "BCIEngProbeInterface.h"

USE_NAMESPACE(std);

//------------------------------------------------------------------------------
class CBCIEngProbeInstrumenter
{
public:
	CBCIEngProbeInstrumenter();
	~CBCIEngProbeInstrumenter();
	void InstrumentClassFile(CSTR i_szName);
	void ParseProbeDescriptors(CSTR i_szName);
	unsigned int InterpretArgs(const char* szArgs);

private:
	CBCIEngInterface*	m_peng;
};


//------------------------------------------------------------------------------
CBCIEngProbeInstrumenter::CBCIEngProbeInstrumenter()
{
	CreateBCIEngine((void**)&m_peng);
	m_peng->SetAllocator(malloc);
}

//------------------------------------------------------------------------------
CBCIEngProbeInstrumenter::~CBCIEngProbeInstrumenter()
{
	DestroyBCIEngine(m_peng);
}

//------------------------------------------------------------------------------
void 
CBCIEngProbeInstrumenter::ParseProbeDescriptors(CSTR i_szName)
{
	char* pchProbe = NULL;
	size_t cbProbe = 0;
	FILE* pfProbe = fopen(i_szName, "r");

	if (NULL == pfProbe)
		throw "Can't open probe descriptors file";
	fseek(pfProbe, 0, SEEK_END);
	cbProbe = ftell(pfProbe) + 2;
	pchProbe = (char*)malloc(cbProbe);
	fseek(pfProbe, 0, SEEK_SET);
	cbProbe = fread(pchProbe, 1, cbProbe, pfProbe);
	m_peng->Initialize(pchProbe, cbProbe);
	free(pchProbe);
	fclose(pfProbe);
}

//------------------------------------------------------------------------------
void 
CBCIEngProbeInstrumenter::InstrumentClassFile(CSTR i_szName)
{
	string strOutName = i_szName;
	size_t cbIn, cbOut;
	void *pIn, *pOut;
	// Open and read class file
	FILE* pfIn = fopen(i_szName, "rb");
	if (NULL == pfIn)
		throw "Can't open class file";
	fseek(pfIn, 0, SEEK_END);
	cbIn = ftell(pfIn);
	pIn = malloc(cbIn);
	fseek(pfIn, 0, SEEK_SET);
	fread(pIn, 1, cbIn, pfIn);
	// Instrument
	m_peng->Instrument(pIn, cbIn, &pOut, &cbOut);
	// Write the result out and cleanup
	free(pIn);	
	fclose(pfIn);
	strOutName += ".bci";
	FILE* pfOut = fopen(strOutName.c_str(), "wb");
	fwrite(pOut, 1, cbOut, pfOut);
	free(pOut);
	fclose(pfOut);
}


//------------------------------------------------------------------------------
static void PrintHelp()
{
	fprintf(stderr, "Usage: InsertProbes engine-script-file class-file [ class-file ... ]\n");
}

//------------------------------------------------------------------------------
main(int argc, char* argv[])
{
	int errorStatus = 0;

	if (argc < 3) {
		PrintHelp();
		return 1;
	}

	CBCIEngProbeInstrumenter eng;

	try {
		eng.ParseProbeDescriptors(argv[1]);
	}
	catch (...) {
		fprintf(stderr, "Problem creating instrumentation engine using engine script %s\n", argv[1]);
		return 1;
	}

	// Operate on each input file.
	// A failure in one doesn't stop us operating on others.
	for (int i = 2; i < argc; i++) {
		try
		{
			eng.InstrumentClassFile(argv[i]);
		}
		catch (CBCIEngProbeInterfaceException& ex)
		{
			unsigned src=ex.GetSource(); 
			unsigned code=ex.GetReason();
			int msgBufSize = 1024;
			char* msgBuf = (char*)malloc(msgBufSize);
			ex.FormatMessage(msgBuf, msgBufSize);
			fprintf(stderr, "Probe Kit Exception number %d from location %d:\n%s\n", code, src, msgBuf);
			free(msgBuf);
			errorStatus = 2;
		}
		catch (char* szMsg)
		{
			fprintf(stderr, "Probe test exception: %s\n", szMsg);
			errorStatus = 2;
		}
	}
	return errorStatus;
}

//= End of ProbeInstrumenter.cpp =======================================================
