/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.util.Enumeration;
import java.util.Vector;
import java.io.IOException;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.LaunchProcessCommand;
import org.eclipse.hyades.internal.execution.local.common.ProcessLaunchedCommand;
import org.eclipse.hyades.internal.execution.local.common.DetailedAgentInfoCommand;
import org.eclipse.hyades.internal.execution.local.common.ErrorCommand;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.ConsoleNotStartedException;
import org.eclipse.hyades.internal.execution.local.common.SimpleAgentInfoCommand;

public class ProcessImpl implements Process, AgentListener, ConnectionListener {
	protected String _name=null;
	protected String _exe=null;
	protected String _params=null;
	protected String _location=null;
	protected String _processId=null;
	protected String _UUID=null;
	protected Node _node=null;
	protected Console _console=null;
	protected boolean _isActive=false;
	protected boolean _isComplete=false;
	
	protected Vector _agents=new Vector(10);
	protected Vector _variables=new Vector(20);
	protected Vector _listeners=new Vector(10);
	
	/* This is a lock for launching the process as this is synchronous */
	private Object _launcherLock=new Object();
	
	/* In eclipse, the model changes need to be done on the UI thread, this requires
	   a bit of additional locking that, by rights, shouldn't be required
	*/
	private Object _eclipseLock=new Object();
	private boolean _requireEclipseLock=false;
	
	private boolean _noSuchAliasExceptionThrown=false;
	


	public ProcessImpl(Node node) {
		_node=node;
	}
	
	public ProcessImpl(Node node, String executable) {
		this(node, executable, null);
	}
	
	public ProcessImpl(Node node, String executable, String parameters) {
		_node=node;
		_exe=executable;
		_params=parameters;
		_console=new Console();
	}
	
	public ProcessImpl(Node node, String executable, long pid) {
		_node=node;
		_exe=executable;
		_processId=new Long(pid).toString();
		_isActive=true;
	}
	
	
	/**
	 * @see Process#launch()
	 */
	public void launch() throws ProcessActiveException, NotConnectedException, NoSuchApplicationException {
		/* Determine if the process is already launched or if it has exited */
		if(isActive() || _isComplete) {
			throw new ProcessActiveException();
		}
		
		
		/* Launch the process */
		ControlMessage message=new ControlMessage();
		LaunchProcessCommand command=new LaunchProcessCommand();
		command.setExe(_exe);
		command.setArgs(_params);
		command.setLocation(_location);
		
		/* Setup the console */
		if(_console==null) {
			_console=new Console();
		}
		_console.start();
		try {
			command.setConsole(_console);
		}
		catch(ConsoleNotStartedException e) {
			/* We just started the console so this isn't going to happen */
		}
		
		Enumeration enum=_variables.elements();
		while(enum.hasMoreElements()) {
			Variable var=(Variable)enum.nextElement();
			command.addEnvironmentVariable(var.getName(), var.getValue());
		}
			
		/* Enumerate all the agents and add the interest only if autoAttach is true */
		enum=_agents.elements();
		while(enum.hasMoreElements()) {
			Agent agent=(Agent)enum.nextElement();
			command.addAgent(agent.getName());
		}
			
		message.appendCommand(command);
		Connection connection=_node.getConnection();
		synchronized(_launcherLock) {
			try {
				/* We need to listen in case this connection gets dropped while waiting for the launch to happen */
				connection.addConnectionListener(this);
				_requireEclipseLock=false;
				connection.sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node, CommandElement command) {
						handleCommand(command);
					}
				});
				
				_requireEclipseLock=true;
			}
			catch(IOException e) {
				throw new NotConnectedException();
			}
			
		
			try {
				/* Never wait morethen 7 seconds in case something goes wrong */
				_launcherLock.wait(7000);
			}
			catch(InterruptedException e) {
				/* We may need to handle this exception */
			}
		}
		
		/* If we have a pending NoSuchApplication error we need to throw it. */
		if(_noSuchAliasExceptionThrown) {
			_noSuchAliasExceptionThrown=false;
			throw new NoSuchApplicationException();	
		}
		
		try {
			/* Inform the listeners that the process is launched */
			synchronized(_listeners) {
				enum=_listeners.elements();
				while(enum.hasMoreElements()) {
					((ProcessListener)enum.nextElement()).processLaunched(this);
				}
			}
			
		}
		finally {
			synchronized(_eclipseLock) {
				_eclipseLock.notify();
			}
			
			/* Remove our connection listener and if the connection is hosed we need to throw a exception */
			connection.removeConnectionListener(this);
			if(!connection.isActive()) {
				throw new NotConnectedException();
			}
			
			/* If our process failed to launch properly then we need to stop the console thread */
			if(!isActive()) {
				_console.close();
			}
			
		}
	}

	/**
	 * @see Process#kill(long)
	 */
	public void kill(long delay) throws InactiveProcessException, NotConnectedException {
		if(_node==null) {
			throw new InactiveProcessException();
		}
		_node.killProcess(this);
	}
	
	/**
	 * @see Process#addAgent()
	 */
	public void addAgent(Agent agent) {
		if(!_agents.contains(agent)) {
			_agents.addElement(agent);
		}
		
	}

	/**
	 * @see Process#listAgents()
	 */
	public Enumeration listAgents() {
		return ((Vector)_agents.clone()).elements();
	}
	
	/**
	 * @see Process#getAgentsByType()
	 */
	public Enumeration getAgentsByType(String type) {
		Vector results=new Vector();
		synchronized(_agents) {
			Enumeration agents=listAgents();
			while(agents.hasMoreElements()) {
				Agent agent=(Agent)agents.nextElement();
				if(agent.getType().equals(type)) {
					results.addElement(agent);
				}
			}
		}
		return results.elements();
	}
	
	
	public Agent getAgent(String name) {
		synchronized(_agents) {
			Enumeration agents=listAgents();
			while(agents.hasMoreElements()) {
				Agent agent=(Agent)agents.nextElement();
				if(agent.getName().equals(name)) {
					return agent;
				}
			}
		}
		return null;
	}
	
	/**
	 * @see Process#getEnvironment()
	 */
	public Enumeration getEnvironment() {
		return _variables.elements();
	}

	/**
	 * @see Process#addProcessListener(ProcessListener)
	 */
	public void addProcessListener(ProcessListener listener) {
		synchronized(_listeners) {
			if(!_listeners.contains(listener)) {
				_listeners.add(listener);
			}	
		}
	}

	/**
	 * @see Process#removeProcessListener(ProcessListener)
	 */
	public void removeProcessListener(ProcessListener listener) {
		synchronized(_listeners) {
			if(_listeners.contains(listener)) {
				_listeners.remove(listener);
			}	
		}
	}

	/**
	 * @see Process#getNode()
	 */
	public Node getNode() {
		return _node;
	}

	/**
	 * @see Process#setName()
	 */
	public void setName(String name) {
		_name=name;
	}

	/**
	 * @see Process#getName()
	 */
	public String getName() {
		return _name;
	}
	
	/**
	 * @see Process#getUUID()
	 */
	public String getUUID() {
		return _UUID;	
	}

	/**
	 * @see Process#isActive()
	 */
	public boolean isActive() {
		return _isActive;
	}

	/**
	 * @see Process#getConsole()
	 */
	public Console getConsole() {
		if(_console==null) {
			_console=new Console();
		}
		return _console;
	}
	
	/**
	 * @see Process#addEnvironmentVariable()
	 */
	public void addEnvironmentVariable(Variable variable) throws ProcessActiveException {
		if(_isActive) {
			throw new ProcessActiveException();
		}
		_variables.add(variable);
		
	}
	
	/**
	 * @see Process#getEnvironmentVariable
	 */
	public Variable getEnvironmentVariable(String name) {
		Enumeration enum=getEnvironment();
		while(enum.hasMoreElements()) {
			Variable var=(Variable)enum.nextElement();
			if(var.getName().equals(name)) {
				return var;
			}
		}
		return null;
	}
	
	/**
	 * @see Process#removeEnvironmentVariable
	 */
	public void removeEnvironmentVariable(Variable variable) throws ProcessActiveException {
		if(_isActive) {
			throw new ProcessActiveException();
		}
		_variables.remove(variable);
		
	}	
	
	/**
	 * @see Process#setExecutable()
	 */
	public void setExecutable(String exe) throws ProcessActiveException {
		if(_isActive) {
			throw new ProcessActiveException();
		}
		_exe=exe;
	}
	
	/**
	 * @see Process#getExecutable()
	 */
	public String getExecutable() {
		return _exe;
	}
	
	/**
	 * @see Process#setParameters()
	 */
	public void setParameters(String params) throws ProcessActiveException {
		if(_isActive) {
			throw new ProcessActiveException();
		}
		_params=params;
	}
	
	/**
	 * @see Process#getParameters()
	 */
	public String getParameters() {
		return _params;
	}
	
	/**
	 * @see Process#setLocation()
	 */
	public void setLocation(String location) throws ProcessActiveException {
		if(_isActive) {
			throw new ProcessActiveException();
		}
		_location=location;
	}
	
	/**
	 * @see Process#getProcessId()
	 */
	public String getProcessId() throws InactiveProcessException {
		if(_processId==null) {
			throw new InactiveProcessException();
		}
		return _processId;
	}
	
	/**
	 * @see AgentListener#agentActive
	 */
	public void agentActive(Agent agent) {
		if(!_agents.contains(agent)) {
			_agents.add(agent);
		}
	}
	
	/**
	 * @see AgentListener#agentInactive
	 */
	public void agentInactive(Agent agent) {
		/* Should we remove this from the agent list? */
	}
	
	/**
	 * @see AgentListener#error
	 */
	public void error(Agent agent, String errorId, String errorMessage) {
		/* Ignore */
	}

	/**
	 * @see AgentListener#handleCommand
	 */
	public void handleCommand(Agent agent, CommandElement command) {
		/* Ignore */
	}
	
	/**
	 * @see ConnectionListener#connectionClosed
	 */
	public void connectionClosed(Connection connection) {	
		/* If there is a pending launch lock we need to release it as our connection has been lost */
		synchronized(_eclipseLock) {
			synchronized(_launcherLock) {
				_launcherLock.notify();
				if(!_requireEclipseLock) {	
					return;
				}
			}
		
			try {
				_eclipseLock.wait();
			}
			catch(InterruptedException e) {
				/* We may need to handle this exception */
			}
		}
	}
	
	
	public String getlocation() {
		return _location;
	}
	
	/**
	 * @see Process#getUUID()
	 */
	public void setUUID(String uuid) {
		_UUID=uuid;	
	}
	
	
	protected void handleCommand(CommandElement command) {
		switch((int)command.getTag()) {
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_LAUNCHED:
			/* Only process a ProcessLaunched command once */
			if(_isActive) {
				break;	
			}
			
			/* Extract all the information for this process */
			ProcessLaunchedCommand plc=(ProcessLaunchedCommand)command;
			_processId=new Long(plc.getProcessId()).toString();
			_UUID=plc.getProcessUUID();
			
			/* Capture all the environment variables */
			_variables.removeAllElements();
			String[] vars=plc.getEnvironment();
			for(int i=0; i<vars.length; i++) {
				int equalsOffset=vars[i].indexOf('=');
				if(equalsOffset>0) {
					String name=vars[i].substring(0, equalsOffset);
					String value=vars[i].substring(equalsOffset+1);
					_variables.addElement(new Variable(name, value));
				}
			}
			
			_isActive=true;
			
			/* Notify that the process is launched so "launch()" can return */
			synchronized(_eclipseLock) {
				synchronized(_launcherLock) {
					_launcherLock.notify();
					if(!_requireEclipseLock) {	
						break;
					}
				}
			
				try {
					_eclipseLock.wait();
				}
				catch(InterruptedException e) {
					/* We may need to handle this exception */
				}
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_ACTIVE:
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_INACTIVE:
			/* When an agent becomes active/inactive we need to ensure it is a member
			   of the agent list for this process.  We also need to inform
			   all the listeners for this agent that it is active/inactive.  This
			   is accomplished by delegating to the handleCommand() of the AgentImpl
			*/
			DetailedAgentInfoCommand aac=(DetailedAgentInfoCommand)command;
			synchronized(_agents) {
				Agent agent=null;
				Enumeration enum=_agents.elements();
				while(enum.hasMoreElements()) {
					Agent current=(Agent)enum.nextElement();
					if(current.getName().equals(aac.getAgentName())) {
						agent=current;
						break;
					}
				}
				if(agent==null) {
					agent=AgentFactory.createAgent(this, aac.getAgentName(), aac.getAgentType());
					((AgentImpl)agent).setUUID(aac.getAgentUUID());
				}
				
				/* Forward this command to the agent so it can handle the details */
				try {
					AgentImpl agentImpl=(AgentImpl)agent;
					if(agent.getUUID()==null) {
						agentImpl.setUUID(aac.getAgentUUID());
					}
					agentImpl.handleCommand(command);
					
				}
				catch(ClassCastException e) {
					/* This is not an instance of AgentImpl, do not attempt to notify it is active */
				}
				
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_ERROR_STRING:
			/* When we are launching a process there could be an error */
			ErrorCommand ec=(ErrorCommand)command;
			if(ec.getErrorId().equals("RAC002")) {
				_noSuchAliasExceptionThrown=true;
				synchronized(_launcherLock) {
					_launcherLock.notify();
				}
			}
			break;
		case (int)org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_EXITED:
			if(!_isComplete) {
				_isComplete=true;
				synchronized(_listeners) {
					Enumeration enum=_listeners.elements();
					while(enum.hasMoreElements()) {
						((ProcessListener)enum.nextElement()).processExited(this);
					}
				}
			}
			break;
		default:
			if(command instanceof SimpleAgentInfoCommand) {
				/* Custom commands can be generated by agents running on this process.  Find
			       the proper agent and forward the message on to it.  This gets invoked only
			       when a launch is done not attach, as these are routed directly to the agent.
			    */
				Agent agent=getAgent(((SimpleAgentInfoCommand)command).getAgentName());
				try {
					if(agent!=null) {
						((AgentImpl)agent).handleCommand(command);	
					}
				}
				catch(ClassCastException e) {
					/* This isn;t our impl so we can ignore the command */
				}
			}
		}		
	}

}

