/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jface.action.*;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.part.*;

public abstract class TraceViewer
	extends ViewPart
	implements IMenuListener, ISelectionListener, IProfileEventListener, IDeleteListener {

	protected boolean _initializedMenu = false;
	protected Hashtable _pages = new Hashtable();
	private PageBook book;
	private IPage currentPage;
	private IPage defaultPage;

	protected Menu fContextMenu;

	public final String GROUP_ADD_VIEWS = "open.views";
	public final String GROUP_GENERIC = "generic.action";
	
	/*
	 * Keep track of the refresh events ignored because the
	 * page view not visible 
	 */
	 protected boolean fRefreshView = false;
	
	/*
	 * Keep track of the selection changed event ignored because the
	 * page view not visible 
	 */	
	 protected boolean fSelectionChanged = false;
	
	/*
	 * Ensure refresh events being processed only if
	 * this part is visible.
	 */
	protected boolean fProcessRefreshEvents= true;

	protected IPartListener2 fPartListener= new IPartListener2() {
		public void partActivated(IWorkbenchPartReference ref) {
		}
		public void partBroughtToTop(IWorkbenchPartReference ref) {
		}
		public void partInputChanged(IWorkbenchPartReference ref) {
		}
		public void partClosed(IWorkbenchPartReference ref) {
		}
		public void partDeactivated(IWorkbenchPartReference ref) {
		}
		public void partOpened(IWorkbenchPartReference ref) {
		}
		public void partVisible(IWorkbenchPartReference ref) {
			
			if(ref != null && ref.getPart(false) == TraceViewer.this)
			{
				fProcessRefreshEvents = true;	
				//refresh the visible page
				refreshVisiblePage();				
			}
		}
		public void partHidden(IWorkbenchPartReference ref) {
			
			if(ref != null && ref.getPart(false) == TraceViewer.this)			
			{
				fProcessRefreshEvents = false;					
			}
		}
	};
    
	/**
	 * TraceViewer constructor comment.
	 */
	public TraceViewer() {
		super();

	}

	/**
	 * Add a page to the workbook. If the mofObject is null, show default empty page
	 */
	public void addViewPage(EObject object) {
		if (book == null) {
			//the widget was disposed...
			//remove the view from the pd listeners
			UIPlugin.getDefault().removeSelectionListener(this);			
			return;
		}

		EObject mofObject = getObjectToView(object);
		setViewTitle(mofObject);

		if (mofObject == null || !isValidObject(mofObject)) {
			showPage(defaultPage);
			return;
		}

		boolean newPage = false;
		TraceViewerPage page = getPage(mofObject);
		if(defaultPage == page)
		  return;
		  
		if (page == null) {
			
			//remove existing pages
			Enumeration pages = _pages.elements();
			while(pages.hasMoreElements())
			{
				((IPage)pages.nextElement()).dispose();
			}
			_pages.clear();

			makeActions();
			page = createPage(mofObject);
			page.createControl(book);

			_pages.put(mofObject, page);
			newPage = true;

		}

		showPage(page);

        page.update(newPage);
	}

	/*
	 * Returns the object that should be viewd if
	 * true if the selObject is selected in the monitors view
	 * For example, if the process node is selected, the view should display
	 * the profiling data if trace collects profiling data
	 * 
	 */
	public EObject getObjectToView(EObject selObject) {
		return selObject;
	}

	/*
	 * Returns true if the object can be viewed using this type of view 
	 * 
	 */
	abstract public boolean isValidObject(Object object);

	/**
	 * Returns the default page.
	 *
	 * @return the default page.
	 */
	protected IPage createDefaultPage(PageBook book) {
		DefaultPage page = new DefaultPage();
		page.createControl(book);
		page.setMessage(UIPlugin.getResourceString("STR_NODATA"));
		return page;
	}
	public abstract TraceViewerPage createPage(EObject mofObject);
	/**
	 * Implementation of IDesktopPart.createPartControl(Composite)
	 */
	public void createPartControl(Composite parent) {
		// Create the page book.
		book = new PageBook(parent, SWT.NONE);

		// Create the default page. 
		defaultPage = createDefaultPage(book);

		//ask the view to display the data for the curently selected object	
		addViewPage(getMofObject());

		UIPlugin.getDefault().addSelectionListener(this);
		UIPlugin.getDefault().addProfileEventListener(this);
		UIPlugin.getDefault().addDeleteListener(this);
		
		getViewSite().getPage().addPartListener(fPartListener);		
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (2/19/2001 2:52:36 PM)
	 * @return com.ibm.eclipse.ui.part.IPage
	 */
	public IPage getCurrentPage() {
		if (currentPage == null || !(currentPage instanceof TraceViewerPage))
			return null;

		return currentPage;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/01/2000 4:01:53 PM)
	 */
	public TraceViewerPage getPage(EObject mofObject) {
		
		Object page = _pages.get(mofObject);
		if (page != null)
			return (TraceViewerPage) page;

		return null;
	}
	/**
	 * 
	 */
	public void dispose() {
		UIPlugin.getDefault().removeSelectionListener(this);
		UIPlugin.getDefault().removeProfileEventListener(this);
		UIPlugin.getDefault().removeDeleteListener(this);

		getSite().getPage().removePartListener(fPartListener);
			
		_pages.clear();
		
		if(book != null)
		{
			book.dispose();
			book = null;
		}
		
		if(currentPage != null)
		{		
			currentPage.dispose();
			currentPage = null;
		}
		
		
		if(defaultPage != null)
		{
			defaultPage.dispose();
			defaultPage = null;
		}
		
	    fPartListener = null;
	    
		super.dispose();
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (2/19/2001 3:37:02 PM)
	 * @param init boolean
	 */
	public void initializedMenu(boolean init) {
		_initializedMenu = init;
	}
	/**
	 * Answer true if we want to track the given page.
	 */
	protected boolean isImportant(IWorkbenchPart part) {
		//We only care about editors
		return true;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (2/19/2001 3:37:34 PM)
	 * @return boolean
	 */
	public boolean isInitializedMenu() {
		return _initializedMenu;
	}
	/**
	 * isSaveNeeded method comment.
	 */
	public boolean isSaveNeeded() {
		return false;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (1/11/2001 2:54:24 PM)
	 */
	public void makeActions() {
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (6/5/2001 2:38:58 PM)
	 * @param obj java.lang.Object
	 */
	protected void refreshPage(Object obj) {
		
		if(_pages == null)
		  return;
		  
		Object page = _pages.get(obj);

		if (page != null && page instanceof TraceViewerPage) {
			((TraceViewerPage) page).refreshPage();
		}

	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (6/5/2001 2:38:58 PM)
	 * @param obj java.lang.Object
	 */
	protected void selectionChanged(Object obj) {
		
		if(obj == null)
		  return;
		  
		Object page = _pages.get(obj);

		if (page != null && page instanceof TraceViewerPage) {
			((TraceViewerPage) page).selectionChanged();
		}

	}
	
	/**
	 * Insert the method's description here.
	 * Creation date: (08/01/2000 4:01:53 PM)
	 */
	public void removePage(Object obj) {
		
		if(obj == null || !(obj instanceof EObject))
		  return;
		  
		Object page = _pages.get(getObjectToView((EObject)obj));
		_pages.remove(obj);

		if (page != null) {
			if (page == currentPage)
				showPage(defaultPage);

			((IPage) page).dispose();
		}
	}
	/**
	 * Asks the part to take focus within the desktop.
	 */
	public void setFocus() {
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (2/19/2001 2:53:35 PM)
	 * @param page com.ibm.eclipse.ui.part.IPage
	 */
	public void showPage(IPage page) {
		book.showPage(page.getControl());
		currentPage = page;

	}

	/*
	 * Selection has changed in the Monitors view (PDProjectExplorer)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		
		if(!fProcessRefreshEvents || part == null
			|| part.getSite().getWorkbenchWindow() != getSite().getWorkbenchWindow())
			{
				if(!fProcessRefreshEvents)
				   fSelectionChanged = true;
				   
				return; //get notifications only from the current workbench window
				
			}
	
		if (selection == null
			|| selection.isEmpty()
			|| !(selection instanceof IStructuredSelection)) {
			//show empty page
			addViewPage((EObject) null);
			return;
		}
	
		Object obj = ((IStructuredSelection) selection).getFirstElement();
		if (obj == null || !(obj instanceof EObject))
			addViewPage((EObject) null);
		else
			addViewPage((EObject) obj);
	}

	public void init(IViewSite site) throws PartInitException {

		super.init(site);
	}
	public void init(IViewSite site, IMemento memento) throws PartInitException {

		super.init(site, memento);
	}
	
	/*
	 * Initial view title
	 */
	 public abstract String getViewTitle();
	
	public void handleProfileEvent(ProfileEvent event) {
		
		int type = event.getType();
		
		if(!fProcessRefreshEvents)
		{
			
			//ignore the events and keep track that the page need to be updated			
			if(type == ProfileEvent.REFRESH_VIEWS)
			   fRefreshView = true;
			   
			return;
		}
		
		
		if(type == ProfileEvent.REFRESH_VIEWS
		  && event.getSource() instanceof EObject)
	    {
			refreshPage(getObjectToView((EObject)event.getSource()));	    	
	    }
	}
	
	protected void setViewTitle(Object selection)
	{
		if(selection != null)
		{
			if(selection instanceof TRCAgentProxy)
			{
				TRCAgentProxy a = (TRCAgentProxy)selection;
				
				  setTitle(getViewTitle() + " - " + HyadesUtil.getProcessName(a.getProcessProxy()));
				  return;
				
			}
		}
		setTitle(getViewTitle());		
	}
	
	public void initializeActionBar()
 	{
 	}
	
	public void setRecordSelection(EObject record, EObject treeSel){
	};

	/**
	 * Insert the method's description here.
	 * Creation date: (10/19/2000 12:05:47 PM)
	 * @return java.lang.Object
	 */
	private EObject getMofObject() {
		IWorkbenchPage page =
			UIPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage();
		if (page == null
			|| !page.getPerspective().getId().equals(PDPerspective.ID_TRACE_PERSPECTIVE))
			return null;

		IViewPart monitorsView = page.findView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
		if (monitorsView == null || !(monitorsView instanceof PDProjectExplorer))
			return null;

		TreeItem item = null;
		PDProjectViewer viewer = ((PDProjectExplorer) monitorsView).getViewer();
		if (viewer != null)
			item = viewer.getTreeSelection();

		if (item == null || !(item.getData() instanceof EObject))
			return null;

		return (EObject) item.getData();
	}
	
	/**
	 * @return
	 */
	public boolean isProcessRefreshEvents() {
		return fProcessRefreshEvents;
	}
	
	/**
	 * Refresh the visible page when the view has become visible
	 *
	 */
	private void refreshVisiblePage()
	{		
		EObject obj = getMofObject();
		if(fSelectionChanged)
		{
			addViewPage(obj);
			fSelectionChanged = false;
		}

		EObject sel = getObjectToView(obj);				
		if(fRefreshView )
		{//the view input should be updated
			refreshPage(sel);
			fRefreshView = false;			
		}
		selectionChanged(sel);
	}

    /**
     * @see IMenuListener
     */
	public void menuAboutToShow(IMenuManager menu) {
		    
	   if(getCurrentPage() != null && getCurrentPage() instanceof TraceViewerPage)
	   {
		   ((TraceViewerPage)getCurrentPage()).fillContextMenu(menu);	
	   }
	   
	   menu.add(new Separator(GROUP_ADD_VIEWS));	
	   menu.add(new Separator(GROUP_GENERIC));	
	   
	   menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));
	   
	}

	/**
	 * Invoked when a top hierarchy object is deleted from the  model.
	 */
	public void deregister(Object obj)
	{
        removePage(obj);
	}

}
