/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.actions;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.actions.SelectionProviderAction;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;

/*
* CONTEXT_ID save0000 for save to local action
*/
public class SaveAction
	extends SelectionProviderAction
	implements ISelectionChangedListener, IRunnableWithProgress {
	private PDProjectExplorer fViewer;
	private Object[] fSelections;
	private IResource fResSelection;
	private IResource[] fResSelections;
	private Shell fShell;
	private IStatus status;

	private final String MONITOR =
		UIPlugin.getResourceString("SAVE_MONITOR_TEXT");
	private final String NODE = UIPlugin.getResourceString("SAVE_NODE_TEXT");
	private final String PROCESS =
		UIPlugin.getResourceString("SAVE_PROCESS_TEXT");
	private final String AGENT = UIPlugin.getResourceString("SAVE_AGENT_TEXT");

	private ArrayList processedURIs = new ArrayList();

	public SaveAction(PDProjectExplorer viewer) {
		super(viewer.getViewer(), UIPlugin.getResourceString("SAVE"));
		setDescription(UIPlugin.getResourceString("SAVE"));
		fViewer = viewer;

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			this,
			UIPlugin.getPluginId() + ".save0000");

	}
	public void run() {
		String title = UIPlugin.getResourceString("TRACE_MSG");
		String msg = "";

		IStructuredSelection selection =
			(IStructuredSelection) fViewer.getViewer().getSelection();
		fSelections = selection.toArray();
		fShell = fViewer.getShell();

		Item[] items = fViewer.getViewer().getTreeSelections();

		if (items.length == 1
			&& items[0].getData() != null
			&& items[0].getData() instanceof IProject)
			msg = UIPlugin.getResourceString("SAVE_PROJECT_Q");
		else
			msg = UIPlugin.getResourceString("SAVE_RES_Q");

		if (!MessageDialog
			.openQuestion(
				fViewer.getShell(),
				UIPlugin.getResourceString("SAVTOLOCACT_DIA_TITLE"),
				msg))
			return;

		fResSelections = new IResource[items.length];
		for (int j = 0; j < items.length; j++) {
			fResSelections[j] =
				fViewer.getSelectionFolder((TreeItem) (items[j]));
		}

		fResSelection = fResSelections[0];
		processedURIs.clear();

		try {
			new ProgressMonitorDialog(fShell).run(
				true,
				true,
				new WorkspaceModifyDelegatingOperation(this));
		} catch (InterruptedException e) {
			try {
				if (fResSelection != null && fResSelection.exists())
					fResSelection.getProject().refreshLocal(
						IResource.DEPTH_INFINITE,
						null);
			} catch (CoreException exc) {
				exc.printStackTrace();
			}

			fResSelection = null;
			fResSelections = null;

			return;

		} catch (InvocationTargetException e) {
			e.printStackTrace();

			fResSelection = null;
			fResSelections = null;

			return;
		}

		try {
			if (fResSelection != null && fResSelection.exists())
				fResSelection.getProject().refreshLocal(
					IResource.DEPTH_INFINITE,
					null);
		} catch (CoreException exc) {
			exc.printStackTrace();
		}

		fResSelection = null;
		fResSelections = null;

	}
	/**
	*	Run self's Operation code, utilizing the passed progress monitor
	*
	*	@param monitor com.ibm.itp.common.IProgressMonitor
	*/
	public void run(IProgressMonitor monitor) {
		ArrayList monitors =
			PDContentProvider.getMonitors((IContainer) fResSelection);

		int tasks = SaveUtil.documentsSize();
		if (monitors.size() != 0) {
			tasks = tasks / monitors.size();
		}

		monitor.beginTask(UIPlugin.getResourceString("SAVE"), tasks);

		Object selection = null;
		for (int i = 0; i < fSelections.length; i++) {
			selection = fSelections[i];
			fResSelection = fResSelections[i];

			if (selection instanceof IContainer) {
				saveContainer((IContainer) selection, monitor);
			} else if (selection instanceof TRCMonitor) {
				TRCMonitor trcMonitor = (TRCMonitor) selection;

				if (validForSave(trcMonitor)) {
					try {
						if (!processedURIs
							.contains(trcMonitor.eResource().getURI())) {
							SaveUtil.save(trcMonitor, monitor);
							processedURIs.add(trcMonitor.eResource().getURI());
						}
					} catch (Exception e) {
						treatSaveException(
							MONITOR,
							trcMonitor.getName(),
							trcMonitor.eResource());
					}
				}
			} else if (selection instanceof TRCNode)
				saveNode((TRCNode) selection, monitor);
			else if (selection instanceof TRCProcessProxy)
				saveProcess((TRCProcessProxy) selection, monitor);
			else if (selection instanceof TRCAgentProxy)
				saveAgent((TRCAgentProxy) selection, monitor);
		}

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (9/14/2001 10:32:44 AM)
	 * @param agent com.ibm.etools.perftrace.TRCAgent
	 */
	private void saveAgent(TRCAgentProxy agent, IProgressMonitor monitor) {
		TRCProcessProxy process = agent.getProcessProxy();
		TRCNode node = process.getNode();
		TRCMonitor mon = node.getMonitor();

		if (isResourcesReadOnly(mon.eResource())) {
			treatSaveReadOnlyException(mon.getName(), mon.eResource());
			return;
		} else if (isResourcesReadOnly(node.eResource())) {
			treatSaveReadOnlyException(node.getName(), node.eResource());
			return;
		} else if (isResourcesReadOnly(process.eResource())) {
			treatSaveReadOnlyException(process.getName(), process.eResource());
			return;
		} else if (isResourcesReadOnly(agent.eResource())) {
			treatSaveReadOnlyException(agent.getName(), agent.eResource());
			return;
		}
		if ((agent.isActive()) && (agent.isMonitored())) {
			treatActiveAgentWhenSave(agent);
			return;
		}

		//save the monitor
		try {
			if (!processedURIs.contains(mon.eResource().getURI())) {
				SaveUtil.saveResource(monitor, mon.eResource());
				processedURIs.add(mon.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(MONITOR, mon.getName(), mon.eResource());
		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the node
		try {
			if (!processedURIs.contains(node.eResource().getURI())) {
				SaveUtil.saveResource(monitor, node.eResource());
				processedURIs.add(node.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(NODE, node.getName(), node.eResource());
		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the process
		try {
			if (!processedURIs.contains(process.eResource().getURI())) {
				SaveUtil.saveResource(monitor, process.eResource());
				processedURIs.add(process.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(PROCESS, process.getName(), process.eResource());
		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the agent (agentProxy already saved with the processProxy)
		try {
			if (agent.getAgent() != null) {
				if (!processedURIs
					.contains(agent.getAgent().eResource().getURI())) {
					SaveUtil.saveResource(
						monitor,
						agent.getAgent().eResource());
					processedURIs.add(agent.getAgent().eResource().getURI());
				}
			}
		} catch (Exception Exc) {
			treatSaveException(AGENT, agent.getName(), agent.eResource());
		}
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (9/14/2001 10:21:22 AM)
	 * @param container org.eclipse.core.resources.IContainer
	 */
	private void saveContainer(
		IContainer container,
		IProgressMonitor monitor) {

		try {
			IResource[] resources = container.members();
			for (int j = 0; j < resources.length; j++) {
				if (monitor.isCanceled())
					throw new OperationCanceledException();
				if (resources[j] instanceof IContainer) {
					saveContainer((IContainer) resources[j], monitor);
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
			return;
		}

		ArrayList monitors = PDContentProvider.getMonitors(container);
		for (int idx = 0; idx < monitors.size(); idx++) {
			if (monitor.isCanceled())
				throw new OperationCanceledException();

			Object mon = monitors.get(idx);
			if (mon instanceof TRCMonitor) {
				TRCMonitor trcMonitor = (TRCMonitor) mon;
				if (validForSave(trcMonitor)) {
					try {
						if (!processedURIs
							.contains(trcMonitor.eResource().getURI())) {
							SaveUtil.save(trcMonitor, monitor);
							processedURIs.add(trcMonitor.eResource().getURI());
						}
					} catch (Exception e) {
						treatSaveException(
							MONITOR,
							trcMonitor.getName(),
							trcMonitor.eResource());
					}
				}

			}
		}
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (9/14/2001 10:32:01 AM)
	 * @param node com.ibm.etools.perftrace.TRCNode
	 */
	private void saveNode(TRCNode node, IProgressMonitor monitor) {
		TRCMonitor mon = node.getMonitor();

		if (isResourcesReadOnly(mon.eResource())) {
			treatSaveReadOnlyException(mon.getName(), mon.eResource());
			return;
		} else if (isResourcesReadOnly(node.eResource())) {
			treatSaveReadOnlyException(node.getName(), node.eResource());
			return;
		}

		Object[] ps = node.getProcessProxies().toArray();
		for (int i = 0; i < ps.length; i++) {
			TRCProcessProxy p = (TRCProcessProxy) ps[i];
			if (p.eIsProxy())
				continue;
			if (isResourcesReadOnly(p.eResource())) {
				treatSaveReadOnlyException(p.getName(), p.eResource());
				return;
			}

			Object[] as = p.getAgentProxies().toArray();
			for (int k = 0; k < as.length; k++) {
				TRCAgentProxy a = (TRCAgentProxy) as[k];
				if (a.eIsProxy())
					continue;

				if (isResourcesReadOnly(a.eResource())) {
					treatSaveReadOnlyException(a.getName(), a.eResource());
					return;
				}
				if ((a.isActive()) && (a.isMonitored())) {
					treatActiveAgentWhenSave(a);
					return;
				}
			}
		}

		//save the monitor
		try {
			if (!processedURIs.contains(mon.eResource().getURI())) {
				SaveUtil.saveResource(monitor, mon.eResource());
				processedURIs.add(mon.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(MONITOR, mon.getName(), mon.eResource());
		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the node
		try {
			if (!processedURIs.contains(node.eResource().getURI())) {
				SaveUtil.saveResource(monitor, node.eResource());
				processedURIs.add(node.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(NODE, node.getName(), node.eResource());
		}

		//save processes under the node
		Iterator processes = node.getProcessProxies().iterator();
		while (processes.hasNext()) {
			if (monitor.isCanceled())
				throw new OperationCanceledException();

			TRCProcessProxy process = (TRCProcessProxy) processes.next();
			if (process.eIsProxy())
				continue;

			try {
				if (!processedURIs.contains(process.eResource().getURI())) {
					SaveUtil.saveResource(monitor, process.eResource());
					processedURIs.add(process.eResource().getURI());
				}
			} catch (Exception Exc) {
				treatSaveException(
					PROCESS,
					process.getName(),
					process.eResource());
			}

			//save agents
			Iterator agents = process.getAgentProxies().iterator();
			while (agents.hasNext()) {
				if (monitor.isCanceled())
					throw new OperationCanceledException();

				TRCAgentProxy agent = (TRCAgentProxy) agents.next();
				if (agent.eIsProxy())
					continue;
				try {
					if (agent.getAgent() != null) {

						if (!processedURIs
							.contains(agent.getAgent().eResource().getURI())) {
							SaveUtil.saveResource(
								monitor,
								agent.getAgent().eResource());
							processedURIs.add(
								agent.getAgent().eResource().getURI());
						}
					}
				} catch (Exception Exc) {
					treatSaveException(
						AGENT,
						agent.getName(),
						agent.eResource());
				}
			}
		}

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (9/14/2001 10:32:27 AM)
	 * @param process com.ibm.etools.perftrace.TRCProcessProxy
	 */
	private void saveProcess(
		TRCProcessProxy process,
		IProgressMonitor monitor) {
		TRCNode node = process.getNode();
		TRCMonitor mon = node.getMonitor();

		if (isResourcesReadOnly(mon.eResource())) {
			treatSaveReadOnlyException(mon.getName(), mon.eResource());
			return;
		} else if (isResourcesReadOnly(node.eResource())) {
			treatSaveReadOnlyException(node.getName(), node.eResource());
			return;
		} else if (isResourcesReadOnly(process.eResource())) {
			treatSaveReadOnlyException(process.getName(), process.eResource());
			return;
		}

		Object[] as = process.getAgentProxies().toArray();
		for (int k = 0; k < as.length; k++) {
			TRCAgentProxy a = (TRCAgentProxy) as[k];
			if (a.eIsProxy())
				continue;

			if (isResourcesReadOnly(a.eResource())) {
				treatSaveReadOnlyException(a.getName(), a.eResource());
				return;
			}
			if ((a.isActive()) && (a.isMonitored())) {
				treatActiveAgentWhenSave(a);
				return;
			}
		}

		//save the monitor
		try {
			if (!processedURIs.contains(mon.eResource().getURI())) {
				SaveUtil.saveResource(monitor, mon.eResource());
				processedURIs.add(mon.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(MONITOR, mon.getName(), mon.eResource());

		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the node
		try {
			if (!processedURIs.contains(node.eResource().getURI())) {
				SaveUtil.saveResource(monitor, node.eResource());
				processedURIs.add(node.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(NODE, node.getName(), node.eResource());
		}

		if (monitor.isCanceled())
			throw new OperationCanceledException();

		//save the process
		try {
			if (!processedURIs.contains(process.eResource().getURI())) {
				SaveUtil.saveResource(monitor, process.eResource());
				processedURIs.add(process.eResource().getURI());
			}
		} catch (Exception Exc) {
			treatSaveException(PROCESS, process.getName(), process.eResource());
		}

		//save agents
		Iterator agents = process.getAgentProxies().iterator();
		while (agents.hasNext()) {
			if (monitor.isCanceled())
				throw new OperationCanceledException();

			TRCAgentProxy agent = (TRCAgentProxy) agents.next();
			if (agent.eIsProxy())
				continue;

			try {
				if (agent.getAgent() != null) {

					if (!processedURIs
						.contains(agent.getAgent().eResource().getURI())) {
						SaveUtil.saveResource(
							monitor,
							agent.getAgent().eResource());
						processedURIs.add(
							agent.getAgent().eResource().getURI());
					}
				}
			} catch (Exception Exc) {
				treatSaveException(AGENT, agent.getName(), agent.eResource());
			}
		}

	}
	public void selectionChanged(IStructuredSelection selection) {
		setEnabled(true);
	}
	private void treatSaveException(
		String objectType,
		String objectName,
		Resource res) {
		if (!processedURIs.contains(res.getURI())) {
			processedURIs.add(res.getURI());
		} else
			return;

		String text = UIPlugin.getResourceString("SAVE_FILE_ERROR_");
		text = TString.change(text, "%1", objectType);

		if (res != null)
			text = TString.change(text, "%2", res.getURI().toString());
		else
			text = TString.change(text, "%2", objectName);

		if (fShell != null) {
			final Status err =
				new Status(
					Status.WARNING,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					text,
					null);
			fShell.getDisplay().asyncExec(new Runnable() {
				public void run() {
					ErrorDialog.openError(
					fShell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			});
		}
	}

	private void treatSaveReadOnlyException(String objectName, Resource res) {

		if (!processedURIs.contains(res.getURI())) {
			processedURIs.add(res.getURI());
		} else
			return;
		String text = UIPlugin.getResourceString("SAVE_READ_ONLY_FILE_ERROR_");
		if (res != null)
			text = TString.change(text, "%2", res.getURI().toString());
		if (objectName != null)
			text = TString.change(text, "%1", objectName);

		if (fShell != null) {
			final Status err =
				new Status(
					Status.WARNING,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					text,
					null);
			fShell.getDisplay().asyncExec(new Runnable() {
				public void run() {
					ErrorDialog.openError(
					    fShell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			});
		}

	}

	private boolean validForSave(TRCMonitor trcMonitor) {
		if (isResourcesReadOnly(trcMonitor.eResource())) {
			treatSaveReadOnlyException(
				trcMonitor.getName(),
				trcMonitor.eResource());
			return false;
		}

		Object[] ns = trcMonitor.getNodes().toArray();
		for (int j = 0; j < ns.length; j++) {
			TRCNode n = (TRCNode) ns[j];
			if (n.eIsProxy())
				continue;

			if (isResourcesReadOnly(n.eResource())) {
				treatSaveReadOnlyException(n.getName(), n.eResource());
				return false;
			}

			Object[] ps = n.getProcessProxies().toArray();
			for (int i = 0; i < ps.length; i++) {
				TRCProcessProxy p = (TRCProcessProxy) ps[i];
				if (p.eIsProxy())
					continue;

				if (isResourcesReadOnly(p.eResource())) {
					treatSaveReadOnlyException(p.getName(), p.eResource());
					return false;
				}

				Object[] as = p.getAgentProxies().toArray();
				for (int k = 0; k < as.length; k++) {
					TRCAgentProxy a = (TRCAgentProxy) as[k];
					if (a.eIsProxy())
						continue;

					if (isResourcesReadOnly(a.eResource())) {
						treatSaveReadOnlyException(a.getName(), a.eResource());
						return false;
					}
					if ((a.isActive()) && (a.isMonitored())) {
						treatActiveAgentWhenSave(a);
						return false;
					}
				}
			}
		}
		return true;
	}

	private boolean isResourcesReadOnly(Resource rsc) {
		IPath path = new org.eclipse.core.runtime.Path(rsc.getURI().toString());
		IResource foundrsc =
			ResourcesPlugin.getWorkspace().getRoot().findMember(path);
		if ((foundrsc != null) && (foundrsc.isAccessible()))
			return foundrsc.isReadOnly();

		return false;
	}

	private void treatActiveAgentWhenSave(TRCAgentProxy a) {
		if (a.getAgent() == null)
			return;
		Resource res = a.getAgent().eResource();
		if (!processedURIs.contains(res.getURI())) {
			processedURIs.add(res.getURI());
		} else
			return;

		String text = UIPlugin.getResourceString("SAVE_ACTIVE_AGENT_ERROR_");
		text = HyadesUtil.change(text, "%1", a.getName());

		if (fShell != null) {
			final Status err =
				new Status(
					Status.WARNING,
					ResourcesPlugin.PI_RESOURCES,
					IResourceStatus.INTERNAL_ERROR,
					text,
					null);
			fShell.getDisplay().asyncExec(new Runnable() {
				public void run() {
					ErrorDialog.openError(
						fShell,
						UIPlugin.getResourceString("TRACE_MSG"),
						UIPlugin.getResourceString("SAVE_FILE_ERROR_TITLE"),
						err);
				}
			});
		}
	}

	public void dispose() {
		
		super.dispose();
		
		fViewer = null;
		fResSelection = null;
		fResSelections = null;
		fShell = null;
		fSelections = null;
	}

}
