/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.console;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.internal.preferences.*;
import org.eclipse.jface.text.*;
import org.eclipse.jface.util.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.widgets.Composite;

/*
* CONTEXT_ID trcv0000 for trace console viewer
*/

public class TraceConsoleViewer extends TextViewer
								implements IPropertyChangeListener
{

	/**
	 * Font used in the underlying text widget
	 */
	protected Font fFont;

	protected InternalDocumentListener fInternalDocumentListener= new InternalDocumentListener();
	/**
	 * Internal document listener.
	 */
	class InternalDocumentListener implements IDocumentListener {
		/*
		 * @see IDocumentListener#documentAboutToBeChanged
		 */
		public void documentAboutToBeChanged(DocumentEvent e) {
		}
		
		/*
		 * @see IDocumentListener#documentChanged
		 */
		public void documentChanged(DocumentEvent e) {
			TraceConsoleDocument doc= (TraceConsoleDocument) getDocument();
			if (doc == null || doc.isClosed()) {
				return;
			}
			revealEndOfDocument();
			if (doc.isReadOnly()) {
				StyledText widget= getTextWidget();
				widget.setEditable(false);
			}
			updateStyleRanges(doc);
		}
	}

	/**
	 * Creates a new console viewer and adds verification checking
	 * to only allow text modification if the text is being modified
	 * in the editable portion of the underlying document.
	 *
	 * @see org.eclipse.swt.events.VerifyListener
	 */	
	public TraceConsoleViewer(Composite parent) {
		super(parent, SWT.H_SCROLL | SWT.V_SCROLL);
		
		getTextWidget().setDoubleClickEnabled(true);
		getTextWidget().addVerifyListener( new VerifyListener() {
			public void verifyText(VerifyEvent e) {
				TraceConsoleDocument doc= (TraceConsoleDocument) getDocument();
				if (doc != null && doc.getStartOfEditableContent() > e.start) {
					e.doit= false;
				}
			}
		});

	UIPlugin.getDefault().getPreferenceStore().addPropertyChangeListener(this);
	FontData data= TraceConsolePreferencePage.getConsoleFontData();
	fFont= new Font(getControl().getDisplay(), data);
	getTextWidget().setFont(fFont);
		
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		this.getControl(),
		UIPlugin.getPluginId()+".trcv0000");
		
	}
	/**
	 * @see IFindReplaceTarget#canPerformFind
	 */
	protected boolean canPerformFind() {
		return (getTextWidget() != null && getVisibleDocument() != null && getVisibleDocument().getLength() > 0);
	}
	/**
	 * Clears the contents of the current document.
	 */
	public void clearDocument() {
		IDocument doc= getDocument();
		if (doc != null) {
			((TraceConsoleDocument) doc).clearDocument();
		}
	}
	/**
	 * Dispose this viewer and resources
	 */
	protected void dispose() {
		fFont.dispose();
	}
	/**
	 * @see IPropertyChangeListener#propertyChange(PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent event) {
		String propertyName= event.getProperty();
		if (!propertyName.equals(TraceConstants.CONSOLE_FONT)) {
			return;
		}
		FontData data= TraceConsolePreferencePage.getConsoleFontData(); 
		Font temp= fFont;
		fFont= new Font(getControl().getDisplay(), data);
		getTextWidget().setFont(fFont);
		temp.dispose();
	}
	/**
	 * Reveals (makes visible) the end of the current document
	 */
	protected void revealEndOfDocument() {
		IDocument doc= getDocument();
		int docLength= doc.getLength();
		if (docLength > 0) {
			revealRange(docLength - 1, 1);
			StyledText widget= getTextWidget();
			widget.setCaretOffset(docLength);

		}
	}
	/**
	 * @see ITextViewer#setDocument(IDocument)
	 */
	public void setDocument(IDocument doc) {
		TraceConsoleDocument oldDoc= (TraceConsoleDocument) getDocument();
		TraceConsoleDocument document= (TraceConsoleDocument)doc;
		if (oldDoc == null && document == null) {
			return;
		}
		if (oldDoc != null) {
			oldDoc.removeDocumentListener(fInternalDocumentListener);
			oldDoc.setConsoleViewer(null);
			if (oldDoc.equals(document)) {
				document.addDocumentListener(fInternalDocumentListener);
				document.setConsoleViewer(this);
				return;
			}
		}

		super.setDocument(document);
		if (document != null) {
			getTextWidget().setEditable(!document.isReadOnly());
			updateStyleRanges(document);
			revealEndOfDocument();
			document.addDocumentListener(fInternalDocumentListener);
			document.setConsoleViewer(this);
		}
	}
	protected void updateStyleRanges(TraceConsoleDocument doc) {

		StyledText widget= getTextWidget();
		StyleRange[] ranges= doc.getStyleRanges();
		int storeLength= doc.getStore().getLength();
		int lastRange= ranges.length;
		if (lastRange > 0 && !(storeLength - 1 < ranges[lastRange - 1].start)) {
			widget.setStyleRanges(ranges);
		}
	}
}
