/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.collection.profiler;

/**
 * This is a singleton class that is used by the application to manage the Java Profiling
 * Agent.  This class is a singleton and cannot be instantiated by the user.  In order
 * to access the singleton the application must use Profiler.getProfiler()
 * @author: Richard K. Duggan
 */
public final class Profiler {

	
	/* Boolean flag that is set only when the profiler is actively tracing */
	private static boolean _isTracing = true;
	
	/* Boolean flag to determine if the profiler was successful linked. */
	private static boolean _isActive=true;
	
	/* Boolean flag to determine if we are currently profiling */
	private boolean _isProfiling=false;
	
	/* The singleton instance of Profiler */
	private static Profiler _profiler;
	
	/* The global exception indicating why the profiler is unavailable */
	private static ProfilerNotAvailableException _errorCondition;
	
	
	/* Link to the profiling agent.*/
	static {
	
		try {
			System.loadLibrary("piAgent");
			_profiler=new Profiler();
		}
		catch(UnsatisfiedLinkError f) {
			_isActive=false;
			_profiler=null;
			_errorCondition=new ProfilerNotAvailableException("Cannot locate Java Profiling Agent.  Both, the IBM Agent Controller must be"
															  +" installed and the piAgent.dll must be provided as"
			                                                  +" a parameter to the JVM and it must be running in \"application\" mode."
			                                                  +" ie. -XrunpiAgent:server=application must be provided as a parameter to"
			                                                  +" the JVM");
		}
		catch(ProfilerNotAvailableException g) {
			_isActive=false;
			_profiler=null;	
		}
		
	}
	
	
	/**
	 * Don't allow users to make an instance of this class.
	 */
	private Profiler()  throws ProfilerNotAvailableException  { 	
		int result=initialize0();
		
		/* Did we startup OK.  If not set the error in the global exception object */
		if(result<0) {
			_errorCondition=new ProfilerNotAvailableException("The Java Profiling Agent is currently not running in \"application\""
				                                         +" mode. ie. -XrunpiAgent:server=application must be provided as a "
				                                         +" parameter to the JVM");
			throw _errorCondition;
		}
	}
	
	/**
	 * Native delegate to startup the profiler.  Checks to see if profiling is possible.
	 */
	 private native int initialize0();
	
	/**
	 * Getter to retieve the singleton Profiler.
	 */
	public static Profiler getProfiler() throws ProfilerNotAvailableException {
		if(!_isActive) {
			throw _errorCondition;
		}
		return _profiler;  	
	}
	
	
	/**
	 * Start a profiling session.  If a profiling session is currently executing the call is ignored.
	 *
	 * @param currentThreadOnly - the profiling information is by default for the entire process.  If
	 *                             currentThreadOnly is true profiling information will be restricted
	 *                             to the thread that is currently executing the startProfiling method.
	 */
	public void startProfiling(boolean currentThreadOnly) {
		startProfiling(currentThreadOnly, 0);
	}
	
	/**
	 * Start a profiling session.  If a profiling session is currently executing the call is ignored.
	 *
	 * @param currentThreadOnly - the profiling information is by default for the entire process.  If
	 *                             currentThreadOnly is true profiling information will be restricted
	 *                             to the thread that is currently executing the startProfiling method.
	 *
	 * @param boundaryDepth - the profiling information does not extend beyond the number of frames
	 *                         indicated by boundaryDepth.  If the boundary depth is zero or negative
	 *                         then the number of frames is unlimited.
	 */
	public void startProfiling(boolean currentThreadOnly, int boundaryDepth) {
		synchronized(this) {
			if(!_isProfiling) {
				if(startProfiling0(currentThreadOnly, boundaryDepth)==0) {
					_isProfiling=true;
				}
			}
		}
	}
	
	/**
	 * Native delegate of startProfiling(boolean currentThreadOnly, int bondaryDepth).
	 * @returns  0 - the profile session was created successfuly.
	 *
	 */
	private native int startProfiling0(boolean currentThreadOnly, int boundaryDepth);
	
	
	
	/**
	 * Stop profiling the information requested in a previous call to startProfiling.  If there is no
	 * current profiling session the call is ignored.  This call need not be on the same thread as the
	 * startProfiling call irregardless of whether singlethreaded profiling was requested.
	 *
	 */
	public void stopProfiling() {
		synchronized(this) {
			if(_isProfiling) {
				stopProfiling0();
				_isProfiling=false;
			}
		}
	}
	
	/**
	 * Native delgate to stopProfiling.
	 */
	private native void stopProfiling0();

	/**
	  * Mark the heap.  Marking the heap causes all the objects currently in the heap to be marked
	  * so that a subsequent call to analyzeHeap will only apply to the newly created objects between
	  * the two calls.
	  */
	public void markHeap(){
		if(_isActive) {
			markHeap0();
		}
	}
	
	/**
	 * Native delegate for markHeap()
	 */
	private native void markHeap0();
	
	/**
	 * Analyze the heap to determine what new objects have been created in the heap since the last
	 * time the heap was marked or the last time it was analyzed.
	 * @param name - a label that will be applied to this analysis session so that it can be identified
	 *                amongst several analysis sessions.  THIS IS CURRENTLY NOT SUPPORTED.
	 */
	public void analyzeHeap(String name){
		if(_isProfiling) {
			analyzeHeap0(name);
		}
	}
	
	/**
	 * Native delegate of analyzeHeap
	 */
	private  native void analyzeHeap0(String name);
	
	
	
	public void runGC() {
		if(_isActive) {
			runGC0();
		}
	}
	
	private native void runGC0();
	
	/**
	 * Is the profiler actively collecting data.
	 */
	public boolean isProfiling() {
		return _isProfiling;
	}

}
