/*
 * Licensed Material - Property of IBM
 * 5724-D14
 * (C) Copyright IBM Corp. 2002 - All Rights Reserved.
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 */

package org.eclipse.hyades.internal.config.generator;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

public class SetConfig {
	protected static	ResourceBundle myResource;
	protected static	String _racHome; // RASERVER_HOME=
	protected static	String _javaPath; // JAVA_HOME=
	protected static	String _allow; // ALLOW=
	protected static	String _hosts; // HOSTS=
	protected static	String _configFileName;
	protected static	String _startFileName;

	protected static	String _platform; // PLATFORM=

	protected static	boolean help = false;
	protected static	boolean silent = false;
	protected static	boolean verifyValues = true;  // verify the values of the parameters passed on the command line
	protected static	boolean overwrite = false; // -overwrite or ignore an existing config file
	
	public static final String copyright = "Licensed Materials -- Property of IBM\n(c) Copyright International Business Machines Corporation, 2002 - All rights reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.";

	public SetConfig() {
	}

	public static void setRACHome(String path) {
		_racHome = path;
	}
	
	public static String getRACHome() {
		return _racHome;
	}

	public static void setJavaHome(String path) {
		_javaPath = path;
	}
	
	public static String getJavaHome() {
		return _javaPath;
	}

	public static void setPlatform(String platform) {
		_platform = platform;
	}
	
	public static String getPlatform() {
		return _platform;
	}
	
	public static void setAllow(String value) {
		_allow = value;
	}
	
	public static String getAllow() {
		return _allow;
	}	

	public static void setHosts(String value) {
		_hosts = value;
	}
	
	public static String getHosts() {
		return _hosts;
	}

	public static void setConfigFileName(String value) {
		_configFileName = value;
	}
	
	public static String getConfigFileName() {
		return _configFileName;
	}

	public static void setStartFileName(String value) {
		_startFileName = value;
	}
	
	public static String getStartFileName() {
		return _startFileName;
	}

	public static void main(String[] args) {
		// Create a common configuration object
		SetConfigCommon cfg;

		cfg = generationMain(args);
		if(cfg != null) {
			writeConfigFile(cfg);
		}
	}

	public static ResourceBundle getResource() {
		ResourceBundle res = null;

		try {
			res = ResourceBundle.getBundle("org.eclipse.hyades.internal.config.generator.SetConfig");
		}
		catch (MissingResourceException mre) {
			System.out.println("Can't find resource file");
		}
		return res;
	}

	public static SetConfigCommon generationMain(String[] args) {

		myResource = getResource();
		if(myResource == null) {
			return null;
		}

		processParameters(args);

		// Help menu
		if(help) {
			printHelpMenu();
			return null;
		}

		// Verify parameter values if we are not in silent mode
		// or if the user hasn't specified the noverify option
		if (verifyValues || !silent) {
			verifyParameters();
		}
		
		// Get Platform
		if(getPlatform() == null) {
			setPlatform(System.getProperty("os.name", "null"));
		}

		// Make sure we have a RAC Home value
		if (getRACHome() == null) {
			setRACHome(getUserRacHome());
			if (getRACHome() == null) {
				// give up
				return null;
			}
		}
		
		// set config file name
		if(getPlatform().startsWith("Windows")) {		
			setConfigFileName(getRACHome() + "\\config\\serviceconfig.xml");
		}
		else {
			setConfigFileName(getRACHome() + "/config/serviceconfig.xml");
		}

		// Create a common configuration object
		SetConfigCommon cfg = new SetConfigCommon(getPlatform());
		
		// If a config file already exists and we do not want to ignore it
      // then load the configuration from the existing file.
		if (!overwrite && fileExists(getConfigFileName())) {

			try {
				if (!cfg.loadExistingConfiguration(getConfigFileName())) {
					System.out.println(myResource.getString("ErrFileLoad") + "  " + getConfigFileName());
					System.out.println(myResource.getString("ErrFileLoadSuggestion"));
					return null;
				}
			}
			catch (FileError e) {
	            Exception ex = e.get_exception();
	            ResourceBundle xmlResource = ResourceBundle.getBundle("org.apache.xerces.impl.msg.XMLMessages");
	
	            // If we got a Premature EOF SaxParserException then the file is empty or
	            // does not contain any valid XML so create a default configuration
            	if (ex.getMessage() != null && ex.getMessage().equals(xmlResource.getString("PrematureEOF"))) {
					   if (!cfg.createDefaultConfiguration()) {
						   System.out.println(myResource.getString("ErrCreateDefCfg"));
						   return null;
					   }
            	}
            	else {
					   System.out.println(myResource.getString("ErrFileLoad") + "  " + e.get_fileName());	            		
//					   ex.printStackTrace();
					   System.out.println(myResource.getString("ErrFileLoadSuggestion"));
					   return null;
            	}

			}
			
		}
		// Otherwise create a default configuration.
		else {
			if (!cfg.createDefaultConfiguration()) {
				System.out.println(myResource.getString("ErrCreateDefCfg"));
				return null;
			}
		}		


		// Set the RAC Home value if the user specified it or we figured it out.
		if (getRACHome() != null && !getRACHome().equals("")) {
			cfg.setRACHome(getRACHome());
		}
			
		
		if (silent) {
			// update the configuration instance with values passed in

			if (getJavaHome() != null && !getJavaHome().equals("")) {
				cfg.setJAVAHome(getJavaHome());
			}

			if (getAllow() != null && !getAllow().equals("")) {

				if(getAllow().equals("CUSTOM")) {
					// Add the hosts specified by the user on the command line if there are any
					if (getHosts() != null && !getHosts().equals("")) {
						// Get the list of hosts
						List hostList = processStringList(getHosts());
						
						// Remove old allow hosts from configuration instance.
						cfg.removeAllowHosts();
						
						// Iterate over the list, adding each one to the configuration
						Iterator iter = hostList.iterator();
						while (iter.hasNext()) {					
							cfg.addAllowHost((String)iter.next());
						}
					}
				}
				else {
					// Remove old allow hosts from configuration instance.
					cfg.removeAllowHosts();

					// Add "ALL" or "LOCAL" to the configuration instance
					cfg.addAllowHost(getAllow());					
				}
			}

		}
		else
		{
			// Ask user for the values not passed in and list default values from
			// the configuration instance.
			// Then update the configuration instance with the values passed in and
			// those specified by the user on the dialog.			

			/*
			 * JAVA_HOME section
			 */
			if (getJavaHome() == null || getJavaHome().equals("")) {
				String defValue;

				defValue = cfg.getJAVAHome();
				if(defValue.equals("%JAVA_PATH%")) {
					defValue = new String(System.getProperty("java.home") + cfg.ps + "bin"); /* Try java.home prop if no JAVA_PATH is specified */
				}

				if(defValue == null) {
					defValue = cfg.getJAVAHome(); /* If can't resolve java.home prop, use the default */
				}

				cfg.setJAVAHome(getUserJavaPath(defValue));
			}
			else {
				cfg.setJAVAHome(getJavaHome());
			}

			/*
			 * Hosts section
			 */
			String listOfHosts = cfg.getAllowHosts();

			if (getAllow() == null || getAllow().equals("")) {
				// Get hosts from configuration to determine the default allow value
				List hostList = processStringList(listOfHosts);
				// Iterate over the list to determine the allow value
				Iterator iter = hostList.iterator();
				String defAllow = new String("LOCAL");
				
				while (iter.hasNext()) {
					String tmpAllow =(String)iter.next();
					if (tmpAllow.equals("ALL")) {
						defAllow = "ALL";
						break;
					}
					else if (!tmpAllow.equals("LOCAL")) {
						defAllow = "CUSTOM";
						break;
					}
				}				
				setAllow(getUserAllow(defAllow));
			}

			// Remove old allow hosts from configuration instance.
			cfg.removeAllowHosts();

			if(getAllow().equals("CUSTOM")) {
				if (getHosts() == null || getHosts().equals("")) {
					// Ask the user for the list of hosts

					List hostList = processStringList(getUserHosts(listOfHosts));

					// Iterate over the list, adding each one to the configuration
					Iterator iter = hostList.iterator();
					while (iter.hasNext()) {
						cfg.addAllowHost((String)iter.next());
					}
				}
				else {
					List hostList = processStringList(getHosts());

					// Iterate over the list, adding each one to the configuration
					Iterator iter = hostList.iterator();
					while (iter.hasNext()) {
						cfg.addAllowHost((String)iter.next());
					}
				}					
			}
			else {
				// Add "ALL" or "LOCAL" to the configuration instance
				cfg.addAllowHost(getAllow());					
			}
		}

		return cfg;
	}
	
	public static void writeConfigFile(SetConfigCommon cfg) {		
		try {
			cfg.writeConfigFile(getConfigFileName());

			// do not generate RAStart.sh script for Windows
			if(!(getPlatform().startsWith("Windows"))) {
				_startFileName = new String(cfg.getRACHome() + "/bin/RAStart.sh");
				cfg.writeToFile(_startFileName, cfg.getStartSh());
			}
		}
		catch (FileError fe) {
			System.out.println(myResource.getString("ErrFileWrite") + "  " + fe.get_fileName());
		}
		
	}

	public static void printHelpMenu() {
		System.out.println(myResource.getString("CmdUsage"));
		System.out.println("    " + myResource.getString("CmdUsage2"));
		System.out.println();
		System.out.println(myResource.getString("CmdFlags"));
		System.out.println(myResource.getString("CmdFlagHelp"));
		System.out.println(myResource.getString("CmdFlagSilent"));
		System.out.println(myResource.getString("CmdFlagNoVerify"));
		System.out.println();
		System.out.println(myResource.getString("CmdParms"));
		System.out.println(myResource.getString("CmdRAServerHome"));
		System.out.println(myResource.getString("CmdJavaHome"));
		System.out.println(myResource.getString("CmdAllow"));
		System.out.println(myResource.getString("CmdCustom"));
		System.out.println();
		System.out.println(myResource.getString("CmdExampleTitle"));
		System.out.println();
		System.out.println(myResource.getString("CmdExample1Head"));
		System.out.println(myResource.getString("CmdExample1"));
		System.out.println();
		System.out.println(myResource.getString("CmdExample2Head"));
		System.out.println(myResource.getString("CmdExample2"));
		System.out.println();
		System.out.println(myResource.getString("CmdExample3Head"));
		System.out.println(myResource.getString("CmdExample3"));
		System.out.println();
		System.out.println(myResource.getString("CmdExample4Head"));
		System.out.println(myResource.getString("CmdExample4"));

		return;
	}

	public static boolean fileExists(String filename) {

		File file1 = new File(filename);
		
		// Check if file exists and is greater in length than the minimum
		// config file.
		return (file1.exists() && file1.length() > 206);

	}

	public static List processStringList(String list) {
		List strList = new ArrayList();
		if (list == null)
			return strList;
			
		String list2 = list.trim();
				
		if (list2.startsWith("\"") && list2.endsWith("\"")) {
			// remove quotes

			list2 = list2.substring(1,list2.length()-1).trim();
		}

		StringTokenizer st = new StringTokenizer(list2, " ,");
		
		while (st.hasMoreTokens()) {
			strList.add(st.nextToken());
		}
		
		return strList;
		
	}
		

	public static void processParameters(String[] args) {
		String parm = null;
		int i;

		for(i = 0; i < args.length; i++) {
			
			if(args[i].equals("-help") || args[i].equals("-?") || args[i].equals("-h")) {
				help = true;
			}
			else if(args[i].equals("-silent") || args[i].equals("-s")) {
				silent = true;
			}
			else if(args[i].equals("-noverify") || args[i].equals("-nov")) {
				verifyValues = false;
			}
			else if(args[i].equals("-overwrite")) {
				overwrite = true;
			}
			else if(args[i].startsWith("RASERVER_HOME=")) {
				parm = args[i].substring(14).trim();
				setRACHome(parm);
			}
			else if(args[i].startsWith("PLATFORM=")) {
				setPlatform(args[i].substring(9));
			}
			else if(args[i].startsWith("JAVA_HOME=")) {
				parm =  args[i].substring(10).trim();
				setJavaHome(parm);
			}
			else if(args[i].startsWith("ALLOW=")) {
				parm = args[i].substring(6);
				setAllow(parm);
			}
			else if(args[i].startsWith("HOSTS=")) {
				parm = args[i].substring(6);
				setHosts(parm);
			}
			else {
				// unknown parm - ignore it for now
				if (args[i] != null && !args[i].equals(""))
					System.out.println(myResource.getString("WarnUnknownOpt") + " >" + args[i] + "<");
			}
															
		} // for
		
		return;
	}	

	public static void verifyParameters() {
			
		if (getRACHome() != null) {
			setRACHome(verifyRacHome(getRACHome()));
		}
		
		if (getJavaHome() != null) {
			setJavaHome(verifyJavaHome(getJavaHome()));
		}
		
		if (getAllow() != null) {
			setAllow(verifyAllow(getAllow(), false));
		}

		
		return;
	}	

	public static String getUserRacHome() {
		String path0 = null;
		String path1 = null;

		// Try to guess the RAC home based on the current directory.

		path0 = System.getProperty("user.dir", "null");
		
		if (path0 != null) {
//			System.out.println("user.dir system property is " + path0);
			int lastslashindex;
			// check if the path ends in a slash
			if(getPlatform().startsWith("Windows")) {			
				lastslashindex = path0.lastIndexOf('\\');
				if (lastslashindex == path0.length()-1) {
					lastslashindex = path0.lastIndexOf('\\',lastslashindex-1);
				}
			}
			else {				
				lastslashindex = path0.lastIndexOf('/');
				if (lastslashindex == path0.length()-1) {
					lastslashindex = path0.lastIndexOf('/',lastslashindex-1);
				}
			}
			
			if (lastslashindex != -1) {
				// assume we are in the RAC bin dir so go back one dir
				path0 = path0.substring(0, lastslashindex);
//				System.out.println("modified path0 is " + path0);
				path1 = verifyRacHome(path0);
			}
		}
//		else
//			System.out.println("user.dir system property is null");
		
		if (path1 == null && silent) {
			return null;
		}
		
		// Read user input only if no command line arg is present
		while(path1 == null) {
			System.out.print(myResource.getString("AskRAServerHome"));
			try {
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				path0 = in.readLine().trim();
			}
			catch(java.io.IOException ioe) {
				System.out.println("Exception caught during RASERVER_HOME resolution: " + ioe);
			}
			path1 = verifyRacHome(path0);
/*			
			if(path1 == null) {
				System.out.println(myResource.getString("AskRAServerHomeErr"));
			}
*/			
		}

		return path1;
	}

	public static String getUserJavaPath(String defPath) {
		String path0 = null;
		String path1 = null;

		// Read user input
		while(path1 == null) {
			if (defPath == null) {
				System.out.print(myResource.getString("AskJavaHome"));
			}
			else {
				System.out.print(myResource.getString("AskJavaHome") + " (" + defPath + ") ");
			}
			try {
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				path0 = in.readLine().trim();
			}
			catch(java.io.IOException ioe) {
				System.out.println("Exception caught during JAVA_HOME resolution: " + ioe);
			}
			
			// If user doesn't enter a path use the default
			if (path0 == null || path0.equals("")) {
				path0 = defPath;
			}
			
			path1 = verifyJavaHome(path0);
/*			
			if(path1 == null) {
				System.out.println(myResource.getString("AskJavaHomeErr"));
			}
*/			
		}

		return path1;
	}

	public static String getPlatformOverride(String[] args) {
		String path0 = null;
		int i;

		for(i = 0; i < args.length; i++) {
			if(args[i].startsWith("PLATFORM="))
				path0 = args[i].substring(9);
		}

		return path0;
	}

	public static String getUserAllow(String defAllow) {
		String allow = null;
		String rc = null;

		// Read user input only if no command line arg is present
		while(rc == null) {
			if (defAllow == null) {			
				System.out.print(myResource.getString("AskAllow"));
			}
			else {
				System.out.print(myResource.getString("AskAllow") + " (" + defAllow + ") ");
			}
			
			try {
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				allow = in.readLine();
			}
			catch(java.io.IOException ioe) {
				System.out.println("Exception caught during allow resolution: " + ioe);
			}
			
			if (allow == null || allow.equals("")) {
				allow = defAllow;
			}
			
			rc = verifyAllow(allow, true);
/*			
			if(rc == null) {
				System.out.println(myResource.getString("AskAllowErr"));
			}
*/			
		}

		return rc;
	}

	public static String getUserHosts(String hostList) {
		String hosts = null;


		// Read user input only if no command line arg is present
		while(hosts == null) {
			if (hostList == null) {
				System.out.print(myResource.getString("AskHosts"));
			}
			else {			
				System.out.print(myResource.getString("AskHosts") + " (" + hostList + ") ");
			}				
			try {
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				hosts = in.readLine();
			}
			catch(java.io.IOException ioe) {
				System.out.println("Exception caught during hosts resolution: " + ioe);
			}
			
			if (hosts == null || hosts.equals("")) {
				hosts = hostList;
			}			
		}

		return hosts;
	}

	public static String getUserSecured(String defValue) {
		String secured = null;
		String rc = null;

		while(rc == null) {
			System.out.print(myResource.getString("AskSecured") + " (" + defValue + ") ");
	
			try {
				BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
				secured = in.readLine().toLowerCase();
			}
			catch(java.io.IOException ioe) {
				System.out.println("Exception caught during security resolution: " + ioe);
			}
				
			if((secured == null) || secured.equals("")) {
				secured = defValue;
			}			
	
			rc = verifySecured(secured, true);
		}
		return rc;
	}

	public static String getUserSecuredUsers(String defValue) {
		String securedUsers = null;

		System.out.print(myResource.getString("AskSecuredUsers") + " (" + defValue + ") ");

		try {
			BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
			securedUsers = in.readLine();
		}
		catch(java.io.IOException ioe) {
			System.out.println("Exception caught during security resolution: " + ioe);
		}
			
		if((securedUsers == null) || securedUsers.equals("")) {
			securedUsers = defValue;
		}			

		return securedUsers;
	}

	public static String verifyRacHome(String rachome) {
		String racFile;
		
		if(getPlatform() == null) {
			setPlatform(System.getProperty("os.name", "null"));
		}
		
		if(getPlatform().startsWith("Windows")) {
			racFile = new String("\\bin\\RAServer.exe");
		}
		else if (getPlatform().startsWith("OS/400")) {
			racFile = new String("/dtd/serviceconfig.dtd");
		}
		else {
			racFile = new String("/bin/RAServer");
		}
		
		File RACFile = new File(rachome, racFile);

		if(RACFile.exists())
			return rachome;
		else {
			if (silent) {
				System.out.println(myResource.getString("AskRAServerHomeErr") + "  " +
									myResource.getString("DefaultUsed"));
			}
			else {
				System.out.println(myResource.getString("AskRAServerHomeErr"));
			}
			return null;
		}			

	}

	public static String verifyJavaHome(String javahome) {
		String javaExe;
		File javaFile;

		if(getPlatform() == null) {
			setPlatform(System.getProperty("os.name", "null"));
		}
		
		if(getPlatform().startsWith("Windows")) {		
			javaExe = new String("\\java.exe");
			javaFile = new File(javahome, javaExe);
		}
		else if (getPlatform().startsWith("OS/400")) {
			javaFile = new File(javahome);
		}		
		else {
			javaExe = new String("/java");
			javaFile = new File(javahome, javaExe);
		}

		if(javaFile.exists()) {
			return javahome;
		}
		else {
			if (silent) {
				System.out.println(myResource.getString("AskJavaHomeErr") + "  " + myResource.getString("DefaultUsed"));
			}
			else {
				System.out.println(myResource.getString("AskJavaHomeErr"));
			}
			return null;
		}
	}

	public static String verifyAllow(String allow, boolean askUser) {
		if(allow.toUpperCase().equals("ALL") || allow.toUpperCase().equals("LOCAL") || allow.toUpperCase().equals("CUSTOM")) {
			return allow.toUpperCase();
		}
		else {
			if (silent) {
				System.out.println(myResource.getString("TellAllowErr") + "  " +
									myResource.getString("DefaultUsed"));
			}
			else {
				if (askUser)				
					System.out.println(myResource.getString("AskAllowErr"));
				else
					System.out.println(myResource.getString("TellAllowErr"));					
			}
			
			return null;			

		}
	}

	public static String verifySecured(String secured, boolean askUser) {
		if(secured.toLowerCase().equals("yes") || secured.toLowerCase().equals("no")) {
			return secured.toLowerCase();
		}
		else {
			if (silent) {
				System.out.println(myResource.getString("TellSecuredErr") + "  " +
				myResource.getString("DefaultUsed"));
			}
			else {
				if (askUser)				
					System.out.println(myResource.getString("AskSecuredErr"));
				else
					System.out.println(myResource.getString("TellSecuredErr"));					
			}
			
			return null;			

		}
	}

}
