/*
 * Licensed Material - Property of IBM
 * 5724-D14
 * (C) Copyright IBM Corp. 2002 - All Rights Reserved.
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 */
package org.eclipse.hyades.internal.config.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.util.StringTokenizer;

public class SetConfigCommon {
	protected static String _packageName = "org.eclipse.hyades.internal.config.generator";
	protected static String _RACDTDPath   = "../dtd/serviceconfig.dtd";

	// Environment variable names
	protected static String _JAVAHomeVar  = "JAVA_PATH";
	protected static String _RACHomeVar   = "RASERVER_HOME";
	protected static String _RACConfigVer = "5.0.0";
	protected static String _RACPort      = "10002";
	protected static String _RACSecPort   = "10003";
	protected static String _RACLogLevel  = "WARNING";
	protected static String _RACLogDetail = "LOW";

	protected String libPath = null;
	protected String dllPath = null;
	protected String _encoding = "";
	protected String _encodingTag = "";
	protected String pgmExt = "";
	protected String dcSize = "30000K"; // data channel size
	protected String _platform = null;
	protected char ps; // path separator "/" or "\"
	protected char cs; // classpath separator ":" or ";"

	protected static	AgentControllerConfiguration iAgentControllerConfiguration;
	protected static	AgentControllerConfigurationFactory iAgentControllerConfigurationFactory;


	public static final String copyright = "Licensed Materials -- Property of IBM\n(c) Copyright International Business Machines Corporation, 2002 - All rights reserved.\nUS Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.";

//-----------------------------------------------------------------------------
// Common section starts
//-----------------------------------------------------------------------------
	public SetConfigCommon(String platformName) {
		if (platformName == null) {
			platformName = getPlatformName();
		}

		setPlatform(platformName);

	}

	public SetConfigCommon() {

		setPlatform(getPlatformName());

	}


	//
	// RAStart.sh generation
	//

	public String getStartSh() {
		StringBuffer strbuf = new StringBuffer();

		// RAStart.sh not for Windows platforms
		if(_platform.startsWith("Windows"))
			return null;

		addShellSection(strbuf);
		addEnvSection(strbuf);
		addStartSection(strbuf);
		addCheckSection(strbuf);

		return strbuf.toString();
	}


	public static boolean fileExists(String filename) {

		File file1 = new File(filename);

		return file1.exists();

	}

	public boolean loadExistingConfiguration(String cfgFileName) throws FileError
	{
		if (fileExists(cfgFileName)) {
	    	if (!loadExistingInstance(cfgFileName)) {
	    		return false;
	    	}
	    	
	    	/* We want to keep our entity references */
			fixEntityRefenences();
	    	return true;
		}
		else {
			return false;
		}
	}

	/**
	  * Load an XML document using the generated AgentControllerConfigurationFactory class
	  *   @param filename An existing XML file name
	  */
	public boolean loadExistingInstance(String filename) throws FileError
	{
		iAgentControllerConfigurationFactory = new AgentControllerConfigurationFactory();
		iAgentControllerConfigurationFactory.setPackageName(_packageName);

		// include DOCTYPE for validation
		iAgentControllerConfigurationFactory.setDTDFileName(_RACDTDPath);

		// encoding for output document
		iAgentControllerConfigurationFactory.setEncoding(_encoding);

		// encoding tag for xml declaration
		iAgentControllerConfigurationFactory.setEncodingTag(_encodingTag);

		try {
			// Load the document
			iAgentControllerConfiguration = (AgentControllerConfiguration) iAgentControllerConfigurationFactory.loadDocument(filename);
		}
		catch(Exception ex) {
			throw new FileError(filename, ex);
		}

		if (iAgentControllerConfiguration == null)
			return false;
		else
			return true;
	}


	/**
	* Create a new XML document using the generated AgentControllerConfigurationFactory class
	*   @param filename The XML file name
	*/
	public void createEmptyNewInstance()
	{
		iAgentControllerConfigurationFactory = new AgentControllerConfigurationFactory();
		iAgentControllerConfigurationFactory.setPackageName(_packageName);

		// include DOCTYPE for validation
		iAgentControllerConfigurationFactory.setDTDFileName(_RACDTDPath);

		// encoding for output document
		iAgentControllerConfigurationFactory.setEncoding(_encoding);

		// encoding tag for xml declaration
		iAgentControllerConfigurationFactory.setEncodingTag(_encodingTag);

		// Create the root element in the document using the specified root element name
		iAgentControllerConfiguration = (AgentControllerConfiguration) iAgentControllerConfigurationFactory.createRoot("AgentControllerConfiguration");

    	createEmptyAgentControllerConfiguration();
	}

	public boolean createDefaultConfiguration()
	{
	    if (!createDefaultNewInstance()) {
	    	return false;
	    }
	    return true;
	}

	public boolean createDefaultNewInstance()
	{
		iAgentControllerConfigurationFactory = new AgentControllerConfigurationFactory();
		iAgentControllerConfigurationFactory.setPackageName(_packageName);

		// include DOCTYPE for validation
		iAgentControllerConfigurationFactory.setDTDFileName(_RACDTDPath);

		// encoding for output document
		iAgentControllerConfigurationFactory.setEncoding(_encoding);

		// encoding tag for xml declaration
		iAgentControllerConfigurationFactory.setEncodingTag(_encodingTag);

		// Create the root element in the document using the specified root element name
		iAgentControllerConfiguration = (AgentControllerConfiguration) iAgentControllerConfigurationFactory.createRoot("AgentControllerConfiguration");

		if (iAgentControllerConfiguration == null) {
			return false;
		}

		if (getPlatformName().startsWith("OS/400")) {
			createDefaultAgentControllerConfigurationForOS400();
		}
		else {
			createDefaultAgentControllerConfiguration();
		}
		return true;

	}

   /**
  * Save an XML document using the generated AgentControllerConfigurationFactory class
  *   @param filename The XML file name to save to
  */
  public void saveInstance(String filename) throws Exception
  {

    // Save the document
    iAgentControllerConfigurationFactory.save(filename);
  }


  /**
  * Create the Java bean Agent
  *   @param name The element tag name
  *   @return Agent The Java bean for this element
  */

  public Agent createAgent()
  {
    Agent iAgent = iAgentControllerConfigurationFactory.createAgent("Agent");

    return iAgent;
  }

  public Agent createAgent(String config, String name, String type, String client, String dcsize, String logfile)
  {
    Agent iAgent = iAgentControllerConfigurationFactory.createAgent("Agent");

    iAgent.setConfiguration(config);
    iAgent.setName(name);
    iAgent.setType(type);
    iAgent.setDataChannelSize(dcsize);
    iAgent.setLogFile(logfile);
    iAgent.setClient(client);

    return iAgent;
  }

  public Agent createAgent(String config, String name, String type, String dcsize)
  {
    Agent iAgent = iAgentControllerConfigurationFactory.createAgent("Agent");

    iAgent.setConfiguration(config);
    iAgent.setName(name);
    iAgent.setType(type);
    iAgent.setDataChannelSize(dcsize);

    return iAgent;
  }

  public Agent createAgent(String config, String name, String type)
  {
    Agent iAgent = iAgentControllerConfigurationFactory.createAgent("Agent");

    iAgent.setConfiguration(config);
    iAgent.setName(name);
    iAgent.setType(type);

    return iAgent;
  }

  /**
  * Create the root element in the document
  */
  public void createDefaultAgentControllerConfiguration()
  {
  	AgentControllerEnvironment iACEnv;
  	Application iApp;
  	Hosts iHosts;
  	Agent iAgent;
  	int index, var;

	// AgentControllerEnvironment creation
    iAgentControllerConfiguration.setAgentControllerEnvironment(createAgentControllerEnvironment("default"));
    iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

	var = 0;
   	iACEnv.setVariable(var++, createVariable(_JAVAHomeVar, "%JAVA_PATH%", "replace"));
   	iACEnv.setVariable(var++, createVariable(_RACHomeVar, "%RASERVER_HOME%", "replace"));
   	iACEnv.setVariable(var++, createVariable(getLibPath(), "%RASERVER_HOME%" + ps + getDllPath(), "prepend"));
	iACEnv.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hcframe.jar", "prepend"));
	iACEnv.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "prepend"));
	iACEnv.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "prepend"));


	/* Note:  the index specified on the following setApplication, setHosts, and setAgent
	 * method calls represents the element index within the AgentControllerConfiguration object.
	 * For example, the AgentControllerEnvironment element above is at index 0 within
	 * AgentControllerConfiguration and the java.exe Application below will be at index 1.
	 * The index specified on the getApplication, getHosts, and getAgent method calls
	 * represents the index of the element type. For example, getApplication(1)
	 * will return the second Application element in AgentControllerConfiguration
	 * and getAgent(0) will return the first Agent element in AgentControllerConfiguration.
	 * Note: indices start at 0.
	 */

    // Create Applications
	index = 0;

	/*
	 * Java application
	 */
    iAgentControllerConfiguration.setApplication(++index, createApplication("default", "java.exe", "%JAVA_PATH%" + ps + "java" + pgmExt, "%JAVA_PATH%"));
    iApp = iAgentControllerConfiguration.getApplication(index - 1);
    var = 0;
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "append"));
	iApp.setVariable(var++, createVariable(getLibPath(), "%RASERVER_HOME%" + ps + getDllPath(), "prepend"));

	/*
	 * Log parser applications
	 */
    iAgentControllerConfiguration.setApplication(++index, createApplication("default", "RemoteLogParserLoader", "%JAVA_PATH%" + ps + "java" + pgmExt, "%JAVA_PATH%"));
    iApp = iAgentControllerConfiguration.getApplication(index - 1);
    iApp.setParameter(0, createParameter("org.eclipse.hyades.logging.parsers.RemoteLogParserLoader", "prepend"));
    var = 0;
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hlcore.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hlevents.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hparse.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "prepend"));

	/*
	 * Hyades Testing Session
	 */
	iAgentControllerConfiguration.setApplication(++index, createApplication("default", "HyadesTestingSession", "%JAVA_PATH%" + ps + "java" + pgmExt, "%JAVA_PATH%"));
	iApp = iAgentControllerConfiguration.getApplication(index - 1);
	iApp.setParameter(0, createParameter("&quot;-Dhyades.test.JVM=%JAVA_PATH%" + ps + "java" + pgmExt + "&quot;", "prepend"));
	var = 0;
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "common.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "java.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "http.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "junit.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "xerces.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "xercesImpl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "manual.runner.jar", "append"));
	iApp.setVariable(var++, createVariable(getLibPath(), "%RASERVER_HOME%" + ps + getDllPath(), "prepend"));

    // Create Hosts
    iAgentControllerConfiguration.setHosts(++index, createHosts("default"));
    iHosts = iAgentControllerConfiguration.getHosts(0);
    iHosts.setAllow(0,createAllow("LOCAL"));

    // Create Agents
    iAgentControllerConfiguration.setAgent(++index, createAgent("default","Java Profiling Agent","Profiler","30000K"));

    // Set AgentContrllerConfiguration attributes
    iAgentControllerConfiguration.setActiveConfiguration("default");
    iAgentControllerConfiguration.setVersion(_RACConfigVer);
    iAgentControllerConfiguration.setPort(_RACPort);
    iAgentControllerConfiguration.setSecurePort(_RACSecPort);
    iAgentControllerConfiguration.setLoggingLevel(_RACLogLevel);
    iAgentControllerConfiguration.setLoggingDetail(_RACLogDetail);
//  iAgentControllerConfiguration.setLoggingDirectory("");


  }

  /**
  * Create the root element in the document - with specific defaults for OS/400
  */
  public void createDefaultAgentControllerConfigurationForOS400()
  {
  	AgentControllerEnvironment iACEnv;
  	Application iApp;
  	Hosts iHosts;
  	Agent iAgent;
  	int index, var;

	// AgentControllerEnvironment creation
    iAgentControllerConfiguration.setAgentControllerEnvironment(createAgentControllerEnvironment("default"));
    iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

	var = 0;
   	iACEnv.setVariable(var++,createVariable(_JAVAHomeVar,"/QSYS.LIB/Q5BVAJVM.PGM","replace"));
   	iACEnv.setVariable(var++,createVariable(_RACHomeVar,"/opt/IBMRAC","replace"));
   	iACEnv.setVariable(var++,createVariable(getLibPath(),"/QSYS.LIB/IBMRAC.LIB","prepend"));
	iACEnv.setVariable(var++,createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hcframe.jar", "prepend"));
	iACEnv.setVariable(var++,createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "prepend"));
	iACEnv.setVariable(var++,createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "prepend"));


	/* Note:  the index specified on the following setApplication, setHosts, and setAgent
	 * method calls represents the element index within the AgentControllerConfiguration object.
	 * For example, the AgentControllerEnvironment element above is at index 0 within
	 * AgentControllerConfiguration and the java.exe Application below will be at index 1.
	 * The index specified on the getApplication, getHosts, and getAgent method calls
	 * represents the index of the element type. For example, getApplication(1)
	 * will return the second Application element in AgentControllerConfiguration
	 * and getAgent(0) will return the first Agent element in AgentControllerConfiguration.
	 * Note: indices start at 0.
	 */

    // Create Applications
	index = 0;

	/*
	 * Java application
	 */
    iAgentControllerConfiguration.setApplication(++index, createApplication("default", "java.exe", "%JAVA_PATH%", "/"));
    iApp = iAgentControllerConfiguration.getApplication(index - 1);
    var = 0;
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "append"));
	iApp.setVariable(var++, createVariable(getLibPath(), "%RASERVER_HOME%" + ps + getDllPath(), "prepend"));

	/*
	 * Log parser applications
	 */
    iAgentControllerConfiguration.setApplication(++index, createApplication("default", "RemoteLogParserLoader", "%JAVA_PATH%", "/"));
    iApp = iAgentControllerConfiguration.getApplication(index - 1);
    iApp.setParameter(0, createParameter("org.eclipse.hyades.logging.parsers.RemoteLogParserLoader", "prepend"));
    var = 0;
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hlcore.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hlevents.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hparse.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "prepend"));
    iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "prepend"));

	/*
	 * Hyades Testing Session
	 */
	iAgentControllerConfiguration.setApplication(++index, createApplication("default", "HyadesTestingSession", "%JAVA_PATH%", "/"));
	iApp = iAgentControllerConfiguration.getApplication(index - 1);
	iApp.setParameter(0, createParameter("&quot;-Dhyades.test.JVM=%JAVA_PATH%" + "&quot;", "prepend"));
	var = 0;
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexr.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "hexcore.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "common.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "java.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "http.runner.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "junit.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "xerces.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "xercesImpl.jar", "append"));
	iApp.setVariable(var++, createVariable("CLASSPATH", "%RASERVER_HOME%" + ps + "lib" + ps + "manual.runner.jar", "append"));
	iApp.setVariable(var++, createVariable(getLibPath(), "%RASERVER_HOME%" + ps + getDllPath(), "prepend"));

    // Create Hosts
    iAgentControllerConfiguration.setHosts(++index, createHosts("default"));
    iHosts = iAgentControllerConfiguration.getHosts(0);
    iHosts.setAllow(0,createAllow("LOCAL"));

    // Create Agents
    iAgentControllerConfiguration.setAgent(++index, createAgent("default","Java Profiling Agent","Profiler","8M"));

    // Set AgentContrllerConfiguration attributes
    iAgentControllerConfiguration.setActiveConfiguration("default");
    iAgentControllerConfiguration.setVersion(_RACConfigVer);
    iAgentControllerConfiguration.setPort(_RACPort);
    iAgentControllerConfiguration.setSecurePort(_RACSecPort);
    iAgentControllerConfiguration.setLoggingLevel(_RACLogLevel);
    iAgentControllerConfiguration.setLoggingDetail(_RACLogDetail);
//  iAgentControllerConfiguration.setLoggingDirectory("");
  }


  /**
  * Create the root element in the document
  */
  public void createEmptyAgentControllerConfiguration()
  {
  	AgentControllerEnvironment iACEnv;

    iAgentControllerConfiguration.setAgentControllerEnvironment(createAgentControllerEnvironment("default"));
    iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();
    iACEnv.setVariable(0,createVariable(_JAVAHomeVar,"%JAVA_PATH%","replace"));
    iACEnv.setVariable(1,createVariable(_RACHomeVar,"%RASERVER_HOME%","replace"));
    iACEnv.setVariable(2,createVariable(getLibPath(),"%RASERVER_HOME%" + ps + dllPath,"prepend"));

    iAgentControllerConfiguration.setActiveConfiguration("default");
    iAgentControllerConfiguration.setVersion(_RACConfigVer);
    iAgentControllerConfiguration.setPort(_RACPort);
    iAgentControllerConfiguration.setSecurePort(_RACSecPort);
    iAgentControllerConfiguration.setLoggingLevel(_RACLogLevel);
    iAgentControllerConfiguration.setLoggingDetail(_RACLogDetail);
//	iAgentControllerConfiguration.setLoggingDirectory("");
  }


  /**
  * Create the Java bean AgentControllerEnvironment
  *   @param name The element tag name
  *   @return AgentControllerEnvironment The Java bean for this element
  */
  public AgentControllerEnvironment createAgentControllerEnvironment(String cfgName)
  {
    AgentControllerEnvironment iAgentControllerEnvironment = iAgentControllerConfigurationFactory.createAgentControllerEnvironment("AgentControllerEnvironment");

	iAgentControllerEnvironment.setConfiguration(cfgName);

    return iAgentControllerEnvironment;
  }


  /**
  * Create the Java bean Allow
  *   @param name The element tag name
  *   @return Allow The Java bean for this element
  */
  public Allow createAllow(String host)
  {
    Allow iAllow = iAgentControllerConfigurationFactory.createAllow("Allow");

    iAllow.setHost(host);

    return iAllow;
  }

  public Allow createAllow(String net, String mask)
  {
    Allow iAllow = iAgentControllerConfigurationFactory.createAllow("Allow");

    iAllow.setNet(net);
    if (mask != null && !mask.equals("")) {
    	iAllow.setMask(mask);
    }

    return iAllow;
  }

  /**
  * Create the Java bean Application
  *   @param name The element tag name
  *   @return Application The Java bean for this element
  */
  public Application createApplication(String cfgName, String exeName, String exePath, String runLocation)
  {
    Application iApplication = iAgentControllerConfigurationFactory.createApplication("Application");

    iApplication.setConfiguration(cfgName);
    iApplication.setExecutable(exeName);
    iApplication.setPath(exePath);
    iApplication.setLocation(runLocation);

    return iApplication;
  }


  /**
  * Create the Java bean Deny
  *   @param name The element tag name
  *   @return Deny The Java bean for this element
  */
  public Deny createDeny(String host)
  {
    Deny iDeny = iAgentControllerConfigurationFactory.createDeny("Deny");

    iDeny.setHost(host);

    return iDeny;
  }

  public Deny createDeny(String net, String mask)
  {
    Deny iDeny = iAgentControllerConfigurationFactory.createDeny("Deny");

    iDeny.setNet(net);
    if (mask != null && !mask.equals("")) {
    	iDeny.setMask(mask);
    }

    return iDeny;
  }

  /**
  * Create the Java bean Hosts
  *   @param name The element tag name
  *   @return Hosts The Java bean for this element
  */
  public Hosts createHosts(String cfgName)
  {
    Hosts iHosts = iAgentControllerConfigurationFactory.createHosts("Hosts");

    iHosts.setConfiguration(cfgName);

    return iHosts;
  }

  /**
  * Create a new XML document using the generated AgentControllerConfigurationFactory class
  *   @param filename The XML file name
  */
  public void createNewInstance(String filename) throws Exception
  {
    iAgentControllerConfigurationFactory = new AgentControllerConfigurationFactory();
    iAgentControllerConfigurationFactory.setPackageName(_packageName);

    // include DOCTYPE for validation
    iAgentControllerConfigurationFactory.setDTDFileName(_RACDTDPath);

    // encoding for output document
    iAgentControllerConfigurationFactory.setEncoding("UTF8");

    // encoding tag for xml declaration
    iAgentControllerConfigurationFactory.setEncodingTag("UTF-8");

    // Create the root element in the document using the specified root element name
    iAgentControllerConfiguration = (AgentControllerConfiguration) iAgentControllerConfigurationFactory.createRoot("AgentControllerConfiguration");

    createDefaultAgentControllerConfiguration();

    iAgentControllerConfigurationFactory.save(filename);
  }


  /**
  * Create the Java bean Option
  *   @param name The element tag name
  *   @return Option The Java bean for this element
  */
  public Option createOption(String type, String name, String value)
  {
    Option iOption = iAgentControllerConfigurationFactory.createOption("Option");

    iOption.setType(type);
    iOption.setName(name);
    iOption.setValue(value);

    return iOption;
  }


  /**
  * Create the Java bean Parameter
  *   @param name The element tag name
  *   @return Parameter The Java bean for this element
  */
  public Parameter createParameter(String value, String position)
  {
    Parameter iParameter = iAgentControllerConfigurationFactory.createParameter("Parameter");

    iParameter.setValue(value);
    iParameter.setPosition(position);

    return iParameter;
  }

  /**
  * Create the Java bean Service
  *   @param name The element tag name
  *   @return Service The Java bean for this element
  */
  public Service createService(String cfgName, String serviceName, String servicePath)
  {
    Service iService = iAgentControllerConfigurationFactory.createService("Service");

    iService.setConfiguration(cfgName);
    iService.setName(serviceName);
    iService.setPath(servicePath);

    return iService;
  }

	/**
	* Create the Java bean Variable
	*   @param name The element tag name
	*   @return Variable The Java bean for this element
	*/
	public Variable createVariable(String name, String value, String position) {
		Variable iVariable = iAgentControllerConfigurationFactory.createVariable("Variable");

		iVariable.setName(name);
		iVariable.setValue(value);
		iVariable.setPosition(position);

		return iVariable;
	}


	public void addShellSection(StringBuffer strbuf) {
		if(_platform.equals("z/OS") || _platform.equals("OS/390")) {
			strbuf.append("#!/bin/ksh\n");
		}
		else {
			strbuf.append("#!/bin/sh\n");
		}
		strbuf.append("# Platform = ");
		strbuf.append(_platform);
		strbuf.append("\n");

		return;
	}


	public void addEnvSection(StringBuffer strbuf) {
		if(_platform.equals("z/OS") || _platform.equals("OS/390")) {
			strbuf.append("export RASERVER_HOME=");
			strbuf.append(getRACHome());
			strbuf.append("\n");
			strbuf.append("export PATH=$RASERVER_HOME/bin:$PATH\n");
			strbuf.append("export LIBPATH=$RASERVER_HOME/lib:");
			strbuf.append(findClassicLib(getJAVAHome()) + "/classic:");
			strbuf.append(findClassicLib(getJAVAHome()) + ":");
			strbuf.append("$LIBPATH\n");
			strbuf.append("\n");
		}
		else {
			strbuf.append("RASERVER_HOME=");
			strbuf.append(getRACHome());
			strbuf.append("\n");
			strbuf.append("PATH=$RASERVER_HOME/bin:$PATH\n");
			strbuf.append(libPath);
			strbuf.append("=$RASERVER_HOME/");
			strbuf.append(dllPath + ":");
			strbuf.append(findClassicLib(getJAVAHome()));
			if(_platform.equals("HP-UX")) {
				strbuf.append("/../lib/PA_RISC/classic:");
				strbuf.append(findClassicLib(getJAVAHome()) + "/../lib/PA_RISC:");
			}
			else if(_platform.equals("SunOS")) {
				strbuf.append("/../lib/sparc/server:");
				strbuf.append(findClassicLib(getJAVAHome()) + "/../lib/sparc:");
			}
			else {
				strbuf.append("/classic:");
				strbuf.append(findClassicLib(getJAVAHome()) + ":");
			}
			strbuf.append("$");
			strbuf.append(libPath);
			strbuf.append("\n");
			strbuf.append("export RASERVER_HOME\n");
			strbuf.append("export PATH\n");
			strbuf.append("export ");
			strbuf.append(libPath);
			strbuf.append("\n");
		}

		return;
	}

	public void addStartSection(StringBuffer strbuf) {
		strbuf.append("\n");
		if(_platform.equals("z/OS") || _platform.equals("OS/390")) {
			strbuf.append("ps -e -opid,comm | grep \"RAServer\" | wc -l | awk \'{\n");
		}
		else if(_platform.equals("SunOS")) {
			strbuf.append("ps -ae | grep \"RAServer\" | wc -l | nawk \'{\n");
		}
		else {
			strbuf.append("ps -ae | grep \"RAServer\" | wc -l | awk \'{\n");
		}
		strbuf.append("\tif( $1!=\"0\" ) {\n");
		strbuf.append("\t\tprint \"RAServer is already running\"\n");
		strbuf.append("\t}\n");
		strbuf.append("\telse {\n");
		strbuf.append("\t\tsystem(\"RAServer\")\n");
		strbuf.append("\t}\n");
		strbuf.append("}\'\n");

		return;
	}

	public void addCheckSection(StringBuffer strbuf) {
		strbuf.append("\n");
		strbuf.append("sleep 2\n");
		strbuf.append("\n");


		if(_platform.equals("z/OS") || _platform.equals("OS/390")) {
			strbuf.append("ps -e -opid,comm | grep \"RAServer\" | wc -l | awk \'{\n");
		}
		else if(_platform.equals("SunOS")) {
			strbuf.append("ps -ae | grep \"RAServer\" | wc -l | nawk \'{\n");
		}
		else {
			strbuf.append("ps -ae | grep \"RAServer\" | wc -l | awk '{\n");
		}

		strbuf.append("\tif( $1!=\"0\" ) {\n");
		strbuf.append("\t\tprint \"RAServer started successfully\"\n");
		strbuf.append("\t}\n");
		strbuf.append("\telse {\n");
		strbuf.append("\t\tprint \"RAServer failed to start\"\n");
		strbuf.append("\t}\n");
		strbuf.append("}\'\n");

		return;
	}

	public String getAllowHosts() {
		Hosts iHosts;

		// Check if no hosts exist.
		if (iAgentControllerConfiguration.getHostsCount() == 0 ) {
		    return null;
		}

		// Get the allows from the default Hosts element.
	    iHosts = iAgentControllerConfiguration.getHosts(0);

	    // Check if no allows exist
	    if (iHosts.getAllowCount() == 0) {
	       	return null;
	    }

	    // Create the list of allows from the Hosts element.

		Allow iAllow = iHosts.getAllow(0);
		String allowList = iAllow.getHost();

	    for (int i = 1; i < iHosts.getAllowCount(); i++) {
			iAllow = iHosts.getAllow(i);

		    allowList += "," + iAllow.getHost();

		}

    	return allowList;
	}

	public void removeAllowHosts() {
		Hosts iHosts;

		// Check if no hosts exist.
		if (iAgentControllerConfiguration.getHostsCount() == 0 ) {
		    return;
		}

		// Get the allows from the default Hosts element.
	    iHosts = iAgentControllerConfiguration.getHosts(0);

	    // Check if no allows exist
	    if (iHosts.getAllowCount() == 0) {
	       	return;
	    }

	    // Create the list of allows from the Hosts element.

		int hostCount = iHosts.getAllowCount();

	    for (int i = hostCount-1; i >= 0; i--) {

	    	iHosts.removeAllow(i);

		}

    	return;
	}


	public void addAllowHost(String hostName) {
		Hosts iHosts;

		// Create a Hosts element if one doesn't exist.
		if (iAgentControllerConfiguration.getHostsCount() == 0 ) {
		    iAgentControllerConfiguration.setHosts(0,createHosts("default"));
		}

		// Add the allow to the default Hosts element.
	    iHosts = iAgentControllerConfiguration.getHosts(0);

	    // First check whether it is already in the host list because we don't
	    // need two copies of it.
	    for (int i = 0; i < iHosts.getAllowCount(); i++) {
			Allow iAllow = iHosts.getAllow(i);

			if (iAllow.getHost().equals(hostName)) {
				return;
			}
		}

    	iHosts.setAllow(iHosts.getAllowCount(),createAllow(hostName));
	}

	/*
	 *  Adds a list of hosts, delimited by spaces or commas
	 */

	public void addAllowHostList(String hostList) {
		Hosts iHosts;

		// Create a Hosts element if one doesn't exist.
		if (iAgentControllerConfiguration.getHostsCount() == 0 ) {
		    iAgentControllerConfiguration.setHosts(0,createHosts("default"));
		}

		// Add the allow to the default Hosts element.
	    iHosts = iAgentControllerConfiguration.getHosts(0);

		String list = hostList.trim();

		StringTokenizer st = new StringTokenizer(list, " ,");

		while (st.hasMoreTokens()) {
			String hostName = st.nextToken();
			boolean alreadyExists = false;

	    	// First check whether it is already in the host list because we don't
	    	// need two copies of it.
	    	for (int i = 0; i < iHosts.getAllowCount(); i++) {
				Allow iAllow = iHosts.getAllow(i);

				if (iAllow.getHost().equals(hostName)) {
					alreadyExists = true;
					break;
				}
			}

			if (!alreadyExists) {
	    		iHosts.setAllow(iHosts.getAllowCount(),createAllow(hostName));
			}
		}
	}

	public static String removeSlashes(String in_str) {
		String tmp;

		tmp = in_str.trim();
		if(tmp.endsWith("\\") || tmp.endsWith("/")) {
			return tmp.substring(0, tmp.length() - 1);
		}
		return tmp;
	}

	public String getLibPath() {
		return libPath;
	}

	public String getDllPath() {
		return dllPath;
	}

	public void setPlatformSpecificParameters() {

		String platform = getPlatformName();

		if(platform.equals("AIX")) {
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("LIBPATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else if(platform.equals("HP-UX")) {
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("SHLIB_PATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else if(platform.equals("Linux")) {
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("LD_LIBRARY_PATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else if(platform.equals("SunOS")) {
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("LD_LIBRARY_PATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else if(platform.startsWith("Windows")) {
			_encoding = new String("UTF8");
			_encodingTag = new String("UTF-8");
			libPath = new String("PATH");
			dllPath = new String("bin");
			pgmExt = new String(".exe");
			dcSize = new String("64M");
			ps = '\\';
			cs = ';';
		}
		else if(platform.equals("z/OS") || platform.equals("OS/390")) {
			_encoding = new String("IBM-1047");
			_encodingTag = new String("IBM-1047");
//			XERCES does not recognize the following encoding name
//			_encoding = new String("ibm1047-s390");
//			_encodingTag = new String("ibm1047-s390");
			libPath = new String("LIBPATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else if(platform.equals("OS/400")) {
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("LIBPATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
		else {
			/* Platform not recognized - we'll create a default configuration */
			_encoding = "UTF8";
			_encodingTag = new String("UTF-8");
			libPath = new String("LIBPATH");
			dllPath = new String("lib");
			pgmExt = "";
			ps = '/';
			cs = ':';
		}
	}

	public String getLocation(String path) {
		if(path.charAt(0) == '%') {
			return path;
		}
		else if(_platform.startsWith("Windows")) {
			return path.substring(0, path.indexOf('\\', 1) + 1);
		}
		else {
			return path.substring(0, path.indexOf('/', 1));
		}
	}

	public void setPlatform(String platform_name) {
		if((platform_name != null) && !platform_name.equals("")) {
			_platform = platform_name;
			setPlatformSpecificParameters(); // reload the platform specific information
		}
	}

	public String getPlatformName() {
		if(_platform == null)
			return System.getProperty("os.name", "Windows");
		else
			return _platform;
	}

	public void writeConfigFile(String xmlFileName) throws FileError {

		try {
			saveInstance(xmlFileName);

		}
		catch(Exception ex) {
			throw new FileError(xmlFileName, ex);
		}

		return;
	}

	public void writeToFile(String absoluteFileName, String content) throws FileError {
		if(content == null)
			return;

		try {
			FileOutputStream outFile = new FileOutputStream(absoluteFileName);
			outFile.write(content.getBytes());
			outFile.close();
		}
		catch(Exception ex) {
			throw new FileError(absoluteFileName, ex);
		}

		return;
	}

	public String findClassicLib(String javaHome) {
		if(getPlatformName() == null) {
			setPlatform(System.getProperty("os.name", "null"));
		}

		/*
		 * Windows
		 */
		if(getPlatformName().startsWith("Windows")) {
			if(fileExists(javaHome + "\\classic", "jvm.dll")) { // from "$JAVA_HOME/jre/bin"
				return javaHome;
			}
			else if(fileExists(javaHome + "\\..\\jre\\bin\\classic", "jvm.dll")) { // from "$JAVA_HOME/bin"
				return new String(javaHome + "\\..\\jre\\bin");
			}
		}
		/*
		 * AIX
		 */
		else if(getPlatformName().equals("AIX")) {
			if(fileExists(javaHome + "/classic", "libjvm.a")) { // from "$JAVA_HOME/jre/bin"
				return javaHome;
			}
			else if(fileExists(javaHome + "/../jre/bin/classic", "libjvm.a")) { // from "$JAVA_HOME/bin"
				return new String(javaHome + "/../jre/bin");
			}
		}
		/*
		 * HP-UX
		 */
		else if(getPlatformName().equals("HP-UX")) {
			if(fileExists(javaHome + "/../lib/PA_RISC/classic", "libjvm.sl")) { // from "$JAVA_HOME/jre/bin"
				return javaHome;
			}
			else if(fileExists(javaHome + "/../jre/lib/PA_RISC/classic", "libjvm.sl")) { // from "$JAVA_HOME/bin"
				return new String(javaHome + "/../jre/bin");
			}
		}
		/*
		 * Solaris
		 */
		else if(getPlatformName().equals("SunOS")) {
			if(fileExists(javaHome + "/../lib/sparc/server", "libjvm.so")) { // from "$JAVA_HOME/jre/bin"
				return javaHome;
			}
			else if(fileExists(javaHome + "/../jre/lib/sparc/server", "libjvm.so")) { // from "$JAVA_HOME/bin"
				return new String(javaHome + "/../jre/bin");
			}
		}
		/*
		 * Linux, z/OS
		 */
		else { // Linux, z/OS
			if(fileExists(javaHome + "/classic", "libjvm.so")) { // from "$JAVA_HOME/jre/bin"
				return javaHome;
			}
			else if(fileExists(javaHome + "/../jre/bin/classic", "libjvm.so")) { // from "$JAVA_HOME/bin"
				return new String(javaHome + "/../jre/bin");
			}
		}

		/* Can't find it */
		return javaHome;
	}

	public boolean fileExists(String pathName, String fileName) {
		File f;

		f = new File(pathName, fileName);
		return f.exists();
	}

	public String getJAVAHome() {
  		AgentControllerEnvironment iACEnv;
  		Variable iVar;

		iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

		for (int i = 0; i < iACEnv.getVariableCount(); i++) {
			iVar = iACEnv.getVariable(i);
			String varName = iVar.getName();
			if (varName.equals(_JAVAHomeVar)) {
				return iVar.getValue();
			}
		}

		return null;
	}

	public void setJAVAHome(String newJAVAHome) {
		if((newJAVAHome != null) && !newJAVAHome.equals("")) {
			String JAVAHome = removeSlashes(newJAVAHome);
			AgentControllerEnvironment iACEnv;
			Variable iVar;

			iAgentControllerConfiguration.setJvm(findClassicLib(newJAVAHome));

			iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

			if (iACEnv == null) {
				// Create an AgentControllerEnvironment element
    			iAgentControllerConfiguration.setAgentControllerEnvironment(createAgentControllerEnvironment("default"));
    			iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();
			}

			// Try to find the variable so we can replace the value.
			for (int i = 0; i < iACEnv.getVariableCount(); i++) {
				iVar = iACEnv.getVariable(i);
				String varName = iVar.getName();
				if (varName.equals(_JAVAHomeVar)) {
					iVar.setValue(JAVAHome);
					return;
				}
			}

			// If the variable doesn't exist we must create it
			iACEnv.setVariable(iACEnv.getVariableCount(), createVariable(_JAVAHomeVar, JAVAHome, "replace"));
		}
		else {
			// Should throw an exception here
		}
	}

	public String getRACHome() {
  		AgentControllerEnvironment iACEnv;
  		Variable iVar;

		iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

		for (int i = 0; i < iACEnv.getVariableCount(); i++) {
			iVar = iACEnv.getVariable(i);
			String varName = iVar.getName();
			if (varName.equals(_RACHomeVar)) {
				return iVar.getValue();
			}
		}

		return null;
	}

	public void setRACHome(String newRACHome) {

		if((newRACHome != null) && !newRACHome.equals("")) {
			String RACHome = removeSlashes(newRACHome);
			AgentControllerEnvironment iACEnv;
			Variable iVar;

			iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();

			if (iACEnv == null) {
				// Create an AgentControllerEnvironment element
    			iAgentControllerConfiguration.setAgentControllerEnvironment(createAgentControllerEnvironment("default"));
    			iACEnv = iAgentControllerConfiguration.getAgentControllerEnvironment();
			}

			// Try to find the variable so we can replace the value.
			for (int i = 0; i < iACEnv.getVariableCount(); i++) {
				iVar = iACEnv.getVariable(i);
				String varName = iVar.getName();
				if (varName.equals(_RACHomeVar)) {
					iVar.setValue(RACHome);
					return;
				}
			}

			// If the variable doesn't exist we must create it
			iACEnv.setVariable(iACEnv.getVariableCount(), createVariable(_RACHomeVar, RACHome, "replace"));
		}
		else {
			// Should throw an exception here
		}

	}

	/* Return the platform specific path separator */
	public char getPs() {
		return ps;
	}
	
	/* RKD:  Fixup the entity references in te paramter values */
	private void fixEntityRefenences() {
		int appCount=iAgentControllerConfiguration.getApplicationCount();
		for(int i=0; i<appCount; i++) {
			Application app=iAgentControllerConfiguration.getApplication(i);
			int parmCount=app.getParameterCount();
			for(int j=0; j<parmCount; j++) {
				Parameter parm=app.getParameter(j);
				parm.setValue(removeQuotes(parm.getValue()));
			}
		}	
	}

	private String removeQuotes(String str) {
		StringBuffer strbuf = new StringBuffer();

		for(int i = 0; i < str.length(); i++) {
			if(str.charAt(i) == '\"') {
				strbuf.append("&quot;");
			}
			else {
				strbuf.append(str.charAt(i));
			}
		}

		return strbuf.toString();
	}
}
