/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator;

import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IWorkingSet;
import org.eclipse.ui.IWorkingSetManager;
import org.eclipse.ui.ResourceWorkingSetFilter;
import org.eclipse.ui.views.framelist.FrameList;
import org.eclipse.ui.views.framelist.TreeFrame;
import org.eclipse.ui.views.navigator.ResourceNavigatorMessages;

import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * Implementation of the <code>ITreeNavigator</code> interface.
 * 
 * @author marcelop
 * @since 0.2.0
 */
public abstract class TreeNavigator 
extends Navigator implements 	ITreeNavigator, 
								IDoubleClickListener, IPropertyChangeListener
{
	/*
	 * SET_* constants are used in the IDialogSettings
	 */
	protected final static String SET_WORKING_SET = "WORKING_SET";

	/*
	 * TAG* constants are used in the IMemento
	 */
	protected static final String TAG_FRAME = "FRAME";
	protected static final String TAG_EXPAND_STATE = "EXPAND_STATE";
	
	private FrameList frameList;
	private ResourceWorkingSetFilter workingSetFilter;
	
	/**
	 * Constructor for TreeNavigator
	 */
	public TreeNavigator()
	{
		super();

		workingSetFilter = createWorkingSetFilter();
		getPlugin().getWorkbench().getWorkingSetManager().addPropertyChangeListener(this);
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose()
	{
		frameList = null;
		workingSetFilter = null;
		getViewer().removeDoubleClickListener(this);
		getPlugin().getWorkbench().getWorkingSetManager().removePropertyChangeListener(this);
		
		super.dispose();
	}
	
	/**
	 * Creates the WorkingSetFilter.
	 * @return ResourceWorkingSetFilter
	 */
	protected ResourceWorkingSetFilter createWorkingSetFilter()
	{
		return new ResourceWorkingSetFilter();
	}
	
	/**
	 * Returns the WorkingSetFilter.
	 * @return ResourceWorkingSetFilter
	 */
	protected ResourceWorkingSetFilter getWorkingSetFilter()
	{
		return workingSetFilter;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.ITreeNavigator#getTreeViewer()
	 */
	public TreeViewer getTreeViewer()
	{
		return (TreeViewer)getViewer();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadSettings()
	 */
	protected void loadSettings()
	{
		super.loadSettings();
		initWorkingSet();
	}
	
	/**
	 * Initializes the working set.
	 */
	protected void initWorkingSet()
	{
		ResourceWorkingSetFilter workingSetFilter = getWorkingSetFilter();
		if(workingSetFilter == null)
			return;
			
		String workingSetName = getSettings().get(SET_WORKING_SET);
		if((workingSetName != null) && !workingSetName.equals(""))
		{
			IWorkingSetManager workingSetManager = getPlugin().getWorkbench().getWorkingSetManager();
			IWorkingSet workingSet = workingSetManager.getWorkingSet(workingSetName);
			if (workingSet != null)
			{
				// Only initialize filter. Don't set working set into viewer.
				// Working set is set via WorkingSetFilterActionGroup
				// during action creation.
				workingSetFilter.setWorkingSet(workingSet);
			}
		}		
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createViewer(org.eclipse.swt.widgets.Composite)
	 */
	protected StructuredViewer createViewer(Composite parent)
	{
		TreeViewer treeViewer =	new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		treeViewer.addDoubleClickListener(this);
		if(getWorkingSetFilter() != null)
			treeViewer.addFilter(getWorkingSetFilter());
		
		adjustTreeViewer(treeViewer);
		
		return treeViewer;
	}
	
	/**
	 * Subclasses may extend this method to customize the tree viewer.  The method
	 * <code>getTreeViewer()</code> returns null in the scope of this method.
	 * @param treeViewer
	 */
	protected void adjustTreeViewer(TreeViewer treeViewer)
	{
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event)
	{
		if(event.getViewer() == getTreeViewer())
		{
			IStructuredSelection selection = (IStructuredSelection)event.getSelection();
			Object element = selection.getFirstElement();

			if(getTreeViewer().isExpandable(element))
				getTreeViewer().setExpandedState(element, !getTreeViewer().getExpandedState(element));
		}
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadMemento(org.eclipse.ui.IMemento)
	 */
	protected void loadMemento(IMemento memento)
	{
		loadFrame(memento);
		super.loadMemento(memento);	
		loadExpandState(memento);
	}
	
	/**
	 * @see org.eclipse.ui.IViewPart#saveState(org.eclipse.ui.IMemento)
	 */
	public void saveState(IMemento memento)
	{
			saveFrame(memento);
			super.saveState(memento);
			saveExpandState(memento);
	}

	/**
	 * Loades the previous frame state from the memento returning whether the frame
	 * state was restored.
	 * @param memento
	 * @return boolean
	 */
	protected boolean loadFrame(IMemento memento)
	{
		FrameList frameList = getFrameList();
		if(frameList == null)
			return false;
		
		IMemento frameMemento = memento.getChild(TAG_FRAME);
		if(frameMemento == null)
			return false;
		
		TreeFrame frame = new TreeFrame(getTreeViewer());
		frame.restoreState(frameMemento);
		if(frame.getInput() == null)
			return false;
	
		/*
		 * Makes sure the frame input is one of the elements in the
		 * tree.  This is usefull when objects are logically equals.  
		 */
		getTreeViewer().getTree().deselectAll();
		getTreeViewer().setSelection(new StructuredSelection(frame.getInput()));
		IStructuredSelection structuredSelection = getStructuredSelection();
		if(!structuredSelection.isEmpty())
			frame.setInput(structuredSelection.getFirstElement());

		frame.setName(getFrameName(frame.getInput()));
		frame.setToolTipText(getFrameToolTipText(frame.getInput()));
		frameList.gotoFrame(frame);
		return true;
	}

	/**
	 * Loads the previous expand state from the memento.
	 * @param memento
	 */
	protected void loadExpandState(IMemento memento)
	{
		IMemento expandedMemento = memento.getChild(TAG_EXPAND_STATE);
		if(expandedMemento == null)
			return;
			
		List expanded = loadElements(expandedMemento);
		if(!expanded.isEmpty())
			((TreeViewer)getViewer()).setExpandedElements(expanded.toArray());
	}
	
	/**
	 * Saves the previous frame state in the memento returning whether the 
	 * frame information was saved.
	 * @param memento
	 * @return boolean
	 */
	protected boolean saveFrame(IMemento memento)
	{
		FrameList frameList = getFrameList();
		if(frameList == null)
			return false;
		
		if(frameList.getCurrentIndex() == 0)
			return false;
		
		//save frame, it's not the "home"/workspace frame
		TreeFrame currentFrame = (TreeFrame)frameList.getCurrentFrame();
		IMemento frameMemento = memento.createChild(TAG_FRAME);
		currentFrame.saveState(frameMemento);
		return true;
	}

	/**
	 * Saves the current expand state in the memento.
	 * @param memento
	 */
	protected void saveExpandState(IMemento memento)
	{
		IStructuredSelection structuredSelection = convertFromViewer(new StructuredSelection(((TreeViewer)getViewer()).getExpandedElements()));
		
		IMemento expandedMemento = memento.createChild(TAG_EXPAND_STATE);
		for(Iterator i = structuredSelection.iterator(); i.hasNext();)
			saveElement(expandedMemento, i.next());
	}
	
	/**
	 * Sets the working set for this view, or <code>null</code> to clear it.
	 * @param workingSet the working set, or <code>null</code> to clear it
	 */
	public void setWorkingSet(IWorkingSet workingSet)
	{
		ResourceWorkingSetFilter workingSetFilter = getWorkingSetFilter();
		if(workingSetFilter == null)
			return;
			
		TreeViewer treeViewer = (TreeViewer)getViewer();
		Object[] expanded = treeViewer.getExpandedElements();
		ISelection selection = treeViewer.getSelection();
		
		workingSetFilter.setWorkingSet(workingSet);
		if (workingSet != null)
		{
			getSettings().put(SET_WORKING_SET, workingSet.getName());
		}
		else
		{
			getSettings().put(SET_WORKING_SET, "");
		}
		updateTitle();
		treeViewer.refresh();
		treeViewer.setExpandedElements(expanded);
		if((!selection.isEmpty()) && (selection instanceof IStructuredSelection))
		{
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			treeViewer.reveal(structuredSelection.getFirstElement());
		}
	}
	
	/**
	 * Returns the working set for this view or <code>null</code> if none is 
	 * defined.
	 * @return IWorkingSet
	 */
	public IWorkingSet getWorkingSet()
	{
		ResourceWorkingSetFilter workingSetFilter = getWorkingSetFilter();
		if(workingSetFilter == null)
			return null;
			
		return workingSetFilter.getWorkingSet();
	}
	
	/**
	 * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent event)
	{
		String property = event.getProperty();
		Object newValue = event.getNewValue();
		Object oldValue = event.getOldValue();
		IWorkingSet filterWorkingSet = getWorkingSet();

		if(IWorkingSetManager.CHANGE_WORKING_SET_REMOVE.equals(property) && (oldValue == filterWorkingSet))
		{
			setWorkingSet(null);
		}
		else if(IWorkingSetManager.CHANGE_WORKING_SET_NAME_CHANGE.equals(property) && (newValue == filterWorkingSet))
		{
			updateTitle();
		}
		else if(IWorkingSetManager.CHANGE_WORKING_SET_CONTENT_CHANGE.equals(property) && (newValue == filterWorkingSet))
		{
			getViewer().refresh();
		}
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.ITreeNavigator#getFrameList()
	 */
	public FrameList getFrameList()
	{
		return frameList;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.ITreeNavigator#getFrameName(java.lang.Object)
	 */
	public String getFrameName(Object element)
	{
		return ((ILabelProvider) getTreeViewer().getLabelProvider()).getText(element);
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.ITreeNavigator#getFrameToolTipText(java.lang.Object)
	 */
	public String getFrameToolTipText(Object element)
	{
		return getFrameName(element);
	}
	
	/**
	 * Creates the frame source and frame list, and connects them.
	 * @return FrameList
	 */
	protected FrameList createFrameList()
	{
		TreeNavigatorFrameSource frameSource = createFrameSource();
		if(frameSource == null)
			return null;
			
		FrameList frameList = new FrameList(frameSource);
		frameSource.connectTo(frameList);
		return frameList;
	}
	
	/**
	 * Creates the frame source to be associated with this navigator.
	 * @return TreeNavigatorFrameSource
	 */
	protected TreeNavigatorFrameSource createFrameSource()
	{
		return new TreeNavigatorFrameSource(this);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#initContextMenu()
	 */
	protected void initContextMenu()
	{
		super.initContextMenu();
		frameList = createFrameList();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#updateTitle()
	 */
	public void updateTitle()
	{
		Object input = getViewer().getInput();
		String viewName = getConfigurationElement().getAttribute("name");
		IWorkingSet workingSet = workingSetFilter.getWorkingSet();
					
		if (input == null || input.equals(getInitialViewerInput()))
		{
			setTitle(viewName);
			if (workingSet != null)
				setTitleToolTip(ResourceNavigatorMessages.format("ResourceNavigator.workingSetToolTip", new Object[] {workingSet.getName()}));
			else
				setTitleToolTip(""); //$NON-NLS-1$
		} 
		else
		{
			ILabelProvider labelProvider = (ILabelProvider) getTreeViewer().getLabelProvider();
			String inputToolTip = getFrameToolTipText(input);
			
			setTitle(HyadesUIPlugin.getString("TREE_NAV_TTL", new String[] {viewName, labelProvider.getText(input)}));
			if (workingSet != null)
				setTitleToolTip(workingSet.getName());
			else
				setTitleToolTip(inputToolTip);
		}
	}
}
