package org.eclipse.hyades.logging.java;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.CbeFormatter;
import org.eclipse.hyades.logging.events.IAssociatedEvent;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IContextDataElement;
import org.eclipse.hyades.logging.events.IExtendedDataElement;
import org.eclipse.hyades.logging.events.IMsgDataElement;

/** 
 * Extension of the Java Logging <code>java.util.logging.LogRecord</code>
 * class used to encapsulate an <code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code> 
 * within a <code>java.util.logging.LogRecord</code>.
 * <p>
 * A log record instance is created via the constructor and configured (e.g. setting an ICommonBaseEvent) before 
 * being logged to a logger.
 * <p>
 * NOTE:  The Java Logging classes (e.g. JDK V1.4.0 and above) must be on the CLASSPATH at runtime to utilize this filter class
 *  
 * 
 * @author		Paul Slauenwhite
 * @version	March 7, 2003
 * @see		java.util.logging.LogRecord
 * @see		org.eclipse.hyades.logging.events.ICommonBaseEvent
 */
public class CommonBaseEventLogRecord extends LogRecord implements IExternalizableToXml {

	/**
	 * Instance variable which holds a reference to a <code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code>.
	 */
	private ICommonBaseEvent commonBaseEvent = null;

	/**
	 * @deprecated Use the constructor: CommonBaseEventLogRecord (Level)
	 * 
	 * @param level The logging level of the newly created log record.
	 * @param message The log message of the newly created log record.
	 */
	public CommonBaseEventLogRecord(Level level, String message) {
		super(level, message);
		
	}
	
	
	/**
	 * Constructor to create a Common Base Event log record with the parameter logging leve.
	 * 
	 * @param level The logging level of the newly created log record.
	 */
	public CommonBaseEventLogRecord (Level level)
	{
		super (level, "");
	}

	/**
	 * Get event time in milliseconds since 1970.  
	 * <p>
	 * This method overrides the <code>java.util.logging.LogRecord</code>'s <code>getMillis()</code> API.
	 *
	 * @return If a common base event is associated with this object then an event time in millis is 
	 * 			returned; otherwise -1 is returned. 
	 */
	public long getMillis() {

		long millis = getCreationTimeInMillis(commonBaseEvent);
		return millis;
	}

	/**
	 * Get the "raw" log message, before localization or formatting.
	 * <p>
	 * This method overrides the <code>java.util.logging.LogRecord</code>'s <code>getMessage()</code> API.
	 *
	 * @return The raw message string if a common base event is associated with this object; otherwise null is returned
	 */
	public String getMessage() {

		if (commonBaseEvent == null)
			return null;
			
		StringBuffer message = new StringBuffer();
		String msg = commonBaseEvent.getMsg();
		String msgId = null;

		IMsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

		if (msgDataElement != null)
			msgId = msgDataElement.getMsgId();

		if ((msgId != null) && (msgId.length() != 0))
			message.append(msgId).append(": ");

		if ((msg != null) && (msg.length() != 0)) 
			message.append(msg);
		

		return (message.toString());
	}


	/** 
	 * Get the sequence number.
	 * <p>
	 * This method overrides the <code>java.util.logging.LogRecord</code>'s <code>getSequenceNumber()</code> API.
	 * 
	 * @return If the associated common base event is not null then the sequence number is returned; otherwise -1
	 * 			will be returned.
	 */
	public long getSequenceNumber() {

		if (commonBaseEvent == null)
			return -1;
			
		long sequenceNumber = commonBaseEvent.getSequenceNumber();
		return sequenceNumber;		
	}

	/**
	 * Get the AssociatedEvents, ContextDataElements, and ExtendedDataElements of the log message
	 * <p>
	 * This method overrides the <code>java.util.logging.LogRecord</code>'s <code>getParameters()</code> API.
	 *
	 * @return An object array containing the AssociatedEvents, ContextDataElements, and ExtendedDataElements associated 
	 * 			with the common base event.  If the common base event associated with this object is not set, then 
	 * 			null will be returned.
	 */
	public Object[] getParameters() {

		if (commonBaseEvent == null)
			return null;
		
		
		Object[] returnArray, associatedEvents, contextDataElement, extendedDataElement;
		int length = 0, offset = 0; 

		/* References to the elements that will be returned as the parameter */
		associatedEvents = commonBaseEvent.getAssociatedEvents();
		contextDataElement = commonBaseEvent.getContextDataElements();
		extendedDataElement = commonBaseEvent.getExtendedDataElements();
		
		length += (associatedEvents != null ? associatedEvents.length : 0);
		length += (contextDataElement != null ? contextDataElement.length : 0);
		length += (extendedDataElement != null ? extendedDataElement.length : 0);

		/* Instantiate returnArray */
		returnArray = new Object[length];

		/* Copy and return the parameters */
		if (associatedEvents != null) System.arraycopy(associatedEvents,0,returnArray,offset,associatedEvents.length); offset+= associatedEvents.length; 
		if (contextDataElement != null) System.arraycopy(contextDataElement,0,returnArray,offset,contextDataElement.length); offset+= contextDataElement.length;
		if (extendedDataElement != null) System.arraycopy(extendedDataElement,0,returnArray,offset,extendedDataElement.length); offset+= extendedDataElement.length;

		return returnArray;
	}


	/**
	 * Gets the value of the commonBaseEvent instance variable.
	 * 
	 * @return The value of the commonBaseEvent instance variable, or <code>null</code> if the value of the commonBaseEvent instance variable has not been set or is set to <code>null</code>.
	 */
	public ICommonBaseEvent getCommonBaseEvent() {
		return commonBaseEvent;
	}

	/**
	 * Sets value of the commonBaseEvent instance variable to the specified value.
	 * 
	 * @param cbeLogRecord_commonBaseEvent The new value of the commonBaseEvent instance variable.
	 */
	public void setCommonBaseEvent(ICommonBaseEvent cbeLogRecord_commonBaseEvent) {
		commonBaseEvent = cbeLogRecord_commonBaseEvent;
	}


	/**
	 * This method overrides the setter method of the super class in order to set the 
	 * message field associated with the common base event
	 * 
	 * @param message Used to set the message field.
	 * @throws  java.lang.NullPointerException if the value returned by getCommonBaseEvent() is null
	 */
	public void setMessage(String message) {
		commonBaseEvent.setMsg(message);
	}
		
		
	
	/**
	 * This method overrides the setter method of the super class in order to set the 
	 * creation time field associated with the common base event
	 * 
	 * @param millis Time in milliseconds (1970 is used as the base)
	 * @throws java.lang.NullPointerException if the value returned by getCommonBaseEvent() is null
	 */
	public void setMillis(long millis) {
		commonBaseEvent.setCreationTime(String.valueOf(millis));
	}
		
	
	
	/**
	 * This method overrides the setter method of the super class in order to add to the 
	 * valid fields defined in the common base event.  The only supported elements are
	 * IAssociatedEvent, IContextDataElement, and IExtendedDataElement.
	 * 
	 * @param parameters[] The argument passed should be an array consisting of individual
	 *         elements that are an instance of either IAssociatedEvent, IContextDataElement, or IExtendedDataElement,
	 * 
	 * @throws java.lang.NullPointerException if the value returned by getCommonBaseEvent() is null
	 * @throws java.lang.IllegalArgumentException if parameters[] consists of an element that is not an instance of
	 *          IAssociatedEvent, IContextDataElement, or IExtendedDataElement. 
	 */	
	public void setParameters(Object parameters[]) {
			
		for (int i = parameters.length - 1; i >= 0; i--)
		{
			if (parameters[i] instanceof IAssociatedEvent)
				commonBaseEvent.addAssociatedEvent((IAssociatedEvent)parameters[i]);
			else if (parameters[i] instanceof IContextDataElement)
				commonBaseEvent.addContextDataElement((IContextDataElement) parameters[i]);
			else if (parameters[i] instanceof IExtendedDataElement)
				commonBaseEvent.addExtendedDataElement((IExtendedDataElement) parameters[i]);
			else
				throw new java.lang.IllegalArgumentException("Type of element " + i + " in the passed argument is not supported");
			
		}
	}	
		
	
	/**
	 * This method overrides the setter method of the super class in order to set the 
	 * sequence number field associated with the common base event
	 * 
	 * @param seq The sequence number
	 * @throws java.lang.NullPointerException if the value returned by getCommonBaseEvent() is null
	 */	
	public void setSequenceNumber(long seq) {
		commonBaseEvent.setSequenceNumber(seq);
	}	
	
	/**
	 * Generates the XML representation of the Common Base Event in this log record.
	 * 
	 * The XML representation of the commonBaseEvent instance variable is generated and not the 
	 * XML representation of the complete <code>java.util.logging.LogRecord</code>.
	 *      * @return The XML representation of the Common Base Event in this log record, or <code>null</code> if the value of the commonBaseEvent instance variable has not been set or is set to <code>null</code>.
	 */
	public String externalizeCanonicalXmlString() {

		if (commonBaseEvent != null) {
			return (CbeFormatter.toCanonicalXMLString(commonBaseEvent));
		}

		return null;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.core.IExternalizableToXml#externalizeCanonicalXmlDocString()
	 */
	public String externalizeCanonicalXmlDocString() {
		if (commonBaseEvent != null) {
			return (CbeFormatter.toCanonicalXMLDocString(commonBaseEvent));
		}
		return null;
	}

	//i.e. January 29, 2003 10:19:11 PM EST --> 2003-01-29T22:19:11.000000-05:00
	private long getCreationTimeInMillis(ICommonBaseEvent commonBaseEvent) {

		if (commonBaseEvent != null) {

			//Retrieve the creationTime from the Common Base Event:
			String recordTimeStampString = commonBaseEvent.getCreationTime();

			if ((recordTimeStampString != null) && (recordTimeStampString.length() != 0)) {

				//Find the last dash (i.e. time zone offset) in the creationTime:
				int lastDashIndex = recordTimeStampString.lastIndexOf('-');

				if ((lastDashIndex != -1)) {

					//Add the 'GMT' acronym before the time zone offset since no time zone name is known:
					recordTimeStampString = (recordTimeStampString.substring(0, lastDashIndex).concat("GMT").concat(recordTimeStampString.substring(lastDashIndex)));

					//i.e. 2003-01-29T22:19:11.000000GMT-05:00
					SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSSzzzz");

					//Parse the creationTime string into a java.util.Date:
					Date creationDate = formatter.parse(recordTimeStampString, new ParsePosition(0));

					if (creationDate != null)
						return (creationDate.getTime());
				}
			}
		}

		return -1;
	}
}
