/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.wizards;

import java.net.URL;
import java.text.DateFormat;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.common.*;
import org.eclipse.hyades.internal.execution.local.control.*;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.TraceMergeUIDialog;
import org.eclipse.hyades.logging.parsers.*;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.security.util.TString;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.*;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.*;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.*;
import org.eclipse.ui.plugin.AbstractUIPlugin;


/**
 * Standard workbench wizard for importing the binary WebSphere activity
 * log file (activity.log) from a local or remote file system into the
 * workspace.
 */
public class ImportLogWizard extends Wizard implements INewWizard, IRunnableWithProgress, ISetParser {
    //~ Instance fields ----------------------------------------------------------------------------

    protected IStructuredSelection selection;
    protected IWorkbench workbench;
    protected ImportLogHostPage hostPage;
    protected ImportLogLocationPage locPage;
    protected ImportLogUserPage userPage;
    protected LogParserItem _parser;
    protected LogParsersPage parserPage;
    protected PDProjectExplorer fViewer;
    protected Shell fShell;
    protected String errorMessage = null;
    protected String logParserId;
    protected String reasonAction = null;
    private LogParserItem currentSelectedParser;
    private boolean bAppend = false;
	private final String CANCEL = "cancel";
    //need to add this flag.  In some cases we don't want
    //a parser page
    private boolean addParserPage = true;

    //~ Constructors -------------------------------------------------------------------------------

    public ImportLogWizard() {
        AbstractUIPlugin plugin = (AbstractUIPlugin) Platform.getPlugin(PlatformUI.PLUGIN_ID);

        IDialogSettings workbenchSettings = plugin.getDialogSettings();

        IDialogSettings section = workbenchSettings.getSection("com.ibm.etools.pd.ras.wizard.ImportLogFile");

        if (section == null) {
            section = workbenchSettings.addNewSection("com.ibm.etools.pd.ras.wizard.ImportLogFile");
        }

        setDialogSettings(section);
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * Sets the addParserPage.
     * @param addParserPage The addParserPage to set
     */
    public void setAddParserPage(boolean addParserPage) {
        this.addParserPage = addParserPage;
    }

    /**
     * Returns the addParserPage.
     * @return boolean
     */
    public boolean isAddParserPage() {
        return addParserPage;
    }

    /**
     * Sets the currentSelectedParser.
     * @param currentSelectedParser The currentSelectedParser to set
     */
    public void setCurrentSelectedParser(LogParserItem newSelectedParser) {
        if (currentSelectedParser != newSelectedParser) {
            userPage.initializePage();
            getContainer().updateButtons();
        }

        currentSelectedParser = newSelectedParser;
    }

    /**
     * Returns the currentSelectedParser.
     * @return LogParserItem
     */
    public LogParserItem getCurrentSelectedParser() {
        return currentSelectedParser;
    }

    public String getHostName() {
        return hostPage.getHost();
    }

    public ArrayList getIndexList() {
        return userPage.getIndexList();
    }

    public boolean isLocalHost() {
        return hostPage.isLocaHost();
    }

    public String getLogAgentName() {
        return getLogParser().getName();
    }

    public LogParserItem getLogParser() {
        if (addParserPage) {
            return parserPage.getParser();
        }

        return _parser;
    }

    /**
     * Sets the logParserId.
     * @param logParserId The logParserId to set
     */
    public void setLogParserId(String logParserId) {
        this.logParserId = logParserId;
        addParserPage = false;
        initializeValues();
    }

    /**
     * Returns the logParserId.
     * @return String
     */
    public String getLogParserId() {
        return logParserId;
    }

    public Parser getLogParserInstance() {
        return getLogParser().getParserInstance();
    }

    public String getParserClass() {
        return getLogParser().getParserClass();
    }

    public int getPort() {
        return hostPage.getPort();
    }

    public Hashtable getUserInput() {
        return userPage.getUserInput();
    }

    public void addPages() {
        super.addPages();

        setWindowTitle(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_WIZARD_TITLE"));

        if (addParserPage) {
            parserPage = new LogParsersPage("parserPage");
        }

        hostPage = new ImportLogHostPage();
        userPage = new ImportLogUserPage(workbench, selection);
        locPage = new ImportLogLocationPage();

        addAllPages();
    }

    public void init(IWorkbench workbench, IStructuredSelection currentSelection) {
        this.workbench = workbench;

        selection = currentSelection;

        setNeedsProgressMonitor(true);
    }

    public boolean performFinish() {
        if(!performPageFinish()){
        	return false;
        }

		if(locPage.getSelection()){
			TraceMergeUIDialog dialog = new TraceMergeUIDialog(getShell(), LogUIPlugin.getResourceString("STR_EXISTING_DATA_DESC_2"));
			int ret = dialog.open();
			if(ret==Window.CANCEL){
				return false;
			}
			if(ret==2){
				bAppend = true;
			}
			if(ret==3){
				bAppend = false;
			}
		}

        errorMessage = null;

        try {
            getContainer().run(false, true, this);
        } catch (Exception e) {
            e.printStackTrace();
            errorMessage = e.toString();
        }

        if (errorMessage != null) {
			if(errorMessage.startsWith(CANCEL)){
				return false;
			}            
            if (reasonAction == null) {
                MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errorMessage);
            } else {
                Status errorStatus = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, reasonAction, null);

                ErrorDialog.openError(LogUIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errorMessage, errorStatus);
            }

            return false;
        }

        return true;
    }

    public void run(IProgressMonitor mon) {
        final String project = locPage.getProject();
        final String monitor = locPage.getMonitor();
        final String host = getHostName();
        boolean localHost = true;

        Node node;

        mon.beginTask(LogUIPlugin.getResourceString("IMPORT_LOG_FILE_PROGRESS_DESC"), 5);

        try {
            node = NodeFactory.createNode(host);

            Node lnode = NodeFactory.createNode("localhost");

            localHost = (node.getName().equals(lnode.getName()));
        } catch (Exception e) {
            errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_HOST_ERROR_");
            reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_HOST_REASON");
            reasonAction = HyadesUtil.change(reasonAction, "%1", host);

            return;
        }

        try {
            IWorkbenchPage page = UIPlugin.getActivePage();

            fViewer = ((PDProjectExplorer) (page.showView(PDPerspective.ID_PD_NAVIGATOR_VIEW)));
            fShell = fViewer.getShell();
            
			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			store.setValue(TraceConstants.SHOW_MON_KEY, true);
			store.setValue(TraceConstants.SHOW_NODE_KEY, true);
			
			PDProjectViewer pview = fViewer.getViewer();
			pview.showMonitorAction().setChecked(true, false);
			pview.showNodesAction().setChecked(true, false);
			            
        } catch (Exception e) {
            errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_DEST_ERROR_");
            reasonAction = null;

            return;
        }

        TRCAgent _agent = locPage.getAgentProxy()!=null ? locPage.getAgentProxy().getAgent():null;
        TRCProcessProxy _process;
        final Hashtable input = getUserInput();

        if (_agent == null) {
            IContainer container = PDCoreUtil.createContainer(new Path(project));

            TRCMonitor trcMonitor = PDCoreUtil.createMonitor(container, monitor);

            fViewer.getViewer().refresh(container);
            fViewer.getViewer().setExpandedState(container, true);

            String portNb = UIPlugin.getDefault().getPreferenceStore().getString(HyadesConstants.LOCALHOST_PORT);
            TRCNode trcNode = PDCoreUtil.createNode(trcMonitor, node.getName(), portNb);

            fViewer.getViewer().refresh(trcMonitor);
            fViewer.getViewer().setExpandedState(trcMonitor, true);

            _process = createProcess(container, trcNode);
            fViewer.getViewer().refresh(trcNode);
            fViewer.getViewer().setExpandedState(trcNode, true);

            _agent = createAgent(container, _process);
			if(errorMessage!=null && errorMessage.startsWith(CANCEL)){
				return;
			}
            
            fViewer.getViewer().refresh(_process);
            fViewer.getViewer().setExpandedState(_process, true);
        } else {
            _process = _agent.getAgentProxy().getProcessProxy();

            if (!bAppend) {
                _agent.getDefaultEvents().clear();
            }

        }

        final TRCAgent trcAgent = _agent;
        final TRCProcessProxy trcProcess = _process;

        mon.worked(1);

        if (localHost) {
 
		try {
			
			XMLLoader xmlLoader = new XMLLoader(trcAgent);
			xmlLoader.loadEvent("<CommonBaseEvents>".getBytes(), "<CommonBaseEvents>".getBytes().length);
			
			LocalLogParserLoader parserLoader = new LocalLogParserLoader(getLogParserInstance(), input);						
			parserLoader.setXMLLoader(xmlLoader);
			parserLoader.startParsing();
			xmlLoader.loadEvent("</CommonBaseEvents>".getBytes(), "</CommonBaseEvents>".getBytes().length);
			
			try {
				xmlLoader.cleanUp();				
			}
			catch(Exception exc)
			{
			}
						
	
		} catch (LogParserException l) {
			errorMessage = l.getMessage();
		}


            mon.worked(1);
            fViewer.getViewer().setExpandedState(trcProcess, true);
            fViewer.selectObject(trcAgent.getAgentProxy());

            mon.done();
        } else {
            node = PDCoreUtil.profileConnect(host, String.valueOf(Constants.CTL_PORT_NUM_SERVER));

            if (node == null) {
                mon.done();

                return;
            }

            mon.done();

            Process process = ProcessFactory.createProcess(node, ParserConstants.REMOTE_LOG_LOADER_RAC_APPLICATION, getParserCommand());
            process.addProcessListener(new ProcessListenerImpl(trcProcess, trcAgent.getAgentProxy()));

            final Agent agent = AgentFactory.createAgent(process, ParserConstants.REMOTE_LOG_LOADER_AGENT_NAME, HyadesConstants.LOG_AGENT_TYPE);
			final TRCAgentProxy aproxy = trcAgent.getAgentProxy();

            agent.setAutoAttach(true);
            agent.addAgentListener(new LogAgentListener(trcAgent, aproxy, trcProcess));

            final Agent errorAgent = AgentFactory.createAgent(process, ParserConstants.REMOTE_LOG_LOADER_ERROR_AGENT_NAME, HyadesConstants.LOG_AGENT_TYPE);

            errorAgent.setAutoAttach(true);
            errorAgent.addAgentListener(new ErrorAgentListener());

            try {
                process.launch();
            }
            // The RemoteLogParserLoader application is not configured in the RAC's serviceconfig.xml file: 
             catch (NoSuchApplicationException n) {
                
                   final String parserClass = getParserClass().trim();
                   if ((!parserClass.equals("com.ibm.etools.logging.parsers.WASV4ActivityLogParser")) && (!parserClass.equals("com.ibm.etools.logging.parsers.WASV5ActivityLogParser"))) {
                       errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
                       errorMessage = TString.change(errorMessage, "%1", host);
                       reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON");
                       reasonAction = TString.change(reasonAction, "%1", host);
                       return;
                   }
                   Hashtable userInput = getUserInput();
                   String was_home = ((String) (userInput.get("was_home")));
                   final String activityPath = ((String) (userInput.get("file_path")));
                   Process preV5Process = ProcessFactory.createProcess(node, "ActivityLogParser", ("com.ibm.etools.logging.was.WASActivityLogParser".concat(" \"").concat(activityPath).concat("\" \"").concat(was_home).concat("\"")));
                   preV5Process.addProcessListener(new ProcessListenerImpl(trcProcess, trcAgent.getAgentProxy()));
                   final Agent preV5ProcessAgent = AgentFactory.createAgent(preV5Process, "WAS Activity Log Agent", HyadesConstants.LOG_AGENT_TYPE);
                   preV5ProcessAgent.setAutoAttach(true);
                   preV5ProcessAgent.addAgentListener(new LogAgentListener(trcAgent, aproxy, trcProcess));
                   final Agent preV5ErrorAgent = AgentFactory.createAgent(preV5Process, "WAS Activity Log Error Agent", HyadesConstants.LOG_AGENT_TYPE);
                   preV5ErrorAgent.setAutoAttach(true);
                   preV5ErrorAgent.addAgentListener(new ErrorAgentListener());
                   try {
                       preV5Process.launch();
                   }
                   catch (Exception e) {
                       errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
                       errorMessage = TString.change(errorMessage, "%1", host);
                       reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON");
                       reasonAction = TString.change(reasonAction, "%1", host);
                       return;
                   }
                 
            } catch (Exception e) {
                errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_ERROR_");
                errorMessage = HyadesUtil.change(errorMessage, "%1", host);
                reasonAction = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_RAC_REASON");
                reasonAction = HyadesUtil.change(reasonAction, "%1", host);

                return;
            }
        }

        openLogView(trcAgent);
    }

    protected void addAllPages() {
        if (addParserPage) {
            addPage(parserPage);
        }

        addPage(hostPage);
        addPage(userPage);
        addPage(locPage);
    }

    protected boolean performPageFinish() {
        return hostPage.finish() && userPage.finish() && locPage.finish();
    }

    private String getParserCommand() {
        String cmd = getParserClass().concat(" ");

        Hashtable input = getUserInput();
        Enumeration enum = input.keys();

        while (enum.hasMoreElements()) {
			Object key = enum.nextElement();
			String value = input.get(key).toString();
			cmd = cmd.concat("\"").concat(key.toString()).concat("=");
			if(value.endsWith("\\") || value.endsWith("/")){
				cmd = cmd.concat(value.substring(0,value.length()-1)).concat("\" ");
			}
			else{
				cmd = cmd.concat(value).concat("\" ");			
			}            
        }

        return cmd.trim();
    }

    /*
     *
     */
    private TRCAgent createAgent(IContainer container, TRCProcessProxy process) {
        TRCAgentProxy agent = null;
        String name = getLogAgentName();
        String type = HyadesConstants.LOG_AGENT_TYPE;

        EList agents = process.getAgentProxies();
        Iterator i = agents.iterator();

        while (i.hasNext()) {
            agent = (TRCAgentProxy) i.next();

            if (agent.getName().equals(name)) {
				if(!locPage.getSelection()){

					String msg = LogUIPlugin.getResourceString("STR_EXISTING_DATA_DESC_1");
					msg = TString.change(msg, "%1", name);
					TraceMergeUIDialog dialog = new TraceMergeUIDialog(getShell(), msg);
					int ret = dialog.open();
					if(ret==Window.CANCEL){
						errorMessage = CANCEL;
						return null;					
					}
					if(ret==2){
						bAppend = true;
					}
					if(ret==3){
						bAppend = false;
					}

				}
                
                if (bAppend) {
                    return agent.getAgent();
                } else {
                    agent.getAgent().getDefaultEvents().clear();

                    return agent.getAgent();
                }
            }
        }

        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
        String rID = timestamp;

        rID = HyadesUtil.change(rID, " ", "");
        rID = HyadesUtil.change(rID, ":", "");

        String agentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(name).toString();

        agentName = HyadesUtil.change(agentName, " ", "");

        String pPath = process.eResource().getURI().toString();
        IPath path = new Path(pPath);

        if (path.segmentCount() > 1) {
            pPath = path.removeLastSegments(1).toString();
        }

        String fileName = agentName + "." + TraceConstants.AGENT_EXT;
        IPath filePath = new Path(pPath).append(fileName);

        URI uri = URI.createURI(filePath.toString());

        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList agExt = agDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        //create process element
        agent = factory.createTRCAgentProxy();
        agent.setName(name);
        agent.setType(type);

        agent.setProcessProxy(process);
        process.getAgentProxies().add(agent);

        TRCAgent ag = factory.createTRCAgent();

        ag.setAgentProxy(agent);
        ag.setType(type);
        agExt.add(ag);

        // adding support for multiple files
        SaveUtil.addDocument(agDoc);

        return ag;
    }

    /*
     *
     */
    private TRCProcessProxy createProcess(IContainer container, TRCNode node) {
        int pID = 0;

        String logPath = "";
        Hashtable elems = getUserInput();
        ArrayList index = getIndexList();

        Iterator iterator = index.iterator();

        if (iterator.hasNext()) {
            logPath += elems.get(iterator.next()).toString();
        }

        while (iterator.hasNext()) {
            logPath += (" " + elems.get(iterator.next()).toString());
        }

        TRCProcessProxy process = null;
        EList processes = node.getProcessProxies();
        Iterator i = processes.iterator();
        int counter = 0;

        while (i.hasNext()) {
            process = (TRCProcessProxy) i.next();
            counter++;

            if ((process.getName() != null) && process.getName().equals(logPath)) {
                return process;
            }
        }

        TRCMonitor monitor = node.getMonitor();

        String rID = (new Date()).toString();

        rID = HyadesUtil.change(rID, " ", "");
        rID = HyadesUtil.change(rID, ":", "");

        String processName = node.getMonitor().getName() + "_" + node.getName() + "_" + counter + "_" + pID;

        String fileName = processName + "." + TraceConstants.PROCESS_EXT;
        IPath path = container.getFullPath().append(fileName);
        URI uri = URI.createURI("platform:/resource" + path.toString());

        Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList pExt = pDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

        // adding support for multiple files
        SaveUtil.addDocument(pDoc);

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        process = factory.createTRCProcessProxy();
        process.setPid(pID);

        /* Ensure the runtime UUID is set on the process object */
        process.setRuntimeId(String.valueOf(pID));

        process.setName(logPath);
        process.setLaunchMode(0); //attach                       

        process.setNode(node);
        pExt.add(process);

        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

        event.setSource(process);
        event.setType(ProfileEvent.UNSPECIFIED);
        UIPlugin.getDefault().notifyProfileEventListener(event);

        return process;
    }

    /**
     * Initializes states of the controls from the preference store.
     */
    private void initializeValues() {
        IPluginRegistry reg = Platform.getPluginRegistry();

        IConfigurationElement[] config = reg.getConfigurationElementsFor("org.eclipse.hyades.logging.parsers", "logParser");

        for (int idx = 0; idx < config.length; idx++) {
            IConfigurationElement elem = config[idx];

            if (logParserId.trim().equals(elem.getAttribute("id").trim())) {
                _parser = new LogParserItem(elem.getAttribute("id"), elem.getAttribute("name"), elem.getAttribute("ui_name"), elem.getAttribute("description"));
                _parser.setConfigurationElement(elem);
                _parser.setClass(elem.getAttribute("class"));
                _parser.setUIType(elem.getAttribute("ui_type"));

                String iconPath = elem.getAttribute("icon");

                if ((iconPath != null) && !iconPath.equals("")) {
                    URL pluginURL = elem.getDeclaringExtension().getDeclaringPluginDescriptor().getInstallURL();

                    try {
                        URL url = new URL(pluginURL, iconPath);
                        ImageDescriptor image = ImageDescriptor.createFromURL(url);

                        _parser.setImageDescriptor(image);
                    } catch (Exception exc) {
                        // if the URL does not have a valid format, just log and ignore the exception
                        IStatus status = new Status(IStatus.ERROR, LogUIPlugin.getPluginId(), 2, "exception.malformedURL", exc); //$NON-NLS-1$

                        LogUIPlugin.getDefault().getLog().log(status);

                        continue;
                    }
                }

                IConfigurationElement[] felems = elem.getChildren("field");

                for (int i = 0; i < felems.length; i++) {
                    IConfigurationElement felem = felems[i];
                    ParserField pfield = new ParserField(felem.getAttribute("id"), felem.getAttribute("name"));

                    pfield.setDefaultValue(felem.getAttribute("defaultValue"));
                    pfield.setContextId(felem.getAttribute("helpContextId"));
                    pfield.setTooltip(felem.getAttribute("tooltip"));
                    pfield.setFilters(felem.getAttribute("browseType"));
                    pfield.useBrowse(felem.getAttribute("useBrowse").equals("true"));
                    pfield.setUiType(felem.getAttribute("ui_type"));

                    _parser.addEntry(pfield);
                }
            }
        }
    }

    private void openLogView(TRCAgent agent) {
    	
    	if(agent.getDefaultEvents().size() > 10000)
    	  return; //don't open the log view if the log files has more than 10000 records
    	  
		IAssociationMappingRegistry registry = TraceAssociationManager.getTraceViewMappingRegistry();
		IAssociationMapping mapping = registry.getAssociationMapping(HyadesUIPlugin.EP_ANALYZER_EXTENSIONS);
    	
		if(mapping == null)
			return;
    	
		IAssociationDescriptor[] array = mapping.getAssociationDescriptors(agent.getType());

		if(array.length > 0)
		{
			IAssociationDescriptor desc = array[0];
			try {
					
				IAction action = (IAction)desc.createImplementationClassInstance();
				
				if(action != null)
				{					
					action.run();
				}								
			} catch (Exception exc) {
				exc.printStackTrace();
			}			
		}
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    /* RKD:  ProcessListenerImpl to be instantiated and used by the importers */
    class ProcessListenerImpl implements ProcessListener {
        TRCAgentProxy trcAgent;
        TRCProcessProxy trcProcess;

        public ProcessListenerImpl(TRCProcessProxy process, TRCAgentProxy agent) {
            trcProcess = process;
            trcAgent = agent;
        }

        public void processExited(Process process) {
            trcAgent.setActive(false);
            trcAgent.setAttached(false);
            trcAgent.setMonitored(false);

            Display d = fShell.getDisplay();

            d.asyncExec(new Runnable() {
                    public void run() {
                        //update ui
                        fViewer.getViewer().refresh(trcAgent);
                    }
                });
        }

        public void processLaunched(Process process) {
            try {
                if (trcProcess != null) {
                    trcProcess.setPid(Integer.parseInt(process.getProcessId()));
                    trcProcess.setRuntimeId(process.getUUID());
                    trcProcess.setActive(true);

                    Display d = fShell.getDisplay();

                    d.asyncExec(new Runnable() {
                            public void run() {
                                //update ui
                                fViewer.getViewer().refresh(trcProcess);
                                fViewer.selectObject(trcAgent);
                            }
                        });
                }
            } catch (Exception exc) {
                exc.printStackTrace();
            }
        }
    }
    
    class LogAgentListener implements AgentListener{
		
		private TRCAgentProxy aproxy;
		private TRCAgent trcAgent ;
		private TRCProcessProxy trcProcess ;

		public LogAgentListener(TRCAgent trcAgent, TRCAgentProxy aproxy, TRCProcessProxy trcProcess) {
			this.aproxy = aproxy;
			this.trcAgent = trcAgent;
			this.trcProcess = trcProcess;
		}
		public void agentActive(Agent agent) {
	                    
			aproxy.setActive(true);
			aproxy.setAttached(true);
			aproxy.setMonitored(true);
			LoadersUtils.registerAgentInstance(aproxy, agent);
	

			Display d = fShell.getDisplay();

			d.asyncExec(new Runnable() {
					public void run() {
						//update ui
						ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

						event.setSource(aproxy);
						event.setType(ProfileEvent.START_MONITOR);
						UIPlugin.getDefault().notifyProfileEventListener(event);
					}
				});

			try {
				XMLTraceDataProcessor dataProcessor = new XMLTraceDataProcessor(trcAgent.getAgentProxy()){
					/* (non-Javadoc)
					 * @see org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor#dataServerExited()
					 */
					public void dataServerExited()
					{
						super.dataServerExited();
						LoadersUtils.deregisterAgentInstance(aproxy);
				
//											loadEvent("</CommonBaseEvents>".getBytes(), "</CommonBaseEvents>".getBytes().length);
					}
				};
//									dataProcessor.loadEvent("<CommonBaseEvents>".getBytes(), "<CommonBaseEvents>".getBytes().length);

				agent.startMonitoring(dataProcessor);
        
			} catch (Exception e) {
				final String agentErrorMessage = e.toString();

				Display d2 = fShell.getDisplay();

				d2.asyncExec(new Runnable() {
						public void run() {
							MessageDialog.openError(UIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
						}
					});
			}
		}

		public void agentInactive(Agent agent) {
			aproxy.setActive(false);
			aproxy.setAttached(false);
			aproxy.setMonitored(false);

			trcProcess.setActive(false);

			Display d = fShell.getDisplay();

			d.asyncExec(new Runnable() {
					public void run() {
						//update ui
						ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

						event.setSource(aproxy);
						event.setType(ProfileEvent.TERMINATE);
						UIPlugin.getDefault().notifyProfileEventListener(event);
					}
				});
		}

		public void error(Agent agent, String errorId, String message) {
			final String errMsg = errorId.concat(": ").concat(message);
			Display d = fShell.getDisplay();

			d.asyncExec(new Runnable() {
					public void run() {
						MessageDialog.openError(UIPlugin.getActiveWorkbenchShell(), LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errMsg);
					}
				});
		}

		public void handleCommand(Agent agent, CommandElement command) {
		}
	    
    }
    
    class ErrorAgentListener implements AgentListener{

		public ErrorAgentListener(){
		}
		
		public void agentActive(Agent agent) {
 
			try {
				agent.startMonitoring(new DataProcessor() {
					public void incommingData(byte[] buffer, int length, java.net.InetAddress peer) {
						final String agentErrorMessage = new String(buffer, 0, length);

						Display d = fShell.getDisplay();

						d.asyncExec(new Runnable() {
							public void run() {
								MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
							}
						});
					}

					public void incommingData(char[] buffer, int length, java.net.InetAddress peer) {
					}

					public void invalidDataType(byte[] data, int length, java.net.InetAddress peer) {
					}

					public void waitingForData() {
					}
				});
			} catch (Exception e) {
				final String agentErrorMessage = e.toString();

				Display d2 = fShell.getDisplay();

				d2.asyncExec(new Runnable() {
						public void run() {
							MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), agentErrorMessage);
						}
					});
			}
		}

		public void agentInactive(Agent agent) {
		}

		public void error(Agent agent, String errorId, String message) {
			final String errMsg = errorId.concat(": ").concat(message);
			Display d = fShell.getDisplay();

			d.asyncExec(new Runnable() {
					public void run() {
						MessageDialog.openError(fShell, LogUIPlugin.getResourceString("LOGGING_MESSAGE"), errMsg);
					}
				});
		}

		public void handleCommand(Agent agent, CommandElement command) {
		}
	}    
}
