/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import java.util.Comparator;

/**
 * This is the base application implementation with the basic correlator management funcitons in place.
 * 
 * @author Qiyan Li
 */
public abstract class BaseApplicationImpl implements IApplication {
    private static final int CORRELATOR_POOL_SIZE = 20; // the initial size of the correlator pool
    private static final int CORRELATOR_POOL_INCREMENT = 10; // the increment of the correlator pool
    private static final int CORRELATOR_STACK_SIZE = 10; // the initial size of each correlator stack
    private static final int CORRELATOR_STACK_INCREMENT = 10; // the increment of each correlator stack
    private static final int CORRELATOR_STACKS_SIZE = 10; // the initial size of the stack of correlator stacks
    private static final int CORRELATOR_STACKS_INCREMENT = 10; // the increment of the stack of correlator stacks
    private static final int CORRELATOR_ASSOCIATION_POOL_SIZE = 20; // the initial size of the association pool
    private static final int CORRELATOR_ASSOCIATION_POOL_INCREMENT = 10; // the increment of the association pool
    private long applicationCounter; // the application counter indicating the most recent event
    private Stack correlatorPool; // a pool of reuseable correlators
    private Stack correlatorStacks; // a collection of stacks of correlator stacks
    private IContextHandler handler; // the context handler for handling all events
    private Stack correlatorAssocationPool; // a pool of reuseable correlator associations

    /**
     * Instantiates a new instance of the correlator.  Sub-classes must override this method so that the correlator
     * pool can obtain new correlators when correlators are needed and the pool is empty.
     * 
     * @return  a newly-created instance of the correlator.
     */
    public abstract BaseCorrelatorDataImpl instantiateCorrelator();

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#initialize()
     */
    public void initialize() {

        applicationCounter = 0;
        correlatorPool = new Stack(CORRELATOR_POOL_SIZE, CORRELATOR_POOL_INCREMENT);
        correlatorStacks = new Stack(CORRELATOR_STACKS_SIZE, CORRELATOR_STACKS_INCREMENT);
        correlatorAssocationPool = new Stack(CORRELATOR_ASSOCIATION_POOL_SIZE, CORRELATOR_ASSOCIATION_POOL_INCREMENT);
        handler = null;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#createCorrelatorData()
     */
    public ICorrelatorData createCorrelatorData() {
        BaseCorrelatorDataImpl correlator; // the new correlator requested
        
        // Try to draw a correlator from the pool, or create a new instance if the pool is empty.
        correlator = (BaseCorrelatorDataImpl) correlatorPool.pop();
        if (correlator == null) {
            correlator = instantiateCorrelator();
        } else {
            correlator.invalidate();
        }
        correlator.setApplication(this);
        return correlator;
    }

    /**
     * Returns a correlator to the pool for later reuse.
     * 
     * @param correlator    the correlator to be returned
     */
    public void returnCorrelatorData(ICorrelatorData correlator) {

        correlatorPool.push(correlator);
    }

    /**
     * Creates a correlator association either by drawing an existing instance from the association pool,
     * or by instantiating a new instance.
     * 
     * @return  an instance of correlator association.
     */
    protected CorrelatorAssociation createCorrelatorAssociation() {

        // Try to draw an existing instance of correlator association from the pool, or instantiate one
        // if the pool is empty.
        CorrelatorAssociation association = (CorrelatorAssociation) correlatorAssocationPool.pop();
        return (association == null) ? new CorrelatorAssociation() : association;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#handleCall(Comparator)
     */
    public CorrelatorAssociation handleCall(Comparator key) {
        CorrelatorAssociation association; // the correlator association containing a parent correlator only
        CorrelatorStack stack; // the correlator stack for this key

        // Increment the application counter to indicate a new event.  The operation counter does not change
        // because there is no new frame being pushed onto the stack.  However, this event is not recorded
        // in the context data.
        incrementCounter();

        // Locate the stack for this key.  If there is no stack for the key, create a new stack.
        stack = findCorrelatorStack(key);
        if (stack == null) {
            stack = createCorrelatorStack(key);
        }
        
        // The complete association contains the parent only, since the child (receiver) context is not yet available
        // at this time.
        association = createCorrelatorAssociation();
        association.setParent((ICorrelatorData) stack.peek());
        association.setChild(null);
        if (handler != null) {
            handler.handleSend(association);
        }
        return association;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#handleReturn(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation handleReturn(Comparator key, ICorrelatorData parent) {
        CorrelatorAssociation association; // the correlator association containing a parent correlator only
        CorrelatorStack stack; // the correlator stack for this key

        // Locate the stack for this key, which should always be sucessful because a handleCall() should
        // happen before this method is called.
        stack = findCorrelatorStack(key);
        
        // The application counter is incremented to indicate a new event on this application.  However, the new
        // value of the counter is not recorded in any context data.
        incrementCounter();

        // The complete association consists of the parent only, which is the original parent frame, plus
        // any additional information that may be added by the remote method.
        association = createCorrelatorAssociation();
        association.setParent(parent);
        association.setChild(null);
        if (handler != null) {
            handler.handleReturn(association);
        }
        return association;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#pushChild(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation pushChild(Comparator key, ICorrelatorData child) {
        ICorrelatorData parent; // the parent correlator, i.e., the current top stack frame (if one exists)
        CorrelatorAssociation association; // the correlator association containing a parent and a child
        CorrelatorStack stack; // the correlator stack for this key

        // Locate an existing stack using this key, or create a new stack if none is found.
        stack = findCorrelatorStack(key);
        if (stack == null) {
            stack = createCorrelatorStack(key);
        }

        // Get the parent correlator from the top of the stack.  If there is no parent, the correlator is null.
        parent = (ICorrelatorData) stack.peek();

        // Increment the application and operation counters, and saves their current values in the child context,
        // which will then be pushed onto the stack.
        ((BaseCorrelatorDataImpl) child).setApplicationCounter(incrementCounter());
        ((BaseCorrelatorDataImpl) child).setOperationCounter(stack.incrementOperationCounter());
        stack.push(child);

        // The complete association consists of a parent (if any) and a child, which will then be handed to
        // the customized event handler.
        association = createCorrelatorAssociation();
        association.setParent(parent);
        association.setChild(child);
        if (handler != null) {
            handler.handleEntry(association);
        }
        return association;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#popChild(Comparator)
     */
    public CorrelatorAssociation popChild(Comparator key) {
        CorrelatorAssociation association; // the complete association containing the a parent and a child
        CorrelatorStack stack; // the correlator stack for this key
        BaseCorrelatorDataImpl child; // the child correlator, i.e., the current top frame on the stack

        // Locate the stack for this key, which should always be sucessful given that a pushChild() should
        // happen before this.
        stack = findCorrelatorStack(key);

        // Increment the application counter only, and saves the new value in the child context data.  The operation
        // counter remains unchanged because nothing is pushed onto the stack.
        child = (BaseCorrelatorDataImpl) stack.pop();
        child.setApplicationCounter(incrementCounter());
            
        // The complete association consists of a parent (if one exists) and a child.
        association = createCorrelatorAssociation();
        association.setParent((ICorrelatorData) stack.peek());
        association.setChild(child);
        if (handler != null) {
            handler.handleExit(association);
        }
        return association;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#pushParent(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation pushParent(Comparator key, ICorrelatorData parent) {
        CorrelatorStack stack; // the correlator stack for this key
        CorrelatorAssociation association; // the complete association containing a parent only

        // Locate an existing stack for the thread, or create a new one if none is found.
        stack = findCorrelatorStack(key);
        if (stack == null) {
            stack = createCorrelatorStack(key);
        }

        // Increment the application counter and the operation counter, since a new event has occurred and
        // a new frame has been pushed onto the stack.  However, neither counter is recorded in the context data.
        incrementCounter();
        stack.incrementOperationCounter();
        stack.push(parent);

        // The complete association consists of the parent context only.
        association = createCorrelatorAssociation();
        association.setParent(parent);
        association.setChild(null);
        if (handler != null) {
            handler.handleReceive(association);
        }
        return association;
    }

    /**
    * @see org.eclipse.hyades.collection.correlation.IApplication#popParent(Comparator)
     */
    public CorrelatorAssociation popParent(Comparator key) {
        CorrelatorStack stack; // the correlator stack for this key
        CorrelatorAssociation association; // the complete association containing a parent only
        
        // Locate the stack for this key, which should always be sucessful given that a pushParent() should
        // happen before this.
        stack = findCorrelatorStack(key);

        // Increment the application counter only, since a new event has occurred and no new frame has been pushed
        // onto the stack.  However, neither counter is recorded in the context data.
        incrementCounter();

        // The complete context consists of the parent only.  Note that this parent context is the current top frame
        // on the stack, i.e., the same frame as the parent context in the corresponding pushParent().
        association = createCorrelatorAssociation();
        association.setParent((ICorrelatorData) stack.pop());
        association.setChild(null);
        if (handler != null) {
            handler.handleReply(association);
        }
        return association;
    }

    /**
     * Increments and returns the application counter, which can be used to determine the total ordering of events
     * on a single instance of this application cross multiple threads.  Since this counter can be shared by multiple
     * threads, it must be synchronized.
     * 
     * @return the current application counter, plus 1.
     */
    protected synchronized long incrementCounter() {
        return ++applicationCounter;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#getActiveCorrelatorByKey(Comparator)
     */
    public ICorrelatorData getActiveCorrelatorByKey(Comparator key) {
        CorrelatorStack stack; // the correlator stack for this key

        // Locate the stack for this key, and return the most recent context data.  If no such key exists, return null.
        stack = findCorrelatorStack(key);
        return (stack == null) ? null : (ICorrelatorData) stack.peek();
    }

    /**
     * Creates a new stack for this key, and adds it to the existing collection of correlator stacks.  It is
     * the responsibility of the caller of this method to ensue that there is no stack already created for this key.
     * 
     * @return  a newly-created correlator stack.
     */
    protected CorrelatorStack createCorrelatorStack(Comparator key) {
        
        // Create a new correlator stack for this key, and add it to the collection of correlator stacks.
        CorrelatorStack newStack = new CorrelatorStack(key, CORRELATOR_STACK_SIZE, CORRELATOR_STACK_INCREMENT);
        correlatorStacks.push(newStack);
        return newStack;
    }
    
    /**
     * Removes the stack for this key from the collection of correlator stacks.  If there is no match, nothing
     * is done to the collection.
     */
    protected synchronized void destroyCorrelatorStack(Comparator key) {
        
        // Try to locate the stack for this key.  If found, swap the top of the stack with this entry.
        for (int i = 0; i <= correlatorStacks.top; i++) {
            if (((CorrelatorStack) correlatorStacks.stack[i]).getKey().equals(key)) {
                correlatorStacks.stack[i] = correlatorStacks.stack[correlatorStacks.top];
                correlatorStacks.stack[correlatorStacks.top--] = null;
                return;
            }
        }
        return;
    }
    
    /**
     * Tries to locate the correlator stack for this key.
     * 
     * @param key   the key for the target correlator stack
     * @return      the correlator stack for this key, or <code>null</code> if none is found.
     */
    protected synchronized CorrelatorStack findCorrelatorStack(Comparator key) {
        
        // Look through the stack to locate this key.
        for (int i = 0; i <= correlatorStacks.top; i++) {
            if (((CorrelatorStack) correlatorStacks.stack[i]).getKey().equals(key)) {
                return (CorrelatorStack) correlatorStacks.stack[i];
            }
        }
        return null;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.IApplication#releaseAssociation(CorrelatorAssociation)
     */
    public void releaseAssociation(CorrelatorAssociation association) {
        
        // Return the association to the correlator association pool.
        correlatorAssocationPool.push(association);
    }

    /**
     * Registers a context event handler with the application, so that all later events are handled through this
     * handler.
     * 
     * @param handler   a context event handler
     */
    public void registerHandler(IContextHandler handler) {
        this.handler = handler;
    }

    /**
     * Deregisters any current event handler.
     */
    public void deregisterHandler() {
        handler = null;
    }
}
