/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import java.util.Comparator;

import org.eclipse.hyades.collection.correlation.exceptions.InsufficientBufferLengthException;
import org.eclipse.hyades.collection.correlation.exceptions.InvalidDataException;
import org.eclipse.hyades.collection.correlation.exceptions.MoreDataRequiredException;

/**
 * This is the helper class for encoding and decoding the correlator messages.
 * 
 * @author Qiyan Li
 */
public class TransportMediator {

    /**
     * Creates an array of bytes for holding this correlator.
     * 
     * @param correlator    the correlator to be written
     * @param applicationId the application ID of the correlator
     * @return              an array of bytes with the application counter, operation counter, and application ID.
     */
    public static byte[] writeMessage(ICorrelatorData correlator, int applicationId) {
        byte[] buffer; // the data buffer where the converted correlator is stored
        int offset; // the offset of the next available byte in the data buffer

        // The context data is written to the buffer starting at the 7th byte.  Expand the buffer if necessary.
        buffer = new byte[1024];
        offset = 0;
        try {
            offset = correlator.writeBinary(buffer, 7, buffer.length);
        } catch (InsufficientBufferLengthException e) {
            byte[] newBuffer = new byte[e.getRequiredLength()];
            buffer = newBuffer;
            try {
                offset = correlator.writeBinary(buffer, 7, buffer.length);
            } catch (InsufficientBufferLengthException ePrime) {
                // This exception should not occur because of the new buffer just created.
            }
        }

        // Save the offset of the next empty byte in the buffer, i.e., the length of the data written.
        buffer[0] = (byte) (offset >> 8);
        buffer[1] = (byte) offset;

        // Save the correlator format.  (unused)
        buffer[2] = 0;

        // Save the flags.  (unused)
        buffer[3] = 0;

        // Save the header extension flags.  (unused)
        buffer[4] = 0;

        // Save the application ID.
        buffer[5] = (byte) (applicationId >> 8);
        buffer[6] = (byte) applicationId;
        byte[] newArray = new byte[buffer.length];
        System.arraycopy(buffer, 0, newArray, 0, buffer.length);
        return newArray;
    }
    
    /**
     * Handles the call context and transports the context information.
     * 
     * @param applicationId the application ID of the correlator
     * @param key           the key identifying the context to be transported
     * @param transport     the object to carry out the actual transport
     */
    public static void getDataStream(int applicationId, Comparator key, ITransport transport) {
        IApplication application; // the applications in a single environment
        byte[] buffer; // the buffer for the data to be sent
        CorrelatorAssociation association; // the complete association containing a parent only

        // Locate the application and then locate the correlator stack of interest.
        application = ServiceRegistry.getRegistry().getApplicationById(applicationId);
        association = application.handleCall(key);

        // Write the parent correlator to the data buffer.
        buffer = writeMessage(association.getParent(), applicationId);

        // Since the parent is still on the stack, and there is no child, release the record context only.
        ((BaseApplicationImpl) application).releaseAssociation(association);

        // The data in the buffer will be sent by a customized transport mechanism.
        transport.sendRequest(buffer, 0, buffer.length);
    }

    /**
     * Handles the receive context.
     * 
     * @param key       the key identifying the context to be transported
     * @param buffer    the data buffer holding the correlator from the sender
     * @param offset    the index of the next byte to be read from the buffer
     * @param length    the total length of the data buffer
     * @return          the index of the next byte to be read.
     */
    public static int setRemoteDataStream(Comparator key, byte[] buffer, int offset, int length)
        throws MoreDataRequiredException, InvalidDataException {
            
        // Find the application the data is targeting.
        int applicationId = ((int) buffer[offset + 5] & 0xff) << 8 | ((int) buffer[offset + 6] & 0xff);
        BaseApplicationImpl application =
            (BaseApplicationImpl) ServiceRegistry.getRegistry().getApplicationById(applicationId);
        if (application == null) {
            throw new InvalidDataException();
        }

        try {
            ICorrelatorData parent; // the parent correlator
            CorrelatorAssociation association; // the complete association containing a parent only

            // Draw a context data from the pool, and construct it from the data.
            parent = application.createCorrelatorData();
            offset = parent.readBinary(buffer, offset + 7, length);

            // Push the context data onto the stack.
            association = application.pushParent(key, parent);

            // Since the parent is on the stack, and there is no child, release the association only.
            application.releaseAssociation(association);
            return offset;

        } catch (InsufficientBufferLengthException e) {
            throw new MoreDataRequiredException();
        }
    }

    /**
     * Handles the reply context and transports the context information.
     * 
     * @param applicationId the application ID of the correlator
     * @param key           the key identifying the context to be transported
     * @param transport     the object to carry out the actual transport
     */
    public static void getRemoteDataStream(int applicationId, Comparator key, ITransport transport) {
        IApplication application; // the applications in a single environment
        byte[] buffer; // the buffer for the data to be sent
        CorrelatorAssociation association; // the complete association containing a parent only

        // Locate the application and the stack.
        application = ServiceRegistry.getRegistry().getApplicationById(applicationId);
        association = application.popParent(key);

        // Write the parent correlator to the buffer.
        buffer = writeMessage(association.getParent(), applicationId);

        // Since the parent is off the stack, and there is no child, release the parent and the association.
        ((BaseApplicationImpl) application).returnCorrelatorData(association.getParent());
        ((BaseApplicationImpl) application).releaseAssociation(association);

        // The data in the buffer will be sent by a customized transport mechanism.
        transport.sendReply(buffer, 0, buffer.length);
    }

    /**
     * Handles the return context.
     * 
     * @param key       the key identifying the context to be transported
     * @param buffer    the data buffer holding the correlator from the sender
     * @param offset    the index of the next byte to be read from the buffer
     * @param length    the total length of the data buffer
     * @return          the index of the next byte to be read.
     */
    public static int setDataStream(Comparator key, byte[] buffer, int offset, int length)
        throws MoreDataRequiredException, InvalidDataException {

        // Find the application the data is targeting.
        int applicationId = ((int) buffer[offset + 5] & 0xff) << 8 | ((int) buffer[offset + 6] & 0xff);
        BaseApplicationImpl application =
            (BaseApplicationImpl) ServiceRegistry.getRegistry().getApplicationById(applicationId);
        if (application == null) {
            throw new InvalidDataException();
        }

        try {
            ICorrelatorData parent; // the parent correlator
            CorrelatorAssociation association; // the complete association containing a parent only

            // Draw a correlator from the pool and fill it with the data.
            parent = application.createCorrelatorData();
            offset = parent.readBinary(buffer, offset + 7, length);

            // Handle the return event.
            association = application.handleReturn(key, parent);

            // Release the parent correlator and association, because there is no child context.            
            application.returnCorrelatorData(parent);
            application.releaseAssociation(association);
            return offset;
            
        } catch (InsufficientBufferLengthException e) {
            throw new MoreDataRequiredException();
        }
    }
}
