/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.hyades.loaders.util.*;
import org.eclipse.hyades.loaders.util.GenericLookupService;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.trace.TRCArrayClass;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCHeapDump;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCSourceInfo;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.impl.TRCArrayClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCFullMethodInvocationImpl;
import org.eclipse.hyades.models.trace.impl.TRCFullTraceObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCHeapDumpImpl;
import org.eclipse.hyades.models.trace.impl.TRCHeapObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.models.trace.impl.TRCProcessImpl;
import org.eclipse.hyades.models.trace.impl.TRCSourceInfoImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadImpl;
/**
 * @author slavescu
 */
public class TraceLookupService extends GenericLookupService {
    //~ Constructors -------------------------------------------------------------------------------

    public TraceLookupService() {
        super();
        types.add(TRCProcessImpl.class);
        types.add(TRCThreadImpl.class);
        types.add(TRCPackageImpl.class);
        types.add(TRCArrayClassImpl.class);
        types.add(TRCClassImpl.class);
        types.add(TRCMethodImpl.class);
        types.add(TRCHeapDumpImpl.class);
        types.add(TRCObjectEntry.class);

        types.add(TRCHeapObjectImpl.class);
        types.add(TRCFullTraceObjectImpl.class);

        types.add(VirtualObjectInfo.class);
        types.add(TRCSourceInfoImpl.class);
        types.add(TRCFullMethodInvocationImpl.class);
        types.add(ForwardInvocations.class);
        types.add(CallStackPerThread.class);
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     *
     */
    public boolean deregister(HierarchyContext context, Class type, Object key) {
		if (isObjectType(type)) {
            Map typeMap = getTypeMap(context, TRCObjectEntry.class, false);

            if (typeMap == null) {
                return false;
            }

            TRCObjectEntry objectEntry = (TRCObjectEntry) typeMap.get(key);

            if (objectEntry == null) {
                return false;
            }

            return objectEntry.remove(type);
        }

        return super.deregister(context, type, key);
    }

    /**
     *
     */
    public boolean deregister(HierarchyContext context, Class type) {
		if (isObjectType(type)) {
            Map typeMap = getTypeMap(context, TRCObjectEntry.class, false);

            if (typeMap == null) {
                return false;
            }

            boolean ret = false;

            for (Iterator iter = typeMap.values().iterator(); iter.hasNext();) {
                TRCObjectEntry objectEntry = (TRCObjectEntry) iter.next();

                if (objectEntry.remove(type)) {
                    ret = true;
                }
            }

            return ret;
        }

        return super.deregister(context, type);
    }

    private boolean isObjectType(Class type) {
		return type == TRCHeapObjectImpl.class || type == TRCFullTraceObjectImpl.class;
	}

	/**
     *
     */
    public Object locate(HierarchyContext context, Class type, Object key) {
		if (isObjectType(type)) {
            Map typeMap = getTypeMap(context, TRCObjectEntry.class, false);

            if (typeMap == null) {
                return null;
            }

            TRCObjectEntry objectEntry = (TRCObjectEntry) typeMap.get(key);

            if (objectEntry == null) {
                return null;
            }

            return objectEntry.get(type);
        }

        return super.locate(context, type, key);
    }

    /**
     *
     */
    protected Map getTypeMap(HierarchyContext context, Class type, boolean create) {
		if (isObjectType(type)) {
            return super.getTypeMap(context, TRCObjectEntry.class, create);
        }

        return super.getTypeMap(context, type, create);
    }

    protected boolean doProcessObject(Map typeMap, Class type, Object object, boolean remove) {
        if (object == null) {
            return false;
        }

        //        Class type = object.getClass();
        int classNameInt = LoadersUtils.getHashCode(type.getName());

        switch (classNameInt) {
        case TraceConstants.TRCFullMethodInvocationImpl_int:

            TRCFullMethodInvocationImpl inv = (TRCFullMethodInvocationImpl) object;

            if (inv.getThread() == null) {
                return false;
            }

            String id = inv.getThread().getId() + "-" + inv.getTicket();

            addOrRemove(typeMap, LoadersUtils.getLookUpKey(id), object, remove);

            return true;

        case TraceConstants.TRCProcessImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCProcess) object).getId()), object, remove);

            return true;

        case TraceConstants.TRCThreadImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCThread) object).getId()), object, remove);

            return true;

        case TraceConstants.TRCPackageImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCPackage) object).getName()), object, remove);

            return true;

        case TraceConstants.TRCClassImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCClass) object).getId()), object, remove);

            return true;

        case TraceConstants.TRCArrayClassImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCArrayClass) object).getId()), object, remove);

            return true;

        case TraceConstants.TRCMethodImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCMethod) object).getId()), object, remove);

            return true;

        case TraceConstants.TRCHeapObjectImpl_int:
        case TraceConstants.TRCFullTraceObjectImpl_int:

            Object key = LoadersUtils.getLookUpKey(((TRCObject) object).getId());
            TRCObjectEntry objectEntry = (TRCObjectEntry) typeMap.get(key);

            if (objectEntry == null) {
                objectEntry = new TRCObjectEntry();
                addOrRemove(typeMap, key, objectEntry, remove);
            }

            objectEntry.put(type, (TRCObject) object);

            return true;

        case TraceConstants.TRCObjectEntry_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCObjectEntry) object).getId()), object, remove);

            return true;

        //		  case TraceConstants.VirtualObjectInfo_int:
        //			  addOrRemove(typeMap, LoadersUtils.getLookUpKey(((VirtualObjectInfo) object).objId), object, remove);
        //
        //			  return true;
        case TraceConstants.TRCSourceInfoImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCSourceInfo) object).getLocation()), object, remove);

            return true;

        case TraceConstants.TRCHeapDumpImpl_int:
            addOrRemove(typeMap, LoadersUtils.getLookUpKey(((TRCHeapDump) object).getId()), object, remove);

            return true;

        default:
            break;
        }

        return false;
    }
}
