package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import org.eclipse.hyades.logging.core.ILogger;
import org.eclipse.hyades.logging.events.IExtendedDataElement;

/**
 * AbstractAccessLogParser is the abstract superclass for the ApacheAccessLogParser.
 * <p>
 * For each access log record, this class parses the client IP address, userid, time 
 * stamp, method, file name, return code, and file size.  This parsed information and 
 * information about the local machine (machine where the parse is performed, but not 
 * necessarily the machine that produced the access log) is used to produce a
 * {@link org.eclipse.hyades.logging.events.ICommonBaseEvent} object for each record.
 */
public abstract class AbstractAccessLogParser extends Parser {

    //  Constants

    /**
     * Constant for target time format (XML DateTime).
     */
    private final static String TARGET_FORMAT = "yyyy-MM-dd HH:mm:ss";

    /**
     * Constant for time stamp format.
     */
	private final static String TIME_STAMP_FORMAT = "dd/MMM/yyyy:HH:mm:ss";

    // Variables to hold values parsed from access log records.  

    /**
     * Parsed value of the client IP address field. 
     */
    protected String host;

    /**
     * Parsed value of the method field. 
     */
    protected String method;

    /**
     * Parsed value of the file name field. 
     */
    protected String file;

    /**
     * Parsed value of the return code field. 
     */
    protected String rcode;

    /**
     * Parsed value of the file size field. 
     */
    protected String fileSize;

    /**
     * Parsed value of the userid field. 
     */
    protected String userid;

    /**
     * Parsed value of the message portion of the access log record.  The message includes
     * the method, file name, protocol, return code, and file size fields. 
     */
    protected String message;

    /**
     * Parsed value of the time stamp field converted to the target String. 
     */
    protected StringBuffer creationTime;

    // Variable to hold user-supplied value

    /**
     * User-supplied value of the version of the web server product.  Produced in 
     * the setUserInput method of the subclass.
     */
    protected String productVersion;

    // Variable to hold product information

    /**
     * Web server product name and version supplied by the subclass.   
     */
    protected String sourceId;

    // Variables associated with counting messages with identical time stamp values

    /**
     * XML DateTime String representation of time stamp in previous access log record. 
     */
    private String currentCreationTime = "";

    /**
     * Count of records with identical time stamp values. 
     */
    private long duplicateCreationTimeCounter = 0;

    /**
     * Parses each access log record and produces a CommonBaseEvent object that is 
     * written to the logger.
     *
     * @param argLogger  logger to which CommonBaseEvent objects are written  
     * @exception LogParserException thrown if the parser is unable to parse the access log
     */
    public void parse(ILogger argLogger) throws LogParserException {
        super.parse(argLogger);
        
        try {
            while (curLine != null) {
                reset();
                if (parseData()) {
                	
					messages[arrayIndex].init();
					messages[arrayIndex].setSourceComponentId(eventFactory.createComponentIdentification());
					messages[arrayIndex].getSourceComponentId().init();
					messages[arrayIndex].setMsgDataElement(eventFactory.createMsgDataElement());
					messages[arrayIndex].getMsgDataElement().init();
                	
                    messages[arrayIndex].getSourceComponentId().setLocation(localHostId);
                    messages[arrayIndex].getSourceComponentId().setLocationType(localHostIdFormat);
                    messages[arrayIndex].getSourceComponentId().setComponent(sourceId);

                    // Compose the StringBuffer used to set the id field in the CommonBaseEvent
                    StringBuffer idString = new StringBuffer(localHostId);
                    if (file_path != null) {
                        idString.append(' ');
                        idString.append(file_path);
                    }
                    if (localHostName != null) {
                        idString.append(' ');
                        idString.append(localHostName);
                    }
                    idString.append(' ');

                    // Increment the message counter when duplicate timestamp values are detected.
                    if (currentCreationTime.equals(creationTime.toString().trim()))
                        messages[arrayIndex].setSequenceNumber(++duplicateCreationTimeCounter);
                    else {
                        currentCreationTime = creationTime.toString().trim();
                        duplicateCreationTimeCounter = 0;
                    }

                    //CBE               idString.append(creationTime.toString());
                    //CBE               idString.append("_");
                    //CBE               idString.append(messages[arrayIndex].getSequenceNumber());
                    //CBE               messages[arrayIndex].setGlobalInstanceId(idString.toString());

                    messages[arrayIndex].getSourceComponentId().setSubComponent(ParserConstants.UNKNOWN);
                    messages[arrayIndex].getMsgDataElement().setMsgIdType(ParserConstants.NOT_APPLICABLE);
                    messages[arrayIndex].getMsgDataElement().setMsgId(ParserConstants.NONE);
                    //CBE               messages[arrayIndex].setLocaleOfOrigin(originLocale);
                    messages[arrayIndex].setCreationTime(creationTime.toString());

                    messages[arrayIndex].setMsg(message);

                    messages[arrayIndex].getMsgDataElement().setMsgLocale(ParserConstants.LOCALE_EN_US);
                    messages[arrayIndex].setSeverity(ParserConstants.SEVERITY3);

                    // construct the token array
                    if (method != null) {
                        IExtendedDataElement methodToken = createStringEDE("method", method);
                        messages[arrayIndex].addExtendedDataElement(methodToken);
                    }
                    if (host != null) {
                        IExtendedDataElement clientToken = createStringEDE(ParserConstants.CLIENT, host);
                        messages[arrayIndex].addExtendedDataElement(clientToken);
                    }
                    if (file != null) {
                        IExtendedDataElement fileToken = createStringEDE(ParserConstants.FILE, file);
                        messages[arrayIndex].addExtendedDataElement(fileToken);
                    }
                    if (rcode != null) {
                        IExtendedDataElement rcodeToken = createStringEDE("return_code", rcode);
                        messages[arrayIndex].addExtendedDataElement(rcodeToken);
                    }
                    if (fileSize != null) {
                        IExtendedDataElement fileSizeToken = createStringEDE("file_size", fileSize);
                        messages[arrayIndex].addExtendedDataElement(fileSizeToken);
                    }
                    if (userid != null) {
                        IExtendedDataElement useridToken = createStringEDE("userid", userid);
                        messages[arrayIndex].addExtendedDataElement(useridToken);
                    }

                    // If an array of CommonBaseEvents objects is to be written to the logger,
                    // substitute the commented out code for the two instructions
                    // thereafter following.

                    //if (arrayIndex == PDMessageArraySize - 1) {
                    //    logger.write(messages);
                    //    for (int i = 0; i < PDMessageArraySize; i++) {
                    //        messages[i].init();
                    //    }
                    //    arrayIndex = 0;
                    //}
                    //else
                    //    arrayIndex++;

                    logger.write(messages[arrayIndex]);
                }
                curLine = readLine();
            }
            writePartialArray(); // for now just closes file
        }
        catch (Throwable throwable) {
            LogParserException lpe = new LogParserException(ParserUtilities.getResourceString("ACCESS_LOG_PARSER_ERROR_"), throwable);
            lpe.fillInStackTrace();
            throw lpe;
        }
    }

    /**
     * Parses a time stamp from an access log record, produces a Date object from the
     * parsed information, and converts the Date object into a XML DateTime String.
     *
     * @param start  Position of time stamp in curLine (always set to 0) 
     * @param mode   0 if time stamp is to be parsed; 1 if time stamp is only checked for validity
     * @return       true if time stamp is valid; false otherwise 
     */
    protected boolean parseDate(int start, int mode) {
        String temp = "";
        curLine = curLine.trim();
        curLine = curLine + "\n";
        if (isChar(curLine, start + 0, '[') && isNum(curLine, start + 1, 2) && isChar(curLine, start + 3, '/') && isLet(curLine, start + 4, 3) && isChar(curLine, start + 7, '/') && isNum(curLine, start + 8, 4) && isChar(curLine, start + 12, ':') && isNum(curLine, start + 13, 2) && isChar(curLine, start + 15, ':') && isNum(curLine, start + 16, 2) && isChar(curLine, start + 18, ':') && isNum(curLine, start + 19, 2) && isChar(curLine, start + 21, ' ') && (isChar(curLine, start + 22, '-') || isChar(curLine, start + 22, '+')) && isNum(curLine, start + 23, 4) && isChar(curLine, start + 27, ']')) {
            temp += curLine.substring(1, 21);
            SimpleDateFormat formatter = new SimpleDateFormat(TIME_STAMP_FORMAT, Locale.US);
            ParsePosition parsePosition = new ParsePosition(0);
            Date artifactDate = formatter.parse(temp, parsePosition); 
            if (artifactDate != null) {
                char sign = curLine.charAt(22);
                int zoneOffset = Integer.parseInt(curLine.substring(23, 25));
                int zoneOffsetMinutes = Integer.parseInt(curLine.substring(25, 27));
                if (sign == '+')
                    artifactDate.setTime(artifactDate.getTime());   
                else
                    artifactDate.setTime(artifactDate.getTime());  
                if (mode == 0) {
                    formatter = new SimpleDateFormat(TARGET_FORMAT);
                    creationTime = new StringBuffer(formatter.format(artifactDate));
                    creationTime.replace(10, 11, "T");
                    creationTime.append(ParserConstants.SIX_ZERO);

                    // Time Zone (+/-hh:mm)

                    creationTime.append(sign);

                    String numberHours = String.valueOf(zoneOffset);
                    if (numberHours.length() == 1)
                        creationTime.append("0");
                    creationTime.append(numberHours);
                    creationTime.append(":");

                    String numberMinutes = String.valueOf(zoneOffsetMinutes);
                    if (numberMinutes.length() == 1)
                        creationTime.append("0");
                    creationTime.append(numberMinutes);

                    curLine = curLine.substring(28);
                }
                return true;
            }
        }
        return false;
    }

    /**
     * Resets the parsed values of a log record before next record is parsed.
     */
    protected void reset() {
        userid = null;
        method = null;
        rcode = null;
        host = null;
        file = null;
        fileSize = null;
        message = null;
        creationTime = null;
    }

    /**
     * Main parsing routine for an access log record.
     *
     * @return   true if time stamp is successfully parsed; false otherwise 
     */
    protected boolean parseData() {
        curLine = curLine.trim();
        curLine = curLine + "\n";

        // parse client ip address
        host = curLine.substring(0, curLine.indexOf(" "));
        curLine = curLine.substring(curLine.indexOf(" ")).trim() + "\n";

        // skip RFC1413 identity of client
        curLine = curLine.substring(curLine.indexOf(" ")).trim() + "\n";

        // parse userid of person requesting document
        userid = curLine.substring(0, curLine.indexOf(" "));
        if (userid.equals("-"))
            userid = null;

        // parse timestamp
        curLine = curLine.substring(curLine.indexOf("["));
        curLine = curLine.trim();
        curLine = curLine + "\n";
        if (!parseDate(0, 0))
            return false;
        curLine = curLine.trim();
        curLine = curLine + "\n";

        // parse method
        int firstQuote = curLine.indexOf('\"');
        method = curLine.substring(firstQuote + 1, curLine.indexOf(" ", firstQuote + 1));

        // parse message
        message = curLine.substring(firstQuote).trim();
        curLine = curLine.substring(curLine.indexOf(' ', firstQuote));
        curLine = curLine.trim();

        // parse file
        file = curLine.substring(0, curLine.indexOf(' '));
        curLine = curLine.substring(curLine.indexOf('\"') + 1);
        curLine = curLine.trim();

        // parse return code and file size
        rcode = curLine.substring(0, curLine.indexOf(' '));
        curLine = curLine.substring(curLine.indexOf(' ')).trim();

		// parse file size
		int nextSpace = curLine.indexOf(' ');
		if (nextSpace != -1) {
			fileSize = curLine.substring(0, nextSpace);
		} else {
			fileSize = curLine.trim();
		}

        return true;
    }

}