package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.InetAddress;
import java.util.Hashtable;

import org.eclipse.hyades.logging.core.ILogger;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.hyades.logging.events.IExtendedDataElement;
import org.eclipse.hyades.logging.events.ISimpleEventFactory;
import org.eclipse.hyades.logging.events.SimpleEventFactoryImpl;

/**
 * Parser is the abstract superclass for AbstractAccessLogParser and
 * AbstractErrorLogParser.  Subclasses should implement setUserInput(Hashtable table)
 * and parse(ILogger argLogger), first calling super.parse(argLogger) and 
 * super.setUserInput(table) if the functions implemented here are needed.  Abstract
 * methods, getName() and getVersion(), must also be implemented by concrete subclasses
 * to identify the type of log associated with the parser and the version of
 * the parser, respectively.
 */
public abstract class Parser {

    /**
     * Constant for String "Unknown Host".
     */
    protected final static String UNKNOWN_HOST = "Unknown Host";
    
    
   /**
     * Final static field to hold the default CommonBaseEvent array size.
     * Currently, single CommonBaseEvent objects are written to an ILogger; for
     * that reason this field is set to 1.  If in the future, arrays are 
     * written to the ILogger, this field will hold the default size of 
     * such arrays.
     */
    public final static int defaultArraySize = 1;
    
    /**
     * Field to hold the CommonBaseEvent array size.  Currently, single CommonBaseEvent 
     * objects are written to an ILogger, and there is no mechanism to change 
     * the value of this field; for that reason this field is set to a default
     * array size of 1.  If in the future, arrays are written to the ILogger, 
     * this field will hold the size of such arrays.
     */
    protected int MessageArraySize = defaultArraySize;

    /**
     * Array of CommonBaseEvent objects to be written to the ILogger.  Currently, single 
     * CommonBaseEvent objects are written to an ILogger.
     */
    protected ICommonBaseEvent[] messages;
    
    /**
     * Field to hold index into CommonBaseEvent array. Currently, single 
     * CommonBaseEvent objects are written to an ILogger.  For that reason, this
     * field is set to 0 in the parse(ILogger argLogger) method.
     */
    protected int arrayIndex;
    
    // Variable to hold values parsed from log records.
    
    /**
     * Parsed value of the locale where the log was produced.  This field
     * may be set to the locale of the parsing machine if the actual locale
     * information is not available in the log.
     */
    protected String originLocale;
    
    // Variables to hold information about the parsing machine.
    
    /**
     * IP address of the machine parsing the log file. 
     */
    protected String localHostId;

    /**
     * Format of the IP address of the machine parsing the log file.  This
     * field will be set either to the String "IPV4" or the String "IPV6".
     */
    protected String localHostIdFormat;
    
    /**
     * Fully-qualified host name of the machine parsing the log file. 
     */
    protected String localHostName;

    // Variable associated with the log file to be parsed.
    
    /**
     * Path and name of log file to be parsed. 
     */
    protected String file_path;
    
    /**
     * Reader for log file to be parsed. 
     */
    protected BufferedReader logFile;
    
    /**
     * String to hold lines read from log file to be parsed. 
     */
    protected String curLine = "";
    
    /**
     * Factory for creating events
     */
	protected static ISimpleEventFactory eventFactory = SimpleEventFactoryImpl.getInstance();

    /**
     * ILogger to which CommonBaseEvent objects produced during parsing
     * will be written.  The value for this field is passed to the parser 
     * when the parse(ILogger argLogger) method is called.
     */
    protected ILogger logger;
    
    /**
     * This function is called to perform the parsing of the log file.  
     * Minimal function is provided by this abstract superclass.  Subclasses 
     * will override this method to provide parsing function.
     *
     * @param argLogger  logger to which CommonBaseEvent objects are written  
     * @exception LogParserException thrown if exception is detected while executing function of this superclass
     */
    public void parse(ILogger argLogger) throws LogParserException {
        try {
            logger = argLogger;
            curLine = readLine();
            getLocalHostId();
            originLocale = getLocale();
            arrayIndex = 0;
        }
        catch (Throwable e) {
            LogParserException lpe = new LogParserException(e);
            lpe.fillInStackTrace();
            throw lpe;
        }
    }

    /**
     * Abstract method used to retrieve the name of the parser.  Each concrete
     * subclass will implement this method to provide information about the type of log 
     * that it parses.
     *
     * @return the name of this parser 
     */
    public abstract String getName();

    /**
     * Abstract method used to retrieve the version of the parser.  Each concrete
     * subclass will implement this method to provide its version. 
     *
     * @return the version of this parser 
     */
    public abstract String getVersion();

    /**
     * This function is called to provide user-specified information to the parser.
     * Currently, this superclass method handles the path and name of the log file
     * to be parsed.  Subclasses will override this method to handle additional
     * user-specified information.
     *
     * @param table  Hashtable containing keys and values of user-specified information  
     * @exception LogParserException thrown if user-specified path and name for the log file does not exist on the parsing machine
     */
    public void setUserInput(Hashtable table) throws LogParserException {
        file_path = null;
        
        // Commented code in this method could be used to allow user specification
        // of the size of arrays to be written to the ILogger.
        //String array_size = null;
        try {
            file_path = (String) table.get(ParserConstants.FILE_PATH_KEY);
            if (file_path != null)
                logFile = new BufferedReader(new FileReader(file_path));
            //array_size = (String)table.get("array_size");
            //if (array_size != null)
            //    PDMessageArraySize = Integer.parseInt(array_size);
            messages = new ICommonBaseEvent[MessageArraySize];
            for (int i = 0; i < MessageArraySize; i++) {
                messages[i] = eventFactory.createCommonBaseEvent();
            }
        }
        catch (FileNotFoundException fnfe) {
            LogParserException lpe = new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_INITIALIZATION_ERROR_", file_path), fnfe);
			
            lpe.fillInStackTrace();
            file_path = null;
            throw lpe;
        }
        //catch(NumberFormatException nfe)
        //{
        //    LogParserException lpe = new LogParserException("array_size value, " + array_size + " ,must be an integer.", nfe);
        //    lpe.fillInStackTrace();
        //    throw lpe;
        //}
    }
    
    // utility methods

    /**
     * Function to determine IP address and fully-qualified host name
     * of the machine parsing the log file. 
     */
    protected void getLocalHostId() {
        try {
            localHostId = java.net.InetAddress.getLocalHost().getHostAddress();
            if (localHostId.indexOf(':') != -1)
                localHostIdFormat = ParserConstants.HOST_ID_FORMAT_IPV6;
            else
                localHostIdFormat = ParserConstants.HOST_ID_FORMAT_IPV4;
        }
        catch (Exception e) {
            localHostId = UNKNOWN_HOST;
            localHostIdFormat = ParserConstants.NONE;
        }
        try {
            localHostName = InetAddress.getByName(localHostId).getHostName();  
        }
        catch (Exception e) {
            localHostName = null;
        }
    }

    /**
     * Function to determine the locale (language and country)
     * of the machine parsing the log file. 
     */
    protected String getLocale() {
        String originLanguage = System.getProperty("user.language");
        String originRegion = System.getProperty("user.region");
        String originLocale = "Unknown";
        if (originLanguage != null && originRegion != null)
            originLocale = originLanguage + "-" + originRegion;
        return originLocale;
    }

    /**
     * Function to write a partial array (that is, smaller than the PDMessageArraySize)
     * to the ILogger and close the the log file.  Currently, this method only calls
     * the closeFiles() method. 
     */
    protected void writePartialArray() {
        //if (arrayIndex != 0) {
        //    PD_Message[] partialArray = new PD_Message[arrayIndex];
        //    for (int i = 0; i < arrayIndex; i++) {
        //        partialArray[i] = messages[i];
        //    }
        //    logger.write(partialArray);
        //}
        closeFiles();
    }

    /**
     * Checks that the character in the line at the specified position is 
     * a number.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @return true if the character at the specified position in the string is a number
     */
    protected boolean isNum(String line, int pos) {
        try {
            return Character.isDigit(line.charAt(pos));
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Checks that the characters in the line from the specified position 
     * to that position plus (num - 1)  are all numbers.
     * 
     * @param line The string to check
     * @param pos The position at which to start the check
     * @param num The number of characters to check
     * @return True if all the characters checked are numbers
     */
    protected boolean isNum(String line, int pos, int num) {
        for (int i = 0; i < num; i++) {
            if (!isNum(line, pos + i))
                return false;
        }
        return true;
    }

    /**
     * Checks that the character in the line at the specified position is 
     * a letter.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @return true if the character at the specified position in the string is a letter
     */
    protected boolean isLet(String line, int pos) {
        try {
            return Character.isLetter(line.charAt(pos));
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Checks that the characters in the line from the specified position 
     * to that position + (num - 1) are all letters.
     * 
     * @param line The string to check
     * @param pos The position at which to start the check
     * @param num The number of characters to check
     * @return true if all the characters checked are letters
     */
    protected boolean isLet(String line, int pos, int num) {
        for (int i = 0; i < num; i++) {
            if (!isLet(line, pos + i))
                return false;
        }
        return true;
    }

    /**
     * Checks that the character in the line at the specified position is 
     * the specified character.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @param c The character that should be in this position in the string
     * @return true if the character at the specified position in the string is the specified character
     */
    protected boolean isChar(String line, int pos, char c) {
        try {
            return line.charAt(pos) == c;
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Closes the log file.
     */
    public void closeFiles() {
        try {
            if (logFile != null)
                logFile.close();
        }
        catch (Exception e) {
            // no action required
        }
    }

    /**
     * Reads in the next line from the log file and returns same. 
     *
     * @return   the next line
     */
    protected String readLine() {
        String msgLine = null;
        try {
            msgLine = logFile.readLine();
        }
        catch (IOException e) {
            return null;
        }
        return msgLine;
    }
    
	/**
	 * Method createStringEDE.  Creates an object which implements the 
	 * IExtendedDataElement interface and fills in the name/value pair.
	 * This functions handles values that are single strings.
	 * 
	 * @param name Name portion of the name/value pair
	 * @param value Value portion of the name/value pair
	 * @return IExtendedDataElement
	 */
    protected static IExtendedDataElement createStringEDE(String name, String value) {
		IExtendedDataElement ede = eventFactory.createExtendedDataElement();
		ede.setName(name);
		ede.setType("string");
		String[] values = new String[1];
		values[0] = value;                       
		ede.setValues(values);
		return ede;
    }
    
}
