/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sd.ui.internal.views;

import java.util.*;

import org.eclipse.hyades.sd.ui.internal.model.*;
import org.eclipse.hyades.sd.ui.internal.util.*;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.*;


public class SDViewerUI extends Canvas
						implements ControlListener
	            					, SelectionListener
	            					, PaintListener
	            					, KeyListener
	            					, ISelectionProvider
{
	private Rectangle parentBounds;
	private Graph graph=null;
	int endOfTime = 0;
	int endOfColumns = 16;
	private MouseEvent idEvent = null;
	private DrawAreaToolTip toolTip=null;
	private DrawAreaOverview drawAreaOverview;
	private DrawAreaWeightScale drawAreaWeightScale;
	private DrawAreaFindDialog drawAreaFindDialog=null;	
    private NodeLink selectedNodeLink=null;
	private ColumnNode selectedColumnNode=null;
	private ColumnFigure selectedColumnFigure=null;
	private Composite graphComposite;
    private Map columnPassages;
    private int width=0;
    private Canvas canvas = null;
    private SDViewerPage sdViewerPage;
    private ArrayList _viewsActions = new ArrayList();
	private final String GROUP_ADD_VIEWS = "group.add.views";
	
	public SDViewerUI(Composite c, Graph g, SDViewerPage viewerPage) {
		super(c, 0);
		sdViewerPage = viewerPage;
		parentBounds = c.getClientArea();     //get page book client area
		int weightScaleWidth=LogCDrawUtils.getWeightScaleWidth();
		
		graphComposite = new Composite(this,SWT.V_SCROLL|SWT.H_SCROLL);
		graphComposite.setLocation(weightScaleWidth,0);
		graphComposite.setSize(parentBounds.width-weightScaleWidth, parentBounds.height);
		graphComposite.setBackground(LogCDrawUtils.getBackgroundColor());
		
		canvas = new Canvas(graphComposite, 0);
		canvas.setSize(graphComposite.getSize());
	    canvas.setBackground(LogCDrawUtils.getBackgroundColor());
	   	canvas.setLocation(0,0);
	    canvas.setVisible(true);
	    canvas.setEnabled(true);
		
		ScrollBar vBar = graphComposite.getVerticalBar();
		vBar.addSelectionListener(this);
		vBar.setMinimum(0);
		vBar.setIncrement(LogCDrawUtils.getTimeUnit());
		
		ScrollBar hBar = graphComposite.getHorizontalBar();
		hBar.addSelectionListener(this);
		hBar.setMinimum(0);
		hBar.setIncrement(20);
		
		drawAreaWeightScale = new DrawAreaWeightScale(this,SWT.NULL);
		drawAreaWeightScale.setSize(weightScaleWidth,parentBounds.height);
		
		drawAreaOverview = new DrawAreaOverview(graphComposite,new Point(graphComposite.getBounds().width-vBar.getSize().x,graphComposite.getBounds().height-hBar.getSize().y));
        
        columnPassages = new HashMap();
		
		reloadGraph(g);
		
		this.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				SDViewerUI.this.widgetDisposed(e);
			}
		});
		this.addPaintListener(this);
		this.addControlListener(this);
		graphComposite.addKeyListener(this);
		canvas.addKeyListener(this);
	}
	
	public void reloadGraph(Graph newGraph){
	   resetGraph();
	   width=0;
	   graph = newGraph;
	   if (graph != null){
			buildGraph();
			drawAreaWeightScale.setWeights(graph.getIncrementWeights());
			LogCDrawUtils.setWeightSelection(-1,null);
			if (toolTip == null) 
				toolTip = new DrawAreaToolTip(this);
	   }
	   updateUI();
	   LogCDrawUtils.setBottomMax(getSize().y / LogCDrawUtils.getTimeUnit());
	   drawAreaOverview.redraw();
	   graphComposite.redraw();
	}
	
	private void resetGraph(){
	   if(graph!=null){
	      ColumnFigure columnFigure=null;
	      for(int i = 0;i<graph.getTopNodeCount();i++){
	   	     columnFigure = (ColumnFigure)(graph.getTopNodes()[i]);
	   	     if(columnFigure!=null)
	   	        columnFigure.disposeMe();
	      }
	   }
	   endOfTime = 0;
	   endOfColumns = 16;
       columnPassages.clear();
	}

	/**
	 * the actual drawing calls of the graph
	 */
	private void buildGraph() {
		int timeUnit = LogCDrawUtils.getTimeUnit();
		Vector _nodeConnections = new Vector();
		int i = 0, j = 0, k = 0;
		NodeLink nodeConnection;
		ColumnFigure columnFigure;
		ColumnNode columnNode;
		do {
			if(graph.getTopNodes()[i]!=null){
			    columnFigure = (ColumnFigure)graph.getTopNodes()[i];
				addColumn(i, columnFigure); //build columnFigure
			    for (j = 0; j < columnFigure.getInternalNodeCount(); j++) {
					if(columnFigure.getInternalNodes()[j]!=null){
					columnNode = (ColumnNode)columnFigure.getInternalNodes()[j];
					columnNode.init(); //build columnNode
						for (k = 0; k < columnNode.getSourceConnectionCount(); k++) {
							if(columnNode.getSourceConnections()[k]!=null){
								nodeConnection = (NodeLink) columnNode.getSourceConnections()[k];
								initConnection(nodeConnection, true); //build NodeConnection
							}
						}
					}
				}
			}
			i++;
		} while (i < graph.getTopNodeCount());
		endOfTime = graph.getMaxIncrement();
        if (endOfTime > 0) {
        	adjustVScroll();
        	adjustHScroll();
        	drawAreaOverview.setCanvasSize(endOfTime);
		}
	}
	
	public void addColumn(int no, NodeContainer nodeContainer) {
		String name = nodeContainer.getShortName();
		if(nodeContainer.getType()==NodeContainer.NODE)
		   name+="/"+nodeContainer.getSecondaryName();
		else
		   name = "  "+name+"  ";
		width+=((ColumnFigure)nodeContainer).init(graphComposite, name, width-graphComposite.getHorizontalBar().getSelection(), no, canvas);
	}
	
	private void adjustVScroll(){
		int timeUnit = LogCDrawUtils.getTimeUnit();
		int boxSpace = LogCDrawUtils.getInitBoxSpace();
		ScrollBar vBar = graphComposite.getVerticalBar();
		vBar.setMaximum(endOfTime+boxSpace/timeUnit); //in time
		vBar.setThumb((Math.min(graphComposite.getSize().y,endOfTime*timeUnit+boxSpace))/timeUnit); //in time
		vBar.setPageIncrement((Math.min(graphComposite.getSize().y,endOfTime*timeUnit+boxSpace))/timeUnit); //in time
	}
	
	private void adjustHScroll(){
		ScrollBar hBar = graphComposite.getHorizontalBar();
		endOfColumns = graphComposite.getVerticalBar().getSize().x;
		ColumnFigure column;
		for (int i = 0; i < graph.getTopNodeCount(); i++) {
			column = (ColumnFigure)graph.getTopNodes()[i];
			endOfColumns +=column.getWidth();
		}
		hBar.setMaximum(endOfColumns);
		hBar.setThumb(Math.min(endOfColumns,graphComposite.getSize().x)); //in pixel
		hBar.setPageIncrement(Math.min(endOfColumns,graphComposite.getSize().x)); //in pixel
	}
	
	public void initConnection(NodeLink _nodeConnection, boolean passages) {
		int sourceConnectionType = 26, targetConnectionType = 26, passageType = 26; //set to none
		int sourceContainter = _nodeConnection.getSource().getContainer().getIndexInContainer();
		int targetContainer  = _nodeConnection.getTarget().getContainer().getIndexInContainer();
		int sourceNode = _nodeConnection.getSource().getIndexInContainer();
		int targetNode = _nodeConnection.getTarget().getIndexInContainer();
							
		switch (_nodeConnection.getType()) {
		/* 	nodeConnection type
		 		UNKNOWN      = 0;
				SIMPLE       = 1;
				SYNCHRONOUS  = 2;
				BALKING      = 3;
				TIMEOUT      = 4;
				CALL         = 5;
				ASYNCHRONOUS = 6;
				RETURN       = 7;
		  
		 	source/target connection type
			 	0 - simpleLeftIn
		    	1 - simpleRightIn
				2 - simpleLeftOut
				3 - simpleRightOut
				4 - synchLeft
				5 - synchRight
				6 - balkLeft
				7 - balkRight
				8 - timeoutLeft
				9 - timeoutRight
				10 - callLeft
				11 - callRight
				12 - asynchLeft
				13 - asynchRight
				14 - returnLeft
				15 - returnRight
				16 - returnLeftOut
				17 - returnRightOut
				18 - simpleSelfLeft
				19 - simpleSelfRight
				20 - returnSelfLeft
				21 - returnSelfRight
				22 - log record call left in
				23 - log record call right in
				26 - none
				27 - unknownLeft
				28 - unknownRight			

			Passage Type
				22 - bridgeLeftRight
				23 - bridgeRightLeft
				24 - bridgeLeftRightDashed
				25 - bridgeRightLeftDashed                 */
			
			case (0) : //unknown
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 27;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 28;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (1) : //simple
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 0;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 1;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (2) : //synch
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 4;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 5;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (3) : //balk
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 6;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 7;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (4) : //timeout
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 8;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 9;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (5) : //call
				if (sourceContainter < targetContainer) { //left to right
					sourceConnectionType = 3; 
					targetConnectionType = (graph.getType()==Graph.LOGoverRECORD||graph.getType()==Graph.THREADoverRECORD)?22:10;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) { //right to left
						sourceConnectionType = 2;
						targetConnectionType = (graph.getType()==Graph.LOGoverRECORD||graph.getType()==Graph.THREADoverRECORD)?23:11;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) { //simple self
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (6) : //asynch
				if (sourceContainter < targetContainer) {
					sourceConnectionType = 3;
					targetConnectionType = 12;
					passageType = 22;
				} else
					if (sourceContainter > targetContainer) {
						sourceConnectionType = 2;
						targetConnectionType = 13;
						passageType = 23;
					} else
						if (sourceContainter == targetContainer) {
							sourceConnectionType = 19;
							targetConnectionType = 26;
						}
				break;
			case (7) : //return
				if (sourceContainter < targetContainer) { //left to right
					sourceConnectionType = 17;
					targetConnectionType = 14;
					passageType = 24;
				} else
					if (sourceContainter > targetContainer) { //right to left
						sourceConnectionType = 16;
						targetConnectionType = 15;
						passageType = 25;
					} else
						if (sourceContainter == targetContainer) { //return self
							sourceConnectionType = 21;
							targetConnectionType = 26;
						}
				break;
			default :
				System.out.println("WRONG INDEX FOR NODECONNECTION!!!!!");
				break;
		}

        _nodeConnection.init(sourceConnectionType,targetConnectionType,graph.getType()==Graph.LOGoverRECORD || graph.getType()==Graph.THREADoverRECORD);

		if(passages) {
			if (!(columnPassages.containsKey(_nodeConnection)))
				columnPassages.put(_nodeConnection, new ColumnPassage(passageType, _nodeConnection,graph.getType()==Graph.LOGoverRECORD || graph.getType()==Graph.THREADoverRECORD)); 
		}
	}
	
	public Control getControl() {
		return this;
	}
	
	public void paintControl(PaintEvent  e ){
		toolTip.paintControl();
	}

	public void update() {
        /*if(endOfTime>0){
        	adjustVScroll();   
        }*/
	}

    public void fillContextMenu(IMenuManager menu){
		if(selectedColumnFigure != null){
			toolTip.hideText();
			int containerType = selectedColumnFigure.getNodeContainer().getType();
		}
			
		menu.add(new Separator());
		menu.add(((SDViewer)(sdViewerPage.getTraceViewer())).refresh);
		menu.add(new Separator());
		menu.add(((SDViewer)(sdViewerPage.getTraceViewer())).zoomIn);
		menu.add(((SDViewer)(sdViewerPage.getTraceViewer())).zoomOut);
		menu.add(((SDViewer)(sdViewerPage.getTraceViewer())).home);
		menu.add(new Separator());
		menu.add(((SDViewer)(sdViewerPage.getTraceViewer())).find);
	}
	
	public void removeToolTip(){
		toolTip.hideText();
	}
	public void setToolTip(int x, int y, ColumnFigure columnFigure){
    	
     	int xOffset = columnFigure.getXOffset();
//    	int correctedX = x + columnFigure.getMyCanvas().getLocation().x-graphComposite.getHorizontalBar().getSelection();
		int correctedX = x - graphComposite.getHorizontalBar().getSelection();

    	if(correctedX==parentBounds.x || correctedX==0 || y==parentBounds.y || y==0){
    		toolTip.hideText();
    	    return;
    	}

        if(!((x >= xOffset) && x <=(xOffset+width)))
        {
 	          return;
        }
    	
    	ColumnPassage columnPassage = columnFigure.getColumnPassageAtPosition(y);
    	if(columnPassage != null){
	   	   	  toolTip.showText(columnPassage.getNodeConnection().getName());
	   	   	  return;
    	}
    	
	 	ArrayList columnNodes = columnFigure.getColumnNodesAtPosition(y);
	 	if (columnNodes!=null && columnNodes.size()>0)
	 	{
	   		for (int count=0;count<columnNodes.size();count++) 
	   		{
				ColumnNode columnNode = (ColumnNode)(columnNodes.get(count));
		    	if(columnNode != null && x>=xOffset+columnFigure.getWidth()/2-5 && x<=xOffset+columnFigure.getWidth()/2+5)
		    	{
					toolTip.showText(columnNode.getGraphNode().getName());
			    	return;
		   		} 
		   		else if(columnNode != null)
		   		{
		   	   		NodeLink nodeConnection = columnFigure.getNodeLinkAtPosition(columnNode,x,y);
		   	   		if(nodeConnection != null) {
		   	      		toolTip.showText(nodeConnection.getName());
		   	      		return;
		   	   		}
		   	   		else
		   	   		{
		   	      		toolTip.hideText();
		   	   		}
	
		   		}
	   		}
	 	} 
	   	else if(columnNodes.size()==0 
	   		&& ((x>=xOffset+columnFigure.getWidth()/2-1 
	   		&& x<=xOffset+columnFigure.getWidth()/2+1) || 
		   	((x>=xOffset && x<=xOffset+columnFigure.getWidth()) && (LogCDrawUtils.getBoxSpace()!=0 && y<=LogCDrawUtils.getBoxSpace()))))
		   	{
		   	      toolTip.showText(columnFigure.getFullName());
		   	      return;
		   	}
	   	else if(columnNodes.size()==0)
	   	{
		     toolTip.hideText();
	   	}
    }
    
    public void setToolTip(int x, int y, String tag){
       toolTip.showText(tag);	
    }

    private ColumnFigure getColumnFigureAtLocation(int x){
    	ColumnFigure column;
    	int location=0;
    	x+=graphComposite.getHorizontalBar().getSelection();
			for (int i = 0; i < graph.getTopNodeCount(); i++){
				column = (ColumnFigure) graph.getTopNodes()[i];
				location += column.getWidth();
				if(location>=x)
				   return column;
			}
		return null;
    }
    
    private ColumnFigure getColumnFigureAtLocation(int x, int localPos){
    	ColumnFigure column;
    	x+=graphComposite.getHorizontalBar().getSelection();
			for (int i = 0; i < graph.getTopNodeCount(); i++){
				column = (ColumnFigure) graph.getTopNodes()[i];
				if(localPos<=column.getWidth())
				   return column;
				localPos -= column.getWidth();
			}
		return null;
    }

	void widgetDisposed(DisposeEvent e) {
		if (canvas != null)
			canvas.dispose();
		if (graphComposite != null)
			graphComposite.dispose();			
		graph = null;
		columnPassages.clear();
		idEvent = null;
		toolTip=null;
		drawAreaFindDialog=null;
		drawAreaOverview=null;
		drawAreaWeightScale=null;	
	    selectedNodeLink=null;
		selectedColumnNode=null;
		selectedColumnFigure=null;
	}

	public Rectangle getParentBounds() {
		return parentBounds;
	}
	
	public GraphNode[] getColumnFigures(){
		return graph.getTopNodes();
	}
	
	public int getEndOfTime(){
		return endOfTime;
	}
	
	public Composite getGraphComposite(){
	   return graphComposite;	
	}
	
	public Graph getGraph(){
	   return graph;	
	}
	
	public Point getSize(){
	   return graphComposite.getSize();	
	}
	
	public Iterator getColumnPassages(){
	   return columnPassages.values().iterator();	
	}
	
	public Map getColumnPassagesMap()
	{
		return columnPassages;
	}
    
	public boolean setFocus(){
		graphComposite.forceFocus();
		return graphComposite.setFocus();	
	}
	
	public void setScrollbars(int x, int y){
		/*ScrollBar hbar = graphComposite.getHorizontalBar();
		if(x<0) x=0;
		if(x>hbar.getMaximum()) x=hbar.getMaximum();
		hbar.setSelection(x);
		ScrollBar vbar = graphComposite.getVerticalBar();
		if(y<0) y=0;
		if(y>vbar.getMaximum()) y=vbar.getMaximum();
		vbar.setSelection(y);*/
        if(x<0) x=0;
		if(y<0) y=0;
		graphComposite.getHorizontalBar().setSelection(x);
		graphComposite.getVerticalBar().setSelection(y);
		scrollViewport(x,y);
	}
	
	public void scrollViewport(int x, int y){
		hScroll(x);
		vScroll(y);
	    update();
	}
	
	private void hScroll(int x){
		scrollColumns(x);
		drawAreaOverview.positionShadow();
		updateUI();
	}
	
	private void vScroll(int y){
		LogCDrawUtils.adjustStartPosition(y);
		drawAreaOverview.positionShadow();
		updateUI();
	}
		
    public void updateUI(){
    	drawAreaWeightScale.drawSelf();
    	adjustVScroll();
    	adjustHScroll();
		canvas.redraw();
	}
	
	public void adjustColumns(int widthAdjustment, int columnNumber){
	   width+=widthAdjustment;
	   for(int i=columnNumber+1;i<graph.getTopNodeCount();i++)
	   	  ((ColumnFigure)graph.getTopNodes()[i]).adjustLocation(widthAdjustment);
	   scrollColumns(graphComposite.getHorizontalBar().getSelection());	
	   adjustHScroll();
	   drawAreaOverview.close();
	   drawAreaOverview.adjustParentSize(getSize().x-graphComposite.getVerticalBar().getSize().x,getSize().y-graphComposite.getHorizontalBar().getSize().y);
	}
	
	private void scrollColumns(int scrollShift){
		endOfColumns = graphComposite.getVerticalBar().getSize().x;
		int currentLocation = -scrollShift;
		ColumnFigure column;
		for( int i=0; i<graph.getTopNodeCount();i++){
			column = (ColumnFigure) graph.getTopNodes()[i];
//			column.getMyCanvas().setLocation(new Point(currentLocation, 0));
			column.setOriginalLocation(currentLocation, 0);
			currentLocation += column.getWidth();
			//endOfColumns +=column.getWidth();
		}
	}
	 
	public void goHome(){
		graphComposite.getHorizontalBar().setSelection(0);
		graphComposite.getVerticalBar().setSelection(0);
		scrollViewport(0,0);
		drawAreaOverview.close();
	}
	
	public void setSelectedNode(ColumnNode columnNode, ColumnFigure columnFigure){		
		int width = 0;
		int nodeWidth = LogCDrawUtils.getNodeWidth();
		boolean drawSelected=true;
		if(selectedColumnFigure!=null){
		   if(selectedColumnFigure.getMyCanvas().isDisposed())
		      drawSelected = false;
		   selectedColumnFigure.setSelected(false);
		   if(drawSelected)
		      selectedColumnFigure.getMyCanvas().redraw();	
		}
		if(columnNode == null){
		   if(selectedColumnNode == null)
		      return;
		    else if(selectedColumnNode != null){
				selectedColumnNode.setSelected(false);
				if (drawSelected && selectedColumnFigure != null) {
					width = selectedColumnFigure.getWidth();
					selectedColumnFigure.getMyCanvas().redraw(
						(width - nodeWidth) / 2 - 1,
						selectedColumnNode.getBeginPosition() - 1,
						nodeWidth + 2,
						selectedColumnNode.getPositionRange() + 2,
						true);
				}
				selectedColumnNode = null;
				selectedColumnFigure = null;
		    }
		}else if(columnNode != null){
			   if (selectedColumnNode!=null)
		       		selectedColumnNode.setSelected(false);
		       if(drawSelected && selectedColumnFigure != null){
		          width = selectedColumnFigure.getWidth();
		          selectedColumnFigure.getMyCanvas().redraw(
		          		selectedColumnFigure.getXOffset()+(width-nodeWidth)/2-1,
		          		selectedColumnNode.getBeginPosition()-1,
		          		nodeWidth+2,
		          		selectedColumnNode.getPositionRange()+2,
		          		true);
		       }
		    }
		    selectedColumnNode = columnNode;
		    selectedColumnFigure = columnFigure;
		    selectedColumnNode.setSelected(true);
		    
			if(selectedColumnNode.getUserArea()!=null)
					UIPlugin.getDefault().getSelectionModel(sdViewerPage.getMOFObject()).add(selectedColumnNode.getUserArea());

		    if(drawSelected && selectedColumnFigure != null){
		       width = selectedColumnFigure.getWidth();
		       selectedColumnFigure.getMyCanvas().redraw(
		       		selectedColumnFigure.getXOffset()+(width-nodeWidth)/2-1,
		       		selectedColumnNode.getBeginPosition()-1,
		       		nodeWidth+2,
		       		selectedColumnNode.getPositionRange()+2,
		       		true);
		    }
            if(selectedNodeLink!=null){
	         selectedNodeLink.setSelected(false);	
	         updateUI();
	      }
	      
	      updateUI();
	}
	
	public void setSelectedColumnFigure(ColumnFigure columnFigure){
		//new selected Figure
	   if(selectedColumnFigure == null && columnFigure != null){
	   	   columnFigure.setSelected(true);	
	   	   columnFigure.getMyCanvas().redraw();
	       selectedColumnFigure = columnFigure;
	   }else if(selectedColumnFigure != null && columnFigure != null){
	   	   if (!(selectedColumnFigure.equals(columnFigure))){
	   	   selectedColumnFigure.setSelected(false);
	   	   selectedColumnFigure.getMyCanvas().redraw();
	   	   columnFigure.setSelected(true);
	   	   columnFigure.getMyCanvas().redraw();
	   	   selectedColumnFigure = columnFigure;
	   	   }
	   }
	   
	   if(selectedColumnFigure != null && selectedColumnFigure.getUserArea() != null)
	   {
			UIPlugin.getDefault().getSelectionModel(
				sdViewerPage.getMOFObject()).add(
				selectedColumnFigure.getUserArea());
			
	   }
	   
	   updateUI();
	   
	}
	
	public void setSelectedNodeLink(ColumnFigure columnFigure, NodeLink nodeLink){
//		if(selectedNodeLink!=null)
//		   selectedNodeLink.setSelected(false);
		if(selectedColumnNode!=null)
		   selectedColumnNode.setSelected(false);
		if(selectedColumnFigure!=null)
		   selectedColumnFigure.setSelected(false);	
		nodeLink.setSelected(true);
		updateUI();
		selectedNodeLink = nodeLink;		
	}

	public void scrollToNode(ColumnNode node) {
		int boxHeight = LogCDrawUtils.getBoxHeight();
		int upperSpace = LogCDrawUtils.getUpperSpace();
		int timeUnit = LogCDrawUtils.getTimeUnit();
		
		ScrollBar hBar = graphComposite.getHorizontalBar();
		ScrollBar vBar = graphComposite.getVerticalBar();
		int hScroll=0, vScroll=0;
		ColumnFigure columnFigure =	(ColumnFigure)node.getContainer();

		//horizontalPosition are calcuated in pixel
		int startPosition = columnFigure.getOriginalLocation().x;
		if ((startPosition + columnFigure.getWidth()) > hBar.getSize().x)
					hScroll = startPosition - ((hBar.getSize().x - columnFigure.getWidth()) / 2);

		//verticalPostion are calculate in time weight scale unit
		int verticalPosition = node.getStartPosition() + (upperSpace + boxHeight-graphComposite.getSize().y/2)/timeUnit;
		if (verticalPosition<0)
			vScroll = 0;
		else	
			vScroll = verticalPosition;

		hScroll(hScroll);
		hBar.setSelection(hScroll);
		vScroll(vScroll);
		vBar.setSelection(vScroll);
	}

	public void find() {
		drawAreaFindDialog = new DrawAreaFindDialog(this.getParent().getShell(), graph);
		drawAreaFindDialog.open();
		if (drawAreaFindDialog.getReturnCode() == Window.OK) {
			if (drawAreaFindDialog.getSelectedNode() != null) {
				ScrollBar hBar = graphComposite.getHorizontalBar();
				int scroll = 0;
				ColumnFigure columnFigure =	(ColumnFigure) drawAreaFindDialog.getSelectedNode();
				setSelectedColumnFigure(columnFigure);
				int startPosition = columnFigure.getOriginalLocation().x;
				if ((startPosition + columnFigure.getWidth())> hBar.getSize().x)
					scroll = startPosition - ((hBar.getSize().x - columnFigure.getWidth()) / 2);
				hScroll(scroll);
				hBar.setSelection(scroll);
			}
		}
	}
	
	public MouseEvent getIdEvent() {
		return idEvent;
	}
	public void setIdEvent(MouseEvent idEvent) {
		if ((this.idEvent == null) || (!((this.idEvent.x == idEvent.x) && (this.idEvent.y == idEvent.y)))){
			this.idEvent = idEvent;
			int nodeWidth = LogCDrawUtils.getNodeWidth();
			if(selectedColumnNode != null){
				selectedColumnNode.setSelected(false);
				if (selectedColumnFigure != null) {
					width = selectedColumnFigure.getWidth();
					
					selectedColumnFigure.getMyCanvas().redraw(selectedColumnFigure.getXOffset()+
						(width - nodeWidth) / 2 - 1,
						selectedColumnNode.getBeginPosition() - 1,
						nodeWidth + 2,
						selectedColumnNode.getPositionRange() + 2,
						true);
				}
				selectedColumnNode = null;
				selectedColumnFigure = null;
			}
			if (selectedColumnFigure != null){
			   selectedColumnFigure.setSelected(false);
			   selectedColumnFigure.getMyCanvas().redraw(selectedColumnFigure.getXOffset(), 0, selectedColumnFigure.getXOffset()+width, selectedColumnFigure.getHeight(), true);
			   selectedColumnFigure = null;
			}
		   if(selectedNodeLink!=null){
			  selectedNodeLink.setSelected(false);
			  updateUI();
			  selectedNodeLink = null;
		   }
		    
		}
	}

	public void dispose() {		
		 _viewsActions.clear();
		 super.dispose();
	}

	//Control Listener -> begin
	public void controlMoved(ControlEvent e) {
		/*Object source = e.getSource();
		if (source instanceof ToolItem) {
			scrollColumns(getHorizontalBar().getSelection());
		}*/
	}

	public void controlResized(ControlEvent e) {
		Object source = e.getSource();
		if (source instanceof SDViewerUI) {
			Rectangle parentBounds = getClientArea();
			int weightScaleWidth = LogCDrawUtils.getWeightScaleWidth();
			drawAreaWeightScale.setSize(weightScaleWidth,parentBounds.height);
			graphComposite.setSize(parentBounds.width-weightScaleWidth, parentBounds.height);
			canvas.setSize(parentBounds.width-weightScaleWidth, parentBounds.height);
			ColumnFigure column;
			for (int i = 0; i < graph.getTopNodeCount(); i++){
				   column = (ColumnFigure) graph.getTopNodes()[i];
				   column.setHeight(getClientArea().height);
				   LogCDrawUtils.setBottomMax(getSize().y / LogCDrawUtils.getTimeUnit());
			}
			adjustVScroll();
			adjustHScroll();
			update();
			drawAreaOverview.close();
			drawAreaOverview.adjustParentSize(getSize().x-graphComposite.getVerticalBar().getSize().x,getSize().y-graphComposite.getHorizontalBar().getSize().y);
		}
	}
	//ControlListener -> end

	//SelectionListner -> begin
	public void widgetDefaultSelected(SelectionEvent e) {}

	public void widgetSelected(SelectionEvent e) {
		ScrollBar scrollBar = (ScrollBar) e.getSource();
		if(scrollBar==graphComposite.getHorizontalBar()){
			hScroll(scrollBar.getSelection());
		} else if(scrollBar==graphComposite.getVerticalBar()) {
			vScroll(scrollBar.getSelection());
		}
		update();
	}
	//SelectionListener -> end
	
	//KeyListener -> begin
	public void keyPressed(KeyEvent e){
	   ScrollBar hBar = graphComposite.getHorizontalBar();
	   ScrollBar vBar = graphComposite.getVerticalBar();
	   int timeUnit = LogCDrawUtils.getTimeUnit();
	   int boxSpace = LogCDrawUtils.getBoxSpace();
	   int position;
		 ColumnFigure columnFigure = null;
	   switch(e.keyCode){
		case(SWT.ARROW_RIGHT):
	   	
		if(!(hBar.getPageIncrement() == hBar.getMaximum())){
		   position = hBar.getSelection()+getColumnFigureAtLocation(5).getWidth();
		   if(position>hBar.getMaximum())
			  position=hBar.getMaximum();
		   hScroll(position);
		   hBar.setSelection(position);
		}
		break;
		case(SWT.ARROW_LEFT):
		//System.out.println("LEFT");
		if(hBar.getSelection()>0){
		   position = hBar.getSelection()-getColumnFigureAtLocation(-5).getWidth();
		   if(position<0)
			  position=0;
		   hScroll(position);
		   hBar.setSelection(position);
		}
		break;
		case(SWT.ARROW_DOWN):
		//System.out.println("DOWN");
		if(endOfTime*timeUnit>graphComposite.getSize().y){
		   position = vBar.getSelection()+timeUnit;
		   //if(position>endOfTime-graphComposite.getSize().y/timeUnit+boxSpace/timeUnit)
			  //position=endOfTime-graphComposite.getSize().y/timeUnit+boxSpace/timeUnit;
		   if(position>vBar.getMaximum()-graphComposite.getSize().y/timeUnit)
			  position=vBar.getMaximum()-graphComposite.getSize().y/timeUnit;
		   vScroll(position);
		   vBar.setSelection(position);
		}
		break;
		case(SWT.ARROW_UP):
		//System.out.println("UP");
		if(vBar.getSelection()>0){
		   position = vBar.getSelection()-timeUnit;
		   if(position<0)
			  position=0;
		   vScroll(position);
		   vBar.setSelection(position);
		}
		break;
		case(SWT.PAGE_DOWN):
		//System.out.println("PGDOWN");
		if(endOfTime*timeUnit>graphComposite.getSize().y){
			   position = vBar.getSelection()+graphComposite.getSize().y/timeUnit;
		   if(position>vBar.getMaximum()-graphComposite.getSize().y/timeUnit)
			  position=vBar.getMaximum()-graphComposite.getSize().y/timeUnit;
		   vScroll(position);
		   vBar.setSelection(position);
		}
		break;
		case(SWT.PAGE_UP):
		//System.out.println("PGUP");
		if(vBar.getSelection()>0){
		   position = vBar.getSelection()-graphComposite.getSize().y/timeUnit;
		   if(position<0)
			  position=0;
		   vScroll(position);
		   vBar.setSelection(position); 
		}
		break;
			case(SWT.HOME):
		if(selectedNodeLink!=null){
		   columnFigure = (ColumnFigure)((ColumnNode)selectedNodeLink.getSource()).getContainer();
		   position = columnFigure.getOriginalLocation().x;
		   if(columnFigure.getIndexInContainer()>((ColumnFigure)((ColumnNode)selectedNodeLink.getTarget()).getContainer()).getIndexInContainer())
			  position = position - graphComposite.getBounds().width + vBar.getSize().x + ((ColumnFigure)((ColumnNode)selectedNodeLink.getSource()).getContainer()).getWidth();
		   if(position<0)
			  position=0;
		   hScroll(position);
		   hBar.setSelection(position);
		}
		break;
		case(SWT.END):
		if(selectedNodeLink!=null){
		   columnFigure = (ColumnFigure)((ColumnNode)selectedNodeLink.getTarget()).getContainer();
		   position = columnFigure.getOriginalLocation().x;
		   if(columnFigure.getIndexInContainer()>((ColumnFigure)((ColumnNode)selectedNodeLink.getSource()).getContainer()).getIndexInContainer())
			  position = position - graphComposite.getBounds().width + vBar.getSize().x + ((ColumnFigure)((ColumnNode)selectedNodeLink.getTarget()).getContainer()).getWidth();
		   if(position<0)
			  position=0;
		   hScroll(position);
		   hBar.setSelection(position);
		}
		break;
	   }
	}
	
	public void keyReleased(KeyEvent e) {}
	//KeyListener -> end
    //MenuListener -> end
	      
	/**
	 * Adds a listener for selection changes in this selection provider.
	 * Has no effect if an identical listener is already registered.
	 *
	 * @param listener a selection changed listener
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener)
	{
	}
	/**
	 * Returns the current selection for this provider.
	 * 
	 * @return the current selection
	 */
	public ISelection getSelection()
	{ 
		if(selectedColumnNode!=null && selectedColumnNode.getUserArea()!=null)
		{
			return new StructuredSelection(selectedColumnNode.getUserArea());
		}
		else if(selectedColumnFigure != null && selectedColumnFigure.getUserArea() != null)
		{
			return new StructuredSelection(selectedColumnFigure.getUserArea());
		}
			
		return new StructuredSelection();
	}
	/**
	 * Removes the given selection change listener from this selection provider.
	 * Has no affect if an identical listener is not registered.
	 *
	 * @param listener a selection changed listener
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener)
	{
	}
	/**
	 * Sets the current selection for this selection provider.
	 *
	 * @param selection the new selection
	 */
	public void setSelection(ISelection selection)
	{
	}    
	/**
	 * @return
	 */
	public Canvas getCanvas() {
		return canvas;
	}
	
	public boolean isEmpty()
	{
		//if the graph is null, then the page is empty
		//otherwise return true if number of nodes is > 0
		if (getGraph() == null) return true;
		return (!(getGraph().getTopNodeCount() > 0));
		
	}

}
