/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.internal.junit.editor;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.test.common.TestCommon;
import org.eclipse.hyades.test.common.internal.editor.TestCasesForm;
import org.eclipse.hyades.test.common.internal.editor.action.AddTestCase;
import org.eclipse.hyades.test.http.TestHttpImages;
import org.eclipse.hyades.test.http.TestHttpPlugin;
import org.eclipse.hyades.test.http.internal.preferences.HttpPreferenceUtil;
import org.eclipse.hyades.test.http.internal.util.HttpConstants;
import org.eclipse.hyades.test.http.internal.util.RequestHelper;
import org.eclipse.hyades.test.http.runner.HttpExecutor;
import org.eclipse.hyades.test.http.runner.HttpRequest;
import org.eclipse.hyades.test.http.runner.HttpResponse;
import org.eclipse.hyades.test.ui.internal.editor.form.util.DetailSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class HttpTestCasesForm
extends TestCasesForm
{
	protected static class CheckTestCaseDialog
	extends Dialog implements Listener
	{
		protected HttpRequest request;
		private Text resultText;
		private Cursor cursor;
		private boolean started;
		
		public CheckTestCaseDialog(HttpRequest request)
		{
			super(Display.getCurrent().getActiveShell());			
			this.request = request;
			started = false;
		}

		protected Control createDialogArea(Composite parent)
		{
			getShell().setText(TestHttpPlugin.getString("W_TEST"));
				
			Composite result = new Composite(parent, SWT.NULL);
			GridData gridData = GridDataUtil.createFill();
			gridData.heightHint = 250;
			gridData.widthHint = 400;
			result.setLayoutData(gridData);
			result.setLayout(new GridLayout());

			resultText = new Text(result, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
			resultText.setLayoutData(GridDataUtil.createFill());
			resultText.setEditable(false);
			
			getShell().addListener(SWT.Activate, this);
			return result;
		}
		
		protected void createButtonsForButtonBar(Composite parent)
		{
			createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL,	true);
		}
		
		public void handleEvent(Event event)
		{
			execute();
		}
		
		protected void execute()
		{	
			if(started)
				return;
				
			started = true;
			cursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
			getShell().setCursor(cursor);
			resultText.setText(TestHttpPlugin.getString("test.WaitMessage"));
			
			Thread thread = new Thread("HyadesHttpRequestTester")
			{
				public void run()
				{
					String checkerReturn = null;
					try
					{
						HttpResponse response = new HttpExecutor().execute(request);
						checkerReturn = TestHttpPlugin.getString("test.Response", new String[] {response.getCode()+"", response.getContentType(), "" + response.getContentLength()});
						if(response.getDetail() != null)
							checkerReturn = checkerReturn + "\n\n" + response.getDetail();
						if(response.getBody() != null)
							checkerReturn = checkerReturn + "\n\n" + response.getBody();
					}
					catch (Throwable t)
					{
						checkerReturn = TestHttpPlugin.getString("test.Exception") + "\n" + CoreUtil.getStackTrace(t);
					}
					
					final String cr = checkerReturn;
					Display.getDefault().asyncExec(new Runnable()
					{
						public void run()
						{
							testFinished(cr);
						}
					});
				}
			};
			thread.start();			
		}
		
		protected void testFinished(String text)
		{
			if((getShell() == null) || resultText.isDisposed())
				return;
			
			resultText.setText(text);
			getButton(IDialogConstants.CANCEL_ID).setText(IDialogConstants.OK_LABEL);
				
			getShell().setCursor(null);
			cursor.dispose();					
		}
	}
	
	
	protected static class CheckTestCaseAction
	extends Action implements IDisposable
	{
		private RequestHelper requestHelper;
		
		/**
		 * Constructor for CheckTestCaseAction
		 */		
		public CheckTestCaseAction()
		{
			super(TestHttpPlugin.getString("W_TEST"), TestHttpImages.INSTANCE.getImageDescriptor(TestHttpImages.IMG_CHECK));
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			if(requestHelper != null)
			{
				requestHelper.dispose();
				requestHelper = null;
			}
		}
		
		public boolean isAvailable()
		{
			return ((requestHelper != null) && (requestHelper.getTestCase() != null));
		}
		
		public void setStructuredSelection(IStructuredSelection structuredSelection)
		{
			if(requestHelper != null)
				requestHelper.dispose();
				
			if((structuredSelection != null) && (!structuredSelection.isEmpty()))
			{
				Object object = structuredSelection.getFirstElement();
				if(object instanceof ITestCase)
					setTestCase((ITestCase)object);
			}
			
			setEnabled(isAvailable());
		}
	
		public void setTestCase(ITestCase testCase)
		{
			if(requestHelper == null)
				requestHelper = new RequestHelper();
			else
				requestHelper.dispose();
			
			requestHelper.setTestCase(testCase);
		}
		
		public ITestCase getTestCase()
		{
			if(requestHelper == null)
				return null;
				
			return requestHelper.getTestCase();
		}
		
		/**
		 * @see org.eclipse.jface.action.Action#run()
		 */
		public void run()
		{
			if(requestHelper == null)
				return;
				
			String value = requestHelper.getAttribute(RequestHelper.ATT_HOST);
			if(value == null)
			{
				openErrorMessage(TestHttpPlugin.getString("Host_Is_Blank"), null);
				return;
			}
			
			value = requestHelper.getAttribute(RequestHelper.ATT_ABS_PATH);
			if((value == null) || (!value.startsWith("/")))
			{
				openErrorMessage(TestHttpPlugin.getString("Path_Needs_Slash"), null);
				return;
			} 
			
			HttpRequest request = requestHelper.createHttpRequest();
			if(request == null)
			{
				openErrorMessage(TestHttpPlugin.getString("TEST_ERR_MSG"), null);
				return; 
			}

			CheckTestCaseDialog dialog = new CheckTestCaseDialog(request);
			dialog.open();
		}
				
		protected void openErrorMessage(String text, String detail)
		{
			if(detail != null)
				text = text + " \n\n" + detail;
			MessageDialog.openError(Display.getDefault().getActiveShell(), TestHttpPlugin.getString("W_ERROR"), text);
		}			
	}
	
	protected DetailSection detailSection;
	
	/**
	 * Constructor for HttpTestCasesForm
	 * @param junitEditorExtension
	 * @param widgetFactory
	 */
	public HttpTestCasesForm(HttpEditorExtension httpEditorExtension, WidgetFactory widgetFactory)
	{
		super(httpEditorExtension, widgetFactory);
		setHeadingText(TestHttpPlugin.getString("EDT_GENERAL_INFO"));
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#createLeftColumn(org.eclipse.swt.widgets.Composite)
	 */
	protected void createLeftColumn(Composite parent)
	{
		createTestCaseSection(parent);
		createNamedElementSection(parent);
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#createRightColumn(org.eclipse.swt.widgets.Composite)
	 */
	protected void createRightColumn(Composite parent)
	{
		createDetailSection(parent);
	}

	/**
	 * @see org.eclipse.hyades.test.java.internal.junit.editor.TestCasesForm#getAddTestCaseAction()
	 */
	protected IAction getAddTestCaseAction()
	{
		return new AddTestCase(this, TestCommon.HTTP_JUNIT_TEST_CASE_TYPE, TestHttpPlugin.getString("W_HTTP_REQUEST").toLowerCase())
		{
			protected void adjusTestCase(ITestCase testCase)
			{
				RequestHelper requestHelper = new RequestHelper();
				requestHelper.setTestCase(testCase);
				requestHelper.setAttribute(RequestHelper.ATT_HOST, HttpPreferenceUtil.getInstance().getDefaultHostName());
				requestHelper.setAttribute(RequestHelper.ATT_PORT, HttpPreferenceUtil.getInstance().getDefaultHostPort());
				requestHelper.setAttribute(RequestHelper.ATT_ABS_PATH, HttpPreferenceUtil.getInstance().getDefaultAbsolutePath());
				requestHelper.setAttribute(RequestHelper.ATT_METHOD, HttpConstants.SUPPORTED_REQUEST_METHODS[0]);
				requestHelper.setAttribute(RequestHelper.ATT_VERSION, HttpConstants.HTTP_VERSION);
				
				detailSection.setInput(testCase);				
			}
		};
	}

	/**
	 * @see org.eclipse.hyades.test.java.internal.junit.editor.TestCasesForm#getTestCasesSectionHeaderText()
	 */
	protected String getTestCasesSectionHeaderText()
	{
		return TestHttpPlugin.getString("W_REQUESTS");
	}

	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#registerHelp(int, java.lang.Object)
	 */
	protected void registerHelp(int flag, Object object)
	{
		switch(flag)
		{
			case HELP_NAMED_ELEMENT_CONTROL:
				break;
				
			case HELP_TEST_CASES_SECTION_CONTROL:
				break;
				
			case HELP_TEST_CASES_SECTION_TREE_CONTROL:
				break;
		}
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#addDetails(org.eclipse.hyades.test.ui.internal.editor.form.util.DetailSection)
	 */
	protected void addDetails(DetailSection detailSection)
	{
		this.detailSection = detailSection;
		detailSection.add(ITestCase.class, new HttpTestCaseDetailPage());
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.editor.TestCasesForm#createTestCasesSection()
	 */
	protected EObjectTreeSection createTestCasesSection()
	{
		return new EObjectTreeSection(this, Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestCases(), getAddTestCaseAction())
		{
			private CheckTestCaseAction checkTestCaseAction;
			
			protected void createActions()
			{
				super.createActions();
				checkTestCaseAction = new CheckTestCaseAction();
			}
			
			/**
			 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection#fillContextMenu(org.eclipse.jface.action.IMenuManager)
			 */
			protected void fillContextMenu(IMenuManager menuManager)
			{
				super.fillContextMenu(menuManager);
				
				menuManager.add(new Separator());
				menuManager.add(checkTestCaseAction);
				
				checkTestCaseAction.setStructuredSelection(getStructuredSelection());
			}
		};
	}	
}
