/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.junit;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import junit.extensions.TestDecorator;
import junit.framework.Test;
import junit.framework.TestListener;
import junit.framework.TestResult;
import junit.framework.TestSuite;

/**
 * This class extends the {@link junit.framework.TestSuite} class adding the 
 * expected behavior defined in the 
 * {@link org.eclipse.hyades.test.java.junit.IHyadesTest} interface.
 * 
 * <p>By redefining addTest(Test) and addTestSuite(Class), instances of this class will 
 * be set as the parent of any instance of IHyadesTest  passed as parameter.  This is 
 * done transparently so the test developer won't need to add any additional code to 
 * provide the backward hierarchy navigability.
 * 
 * <p>By redefining run(TestResult) the iteration information is also calculated 
 * in a transparent way.
 * 
 * @author marcelop
 * @since 1.0.1
 */
public class HyadesTestSuite
extends TestSuite implements IHyadesTest
{
	private static class AsynchronousThread
	extends Thread
	{
		private static int counter = 0;
		private Map iterationByTest;
		
		public AsynchronousThread(String name)
		{
			super(name + (++counter));
			iterationByTest = new HashMap();
		}

		public void dispose()
		{
			iterationByTest.clear();
		}

		public void setIteration(Test test, int iteration)
		{
			iterationByTest.put(getName() + test.hashCode(), new Integer(iteration));
		}
	
		public int getIteration(Test test)
		{
			Integer integer = (Integer)iterationByTest.get(getName() + test.hashCode());
			if(integer == null)
				return -1;
			return integer.intValue();
		}
	}
	
	private String id;
	private String testInvocationId;
	private Test parent;
	
	private IHyadesTestArbiter arbiter;
	
	private int iteration = 0;
	private int previousParentIteration = 0;
	
	private boolean synchronous = true;
	private int activeAsynchronousTest;

	/**
	 * Constructor for HyadesTestSuite.
	 */
	public HyadesTestSuite()
	{
		super();
	}

	/**
	 * Constructor for HyadesTestSuite.
	 * @param theClass
	 */
	public HyadesTestSuite(Class theClass)
	{
		super(theClass);
	}

	/**
	 * Constructor for HyadesTestSuite.
	 * @param name
	 */
	public HyadesTestSuite(String name)
	{
		super(name);
	}

	/**
	 * @see IHyadesTest#getId()
	 */
	public String getId()
	{
		return id;
	}

	/**
	 * @see IHyadesTest#setId(String)
	 */
	public IHyadesTest setId(String id)
	{
		this.id = id;
		return this;
	}

	/**
	 * @see IHyadesTest#getTestInvocationId()
	 */
	public String getTestInvocationId()
	{
		return testInvocationId;
	}

	/**
	 * @see IHyadesTest#setTestInvocationId(java.lang.String)
	 */
	public IHyadesTest setTestInvocationId(String testInvocationId)
	{
		this.testInvocationId = testInvocationId;
		return this;
	}

	/**
	 * @see IHyadesTest#setParent(TestSuite)
	 */
	public void setParent(Test parent)
	{
		this.parent = parent;
	}

	/**
	 * @see IHyadesTest#getParent()
	 */
	public Test getParent()
	{
		return parent;
	}

	/**
	 * @see IHyadesTest#getIteration()
	 */
	public int getIteration()
	{
		if((!isSynchronous()) && (Thread.currentThread() instanceof AsynchronousThread))
		{
			int i = ((AsynchronousThread)Thread.currentThread()).getIteration(this);
			if(i >= 0)
				return i;
		}
		
		return iteration;
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.junit.IHyadesTest#setSynchronous(boolean)
	 */
	public IHyadesTest setSynchronous(boolean synchronous)
	{
		this.synchronous = synchronous;
		return this;
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.IHyadesTest#isSynchronous()
	 */
	public boolean isSynchronous()
	{
		return synchronous;
	}

	/**
	 * Sets the arbiter that evaluates this test suite.
	 * @param arbiter
	 * @return this test suite
	 */
	public HyadesTestSuite setArbiter(IHyadesTestArbiter arbiter)
	{
		this.arbiter = arbiter;
		return this; 
	}

	/**
	 * Gets the arbiter that evaluates this test suite.
	 * @param arbiter
	 */
	public IHyadesTestArbiter getArbiter()
	{
		return arbiter; 
	}

	/**
	 * @see TestSuite#addTestSuite(Class)
	 */
	public void addTestSuite(Class testClass)
	{
		addTest(new HyadesTestSuite(testClass));
	}

	/**
	 * Sets this test as the parent of the added Test if it implements 
	 * IHyadesTest.
	 * @param test
	 * @see TestSuite#addTest(Test)
	 */
	public void addTest(Test test)
	{
		super.addTest(test);

		if (test instanceof TestDecorator)
			test = HyadesTestUtil.getDecoratorTest((TestDecorator)test);

		if (test instanceof IHyadesTest)
		{
			IHyadesTest extendedTest = (IHyadesTest)test;
			extendedTest.setParent(this);
		}
	}

	/**
	 * If the test result is an instance of {@link HyadesTestResult} 
	 * this method notifies the start and end events of this test suite to the
	 * registered listeners of the test result.
	 * @see junit.framework.Test#run(junit.framework.TestResult)
	 */
	public void run(TestResult result)
	{
		if(isSynchronous())
		{
			doRun(result);
		}
		else
		{
			final TestResult finalResult = result; 
			Thread t= new AsynchronousThread("Asynchronous HyadesTestSuite")
			{
				public void run()
				{
					try
					{
						doRun(finalResult);
					}
					finally
					{
						if(getParent() instanceof HyadesTestSuite)
							((HyadesTestSuite)getParent()).runFinished(HyadesTestSuite.this);
						dispose();
					}
				}
			};
			if(getParent() instanceof HyadesTestSuite)
				((HyadesTestSuite)getParent()).activeAsynchronousTest++;
			t.start();
		}			
	}
	
	protected void doRun(TestResult result)
	{
		synchronized(this)
		{
			adjustIteration();
			if((!isSynchronous()) && (Thread.currentThread() instanceof AsynchronousThread))
				((AsynchronousThread)Thread.currentThread()).setIteration(this, iteration);
				
			for(Iterator i=HyadesTestUtil.getTestListeners(result); i.hasNext();)
				((TestListener)i.next()).startTest(this);
		}		

		activeAsynchronousTest = 0;
		super.run(result);
		waitUntilFinished();		

		for(Iterator i=HyadesTestUtil.getTestListeners(result); i.hasNext();)
			((TestListener)i.next()).endTest(this);
	}

	/**
	 * @see junit.framework.TestSuite#runTest(junit.framework.Test, junit.framework.TestResult)
	 */
	public void runTest(final Test test, final TestResult result)
	{		
		if((test instanceof IHyadesTest) && (!((IHyadesTest)test).isSynchronous()))
		{
			Thread t= new Thread("Asynchronous Test")
			{
				public void run()
				{
					try
					{
						test.run(result);
					}
					finally
					{
						HyadesTestSuite.this.runFinished(test);
					}
				}
			};
			activeAsynchronousTest++;
			t.start();
		}
		else
			super.runTest(test, result);
	}

	/**
	 * Calculates the current iteration comparing the number of executions
	 * with the number of tests of the suite.
	 */
	protected void adjustIteration()
	{
		if(parent != null)
		{
			int parentIteration = previousParentIteration;
			if(parent instanceof IHyadesTest)
				parentIteration = ((IHyadesTest)parent).getIteration();

			if(parentIteration != previousParentIteration)
			{
				iteration = 0;
				previousParentIteration = parentIteration;
			}
		}		
		iteration++;
	}
	
	protected synchronized void waitUntilFinished()
	{
		while(activeAsynchronousTest > 0)
		{
			try
			{
				wait();
			}
			catch (InterruptedException e)
			{
				return;
			}
		}
	}

	protected synchronized void runFinished(Test test)
	{
		activeAsynchronousTest--;
		notifyAll();
	}
	
	/**
	 * @see IHyadesTest#toJUnitString()
	 */
	public String toJUnitString()
	{
		return super.toString();
	}
	
	/**
	 * Returns this test's hierarchy.
	 * @see Object#toString()
	 */
	public String toString()
	{
		return HyadesTestUtil.getHierarchy(this, false);
	}
}