/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;

import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.MethodStatisticPage;
import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;


public class MethodStatistic extends StatisticView
{
	private final TRCMethodImpl.TRCMethodSnapshot delta1 = new TRCMethodImpl.TRCMethodSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot delta2 = new TRCMethodImpl.TRCMethodSnapshot();

	public class MethodStatisticSorter extends StatisticSorter {

		public MethodStatisticSorter() {
			super();
		}

		public int compare(Viewer viewer, Object e1, Object e2) {
			
			double d = 0;
 			if(e1 instanceof TRCMethod)	
			{
				TRCMethod m1 = (TRCMethod) e1;
				TRCMethod m2 = (TRCMethod) e2;				 
				TRCMethodImpl.TRCMethodSnapshot ms1 = (TRCMethodImpl.TRCMethodSnapshot)m1.retrieveSnapshot();
				TRCMethodImpl.TRCMethodSnapshot ms2 = (TRCMethodImpl.TRCMethodSnapshot)m2.retrieveSnapshot();
				
				switch(_pos)
				{
					case 0://method name
					  StringBuffer buf1 = new StringBuffer();
					  StringBuffer buf2 = new StringBuffer();					  
					  return (int) _sortSequence * (int)buf1.append(m1.getName()).append(m1.getSignature()).toString()
					     .compareToIgnoreCase(buf2.append(m2.getName()).append(m2.getSignature()).toString());

					case 1://class name
					  return (int) _sortSequence * (int)m1.getDefiningClass().getName().compareToIgnoreCase(m2.getDefiningClass().getName());

					case 2://package name
					  return (int) _sortSequence * (int)m1.getDefiningClass().getPackage().getName().compareToIgnoreCase(m2.getDefiningClass().getPackage().getName());
					  					  					  
					case 3: //base time
					  if(!_info.isDeltaColumn())
					  {																														
						  d = ms1.getBaseTime() - ms2.getBaseTime();
					  }
					  else
					  {
						m1.computeDelta(delta1,TracePackage.TRC_METHOD__BASE_TIME);
						m2.computeDelta(delta2,TracePackage.TRC_METHOD__BASE_TIME);
						d = delta1.getBaseTime() - delta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 4: //average base time
					{
					   int calls1 = ms1.getCalls();
					   if(calls1 == 0)
					      calls1 = 1;
					      
					   int calls2 = ms2.getCalls();
					   if(calls2 == 0)
					      calls2 = 1;
					      
					      
					  d = ms1.getBaseTime()/calls1 - ms2.getBaseTime()/calls2;
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					}
					  
					case 5: //cumulative time
					  if(!_info.isDeltaColumn())
					  {																														
						  d = ms1.getCumulativeTime() - ms2.getCumulativeTime();
					  }
					  else
					  {
						m1.computeDelta(delta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						m2.computeDelta(delta2,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						d = delta1.getCumulativeTime() - delta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					  
					case 6: //calls
					  if(!_info.isDeltaColumn())
					  {																														
						  return (int) _sortSequence * (ms1.getCalls() - ms2.getCalls());
					  }
					  else
					  {
						m1.computeDelta(delta1,TracePackage.TRC_METHOD__CALLS);
						m2.computeDelta(delta2,TracePackage.TRC_METHOD__CALLS);
						return (int) _sortSequence * (delta1.getCalls() - delta2.getCalls());					  	
					  }
					  
				}				  				  				 				   				  
			}
			
			return 0;
		}
		
	}

    /**
     * 
     */
	public class MethodStatisticContentProvider implements ITreeContentProvider {
		
		public void dispose() {
		}
		
		public Object getParent(Object element) {
			return null;
		}
		
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		
		public Object[] getElements(Object inputElement)
		{
			Object[] classes =  PerftraceUtil.getAllClasses(_page.getMOFObject());
			ArrayList methods = new ArrayList();
			
			for(int idx=0; idx<classes.length; idx++)
			{
				methods.addAll(((TRCClass)classes[idx]).getMethods());
			}
			
			return methods.toArray();
		}
				
		public Object[] getChildren(Object element)
		{
			tmpList.clear();			
			return tmpList.toArray(); 	
		}
	
		public boolean hasChildren(Object element) {
			return false;
		}
	}

    /**
     * 
     */
	public class MethodStatisticLabelProvider extends LabelProvider implements ITableLabelProvider
	{
		protected StatisticView _viewer;
		
		public MethodStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}
		
		public Image getColumnImage(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			
			
            //TRCMethod				
			TRCMethod m = (TRCMethod)obj;

			if(m.computeDelta(delta1,TracePackage.TRC_METHOD__BASE_TIME)<0)
			{
				if(col == 1)
					return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
				return null;	
			}
						
			switch(pos)
			{
				case 3: //base time
				  m.computeDelta(delta1,TracePackage.TRC_METHOD__BASE_TIME);
				  d = delta1.getBaseTime();
				  if(d<0) state = -1;
				  else if(d>0) state = 1;
				  
				  break;
				  
				case 4: //average base time
				  state = 0;				  
				  break;
				  					  					  
				case 5: //cumulative time
				  m.computeDelta(delta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
				  d = delta1.getCumulativeTime();
				  if(d<0) state = -1;
				  else if(d>0) state = 1;
				  
				  break;
					  
				case 6: //calls
				  m.computeDelta(delta1,TracePackage.TRC_METHOD__CALLS);				
				  state = delta1.getCalls();
				  break;
				  
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
		}

		public String getColumnText(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			TRCMethod m = (TRCMethod)obj;
			TRCMethodImpl.TRCMethodSnapshot ms = (TRCMethodImpl.TRCMethodSnapshot)m.retrieveSnapshot();
			
            //TRCMethod
			switch(pos)
			{				
				case 0://method name
				  StringBuffer buf = new StringBuffer();
				  return buf.append(m.getName()).append(m.getSignature()).toString() ;
				  
				case 1: //class name
				  return m.getDefiningClass().getName();

				case 2: //package name
					return PerftraceUtil.getPackageName(m.getDefiningClass().getPackage(), _page.getMOFObject()).toString();
				 				  
				case 3: //base time
				  if(!info.isDeltaColumn())
				  {				
					  if(isShowPercent())
					  {
					  	  return PerftraceUtil.formatAsPercentage(ms.getBaseTime() / _maxTime);
					  }
					  else
					  {													  													
					  	return TString.formatTimeValue(ms.getBaseTime());
					  }
				  }
				  else
				  {
					m.computeDelta(delta1,TracePackage.TRC_METHOD__BASE_TIME);
					return TString.formatTimeValue(delta1.getBaseTime());					  	
				  }
				  
				case 4: //average base time
				{
				   int calls = ms.getCalls();
				   if(calls == 0)
				      calls = 1;
				      
				  if(isShowPercent())
				  {
				  	  return TString.formatAsPercentage(ms.getBaseTime() / (calls*_maxTime));
				  }
				  else
				  {													  					
					return TString.formatTimeValue(ms.getBaseTime() / calls);
				  }
				}
				  
				case 5: //cumulative time
				  if(!info.isDeltaColumn())
				  {								
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage(ms.getCumulativeTime() / _maxTime);
					  }
					  else
					  {													  									
						return TString.formatTimeValue(ms.getCumulativeTime());
					  }
				  }
				  else
				  {
					m.computeDelta(delta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);					  	
					return TString.formatTimeValue(delta1.getCumulativeTime());					  	
				  }
				  
				case 6: //calls
				  if(!info.isDeltaColumn())
				  {								
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage(ms.getCalls() / _totalCalls);
					  }
					  else
					  {													  									
						return String.valueOf(ms.getCalls());
					  }
				  }
				  else
				  {
					m.computeDelta(delta1,TracePackage.TRC_METHOD__CALLS);
					return String.valueOf(delta1.getCalls());					  	
				  }
				  
			}				  				  				 				   				  
            							
			return "";
		}
	}
	
	public MethodStatistic(Composite parent, TraceViewerPage page) {
		super(parent, page);	
		_viewerFilter = new StatisticFilter();				
	}
	

	public String getColumnsPreferencesKey() {
		return "M60"; // Note this string is not to be translated
	}
	
	public String getDefaultColumnsTemplate() {
		//Method Columns Data
		String methodColumn =
			TraceUIPlugin.getString("STR_ST_METHOD_NAME")			+ ":0:"
				+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
				+ TraceUIPlugin.getString("STR_ST_CLASS_NAME")	+ ":1:"
					+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:200,"
				+ TraceUIPlugin.getString("STR_ST_PACKAGE")	+ ":2:0:left:200,"					
				+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":3:" 
					+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_CPU_TIME_AVERAGE")		+ ":4:" 
					+ String.valueOf(ColumnData.IS_VISIBLE ) + ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_STACK_TIME")	+ ":5:"
					+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
				+ TraceUIPlugin.getString("STR_ST_CALLS")			+ ":6:"
					+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:50";
					
		return methodColumn;
	}

	protected Composite createTable(Composite parent, int options) {
		return new Table(parent, options);
	}
	protected StructuredViewer createTableViewer(Composite table) {
		return new TableViewer((Table) table);
	}

	protected IContentProvider getContentProvider() {
		return new MethodStatisticContentProvider();
	}

	public LabelProvider getTableLabelProvider()
	{
		return new MethodStatisticLabelProvider(this);
	}

	public Table getTable() {
		return (Table) getTableViewer().getControl();
	}

	protected void handleSelectionEvent() {
		
		ITraceSelection model = UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			Object sel = model.getFirstElement();

            if( sel instanceof TRCMethodInvocation)
            {
            	select(((TRCMethodInvocation)sel).getMethod());
            	
            }
			if (sel instanceof TRCMethod) {
				select((TRCMethod)sel);
			}
		}

	}

	private void select(TRCMethod object) {
		Table table = getTable();
		
		for (int idx = 0; idx < table.getItemCount(); idx++) {
			Object item = table.getItem(idx).getData();
			if (item == object)
			{
				table.setSelection(idx);
				setNewSelection();
				return;
			}
		}
		
		table.deselectAll();
	}
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		
		menu.add(getUpdateAction());

		menu.add(fSeparator);
		menu.add(getChooseColumnsAction(getColumnDataList(), getColumnsPreferencesKey()));
	}

	/**
	* Insert the method's description here.
	* Creation date: (8/7/2001 2:12:40 PM)
	*/
	public void updateButtons() {

		((MethodStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());		
		((MethodStatisticPage) getTraceViewerPage()).percentMode().setChecked(isShowPercent());

	}
	public void updateModelSelection() {
		ISelection selection = getTableViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			UIPlugin.getDefault().getSelectionModel(
				_page.getMOFObject()).add(
				sel);
	
			ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
			event.setSource(_page.getMOFObject());
			UIPlugin.getDefault().notifyViewSelectionChangedListener(event);			
		}
	}
	
	public void update() {
		if (_firstTime) {
			getTableViewer().addFilter(getViewerFilter());
			_firstTime = false;
			Table table = getTable();
			TableColumn firstColumn = table.getColumn(0);
			
			_viewerSorter = new MethodStatisticSorter();

			getViewerSorter().setSortedColumn(firstColumn);
			getTableViewer().setSorter(getViewerSorter());
			
		}
		// set the input of the viewer
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());	
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
		}
		
		getTableViewer().setInput(_page.getMOFObject());

        getTable().setRedraw(false);
		getTableViewer().refresh();
        getTable().setRedraw(true);
        
        handleSelectionEvent();
		
	}

	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event)
	{
		if(!getTraceViewerPage().getTraceViewer().isProcessRefreshEvents()
		   || getTable().isFocusControl())
		  return;
		  
		handleSelectionEvent();  
	}
	
	public boolean isEmpty()
	{
		Object[] classes = PerftraceUtil.getAllClasses(_page.getMOFObject());
		int size = 0;
		
		for (int idx = 0; idx < classes.length; idx++)
		{
			size = size + ((TRCClass)classes[idx]).getMethods().size();
		}
		return (!(size > 0));
	}
	
}