/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.internal.ui;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.actions.*;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.launcher.*;
import org.eclipse.hyades.trace.ui.internal.util.UnresolvedResourceHelper;
import org.eclipse.hyades.trace.ui.internal.wizard.*;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.*;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jface.action.*;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.actions.*;
import org.eclipse.ui.dialogs.PropertyDialogAction;
import org.eclipse.ui.part.ViewPart;

public class PDProjectExplorer extends ViewPart implements IMenuListener
															  , IProfileEventListener
															  , IPropertyListener
{

	private PDProjectViewer fViewer;
	protected DeleteAction fDeleteAction;
	protected OpenWizardAction fOpenMonitorAction;
	protected OpenWizardAction fOpenNodeAction;
	protected OpenWizardAction fOpenProfAgentAction;
	protected OpenWizardAction fOpenLogAgentAction;
	protected RefreshFromLocalAction fRefreshLocal;
	protected RefreshTreeAction fRefreshTree;
	protected SaveAction fSaveToFile;
	protected StartTraceAction fStartTraceAction;
	protected StopTraceAction fStopTraceAction;
	protected PropertyDialogAction fPropertyDialogAction;
	protected OpenWizardAction fOpenProcessAction;
	protected Menu fContextMenu;
	
	protected Map traceViewsMap;
	
	protected ActionGroup fImportActionGroup;
	protected Separator fAttachGrp;
	protected Separator fGCGroup;	
	protected Separator fMonitorGrp;
	protected Separator fTerminateGrp;
	protected Separator fRefreshGrp;
	protected RelaunchTraceAction fRelaunchAction;

	private static Hashtable propertyPageActionMap = null;
	
	//instead of adding separator before group, we create group with separator
	private final String GROUP_ADD_VIEWS = "group.add.views";
	private final String GROUP_REFRESH = "group.refresh";
	private final String GROUP_SAVE = "group.save";
	private final String GROUP_OPEN_PROPERTY = "group.open.property";

   class ImportActionGroup extends ActionGroup
   {
	
		private final String GROUP_IMPORT= "group.import";
		
		private ImportResourcesAction fImportAction;
		private ExportResourcesAction fExportAction;
	
		/**
		 * Creates a new <code>ImportActionGroup</code>. The group 
		 * requires that the selection provided by the part's selection provider 
		 * is of type <code>org.eclipse.jface.viewers.IStructuredSelection</code>.
		 * 
		 * @param part the view part that owns this action group
		 */
		public ImportActionGroup(IViewPart part) {
			IWorkbenchWindow workbench = part.getSite().getWorkbenchWindow();
			fImportAction= new ImportResourcesAction(workbench);
			fExportAction= new ExportResourcesAction(workbench);			
		}
		
		/* (non-Javadoc)
		 * Method declared in ActionGroup
		 */
		public void fillContextMenu(IMenuManager menu) {
			menu.appendToGroup(IContextMenuConstants.GROUP_REORGANIZE, new Separator(GROUP_IMPORT));
			menu.appendToGroup(GROUP_IMPORT, fImportAction);
			menu.appendToGroup(GROUP_IMPORT, fExportAction);
			
			super.fillContextMenu(menu);
		}	
	}

	public PDProjectExplorer() {
		super();
		
		/* set project explorer to resource helper such that it can be notified dirty for save */
		Object ue = HierarchyResourceSetImpl.getInstance().getUnresolvedException();
		if ((ue != null) && (ue instanceof UnresolvedResourceHelper))
			 ((UnresolvedResourceHelper) ue).setPDProjectExplorer(this);
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/24/2000 2:06:10 PM)
	 * @param menu com.ibm.jface.MenuManager
	 */
	protected void addViews(IMenuManager menu, Object obj) {
		
		if(obj == null || !(obj instanceof EObject))
		  return;

		MenuManager popup = new MenuManager(UIPlugin.getResourceString("MENU_MANAGER"));
		menu.appendToGroup(GROUP_ADD_VIEWS,popup);
		  
		if(obj instanceof TRCMonitor)
		{
			TRCMonitor monitor = (TRCMonitor)obj;
			IAction[] actions = getViewMappingForType(TraceConstants.MONITOR_TYPE);
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
			
			ArrayList typesList = new ArrayList();
			EList nodes = monitor.getNodes();
			if(nodes.size() > 0)
			{ 
				for(int idx=0; idx<nodes.size(); idx++)
				{
					TRCNode node = (TRCNode) nodes.get(idx);
					
					EList processes = node.getProcessProxies();
					for(int i=0; i<processes.size(); i++)
					{
						TRCProcessProxy process = (TRCProcessProxy)processes.get(i);
						EList agents = process.getAgentProxies();
						
						for(int j=0; j<agents.size(); j++)
						{
							TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
							
							if(typesList.contains(agent.getType()))
							  continue; //actions already added to the popup
							
							typesList.add(agent.getType());
							actions = getViewMappingForType(agent.getType());
							if(actions.length > 0)
							{//add the actions to the popup
								popup.add(new Separator());								
								for(int k=0; k<actions.length; k++)
									popup.add(actions[k]);
							}
						}
					}
				}
			}
		}
		else if(obj instanceof TRCNode)
		{
			IAction[] actions = getViewMappingForType(TraceConstants.HOST_TYPE);
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
			
			ArrayList typesList = new ArrayList();
			EList processes = ((TRCNode) obj).getProcessProxies();
			for(int i=0; i<processes.size(); i++)
			{
				TRCProcessProxy process = (TRCProcessProxy)processes.get(i);
				EList agents = process.getAgentProxies();
				
				for(int j=0; j<agents.size(); j++)
				{
					TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
					
					if(typesList.contains(agent.getType()))
					  continue; //actions already added to the popup
					
					typesList.add(agent.getType());
					actions = getViewMappingForType(agent.getType());
					if(actions.length > 0)
					{//add the actions to the popup
						popup.add(new Separator());						
						for(int k=0; k<actions.length; k++)
							popup.add(actions[k]);
					}
				}
			}
			
		}
		else if(obj instanceof TRCProcessProxy)
		{
			ArrayList typesList = new ArrayList();
				EList agents = ((TRCProcessProxy)obj).getAgentProxies();
				
			for(int j=0; j<agents.size(); j++)
			{
				TRCAgentProxy agent = (TRCAgentProxy)agents.get(j);
				
				if(typesList.contains(agent.getType()))
				  continue; //actions already added to the popup
				
				typesList.add(agent.getType());
				IAction[] actions = getViewMappingForType(agent.getType());
				if(actions.length > 0)
				{//add the actions to the popup
					popup.add(new Separator());
					for(int k=0; k<actions.length; k++)
						popup.add(actions[k]);
				}
			}			
		}
		else if(obj instanceof TRCAgentProxy)
		{
			IAction[] actions = getViewMappingForType(((TRCAgentProxy)obj).getType());
			for(int idx=0; idx<actions.length; idx++)
			{
				if(actions[idx] != null)
				   popup.add(actions[idx]);
			}
		}
	}
	/**
	 * Creates the SWT controls for a part.  
	 *
	 * TIP : This method must call hookFocus() on some control (most likely
	 * the top-level control), within the view.  This translates OS-level
	 * focus to part focus within the desktop.
	 */
	public void createPartControl(Composite parent) {
	
	    traceViewsMap = new HashMap();
	    TraceAssociationManager.getTraceViewMappingRegistry().addCommitChangeListener(this);
	    
		fViewer = new PDProjectViewer(parent);
	
		fViewer.setContentProvider(new PDContentProvider(this));
		fViewer.setLabelProvider(new PDLabelProvider());
		fViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
	
		MenuManager menuMgr= new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);
		fContextMenu= menuMgr.createContextMenu(fViewer.getTree());
		fViewer.getTree().setMenu(fContextMenu);
		
		makeActions();
		
		// Register viewer with site. This must be done before making the actions.
		IWorkbenchPartSite site= getSite();
		site.registerContextMenu(menuMgr, fViewer);
		site.setSelectionProvider(fViewer);

		fillLocalToolBar();
			
		// register this viewer as the debug UI selection provider
		UIPlugin.getDefault().addSelectionProvider(fViewer, this);
		fViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				handleDoubleClick(event);
			}
		});
	
		getSite().setSelectionProvider(fViewer);
		
		UIPlugin.getDefault().addProfileEventListener(this);				
	}

	/**
	 * @see IWorkbenchPart
	 */
	public void dispose() {
		
		TraceAssociationManager.getTraceViewMappingRegistry().removeCommitChangeListener(this);
		
		if (fContextMenu != null && !fContextMenu.isDisposed())
			fContextMenu.dispose();
		
		UIPlugin.getDefault().removeProfileEventListener(this);		
		
		if (fViewer != null)
			UIPlugin.getDefault().removeSelectionProvider(fViewer, this);

        if(fImportActionGroup != null)
           fImportActionGroup.dispose();
           
		fAttachGrp = null;
		fRefreshGrp = null;
		fTerminateGrp = null;
		fMonitorGrp = null;
		
		fSaveToFile.dispose();
		fDeleteAction.dispose();
		fOpenLogAgentAction.dispose();
		fOpenMonitorAction.dispose();
		fOpenNodeAction.dispose();
		fOpenProcessAction.dispose();
		fPropertyDialogAction.dispose();
		fRefreshLocal.dispose();
		fRefreshTree.dispose();
		fSaveToFile.dispose();
           
 		try {
 			
 			boolean modified = false;
			if (SaveUtil.documentsSize() > 0) {
				
				Enumeration docs = SaveUtil.getDocuments().elements();
				while(docs.hasMoreElements())
				{
					org.eclipse.emf.ecore.resource.Resource res = (org.eclipse.emf.ecore.resource.Resource) docs.nextElement();
					if(res != null && res.isLoaded() && res.isModified())
					{
						modified = true;
						break;
					}
				}
				
				if(modified)
				{
					String title = UIPlugin.getResourceString("TRACE_MSG");
	
					String msg = UIPlugin.getResourceString("SAVE_PROJECT_SQ");
	
					if (MessageDialog.openQuestion(null, title, msg)) {
						SaveUtil.saveDocuments();
	
						IResource res = UIPlugin.getPluginWorkbench().getRoot();
						res.refreshLocal(IResource.DEPTH_INFINITE, null);
					}
				}
				
				//clear documents list
				SaveUtil.getDocuments().clear();

			}
		} catch (Exception exc) {
			exc.printStackTrace();
			//do nothing
		}


        fViewer.dispose();
        fViewer = null; 
        

		super.dispose();
	}
	
	public final boolean hasFilter(int filter){
		return fViewer.hasFilter(filter);		
	}

	public Control getControl(){
		return fViewer.getControl();
	}
	
	public void refresh(){
		fViewer.refresh();
	}	
	private boolean areAllProcessActive(Object[] selections) {
		for (int idx = 0; idx < selections.length; idx++) {
			if (!((TRCProcessProxy) selections[idx]).isActive())
				return false;
		}
		return true;
	}

	private boolean anyProcessActive(Object[] selections) {
		for (int idx = 0; idx < selections.length; idx++) {
			if (((TRCProcessProxy) selections[idx]).isActive())
				return true;
		}
		return false;
	}

	private boolean areSameClass(Object[] selection) {
		Class objClass = selection[0].getClass();
		for (int idx = 1; idx < selection.length; idx++) {
			if (!(objClass.equals(selection[idx].getClass()))) {
				return false;
			}
			objClass = selection[idx].getClass();
		}
		return true;
	}

	public void fillContextMenu(IMenuManager menu) {
		
		String objType = "";
		IStructuredSelection selection = (IStructuredSelection) fViewer.getSelection();
		if (selection.isEmpty()) {
			
			addLaunchOptions(menu);
			
			menu.add(new Separator(GROUP_REFRESH));
			menu.appendToGroup(GROUP_REFRESH,fRefreshTree);
	
			menu.add(new Separator(IContextMenuConstants.GROUP_REORGANIZE));
			menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));
			return;
		}
	
		Object[] selections = selection.toArray();
		boolean sameClass = false;
		boolean multiSelection = (selections.length > 1);
		if (multiSelection)
			sameClass = areSameClass(selections);
	
		Object sel = selection.iterator().next();
	
		if (sel instanceof IContainer)
		{
			addLaunchOptions(menu);			
	    }
	    else if(sel instanceof TRCMonitor)
	    {
			addLaunchOptions(menu);	
			menu.add(new Separator(GROUP_ADD_VIEWS));		
			addViews(menu, sel);
			menu.add(fRefreshGrp);						    	
	    }
	    else if(sel instanceof TRCNode)
	    {
			addLaunchOptions(menu);	
	   		menu.add(new Separator(GROUP_ADD_VIEWS));			
			addViews(menu, sel); 
			menu.add(fRefreshGrp);
					
	    	
	    }
	    else if(sel instanceof TRCProcessProxy)
	    {
			if (!multiSelection) {
				menu.add(fGCGroup);
			    menu.add(fTerminateGrp);
				if (((TRCProcessProxy) sel).getLaunchMode() == 1) //launch mode
				{
					menu.add(fRelaunchAction);
				}
				
				menu.add(new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.add(fRefreshGrp);
				
			}
	    	
	    }
	    else if(sel instanceof TRCAgentProxy)
	    {
			if (!multiSelection || sameClass) {
				menu.add(fMonitorGrp);
				menu.add(fGCGroup);
				menu.add(fAttachGrp);				
			}
			
			if(!multiSelection)
			{
			    menu.add(fTerminateGrp);
				if (((TRCAgentProxy) sel).getProcessProxy().getLaunchMode() == 1) //launch mode
				{
					menu.add(fRelaunchAction);
				}
				
				menu.add(new Separator(GROUP_ADD_VIEWS));		
				addViews(menu, sel);
				menu.add(fRefreshGrp);
			}
	    	
	    }
	
		menu.add(new Separator(GROUP_SAVE));
		menu.appendToGroup(GROUP_SAVE,fSaveToFile);
	
		menu.add(new Separator(GROUP_REFRESH));
		menu.appendToGroup(GROUP_REFRESH,fRefreshLocal);
		menu.appendToGroup(GROUP_REFRESH,fRefreshTree);
		((DeleteAction) fDeleteAction).setSelections(selections);
		menu.appendToGroup(GROUP_REFRESH,fDeleteAction);
	
	
		if (!multiSelection) {
			
			menu.add(new Separator(GROUP_OPEN_PROPERTY));
			if (fPropertyDialogAction.isApplicableForSelection())
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fPropertyDialogAction);
	
			if (sel instanceof TRCMonitor) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenMonitorAction);
			} else if (sel instanceof TRCNode) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenNodeAction);
			} else if (sel instanceof TRCProcessProxy) {
				menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenProcessAction); 
			} else if (sel instanceof TRCAgentProxy) {
				if (((TRCAgentProxy) sel).getType().equals(TraceConstants.PROFILE_AGENT_TYPE)) {
					menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenProfAgentAction);
				} else {
					if (propertyPageActionMap != null) {
						String aType = ((TRCAgentProxy) sel).getType();
						if (propertyPageActionMap.containsKey(aType)) {
							Object propertyAction = propertyPageActionMap.get(aType);
							if (propertyAction instanceof Action) {
								menu.appendToGroup(GROUP_OPEN_PROPERTY,(Action) propertyAction);
							}
						} else
							menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenLogAgentAction);
					} else {
						menu.appendToGroup(GROUP_OPEN_PROPERTY,fOpenLogAgentAction);
					}
				}
			}
		}
		
		menu.add(new Separator(IContextMenuConstants.GROUP_REORGANIZE));
		menu.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));
	
	}

	/**
	 *  Fill the local tool bar with actions.
	 */
	protected void fillLocalToolBar() {
		
		IActionBars actionBars = getViewSite().getActionBars();
		IToolBarManager toolbar = actionBars.getToolBarManager();

		IMenuManager menu = actionBars.getMenuManager();

		menu.add(fViewer.showFolderAction());
		menu.add(fViewer.showMonitorAction());
		menu.add(fViewer.showNodesAction());
		menu.add(fViewer.showProcessAction());
		menu.add(fViewer.showProfAgentsAction());

		if (fViewer.showLogAgentsAction() != null)
			menu.add(fViewer.showLogAgentsAction());

		actionBars.updateActionBars();

	}
	/**
	 * Returns the trace explorer part of the active perspective. If 
	 * there isn't any trace explorer part <code>null</code> is returned.
	 */
	public static PDProjectExplorer getFromActivePerspective() {
		
		try {
			IViewPart view =
				UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
			if (view instanceof PDProjectExplorer)
				return (PDProjectExplorer) view;
	
		}
		catch(PartInitException exc)
		{
			exc.printStackTrace();
			return null;
		}
		
		return null;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (10/02/2000 1:19:12 PM)
	 * @return com.ibm.itp.core.api.resources.IFolder
	 * @param selection java.lang.Object
	 */
	public IResource getSelectionFolder() {
		return fViewer.getSelectionFolder();
	}

	public IResource getSelectionFolder(TreeItem selection) {
		return fViewer.getSelectionFolder(selection);
	}
	/**
	 * Returns the shell to use for opening dialogs.
	 * Used in this class, and in the actions.
	 */
	public Shell getShell() {
		return fViewer.getTree().getShell();
	}
	/**
	 * Returns the tool tip text for the view.
	 */
	protected String getToolTipText() {
		return UIPlugin.getResourceString("PDPROEXPTOOLTIP");
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (1/18/2001 5:10:11 PM)
	 * @return com.ibm.etools.pd.core.PDProjectViewer
	 */
	public PDProjectViewer getViewer() {
		return fViewer;
	}
	/**
	 * Handles double clicks in viewer.
	 * Opens editor if file double-clicked.
	 */
	protected void handleDoubleClick(DoubleClickEvent event) {
		IStructuredSelection s = (IStructuredSelection) event.getSelection();
		Object obj = s.getFirstElement();

		if(obj == null || !(obj instanceof EObject))
		  return;

		if(obj instanceof TRCMonitor)
		{
			IAction[] actions = getViewMappingForType("Monitor");
			if(actions.length > 0 && actions[0] != null)
			    actions[0].run();
		}
		else if(obj instanceof TRCNode)
		{
			IAction[] actions = getViewMappingForType("Host");
			if(actions.length > 0 && actions[0] != null)
				actions[0].run();
		}
		else if(obj instanceof TRCProcessProxy)
		{
			EList agents = ((TRCProcessProxy)obj).getAgentProxies();
			for(int idx=0; idx<agents.size(); idx++)
			{
				TRCAgentProxy agent = ((TRCAgentProxy)agents.get(idx));
				if(agent == null || agent.eIsProxy())
				  continue;
				  
				IAction[] actions = getViewMappingForType(agent.getType());
				if(actions.length > 0 && actions[0] != null)
				{
					actions[0].run();					
					return;
				}				  				 
			}
		}
		else if(obj instanceof TRCAgentProxy)
		{
			IAction[] actions = getViewMappingForType(((TRCAgentProxy)obj).getType());
			if(actions.length > 0 && actions[0] != null)
				actions[0].run();
		}
	}
	/* (non-Javadoc)
	 * Method declared on IViewPart.
	 */
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
	}
	public static Hashtable getPropertyPageActionMap() {
		return propertyPageActionMap;
	}

	/**
	 *  Create self's action objects
	 */
	protected void makeActions() {
		
		fImportActionGroup = new ImportActionGroup(this);
		
		fAttachGrp = new Separator(TraceConstants.PROFILE_ATTACH_GROUP);
		fGCGroup = new Separator(TraceConstants.PROFILE_GC_GROUP);
		fRefreshGrp = new Separator(TraceConstants.PROFILE_REFRESH_GROUP);
		fTerminateGrp = new Separator(TraceConstants.PROFILE_TERMINATE_GROUP);
		fMonitorGrp = new Separator(TraceConstants.PROFILE_MONITOR_GROUP);
		
		fRefreshTree = new RefreshTreeAction(this);
	
		fRefreshLocal = new RefreshFromLocalAction(this);
		fSaveToFile = new SaveAction(this);
	
		fOpenMonitorAction =
			new OpenWizardAction(
				this,
				OpenMonitorWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProfAgentAction =
			new OpenWizardAction(
				this,
				OpenProfAgentWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenLogAgentAction =
			new OpenWizardAction(
				this,
				OpenLogAgentWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenNodeAction =
			new OpenWizardAction(
				this,
				OpenNodeWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
		fOpenProcessAction =
			new OpenWizardAction(
				this,
				OpenProcessWizard.class,
				UIPlugin.getResourceString("PROPERTIES"));
	
		fDeleteAction = new DeleteAction(this);
		fPropertyDialogAction = new PropertyDialogAction(getShell(), getViewer());
	
		fRelaunchAction = new RelaunchTraceAction(this);
	}

	public void menuAboutToShow(IMenuManager menu) {
				
		fillContextMenu(menu);		
		fImportActionGroup.fillContextMenu(menu);
		
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (10/05/2000 1:19:46 PM)
	 */
	public void refreshView(Object selection) {
		fViewer.refresh();

		fViewer.selectObject(selection);

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (09/26/2000 12:35:46 PM)
	 * @param obj java.lang.Object
	 */
	public void selectObject(Object item) {
		fViewer.selectObject(item);
	}
	/**
	 * Asks the part to take focus within the desktop.
	 */
	public void setFocus() {
		if (fViewer != null)
			fViewer.getControl().setFocus();
	}

	private void addLaunchOptions(IMenuManager menu)
	{
		menu.add(new LaunchTraceAction());		
		menu.add(new AttachTraceAction());
	}

    /**
     * 
     */
	public void handleProfileEvent(ProfileEvent event) {
			
		switch(event.getType())
		{
			case ProfileEvent.START_MONITOR:
			
		   	   fViewer.selectObject(event.getSource());
			   
			break;
			
			case ProfileEvent.ATTACH:
			
		   	   fViewer.selectObject(event.getSource());
			   
			break;
						
			default:
				fViewer.refresh(event.getSource());
			
			
		}	
	}

    /**
     * Returns the list of actions available for this trace type
     * @param type
     */
    protected IAction[] getViewMappingForType(String type)
    {
    	Object mapp = traceViewsMap.get(type);
    	
    	if(mapp != null)
    	    return ((IAction[])mapp);
    	    
		IAssociationMappingRegistry registry = TraceAssociationManager.getTraceViewMappingRegistry();
		IAssociationMapping mapping = registry.getAssociationMapping(HyadesUIPlugin.EP_ANALYZER_EXTENSIONS);
		
		if(mapping == null)
		{
			IAction[] actions = new IAction[0];
			traceViewsMap.put(type, actions);
			
			return actions;
		}
    		    		  
		IAssociationDescriptor[] array = mapping.getAssociationDescriptors(type);
		IAction[] actions = new IAction[array.length];
		
		for(int idx=0; idx<array.length; idx++)
		{
			IAssociationDescriptor desc = array[idx];
				
			try {
					
				IAction action = (IAction)desc.createImplementationClassInstance();
				
				if(action != null)
				{
					action.setText(desc.getName());
					action.setImageDescriptor(desc.getImageDescriptor());
				
					actions[idx] = action;										
				}								
			} catch (Exception exc) {
				exc.printStackTrace();
			}
								
		}
		
		traceViewsMap.put(type, actions);
		return actions;
    	        	  
    }

	/**
	 * Indicates that a property has changed.
	 * @see IPropertyListener
	 */
	public void propertyChanged(Object source, int propId)
	{
		//reset trace view actions
		if(source instanceof IAssociationMappingRegistry)
			traceViewsMap.clear();
	}
    
}
