/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.console;

import org.eclipse.ui.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.internal.preferences.*;
import org.eclipse.jface.text.*;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Display;

public class TraceConsoleDocument extends AbstractDocument
{
	public final static int fgMinimumSize= 500;
	public final static int fgMinMaxRation= 5;

	private boolean fClosed= false;
	private int fMinSize= fgMinimumSize;
	private int fMaxSize= fgMinimumSize * fgMinMaxRation;

	private int fLastStreamWriteEnd= 0;
	private int fNewStreamWriteEnd= 0;
	protected boolean fNeedsToStartReading= true;
	
	public static final int OUT= 0;
	public static final int ERR= 1;
	
	protected List fStyleRanges= new ArrayList(2);

	protected ConsoleDataProcessor fDataConsole;
	protected TraceConsoleViewer fConsoleViewer= null;
	public TraceConsoleDocument(ConsoleDataProcessor dataConsole)
	{
		super();
		fDataConsole= dataConsole;
		setTextStore(new ConsoleOutputTextStore(fMaxSize));
		setLineTracker(new DefaultLineTracker());
		
		completeInitialization();
	}
	/**
	 * Adds a new style range if the document is not closed.
	 * Note that the document can be closed by a separate thread.
	 * This is the reason for the copy of the style ranges.
	 */
	protected void addNewStyleRange(StyleRange newRange) {
		List tempRanges= fStyleRanges;
		if (fClosed) {
			return;
		}
		tempRanges.add(newRange);
	}
	protected void clearDocument() {
		fStyleRanges= new ArrayList(2);
		set("");
	}
	public void close() {
		stopReading();

		fClosed= true;
		fStyleRanges= Collections.EMPTY_LIST;
		set("");
	}
	/**
	 * Coalese that last two style ranges if they are similar
	 */
	protected void coaleseRanges() {
		int size= fStyleRanges.size();
		if (size > 1) {
			StyleRange last= (StyleRange) fStyleRanges.get(size - 1);
			StyleRange nextToLast= (StyleRange) fStyleRanges.get(size - 2);
			if (last.similarTo(nextToLast)) {//same color?
				StyleRange newRange= new StyleRange(nextToLast.start, last.length + nextToLast.length, last.foreground, null);
				fStyleRanges.remove(size - 1);
				fStyleRanges.remove(size - 2);
				addNewStyleRange(newRange);
			}
		}
	}
	/**
	 * If the buffer is longer than fMaxSize, 
	 * trim it back to fMinSize.
	 */
	protected void ensureSizeConstraints() {
		if (getLength() > fMaxSize) {
			replace(0, getLength() - fMinSize, "");
		}
	}
	public boolean equals(Object obj)
	{
			boolean correctInstance= obj instanceof TraceConsoleDocument;

			if (fDataConsole != null) {
				return correctInstance && fDataConsole.equals(((TraceConsoleDocument)obj).fDataConsole);
			} else {
				return correctInstance && ((TraceConsoleDocument)obj).fDataConsole == null;
			}
	}
	/**
	 * Fires the <code>DocumentEvent</code>, but also
	 * writes to the proxy if the user is entering input and
	 * has hit "Enter".
	 */
	protected void fireDocumentChanged(DocumentEvent event)
	{
		super.fireDocumentChanged(event);
		String eventText= event.getText();
		if (eventText == null || 0 >= eventText.length() || eventText.length() > 2) {
			return;
		}
		String[] lineDelimiters= event.getDocument().getLegalLineDelimiters();
		for (int i= 0; i < lineDelimiters.length; i++) {
			if (lineDelimiters[i].equals(eventText)) {
				try {
					String inText= event.getDocument().get();
					inText= inText.substring(fNewStreamWriteEnd, inText.length());
					if (inText.length() == 0) {
						return;
					}
					if(fDataConsole != null)
						fDataConsole.write(inText);
						
					fLastStreamWriteEnd= getLength();
					return;
				} catch (Exception ioe) {
					ioe.printStackTrace();
				}
			}
		}
	}
	/**
	 * Returns the position after which editing of the
	 * content is allowable.
	 */
	protected int getStartOfEditableContent() {
		return fLastStreamWriteEnd;
	}
	/**
	 * Make visible to the ConsoleViewer
	 */
	protected ITextStore getStore() {
		return super.getStore();
	}
	protected StyleRange[] getStyleRanges() {
		if (fStyleRanges.isEmpty()) {
			return new StyleRange[]{};
		} 
		StyleRange[] sRanges= new StyleRange[fStyleRanges.size()];
		return (StyleRange[])fStyleRanges.toArray(sRanges);
	}
	public boolean isClosed() {
		return fClosed;
	}
	/**
	 * Returns whether the document's underlying process is
	 * terminated.
	 */
	protected boolean isReadOnly() {
		return false;
	}
	public void replace(int pos, int replaceLength, String text) {
		if (isReadOnly() || pos < getStartOfEditableContent()) {
			return;
		}

		replace0(pos, replaceLength, text);
		int docLength= getLength();
		if (docLength == fNewStreamWriteEnd) {
			//removed all of the user input text
			fStyleRanges.remove(fStyleRanges.size() - 1);
		} else {
			updateInputStyleRange(docLength);
			//notify the viewer that the style ranges have changed.
			fireDocumentChanged(new DocumentEvent(this, 0, 0, ""));
		}
	}
	/**
	 * Replace text used to add content from streams even though
	 * the process is terminated (and therefore the doc is "read only")
	 */
	protected void replace0(int pos, int replaceLength, String text) {
		try {		
			super.replace(pos, replaceLength, text);
		} catch (BadLocationException ble) {
			ble.printStackTrace();
		}
		
		if (text != null && text.length() - replaceLength > fMaxSize / 2) {
			ensureSizeConstraints();
		}
	}
	public void set(String text) {
		fNewStreamWriteEnd= text.length();
		super.set(text);
		fLastStreamWriteEnd= fNewStreamWriteEnd;
		ensureSizeConstraints();
	}
	public void setBufferSize(int minSize, int maxSize) {
		fMinSize= (minSize < fgMinimumSize ? fgMinimumSize : minSize);
		fMaxSize= (maxSize < minSize * fgMinMaxRation ? minSize * fgMinMaxRation : maxSize);

		if (getStore() instanceof ConsoleOutputTextStore)
			 ((ConsoleOutputTextStore) getStore()).setMinimalBufferSize(fMinSize);

		ensureSizeConstraints();
	}
	/**
	 * Sets the console viewer that this document is viewed within.
	 * Can be set to <code>null</code> if no longer currently being
	 * viewed.
	 */
	protected void setConsoleViewer(TraceConsoleViewer viewer) {
		fConsoleViewer = viewer;
	}
	protected void setStyleRanges(List ranges) {
		fStyleRanges= ranges;
	}
	public void startReading()
	{
//		System.out.println("startReading");

	}
	protected void stopReading()
	{
//		System.out.println("stopReading");	
	}
	/**
	 * System out or System error has had text append to it.
	 * Adds the new text to the document.
	 */
	protected void streamAppended(final String text, final int source)
	{
		update(new Runnable() {
			public void run()
			{
				try {
					IWorkbenchPage persp = UIPlugin.getActivePage();
					if(persp != null) 
					    persp.showView(PDPerspective.ID_CONSOLE_VIEW);
				}
				catch(Exception exc)
				{
				}
				
				int appendedLength= text.length();
				fNewStreamWriteEnd= fLastStreamWriteEnd + appendedLength;
				TraceConsoleDocument.this.replace0(fLastStreamWriteEnd, 0, text);
				updateOutputStyleRanges(source);
				fLastStreamWriteEnd= fNewStreamWriteEnd;
			}
		});
	}
	/**
	 * @see IInputStreamListener
	 */
	protected void systemErrAppended(String text)
	{
		streamAppended(text, ERR);
	}
	/**
	 * @see IInputStreamListener
	 */
	protected void systemOutAppended(String text) {
		streamAppended(text, OUT);
	}
	/**
	 * Posts the update code "behind" the running operation if the 
	 * UI will be updated.
	 */
	protected void update(Runnable runnable) {
		if (fConsoleViewer != null) {
			fConsoleViewer.getControl().getDisplay().asyncExec(runnable);
		} else {
			Display display= UIPlugin.getDefault().getConsoleManager().getDisplay(); 
			if (display != null) {
				display.asyncExec(runnable);
			}
		}
	}
	/**
	 * Updates the current input style range.
	 */
	protected void updateInputStyleRange(int docLength) {
		if (fClosed) {
			return;
		}
		
		if (docLength != fNewStreamWriteEnd) {
			StyleRange input= 
				new StyleRange(fNewStreamWriteEnd, docLength - fNewStreamWriteEnd, 
								TraceConsolePreferencePage.getPreferenceColor(TraceConstants.CONSOLE_SYS_IN_RGB),
								null); 
			if (!fStyleRanges.isEmpty()) {
				if (((StyleRange)fStyleRanges.get(fStyleRanges.size() - 1)).similarTo(input)) { 
					//remove the top "input" range...continuing input
					fStyleRanges.remove(fStyleRanges.size() - 1);
				}
			} 
			
			addNewStyleRange(input);
		}
		
	}
	protected void updateOutputStyleRanges(int sourceStream) {
		if (fClosed) {
			return;
		}
		int docLength= getLength();
		if (docLength == 0) {
			return;
		}
		
		if ((fNewStreamWriteEnd == 0) && (0 == fLastStreamWriteEnd)) {
			return;
		}
		
		if (fNewStreamWriteEnd == fLastStreamWriteEnd) {
			return;
		}

		Color newRangeColor= 
			(sourceStream == TraceConsoleDocument.OUT) ? TraceConsolePreferencePage.getPreferenceColor(TraceConstants.CONSOLE_SYS_OUT_RGB) : TraceConsolePreferencePage.getPreferenceColor(TraceConstants.CONSOLE_SYS_ERR_RGB);

		StyleRange newRange= new StyleRange(fLastStreamWriteEnd, fNewStreamWriteEnd - fLastStreamWriteEnd, newRangeColor, null);
		if (!fStyleRanges.isEmpty()) {
			if ((docLength != fNewStreamWriteEnd) && 
				((StyleRange)fStyleRanges.get(fStyleRanges.size() - 1)).foreground ==
				TraceConsolePreferencePage.getPreferenceColor(TraceConstants.CONSOLE_SYS_IN_RGB)) {
				//remove the top "input" range..it will get recalculated in updateInputStyleRanges
				fStyleRanges.remove(fStyleRanges.size() - 1);
			}
		}
		
		addNewStyleRange(newRange);
		coaleseRanges();
		updateInputStyleRange(docLength);
		//notify the viewer that the style ranges have changed.
		fireDocumentChanged(new DocumentEvent(this, 0, 0, null));
		
	}
}
