/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.console;

import java.util.*;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.texteditor.*;

public class TraceConsoleView extends ViewPart
								implements IProfileEventListener
											, ISelectionListener
{
	protected final static String PREFIX= "console_view.";
	protected static final String _title = UIPlugin.getResourceString("STR_TRACE_CONSOLE");

	protected TraceConsoleViewer fConsoleViewer= null;
	protected ClearOutputAction fClearOutputAction= null;

	protected Map fGlobalActions= new HashMap(10);
	protected List fSelectionActions = new ArrayList(7);
	protected Separator fAdditions;
	protected Separator fSeparator;

	/**
	 * @see ViewPart#createChild(IWorkbenchPartContainer)
	 */
	public void createPartControl(Composite parent) {
		
		UIPlugin.getDefault().addProfileEventListener(this);
		UIPlugin.getDefault().addSelectionListener(this);
		
		fConsoleViewer= new TraceConsoleViewer(parent);
		initializeActions();
		initializeToolBar();

		// create context menu
		MenuManager menuMgr= new MenuManager("#PopUp", PDPerspective.ID_CONSOLE_VIEW);
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});
		Menu menu= menuMgr.createContextMenu(fConsoleViewer.getTextWidget());
		fConsoleViewer.getTextWidget().setMenu(menu);
		// register the context menu such that other plugins may contribute to it
		getSite().registerContextMenu(menuMgr.getId(), menuMgr, fConsoleViewer);
		
		fConsoleViewer.getSelectionProvider().addSelectionChangedListener(getSelectionChangedListener());
		setViewerInput(UIPlugin.getDefault().getConsoleManager().getCurrentProcess());
		setTitleToolTip(UIPlugin.getResourceString("TRACECONSOLEVIEW_TOOLTIP"));
	}
	/**
	 * Adds the text manipulation actions to the <code>ConsoleViewer</code>
	 */
	protected void fillContextMenu(IMenuManager menu) {
		Point selectionRange= fConsoleViewer.getTextWidget().getSelection();
		TraceConsoleDocument doc= (TraceConsoleDocument) fConsoleViewer.getDocument();
		if (doc == null) {
			return;
		}
		if (doc.isReadOnly() || selectionRange.x < doc.getStartOfEditableContent()) {
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.COPY));
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.SELECT_ALL));
		} else {
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.CUT));
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.COPY));
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.PASTE));
			menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.SELECT_ALL));
		}


		menu.add(fSeparator);
		menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.FIND));
		menu.add((IAction)fGlobalActions.get(ITextEditorActionConstants.GOTO_LINE));
		
		menu.add(fClearOutputAction);
		menu.add(fAdditions);
	}
	/**
	 * @see WorkbenchPart#getAdapter(Class)
	 */
	public Object getAdapter(Class required) {
		if (IFindReplaceTarget.class.equals(required)) {
			return fConsoleViewer.getFindReplaceTarget();
		}
		return super.getAdapter(required);
	}
	protected final ISelectionChangedListener getSelectionChangedListener() {
		return new ISelectionChangedListener() {
				public void selectionChanged(SelectionChangedEvent event) {
					updateSelectionDependentActions();
				}
			};
	}
	/**
	 * Initialize the actions of this view
	 */
	private void initializeActions()
	{
		fSeparator = new Separator("FIND");
		fAdditions = new Separator(IWorkbenchActionConstants.MB_ADDITIONS);
		fClearOutputAction= new ClearOutputAction(fConsoleViewer);
		fClearOutputAction.setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_CLEAR_CONSOLE));
		
		ResourceBundle bundle= UIPlugin.getDefault().getResourceBundle();
		// In order for the clipboard actions to accessible via their shortcuts
		// (e.g., Ctrl-C, Ctrl-V), we *must* set a global action handler for
		// each action		
		IActionBars actionBars= getViewSite().getActionBars();
		setGlobalAction(actionBars, ITextEditorActionConstants.CUT, new ConsoleViewerAction(bundle, "cut_action.", fConsoleViewer, TraceConsoleViewer.CUT));
		setGlobalAction(actionBars, ITextEditorActionConstants.COPY, new ConsoleViewerAction(bundle, "copy_action.", fConsoleViewer, TraceConsoleViewer.COPY));
		setGlobalAction(actionBars, ITextEditorActionConstants.PASTE, new ConsoleViewerAction(bundle, "paste_action.", fConsoleViewer, TraceConsoleViewer.PASTE));
		setGlobalAction(actionBars, ITextEditorActionConstants.SELECT_ALL, new ConsoleViewerAction(bundle, "select_all_action.", fConsoleViewer, TraceConsoleViewer.SELECT_ALL));
		setGlobalAction(actionBars, ITextEditorActionConstants.FIND, new FindReplaceAction(bundle, "find_replace_action.", this));				
		setGlobalAction(actionBars, ITextEditorActionConstants.GOTO_LINE, new ConsoleGotoLineAction(bundle, "goto_line_action.", fConsoleViewer));				
	
		fSelectionActions.add(ITextEditorActionConstants.CUT);
		fSelectionActions.add(ITextEditorActionConstants.COPY);
		fSelectionActions.add(ITextEditorActionConstants.PASTE);
		updateAction(ITextEditorActionConstants.FIND);

			
	}
	/**
	 * Configures the toolBar.
	 */
	private void initializeToolBar() {
		IToolBarManager tbm= getViewSite().getActionBars().getToolBarManager();
		tbm.add(fClearOutputAction);
		getViewSite().getActionBars().updateActionBars();
	}
	public void markAsSelectionDependentAction(String actionId) {
		if (!fSelectionActions.contains(actionId)) {
				fSelectionActions.add(actionId);
		}
	}
	/**
	 * @see IWorkbenchPart
	 */
	public void setFocus() {
		fConsoleViewer.getControl().setFocus();
	}
	protected void setGlobalAction(IActionBars actionBars, String actionID, IAction action) {
		fGlobalActions.put(actionID, action); 
		actionBars.setGlobalActionHandler(actionID, action);
	}
	/** 
	 * Sets the input of the viewer of this view in the
	 * UI thread.
	 */
	public void setViewerInput(final TRCProcessProxy element) {
		setViewerInput(element, true);		
	}
	/** 
	 * Sets the input of the viewer of this view in the
	 * UI thread.  The current input process is determined
	 * if so specified.
	 */
	public void setViewerInput(final TRCProcessProxy element, final boolean determineCurrentProcess) {
		if (fConsoleViewer == null) {
			return;
		}
		Display display= fConsoleViewer.getControl().getDisplay();
		if (display != null) {
			display.asyncExec(new Runnable() {
				public void run() {
					IDocument doc= UIPlugin.getDefault().getConsoleManager().getConsoleDocument(element, determineCurrentProcess);
					fConsoleViewer.setDocument(doc);
				}
			});
		}
	}
	protected void updateAction(String actionId) {
		IAction action= (IAction)fGlobalActions.get(actionId);
		if (action instanceof IUpdate)
			((IUpdate) action).update();
	}
	protected void updateSelectionDependentActions() {
		Iterator iterator= fSelectionActions.iterator();
		while (iterator.hasNext())
			updateAction((String)iterator.next());		
	}
	
	public void handleProfileEvent(ProfileEvent event) {
		
			setViewTitle(event.getSource());
	}

	private void setViewTitle(Object selection)
	{
		if(selection != null)
		{
            if(selection instanceof EObject)
            	selection = getObjectToView((EObject)selection);
            
			if(selection instanceof TRCAgentProxy)
			{
				TRCAgentProxy a = (TRCAgentProxy)selection;
				
				  String status = HyadesUtil.getAgentLabel(a);
				  setTitle(getViewTitle() + " [" + status+" "+HyadesUtil.getProcessName(a.getProcessProxy())+"]");
				  return;
				
			}
		}
		setTitle(getViewTitle());		
	}
	
/*
 * Returns the object that should be viewd 
 * For example, if the process node is selected, the view should display
 * the profiling data if trace collects profiling data
 * 
 */
public EObject getObjectToView(EObject selObject)
{
	if(selObject == null)
	  return selObject;

	if(selObject instanceof TRCProcessProxy)
	{
		EList agents = ((TRCProcessProxy)selObject).getAgentProxies();
		for(int idx=0; idx<agents.size(); idx++)
		{
			TRCAgentProxy agent = (TRCAgentProxy) agents.get(idx);
			if(agent.eIsProxy())
				continue;
			if(agent.getType().equals(TraceConstants.PROFILE_AGENT_TYPE))
				return agent;
		}
	}

	return selObject;
}
	
/*
 * Initial title view 
 */
 public String getViewTitle()
 {
 	return _title;
 }


	/*
	 * Selection has changed in the Monitors view (PDProjectExplorer)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		if (part == null
			|| part.getSite().getWorkbenchWindow() != getSite().getWorkbenchWindow())
			return; //get notifications only from the current workbench window

		if (selection == null
			|| selection.isEmpty()
			|| !(selection instanceof IStructuredSelection)) {
			//show empty page
			setViewTitle(null);
			return;
		}

		Object obj = ((IStructuredSelection) selection).getFirstElement();
		setViewTitle(obj);
	}
	
public void dispose() {
	super.dispose();
	
	UIPlugin.getDefault().removeSelectionListener(this);	
	UIPlugin.getDefault().removeProfileEventListener(this);
}
	
}
