/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.views;

import java.util.*;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.*;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.*;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.ViewForm;

/**
 * This class is used to provide CBEDefaultEvent entries to the Log Viewer.
 */
public class LogContentProvider
	implements IStructuredContentProvider, ITreeContentProvider {
    
    private int _visible = 0;
    private int _total = 0;    
    private ViewForm _viewer;
    private String _title = LogUIPlugin.getResourceString("STR_LOG_PANE_TITLE");
    private Sorter sorter;     
    
    public LogContentProvider(ViewForm viewer)
    {
    	super();
    	_viewer = viewer;
    }
    

	public void dispose() {
	}

	public Object[] getChildren(Object element) {
		return getElements(element);
	}
	
	
	/**
	 * Returns an array of CBEDefaultEvent elements under a monitor or node or agent filtered 
	 * by severity
	 *
	 * @param element 
	 * @see #getFilteredRecords(List)
	 * @return an array of CBEDefaultEvent objects
	 */ 
	public Object[] getElements(Object element) {

		ArrayList wlist = new ArrayList();
		List pdList = null;			
		_visible = 0;
		_total = 0;
								
		if (element instanceof TRCMonitor) {
			pdList = new ArrayList();			
			getMonitorRecords((TRCMonitor)element, pdList, wlist);
		}
		if (element instanceof TRCNode)
		{
			pdList = new ArrayList();
			getNodeRecords((TRCNode) element, pdList, wlist, false);
		}
		if (element instanceof TRCAgentProxy)
		{
			pdList = new ArrayList();			
			getAgentRecords((TRCAgentProxy)element, pdList, wlist, false);			
		}
		if (element instanceof TRCProcessProxy) {
			pdList = new ArrayList();			
			getProcessRecords((TRCProcessProxy)element, pdList, wlist, false);
		} 
		else if (element instanceof EObject && ((EObject)element).eClass().getEPackage()==CBEPackage.eINSTANCE){
			getFeatureNodes((EObject)element, wlist);
		}
		else if (element instanceof FeatureNode){
			getFeatureNodeElements((FeatureNode)element, wlist);
		}
	
		return wlist.toArray();
	}
	
	public boolean hasChildren(Object element) {
		
		boolean result = false;
		
		//check if the object is in the  CommonBaseEvent model, i.e in the CBE package
		if (element instanceof EObject && ((EObject)element).eClass().getEPackage()==CBEPackage.eINSTANCE){

			final EClass eClass = ((EObject)element).eClass();
			for (Iterator iter = eClass.getEAllStructuralFeatures().iterator(); iter.hasNext();) {
				EStructuralFeature feature = (EStructuralFeature) iter.next();
				// if not string array attribute add feature node (if it's not an empty list)
				if((feature instanceof EReference && feature.getFeatureID()!=CBEPackage.CBE_COMMON_BASE_EVENT__SYMPTOMS)
				    || (feature instanceof EAttribute && feature.isMany() &&  feature.getEType().getInstanceClass() != String.class))
				{
					Object refs = ((EObject)element).eGet(feature, true);
					if(refs != null){ 
						if(refs instanceof EList){
							if(((EList)refs).size()>0){
								
								result = true;
								break;
							}
						}
						else{
							result = true;
							break;
						}
						
					}
				}
			}
		
		}
		else if (element instanceof FeatureNode){

			final EStructuralFeature feature = ((FeatureNode)element).getFeature();
			final EObject cElem = ((FeatureNode)element).getElement();
			if(feature instanceof EReference)
			{
				if(feature.isMany()){				
					EList refs = (EList)cElem.eGet(feature, true);
					result = refs.size() > 0;
				}
				else{
					Object ref = cElem.eGet(feature, true);
					result = ref!=null;					
				}
					
			}
		}
		else if(element instanceof TerminalNode){
			result = false;
		}		
		
		return result;		
	}

	public Object getParent(Object element) {
		return null;
	}
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
	}
	public boolean isDeleted(Object element) {
		return false;
	}
	
	/**
	 * Filters and sorts the list of CBEDefaultEvent's according to the filter and sort 
	 * options saved in the preferences store
	 * CBEDefaultEvent objects are considered to be:
	 * - of severity one if their severity property 
	 * satisfies the condition	50<= severity <=70
	 * - of severity two if 30<= severity <50
	 * - of severity three if 0<= severity <30
	 */
	private ArrayList getFilteredRecords(List pdList) {

		List sortAttributes = parsePreferenceString(LogUIConstants.PD_SORT_LOG_OPTIONS);
		
		if(sortAttributes.size()>0){
			Collections.sort(pdList, getLogSorter(sortAttributes));
		}			

		ArrayList data = new ArrayList();

		String filters =
			LogUIPlugin.getDefault().getPreferenceStore().getString(
				LogUIConstants.PD_SEV_OPTIONS);

		if (filters == null || filters.equals(""))
		{
			data.addAll(pdList);
			_total += pdList.size();
			_visible += pdList.size();
			return data;
		}
		
		
		boolean filterSevOne = false;
		boolean filterSevTwo = false;
		boolean filterSevThree = false;
		int idx = filters.indexOf(" ");
		if (idx != -1) {
			filterSevOne = (filters.substring(0, 1).equals("1"));
			filters = filters.substring(idx + 1);
		}
		idx = filters.indexOf(" ");
		if (idx != -1) {
			filterSevTwo = (filters.substring(0, 1).equals("1"));
			filters = filters.substring(idx + 1);
		}
		idx = filters.indexOf(" ");
		if (idx != -1) {
			filterSevThree = (filters.substring(0, 1).equals("1"));
			filters = filters.substring(idx + 1);
		}

		for (int i = 0; i < pdList.size(); i++) {
			Object  entry = pdList.get(i);

			try{
				int sev = ((CBECommonBaseEvent)entry).getSeverity();
						
				if ((sev >=50 && sev<=70 && filterSevOne)
					|| (sev>=30 && sev< 50 && filterSevTwo)
					|| (sev>=0 && sev< 30 && filterSevThree))
				{	
					data.add(entry);
					_visible++;
				}				
				
			}
			catch(ClassCastException e){
				// we might encounter nodes that are not CommonBaseEvent's and don't have severity 

				data.add(entry);
				_visible++;

			}
			
			_total++;

		}
		
		return data;
	}
	
	/**
	 * Sets the title of the log viewer
	 *
	 */
	private void setTitle()
	{
		String title = (new StringBuffer(_title)
							 .append(" ").append(LogUIPlugin.getResourceString("STR_SHOW_INFO"))).toString();
							 
		title = HyadesUtil.change(title, "%2", String.valueOf(_total));
		title = HyadesUtil.change(title, "%1", String.valueOf(_visible));
		
		((CLabel)_viewer.getTopLeft()).setText(title);
		
	}
	
	
	private List parsePreferenceString(String option){

			ArrayList sortAttributes = null;
			String firstSort = "";
			IPreferenceStore store = LogUIPlugin.getDefault().getPreferenceStore();

			String sortData = LogUIPlugin.getDefault().getPreferenceStore().getString(option);
		
			if(sortData != null)
			{
				int i = sortData.indexOf(" ");
				String column = "";
				int j = 0;
				sortAttributes = new ArrayList();			
				while(i != -1)
				{
					column = sortData.substring(0, i);			
					j = column.indexOf(":");
					if(j>-1 && column.substring(j+1).equals("1")){
						sortAttributes.add(firstCharToUpperCase(column.substring(0, j)));
					} 
					sortData = sortData.substring(i+1);
					i = sortData.indexOf(" ");
				}
				int k = sortData.indexOf(":");
				if(k>-1 && sortData.substring(k+1).equals("1")){
					sortAttributes.add(firstCharToUpperCase(sortData.substring(0, k)));
				} 
						
			}
			
			return sortAttributes;
			
	}

	private String firstCharToUpperCase(String name){
		return name.substring(0,1).toUpperCase() + name.substring(1);
	}
	
	private Sorter getLogSorter(List sortAttributes){
		
		if(sorter==null){		
			sorter = new Sorter(sortAttributes);
		}
		
		sorter.setAttributes(sortAttributes);
		
		return sorter;
	}
	

	private void getMonitorRecords(TRCMonitor element, List pdList, List wlist){
		try {
				
				
			Object[] nodes = ((TRCMonitor) element).getNodes().toArray();
			for(int idx=0; idx<nodes.length; idx++)
			{
				TRCNode node = (TRCNode)nodes[idx];
				getNodeRecords(node, pdList, wlist, true);
			}
			
			wlist.addAll(getFilteredRecords(pdList));

			setTitle();


		} catch (Exception e) {
				
			e.printStackTrace();
			wlist.clear();
				

		}
		finally{

			pdList.clear();
			
		}

	}
	private void getNodeRecords(TRCNode element, List pdList, List wlist, boolean isChild){

		Object[] processes = ((TRCNode)element).getProcessProxies().toArray();
		for(int i=0; i<processes.length; i++)
		{
			getProcessRecords((TRCProcessProxy)processes[i], pdList, wlist, true);
		}
			
		if(!isChild){	
			wlist.addAll(getFilteredRecords(pdList));
			pdList.clear();
			setTitle();			
		}
			
	}

	private void getAgentRecords(TRCAgentProxy element, List pdList, List wlist, boolean isChild){

		if(element!=null){
			pdList.addAll(element.getAgent().getDefaultEvents());
		}

		if(!isChild){		
			wlist.addAll(getFilteredRecords(pdList));
			pdList.clear();			
			setTitle();			
		}		

	}
	
	private void getProcessRecords(TRCProcessProxy element, List pdList, List wlist, boolean isChild){
		try {

			EList agents = ((TRCProcessProxy) element).getAgentProxies();
			for (int idx = 0; idx < agents.size(); idx++) {
				getAgentRecords((TRCAgentProxy) agents.get(idx), pdList, wlist, true);
			}

			if(!isChild){	
				wlist.addAll(getFilteredRecords(pdList));
				pdList.clear();
				setTitle();				
			}


		} catch (Exception e) {
				
			e.printStackTrace();
			wlist.clear();
			pdList.clear();
			
		}

	}
	/**
	 * Creates a list of features of the EObject passed as parameter, wrapped into a FeatureNode object
	 * FeatureNode will be used to get the actual elements of the feature, this is the reason why
	 * the current EObject is also saved into the FeatureNode
	 * 
	 * @param element
	 * @param wlist
	 */	

	private void getFeatureNodes(EObject element, List wlist){
	
		try {
					
			EClass eClass = (element).eClass();
					
			//loop through all the features of the current object and return those who are not the symptom reference or byte arrays
			for (Iterator iter = eClass.getEAllStructuralFeatures().iterator(); iter.hasNext();) {
				EStructuralFeature feature = (EStructuralFeature) iter.next();
				if(feature.getContainerClass().equals(AbstractDefaultEvent.class)){
					continue;
				}
				// if not string array attribute add feature node (if it's not an empty list)
				if((feature instanceof EReference && feature.getFeatureID()!=CBEPackage.CBE_COMMON_BASE_EVENT__SYMPTOMS) 
				   || (feature instanceof EAttribute && feature.isMany() &&  feature.getEType().getInstanceClass() != String.class))
				{
					Object refs = ((EObject)element).eGet(feature, true);
					if(refs != null){ 
						if(refs instanceof EList){
							if(((EList)refs).size()>0){
								wlist.add(new FeatureNode((EObject)element, feature));
							}
						}
						else{
							wlist.add(new FeatureNode((EObject)element, feature));						
						}
					}
	
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			wlist.clear();
		}
	}
	
	/**
	 * Returns in the wlist parameter the list of elements of the feature passed by the element parameter
	 * The model object cached in the FeatureNode paramater is queried for the elements of the feature 
	 * also cached as part of the FeatureNode paramater
	 * 
	 * @param element
	 * @param wlist
	 */
	private void getFeatureNodeElements(FeatureNode element, List wlist){		
		try {

			final EStructuralFeature feature = element.getFeature();
			final EObject cElem = element.getElement();
			if(feature instanceof EReference){
			
				//return the elements of a containment by value or create a terminal node (nodes that don't expand anymore, avoiding an infinite tree)

				if(feature.isMany()){

					EList refs = (EList)cElem.eGet(feature, true);
					if(((EReference)feature).isContainment()){
							
						wlist.addAll(refs);
							
					}else{
								
						int index = 0;

						EObject elem = null;
						String label = "";
						for (Iterator iter = refs.iterator(); iter.hasNext();) {
							elem = (EObject) iter.next();
							//wlist.add(new TerminalNode(elem, elem.eClass().getName()+"["+index+"] - "));
							EStructuralFeature f = elem.eClass().getEStructuralFeature(CBEPackage.CBE_COMMON_BASE_EVENT__MSG);
							
							if(f==CBEPackage.eINSTANCE.getCBECommonBaseEvent_Msg()){
								label = (String)elem.eGet(f);
							}
							else if((f=elem.eClass().getEStructuralFeature(CBEPackage.CBE_DEFAULT_EVENT__EXTENSION_NAME))==CBEPackage.eINSTANCE.getCBEDefaultEvent_ExtensionName()){
							   label = (String)elem.eGet(f);
							}else{
								label = elem.eClass().getName();
							}
							wlist.add(new TerminalNode(elem, label));
							index++;
						}
					}
				}else{
					EObject ref = (EObject)cElem.eGet(feature);
					if(((EReference)feature).isContainment()){
							
						wlist.add(ref);
							
					}else{

						EStructuralFeature f = ref.eClass().getEStructuralFeature(CBEPackage.CBE_ASSOCIATION_ENGINE__NAME);
						
						if(f==CBEPackage.eINSTANCE.getCBEAssociationEngine_Name()){
							wlist.add(new TerminalNode(ref, (String)ref.eGet(f)));
						}
						else{
							wlist.add(new TerminalNode(ref, ref.eClass().getName()));							
						}
					}
				}	
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	
	}
	
	
}
