/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sd.trace.internal.loader;

import java.util.*;

import org.eclipse.emf.common.util.*;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.sd.ui.internal.model.*;

/**
 * TRCModelLoaderUtility provides the common things all loaders that read work with the
 * perftrace model do. It is intended to be extended by a loader.
 */
public class TRCModelLoaderUtility
{
	protected Graph graph = null;
	protected Hashtable containers = new Hashtable();
	protected Hashtable nodes = new Hashtable();
	protected Hashtable agents = new Hashtable();
	protected TRCAgentProxy agentList[] = new TRCAgentProxy[3];
	protected int nextAgent = 0;
	protected double oldestEndTime = 0;
	protected String unknownString = "_unknown";

	/**
	 * TRCModelLoaderUtility default constructor
	 */
	public TRCModelLoaderUtility()
	{
	}

	/**
	 * Set the Graph that is being loaded.
	 */

	protected void setGraph(Graph inputGraph)
	{
		graph = inputGraph;
	}
	/**
	 * Returns the graph that is being loaded.
	 */

	protected Graph getGraph()
	{
		return graph;
	}

	/**
	 * Based on the Graph type being loaded determine the containing object that shoudl be used
	 * as the top node in the graph. For example CLASSover___ will find the correct Class object
	 * and create a container for it.
	 */
	protected NodeContainer resolveContainer(TRCMethodInvocation invocation, int graphType)
	{
		NodeContainer container = null;
		switch (graphType)
		{
			case Graph.NODEoverPROCESS :
			case Graph.NODEoverAGENT :
			case Graph.NODEoverTHREAD :
			case Graph.NODEoverOBJECT :
			case Graph.NODEoverCLASS :
			case Graph.NODEoverMETHOD :
				{
					container = resolveToNode(invocation);
					break;
				}
			case Graph.AGENToverPROCESS :
			case Graph.AGENToverTHREAD :
			case Graph.AGENToverOBJECT :
			case Graph.AGENToverCLASS :
			case Graph.AGENToverMETHOD :
				{
					container = resolveToAgent(invocation);
					break;
				}
			case Graph.PROCESSoverTHREAD :
			case Graph.PROCESSoverOBJECT :
			case Graph.PROCESSoverCLASS :
			case Graph.PROCESSoverMETHOD :
				{
					container = resolveToProcess(invocation);
					break;
				}
			case Graph.THREADoverOBJECT :
			case Graph.THREADoverCLASS :
			case Graph.THREADoverMETHOD :
				{
					container = resolveToThread(invocation);
					break;
				}
			case Graph.OBJECToverMETHOD :
				{
					container = resolveToObject(invocation);
					break;
				}

			case Graph.CLASSoverMETHOD :
			default :
				{
					container = resolveToClass(invocation);
				}
		}
		return container;
	}

	/**
	 * Find the Monitor over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToMonitor(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCMonitor monitor = invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode().getMonitor();

		container = (NodeContainer) containers.get(monitor);
		if (container == null)
		{
			container = addTopNode(monitor);
			containers.put(monitor, container);
		}
		return container;
	}

	/**
	 * Find the Node over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToNode(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCNode node = invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode();

		container = (NodeContainer) containers.get(node);
		if (container == null)
		{
			container = addTopNode(node, invocation);
			containers.put(node, container);
		}
		return container;
	}
	/**
	 * Find the Agent over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToAgent(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCAgent agent = invocation.getThread().getProcess().getAgent();

		container = (NodeContainer) containers.get(agent);
		if (container == null)
		{
			container = addTopNode(agent, invocation);
			containers.put(agent, container);
		}
		return container;
	}

	/**
	 * Find the Process over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToProcess(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCProcess process = invocation.getThread().getProcess();

		container = (NodeContainer) containers.get(process);
		if (container == null)
		{
			container = addTopNode(process, invocation);
			containers.put(process, container);
		}
		return container;
	}

	/**
	 * Find the Thread over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToThread(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCThread thread = invocation.getThread();

		container = (NodeContainer) containers.get(thread);
		if (container == null)
		{
			container = addTopNode(thread, invocation);
			containers.put(thread, container);
		}
		return container;
	}

	/**
	 * Find the Class over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToClass(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCClass c = getClass(invocation);
			
		container = (NodeContainer) containers.get(c);
		if (container == null)
		{
			container = addTopNode(c, invocation);
			containers.put(c, container);
		}
		return container;
	}
	/**
	 * Find the Object over this invocation and create or re-use a container for it.
	 */
	protected NodeContainer resolveToObject(TRCMethodInvocation invocation)
	{
		NodeContainer container = null;
		TRCObject o = invocation.getOwningObject();
		container = (NodeContainer) containers.get(o);
		if (container == null)
		{
			container = addTopNode(o, invocation);
			containers.put(o, container);
		}
		return container;
	}
	/**
	 * Find the Node over this invocation and create or re-use a container for it.
	 */
	protected void resolveMethodNode(TRCMethodInvocation inv, NodeContainer container)
	{
		if (!(inv instanceof TRCFullMethodInvocation) || invocationIsOfInterest(inv) != true)
			return;

		GraphNode graphNode = null;
		
		TRCFullMethodInvocation invocation = (TRCFullMethodInvocation)inv;
		graphNode = (GraphNode) nodes.get(invocation);
		if (graphNode == null)
		{
			graphNode =
				addNode(
					resolveFullMethodName(invocation),
					invocation.getMethod().getName(),
					String.valueOf(invocation.getTicket()),
					GraphNode.METHOD,
					container);
			graphNode.setStartTime(invocation.getEntryTime()
			                       + (getAgentDelta(invocation.getThread().getProcess().getAgent().getAgentProxy()))
			                       );
			if (invocation.getExitTime() != 0)
			{
				graphNode.setEndTime(invocation.getExitTime()
				                     + (getAgentDelta(invocation.getThread().getProcess().getAgent().getAgentProxy()))
			                       );
				setOldestEndTime(graphNode.getEndTime());
			}
			else
			{
				setOldestEndTime(graphNode.getStartTime());
				graphNode.setEndTime(getOldestEndTime());
			}
			nodes.put(invocation, graphNode);
			graphNode.setUserArea(invocation);
		}
	}
	/**
	 * Determine based on the graph type if this invocation crosses a boundary.
	 */
	private boolean invocationIsOfInterest(TRCMethodInvocation invocation)
	{
		if (graph.getFiltered() == false)
			return true;
        
		boolean interest = true;

		switch (getGraph().getType())
		{
			case Graph.NODEoverPROCESS :
			case Graph.NODEoverAGENT :
			case Graph.NODEoverTHREAD :
			case Graph.NODEoverOBJECT :
			case Graph.NODEoverCLASS :
			case Graph.NODEoverMETHOD :
				{
					interest = isOnNodeBoundary(invocation);
					break;
				}
			case Graph.AGENToverPROCESS :
			case Graph.AGENToverTHREAD :
			case Graph.AGENToverOBJECT :
			case Graph.AGENToverCLASS :
			case Graph.AGENToverMETHOD :
				{
					interest = isOnAgentBoundary(invocation);
					break;
				}
			case Graph.PROCESSoverTHREAD :
			case Graph.PROCESSoverOBJECT :
			case Graph.PROCESSoverCLASS :
			case Graph.PROCESSoverMETHOD :
				{
					interest = isOnProcessBoundary(invocation);
					break;
				}
			case Graph.THREADoverOBJECT :
			case Graph.THREADoverCLASS :
			case Graph.THREADoverMETHOD :
				{
					interest = isOnThreadBoundary(invocation);
					break;
				}
			case Graph.OBJECToverMETHOD :
				{
					if(resolveType(invocation.getOwningObject())==GraphNode.OBJECT)
					   interest = isOnObjectBoundary(invocation);
					else if(resolveType(invocation.getOwningObject())==GraphNode.CLASS)
					   interest = isOnClassBoundary(invocation);
					break;
				}

			case Graph.CLASSoverMETHOD :
			    {
					interest = isOnClassNameBoundary(invocation);
					break;
				}

			default :
				{
					interest = true;
				}
		}
		
		return interest;
	}
	
	private boolean isOnClassBoundary(TRCMethodInvocation invocation)
	{
		TRCClass myClass = invocation.getMethod().getDefiningClass();
		// if there is a parent invocation on a different Class return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (!invocation.getInvokedBy().getMethod().getDefiningClass().getName().equals(myClass.getName()))
					return true;
			}
		}
		else
		// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different class return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (!segment.getMethod().getDefiningClass().getName().equals(myClass.getName()))
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;
	}

//this one is for inner classes as well -> used by Class Interactions
private boolean isOnClassNameBoundary(TRCMethodInvocation invocation)
	{
		TRCClass myClass = invocation.getMethod().getDefiningClass();
		// if there is a parent invocation on a different Class return true
		if (invocation.getInvokedBy() != null)
		{
			if (!invocation.getInvokedBy().getMethod().getDefiningClass().getName().startsWith(myClass.getName()) &&
				!myClass.getName().startsWith(invocation.getInvokedBy().getMethod().getDefiningClass().getName()))
					return true;
		}
		else
		// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different class return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (!segment.getMethod().getDefiningClass().getName().startsWith(myClass.getName())&&
					!myClass.getName().startsWith(segment.getMethod().getDefiningClass().getName()))
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;
	}

	
	private boolean isOnObjectBoundary(TRCMethodInvocation invocation)
	{
		TRCObject myObject = invocation.getOwningObject();
		// if there is a parent invocation on a different object return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (invocation.getInvokedBy().getOwningObject() != myObject)
					return true;
			}
		}
		else
		// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different object return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (segment.getOwningObject() != myObject)
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;
	}
	private boolean isOnThreadBoundary(TRCMethodInvocation invocation)
	{
		TRCThread myThread = invocation.getThread();
		// if there is a parent invocation on a different thread return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (invocation.getInvokedBy().getThread() != myThread)
					return true;
			}
		}
		else
			// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different thread return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (segment.getThread() != myThread)
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;
	}
	private boolean isOnProcessBoundary(TRCMethodInvocation invocation)
	{
		TRCProcess myProcess = invocation.getThread().getProcess();
		// if there is a parent invocation on a different thread return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (invocation.getInvokedBy().getThread().getProcess() != myProcess)
					return true;
			}
		}
		else
			// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different thread return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				///if (segment.getInvokes() != null)
				{
					if (segment.getThread().getProcess() != myProcess)
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;

	}
	private boolean isOnAgentBoundary(TRCMethodInvocation invocation)
	{
		TRCAgent myAgent = invocation.getThread().getProcess().getAgent();
		// if there is a parent invocation on a different thread return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (invocation.getInvokedBy().getThread().getProcess().getAgent()
					!= myAgent)
					return true;
			}
		}
		else
			// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different thread return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (segment.getThread().getProcess().getAgent() != myAgent)
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;

	}

	private boolean isOnNodeBoundary(TRCMethodInvocation invocation)
	{
		TRCNode myNode = invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode();
		// if there is a parent invocation on a different thread return true
		if (invocation.getInvokedBy() != null)
		{
			if (invocation.getInvokedBy() != null)
			{
				if (invocation
					.getInvokedBy()					
					.getThread()
					.getProcess()
					.getAgent()
					.getAgentProxy()
					.getProcessProxy()
					.getNode()
					!= myNode)
					return true;
			}
		}
		else
			// top of a call stack is a boundary call because it is the first
			return true;

		// if there is a child invocation on a different thread return true
		for (Iterator segments = invocation.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation segment = (TRCMethodInvocation) segments.next();
			if (segment != null)
			{
				//if (segment.getInvokes() != null)
				{
					if (segment.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode()!=myNode)
						return true;
				}
				//else
					// a segment without an invokes represents an unresolved outbound call
					return true;
			}
		}

		// default to false
		return false;

	}
	
	/**
	 * appends node's name to the container name
	 */
	
	private StringBuffer appendNodeName(StringBuffer buffer, TRCMethodInvocation invocation){
		
	   	if(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode().getMonitor().getNodes().size()>1){
		   buffer.append(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode().getName());
		   buffer.append(":");
	   	}
		return buffer;
	}
	
	/**
	 * appends process' name to the container name
	 */
	
	private StringBuffer appendProcessName(StringBuffer buffer, TRCMethodInvocation invocation){
		
	   if(invocation.getThread().getProcess().getName()!=null)
		  buffer.append(invocation.getThread().getProcess().getName());
	   else
	      buffer.append(unknownString);
	   return buffer;
	}
	
	/**
	 * appends process' id to the container name
	 */
	
	private StringBuffer appendProcessId(StringBuffer buffer, TRCMethodInvocation invocation){
		
		if(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode().getProcessProxies().size()>1){
		   if(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getPid()!=0)
		      buffer.append(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getPid());
     	   else
    		  buffer.append(unknownString);
    	   buffer.append(":");
		}
		return buffer;
	}
	
	/**
	 * appends thread's name to the container name
	 */
	
	private StringBuffer appendThreadName(StringBuffer buffer, TRCMethodInvocation invocation){
		
	   if(invocation.getThread().getName()!=null)
		  buffer.append(invocation.getThread().getName());
	   else
		  buffer.append(unknownString);
	   return buffer;	
	}
	
	/**
	 * appends thread's id to the container name
	 */
	
	private StringBuffer appendThreadId(StringBuffer buffer, TRCMethodInvocation invocation){
		
		if(invocation.getThread().getProcess().getThreads().size()>1){
		   if(invocation.getThread().getName()!=null)
			  buffer.append(invocation.getThread().getName());
		   else
			  buffer.append(invocation.getThread().getId());
		   buffer.append(":");
		}
		return buffer;
	}
		
	/**
	 * Create a String that contains a container name scoped by all the qualifiers between it
	 * and the method.
	 */			   
	
	private String resolveFullContainerName(TRCMethodInvocation invocation){
		
	   	StringBuffer nameBuffer = new StringBuffer();
		switch (getGraph().getType())
		{
			case Graph.NODEoverPROCESS :
			case Graph.NODEoverAGENT :
			case Graph.NODEoverTHREAD :
			case Graph.NODEoverOBJECT :
			case Graph.NODEoverCLASS :
			case Graph.NODEoverMETHOD :
			   {
			   	    nameBuffer.append(invocation.getThread().getProcess().getAgent().getAgentProxy().getProcessProxy().getNode().getName());
					break;
				}
			case Graph.AGENToverPROCESS :
			case Graph.AGENToverTHREAD :
			case Graph.AGENToverOBJECT :
			case Graph.AGENToverCLASS :
			case Graph.AGENToverMETHOD :
			case Graph.PROCESSoverTHREAD :
			case Graph.PROCESSoverOBJECT :
			case Graph.PROCESSoverCLASS :
			case Graph.PROCESSoverMETHOD :
				{
					appendNodeName(nameBuffer, invocation);
					appendProcessName(nameBuffer, invocation);
					break;
				}
			case Graph.THREADoverOBJECT :
			case Graph.THREADoverCLASS :
			case Graph.THREADoverMETHOD :
				{
					appendNodeName(nameBuffer, invocation);
					appendProcessId(nameBuffer, invocation);
					appendThreadName(nameBuffer, invocation);
					break;
				}
			case Graph.OBJECToverMETHOD :
				{
					// if container is an object, qualify the name
					if (invocation.getOwningObject().getIsA() != null)
					{
						appendNodeName(nameBuffer, invocation);
					    appendProcessId(nameBuffer, invocation);
					    appendThreadId(nameBuffer, invocation);
						nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					}
					else
					{
						// if container is a class don't qualify the name
						nameBuffer = appendNodeName(nameBuffer, invocation);
					    appendProcessId(nameBuffer, invocation);
					    appendThreadId(nameBuffer, invocation);
					    nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					}

					break;
				}

			case Graph.CLASSoverMETHOD :
				{
					appendNodeName(nameBuffer, invocation);
					appendProcessId(nameBuffer, invocation);
					appendThreadId(nameBuffer, invocation);
					nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					break;
				}
			default :
				{
					nameBuffer.append(invocation.getMethod().getName());
				}
		}

		return nameBuffer.toString();
	}

	/**
	 * Create a String that contains a method name scoped by all the qualifiers between it
	 * and the container.
	 */
	private String resolveFullMethodName(TRCMethodInvocation invocation)
	{
		StringBuffer nameBuffer = new StringBuffer();
		switch (getGraph().getType())
		{
			case Graph.NODEoverPROCESS :
			case Graph.NODEoverAGENT :
			case Graph.NODEoverTHREAD :
			case Graph.NODEoverOBJECT :
			case Graph.NODEoverCLASS :
			case Graph.NODEoverMETHOD :
				{
					appendProcessId(nameBuffer, invocation);
					appendThreadId(nameBuffer, invocation);
					nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					nameBuffer.append(".");
					nameBuffer.append(invocation.getMethod().getName());

					break;
				}
			case Graph.AGENToverPROCESS :
			case Graph.AGENToverTHREAD :
			case Graph.AGENToverOBJECT :
			case Graph.AGENToverCLASS :
			case Graph.AGENToverMETHOD :
			case Graph.PROCESSoverTHREAD :
			case Graph.PROCESSoverOBJECT :
			case Graph.PROCESSoverCLASS :
			case Graph.PROCESSoverMETHOD :
				{
					appendThreadId(nameBuffer, invocation);
					nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					nameBuffer.append(".");
					nameBuffer.append(invocation.getMethod().getName());

					break;
				}
			case Graph.THREADoverOBJECT :
			case Graph.THREADoverCLASS :
			case Graph.THREADoverMETHOD :
				{
					nameBuffer.append(invocation.getMethod().getDefiningClass().getName());
					nameBuffer.append(".");
					nameBuffer.append(invocation.getMethod().getName());

					break;
				}
			case Graph.OBJECToverMETHOD :
				{
					// if container is an object, qualify the name
					if (invocation.getOwningObject().getIsA() != null)
					{
						nameBuffer.append(invocation.getMethod().getName());
					}
					else
					{
						// if container is a class don't qualify the name
						nameBuffer.append(invocation.getMethod().getName());
					}

					break;
				}

			case Graph.CLASSoverMETHOD :
				{
					nameBuffer.append(invocation.getMethod().getName());
					break;
				}
			default :
				{
					nameBuffer.append(invocation.getMethod().getName());
				}
		}

		return nameBuffer.toString();
	}

	/**
	 * Find the type name for an object/instance.
	 */
	protected String createTypeName(TRCObject o)
	{
		return o.getIsA().getName();
	}

	/**
	 * Given a string, reduce it to the last token that follows a period or slash.
	 * This is typically used to turn a qualified method or class name into a simple anme.
	 */
	protected String createShortClassName(String in)
	{
		// trim down to the last chars in a package qualified class name
		// if / is separator
		if (in.lastIndexOf("/") > -1)
		{
			return in.substring(in.lastIndexOf("/") + 1);
		}
			// if . is separator
		else if (in.lastIndexOf(".") > -1)
			{
				return in.substring(in.lastIndexOf(".") + 1);
			}
			else if (in.indexOf(":") > -1)
		{
			return in.substring(in.lastIndexOf(":")+1);
		}
		else
				// no separators recognized
				{
				return in;
			}

	}

	protected String createShortNodeName(String in)
	{
		// trim down to the last chars in a package qualified class name
		// if / is separator
		if (in.indexOf("/") > -1)
		{
			return in.substring(0, in.lastIndexOf("/"));
		}
		else if (in.indexOf(":") > -1)
		{
			return in.substring(in.lastIndexOf(":")+1);
		}
		else
		{
			// if . is separator
			if (in.indexOf(".") > -1)
			{
				return in.substring(0, in.lastIndexOf("."));
			}
			else
				// no separators recognized
				{
				return in;
			}
		}

	}

	/**
	 * Create a secodnary name for an object.
	 */
	protected String createSecondaryName(TRCObject o)
	{
		if (o.getId() == 0)
			return unknownString;
		else
			return String.valueOf(o.getId());
	}
	/**
	 * Add a top node to represent an Object.
	 */
	protected NodeContainer addTopNode(TRCObject o,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = null;
		if(resolveType(o)==GraphNode.OBJECT){
		   topNode = graph.getContentFactory().createNodeContainer();
		   topNode.setName(resolveFullContainerName(invocation));
		   topNode.setShortName(createShortClassName(topNode.getName()));
		   topNode.setSecondaryName(createSecondaryName(o));
		   topNode.setType(resolveType(o));
		   graph.addTopNode(topNode);
		   topNode.setGraph(graph);
		}else if(resolveType(o)==GraphNode.CLASS){
		   topNode=addTopNode(getClass(invocation),invocation);	
		}
		
		topNode.setUserArea(o);
		
		return topNode;
	}
	/**
	 * Add a top node to represent a Class.
	 */
	protected NodeContainer addTopNode(TRCClass c,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		topNode.setUserArea(c);
		
		topNode.setName(resolveFullContainerName(invocation));
		topNode.setShortName(createShortClassName(topNode.getName()));
		topNode.setSecondaryName(String.valueOf(c.getId()));
		topNode.setType(GraphNode.CLASS);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}
	/**
	 * Add a top node to represent a Thread.
	 */
	protected NodeContainer addTopNode(TRCThread thread,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		topNode.setName(resolveFullContainerName(invocation));
		topNode.setShortName(createShortNodeName(topNode.getName()));
		if (thread.getId() == 0)
			topNode.setSecondaryName(unknownString);
		else
			topNode.setSecondaryName(String.valueOf(thread.getId()));
		topNode.setType(GraphNode.THREAD);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}

	/**
	 * Add a top node to represent a Process.
	 */
	protected NodeContainer addTopNode(TRCProcess process,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		if (process.getAgent().getAgentProxy().getProcessProxy().getName() == null)
			topNode.setName(unknownString);
		else
			topNode.setName(resolveFullContainerName(invocation));
		topNode.setShortName(createShortNodeName(topNode.getName()));
		if (process.getAgent().getAgentProxy().getProcessProxy().getPid() == 0)
			topNode.setSecondaryName(unknownString);
		else
			topNode.setSecondaryName(String.valueOf(process.getAgent().getAgentProxy().getProcessProxy().getPid()));
		topNode.setType(GraphNode.PROCESS);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}
	/**
	 * Add a top node to represent an Agent.
	 */
	protected NodeContainer addTopNode(TRCAgent agent,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		
		if (agent.getName() == null)
			topNode.setName(unknownString);
		else
			topNode.setName(resolveFullContainerName(invocation));
		topNode.setShortName(createShortClassName(topNode.getName()));
		if (agent.getRuntimeId() == null)
			topNode.setSecondaryName(unknownString);
		else
			topNode.setSecondaryName(agent.getRuntimeId());
		topNode.setType(GraphNode.AGENT);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}
	/**
	 * Add a top node to represent a Node.
	 */
	protected NodeContainer addTopNode(TRCNode node,TRCMethodInvocation invocation)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		if (node.getName() == null)
			topNode.setName(unknownString);
		else
			topNode.setName(resolveFullContainerName(invocation));
		topNode.setShortName(createShortNodeName(topNode.getName()));
		if (node.getIpAddress() == null)
			topNode.setSecondaryName(unknownString);
		else
			topNode.setSecondaryName(node.getIpAddress());
		topNode.setType(GraphNode.NODE);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}
	/**
	 * Add a top node to represent a Monitor.
	 */
	protected NodeContainer addTopNode(TRCMonitor monitor)
	{
		NodeContainer topNode = graph.getContentFactory().createNodeContainer();
		if (monitor.getName() == null)
			topNode.setName(unknownString);
		else
			topNode.setName(monitor.getName());
		topNode.setShortName(createShortNodeName(topNode.getName()));
		if (monitor.getName() == null)
			topNode.setSecondaryName(unknownString);
		else
			topNode.setSecondaryName(monitor.getName());
		topNode.setType(GraphNode.MONITOR);
		graph.addTopNode(topNode);
		topNode.setGraph(graph);
		return topNode;
	}

	/**
	 * Create and add a generic node given the names, type and container.
	 */
	protected GraphNode addNode(
		String name,
		String shortName,
		String secondaryName,
		int type,
		NodeContainer container)
	{
		GraphNode node = graph.getContentFactory().createGraphNode();
		node.setName(name);
		node.setShortName(shortName);
		node.setSecondaryName(secondaryName);
		node.setType(type);
		container.addInternalNode(node);
		node.setContainer(container);
		return node;
	}

	protected void addNodeConnections(TRCMethodInvocation mi)
	{
		NodeConnection connection = null;
		// given this invocation create a NodeConnection that uses the source and target MI
		// as keys to the nodes Hashtable to locate source and target GraphNode
		GraphNode source = (GraphNode) nodes.get(mi);
		GraphNode target = null;
		double deltaTime = 0;
		for (Iterator segments = mi.getInvokes().iterator(); segments.hasNext();)
		{
			TRCMethodInvocation seg = (TRCMethodInvocation) segments.next();
			if (seg instanceof TRCFullMethodInvocation && seg != null)
			{			
			   TRCFullMethodInvocation segment = (TRCFullMethodInvocation)seg;
        
				// A segments time stamps are relative to the target method's agent start time there for delta is 
				// calculated as follows 
				deltaTime = getAgentDelta(segment.getThread().getProcess().getAgent().getAgentProxy());
				target = (GraphNode) nodes.get(segment);
				if (target != null)
				{
					//the following removes from filtered graph any self-calls
					if(graph.getFiltered() == true && source.getContainer() == target.getContainer())
					   return;
					// A connection is used to mirror each timestamp in a segment
					// So if the timestamp is missing the connection should not be crested
					// this deals with partial traces
					if (segment.getEntryTime() != 0)
					{
						connection = graph.createNodeConnection();
						connection.setName(target.getName());
						connection.setShortName(target.getShortName());
						connection.setSecondaryName(target.getSecondaryName());
						connection.setTarget(target);
						connection.setSource(source);
						connection.setType(NodeConnection.CALL);
                     
                        // if the target start time >= connection start time, reuse the increment
						// and recover from time missmatches due to network delays etc.
						if (target.getStartTime() >= segment.getEntryTime() + deltaTime)
							connection.setEndIncrement(target.getStartIncrement());
						else
							connection.setEndTime(segment.getEntryTime() + deltaTime);

						// in this release connection start and end times are the same  
						connection.setStartIncrement(connection.getEndIncrement());

						// if the connection start time >= source end time, reuse the increment
						// and recover from time missmatches
						if (connection.getStartTime() > source.getEndTime())
							source.setEndIncrement(connection.getStartIncrement());

						target.addTargetConnection(connection);
						source.addSourceConnection(connection);
					}
					// process return half of the segment
					// determine if it happened based on the times being set
					if (segment.getExitTime() != 0)
					{
						connection = graph.createNodeConnection();
						connection.setName(target.getName() + " return");
						connection.setShortName(target.getShortName());
						connection.setSecondaryName(target.getSecondaryName());
						connection.setTarget(source);
						connection.setSource(target);
						connection.setType(NodeConnection.RETURN);
                              
						// if the target end time >= connection start time, reuse the increment
						// and recover from time missmatches due to network delays etc.
						if (target.getEndTime() > segment.getExitTime() + deltaTime)
							connection.setStartTime(segment.getExitTime() + deltaTime);
						else
							connection.setStartIncrement(target.getEndIncrement());

						// in this release connection start and end times are the same  
						connection.setEndIncrement(connection.getStartIncrement());

						// if the connection start time >= source end time, reuse the increment
						// and recover from time missmatches
						if (connection.getEndTime() > source.getEndTime())
							source.setEndIncrement(connection.getEndIncrement());

						source.addTargetConnection(connection);
						target.addSourceConnection(connection);
					}
				}
				else
				target = null;
			}
		}
	}

	protected int resolveType(TRCObject o)
	{
		if(o.getProcess().getClassClass() == o.getIsA()) //class object
		   return GraphNode.CLASS;
		   
       return GraphNode.OBJECT;
	}

	protected void resolveConnections()
	{
		for (Enumeration nodeList = nodes.keys(); nodeList.hasMoreElements();)
		{
			addNodeConnections((TRCMethodInvocation) nodeList.nextElement());
		}
	}

	protected void setOldestEndTime(double d)
	{
		if (d > oldestEndTime)
			oldestEndTime = d;
	}
	protected double getOldestEndTime()
	{
		return oldestEndTime;
	}
	protected double getAgentDelta(TRCAgentProxy agentProxy)
	{
		double returnValue;
		if (agents.containsKey(agentProxy)){
		     returnValue=((Double) agents.get(agentProxy)).doubleValue() + agentProxy.getProcessProxy().getNode().getDeltaTime();
		}
		else 
		  returnValue=0;

        return returnValue;		   
		
	}
	protected void addAgent(TRCAgentProxy agentProxy)
	{
		// if the agent is in the list already return
		if (agents.containsKey(agentProxy))
			return;

		agents.put(agentProxy, new Double(0));

		try
		{
			agentList[nextAgent] = agentProxy;
		}
		catch (ArrayIndexOutOfBoundsException e)
		{
			TRCAgentProxy[] tempArray = new TRCAgentProxy[agentList.length + 16];
			System.arraycopy(agentList, 0, tempArray, 0, agentList.length);
			agentList = tempArray;
			agentList[nextAgent] = agentProxy;
		}
		nextAgent++;

	}
	protected void loadAllAgents(TRCMonitor monitor)
	{
		if (monitor != null)
		{
			for (Iterator nodes = monitor.getNodes().iterator(); nodes.hasNext();)
			{
				TRCNode node = (TRCNode) nodes.next();
				if (node != null && !node.eIsProxy())
				{
					loadAllAgents(node);
				} // node null test
			} // nodes loop  
		}
	}
	protected void loadAllAgents(TRCNode node)
	{
		for (Iterator processProxies = node.getProcessProxies().iterator(); processProxies.hasNext();)
		{
			TRCProcessProxy processProxy = (TRCProcessProxy) processProxies.next();
			if (processProxy != null && !processProxy.eIsProxy())
			{
				loadAllAgents(processProxy);
			}
		} // processProxies loop
	}

	protected void loadAllAgents(TRCProcessProxy processProxy)
	{
		for (Iterator agents = processProxy.getAgentProxies().iterator(); agents.hasNext();)
		{
			TRCAgentProxy trcAgent = (TRCAgentProxy) agents.next();
			if (trcAgent != null && !trcAgent.eIsProxy())
			{
				addAgent(trcAgent);
			}
		} // agent loop
	}
	protected Hashtable getAgentTable()
	{
		return agents;
	}
	protected void syncAgentTimes()
	{
		if (nextAgent == 0)
			return;
		Hashtable agentTable = getAgentTable();
		TRCAgentProxy oldestAgent = null;
		int firstGoodAgent = 0;
		// skip past the first agents with no startTime
		for (int i = firstGoodAgent; i < nextAgent; i++)
		{
			if (agentList[i].getStartTime() != 0)
			{
				firstGoodAgent = i;
				oldestAgent = agentList[i];
				i = nextAgent;
			}
		}
		// for each agent with a start time, determine if it is older
		for (int i = firstGoodAgent; i < nextAgent; i++)
		{
			try
			{
				if (agentList[i].getStartTime() + agentList[i].getProcessProxy().getNode().getDeltaTime() < oldestAgent.getStartTime() + oldestAgent.getProcessProxy().getNode().getDeltaTime())
					oldestAgent = agentList[i];
			}
			catch (NullPointerException e)
			{
			} // ignore this agent 
		}
		// Calculate the delta from the oldest agent
		for (int i = firstGoodAgent; i < nextAgent; i++)
		{
			if (agentList[i].getStartTime() + agentList[i].getProcessProxy().getNode().getDeltaTime() != 0)
			{
				double temptime = agentList[i].getStartTime() + agentList[i].getProcessProxy().getNode().getDeltaTime() - oldestAgent.getStartTime() - oldestAgent.getProcessProxy().getNode().getDeltaTime();
				agentTable.put(agentList[i], new Double(temptime ));
			}
		}
	}
	
	public void loadInvocationsIntoModel(TRCAgentProxy agentProxy, int graphType) {
		if (agentProxy==null) return;
		
		TRCProcess process = (TRCProcess)agentProxy.getAgent().getProcess();
		if (process != null)
		{
	  		for (Iterator threads = process.getThreads().iterator(); threads.hasNext();)
	  		{
				TRCThread thread = (TRCThread) threads.next();
				if (thread != null)
				{
		  			for (Iterator invocations = thread.getInitialInvocations().iterator(); invocations.hasNext();)
		  			{
						TRCMethodInvocation invocation = (TRCMethodInvocation) invocations.next();
						if (invocation != null)
						{
			 				 NodeContainer container = resolveContainer(invocation, graphType);
			 				 resolveMethodNode(invocation, container);
			 				 call(invocation.getInvokes(),graphType);
			 				 
						} 
		 			 } 
				}
	  		}
		}
		
		resolveConnections();
	}
	
	private void call(EList list,int graphType) {

		 for (Iterator iter = list.iterator(); iter.hasNext();) {
			 TRCMethodInvocation invoker = (TRCMethodInvocation) iter.next();

			NodeContainer container = resolveContainer(invoker, graphType);
			resolveMethodNode(invoker, container);

			call(invoker.getInvokes(),graphType);
		 }
	 }
	
	public void loadInvocationsIntoModel(TRCProcessProxy processProxy, int graphType)
	{
		if (processProxy==null) return;
		
		for (Iterator agentProxies = processProxy.getAgentProxies().iterator(); agentProxies.hasNext();)
		{
		  TRCAgentProxy trcAgentProxy = (TRCAgentProxy) agentProxies.next();
		  if (trcAgentProxy != null && !trcAgentProxy.eIsProxy())
		  {
		  		loadInvocationsIntoModel(trcAgentProxy,graphType);
		  }
		}
	}
	
	public void loadInvocationsIntoModel(TRCNode node, int graphType)
	{
		if (node==null) return;
		
		for (Iterator processProxies = node.getProcessProxies().iterator(); processProxies.hasNext();)
		{
		  TRCProcessProxy processProxy = (TRCProcessProxy) processProxies.next();
		  if (processProxy != null && !processProxy.eIsProxy())
		  {
				loadInvocationsIntoModel(processProxy, graphType);
		  }
		}
	}
	
	public void loadInvocationsIntoModel(TRCMonitor monitor, int graphType)
	{
		if (monitor==null) return;
		
		for (Iterator nodes = monitor.getNodes().iterator(); nodes.hasNext();)
		{
		  TRCNode node = (TRCNode) nodes.next();
		  if (node != null && !node.eIsProxy())
		  {
		  		loadInvocationsIntoModel(node,graphType);
		  }
		}
	}
	
	private TRCClass getClass(TRCMethodInvocation invocation)
	{
		TRCObject o = invocation.getOwningObject();
		if (o.getIsA() != invocation.getProcess().getClassClass())
			return o.getIsA();
		else
			return invocation.getMethod().getDefiningClass();
		
	}
}
