package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.util.Hashtable;
import java.util.Vector;

import org.w3c.dom.Document;
/**
 * Data retrieval methods which retrieve data of interest to <code>SVGPieChart</code> from 
 * documents which conform to the documented data input schemas. 
 * 
 * @version 1.19.1.4
 */
public class PieChartDataRetriever extends CommonDataRetriever {
	/**
	 * Sole constructor.
	 */
	PieChartDataRetriever() {
		super();
	}	
	
	/**
	 * Returns from the data document the text labels which indicate the 
	 * contextual definition of the data values and the summary (total value) 
	 * for the pie, both of which are displayed by the meter graphic.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_CONTEXT_FOUND, NO_SUMMARY_FOUND</code>.
	 * @see IDataInputProcessingExceptionCodes
	 */	
	public Hashtable getTitles(Document dataDocument) throws DataInputProcessingException {	
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
			
		Hashtable titles = new Hashtable(2);		

		String label = getAttributeFromSingleElement(dataDocument, DATARANGE, LABEL);
		titles.put(VALUE_TITLE, (label != null) ? label : "");
		String summary = getAttributeFromSingleElement(dataDocument, DATAUPDATE, SUMMARY);
		titles.put(TOTAL_TITLE, (summary != null) ? summary : "");

		return titles;		
	}	
		
	/**
	 * Returns the ordered array of values found in the data document each of which represents
	 * the total value of a data set and each is rendered as a pie slice value. Values can be 
	 * ordered by specifying their ordinal position; otherwise, the retrieved (document) 
	 * order is used.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_TOTALS_FOUND, MISSING_TOTALS_OR_POSITIONS, 
	 * POSITION_NUMBER_OUT_OF_BOUNDS, DUPLICATE_POSITION_NUMBER, INVALID_NUMBER_FORMAT</code>
	 * @see IDataInputProcessingExceptionCodes
	 */	
	public String [] getDatasetTotals(Document dataDocument) throws DataInputProcessingException {		
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
		
		Vector labels = getAttributeFromElements(dataDocument, DATASET, TOTAL);
		if (labels == null) {
			throwException(NO_TOTALS_FOUND, DATASET);			
		}
		int labelsSize = labels.size();	
		
		// ensure they are numbers
		for (int i = 0; i < labelsSize; i++) {
			double j = 0;
			try {
				j = Double.parseDouble((String)labels.get(i));
			} catch (Exception exc) {
				// NumberFormatException
				throwException(INVALID_NUMBER_FORMAT, DATASET);	
			}	
			// restriction to non-negative numbers
			if (j < 0) {
				throwException(INVALID_NUMBER_FORMAT, SEGMENT_MARKER);
			}					
		}
		
		String [] result = new String [labelsSize];		
		// get the positions, if any
		Vector positions = getAttributeFromElements(dataDocument, DATASET, POSITION);
				
		// use document order
		if (positions == null) {
			for (int i = 0;i < labelsSize; i++) {
				result[i] = (String)labels.get(i);
			}
			return result;
		}		
		if (positions.size() != labelsSize) {
			throwException(MISSING_TOTALS_OR_POSITIONS, DATASET);			
		}				 		
		
		// use positions		
		for (int i = 0;i < labelsSize; i++) {
			short position = 0;			
			try {
				position = Short.parseShort((String)positions.get(i));
			} catch (Exception exc) {
				// NumberFormatException
				throwException(INVALID_NUMBER_FORMAT, DATASET);				
			}					
			// make sure the position is within bounds						
			if (position > (labelsSize - 1)  || position < 0) {
				throwException(POSITION_NUMBER_OUT_OF_BOUNDS, DATASET);
			}				
			// make sure the position is not already taken
			String test = null;		
			try {
				test = result[position];
			}
			catch (Exception e) {}								
				
			if (test != null) {
				throwException(DUPLICATE_POSITION_NUMBER, DATASET);
			}		
			result[position] = (String)labels.get(i);		
		}						
		return result;			
	}			
	
	/**
	 * Returns the ordered array of values found in the data document each of which 
	 * represents the total value of a data set as a percentage of the sum of the total
	 * values of all the data sets. Each of these values is rendered as a pie 
	 * slice percentage value. Values can be ordered by specifying their ordinal 
	 * position; otherwise, the retrieved (document) order is used.
	 * 
	 * @throws DataInputProcessingException for codes <code>NO_DATA_DOCUMENT, NO_PERCENTAGE_OF_DATAPOINT_TOTAL_FOUND,
	 * MISSING_PERCENTAGE_OF_DATAPOINT_TOTAL_OR_POSITION, INVALID_NUMBER_FORMAT</code>
	 * @see IDataInputProcessingExceptionCodes
	 */	
/*
	public String [] getDatasetPercentages(Document dataDocument) throws DataInputProcessingException {		
		if (dataDocument == null) {
			throwException(NO_DATA_DOCUMENT, EMPTY_STRING);
		}	
		
		Vector labels = getAttributeFromElements(dataDocument, DATASET, PERCENTAGE_OF_DATAPOINT_TOTAL);
		if (labels == null) {
			throwException(NO_PERCENTAGE_OF_DATAPOINT_TOTAL_FOUND, DATASET);				
		}			
		int labelsSize = labels.size();	
		// ensure they are numbers
		for (int i = 0; i < labelsSize; i++) {
			double j = 0;
			try {
				j = Double.parseDouble((String)labels.get(i));
			} catch (Exception exc) {
				// NumberFormatException
				throwException(INVALID_NUMBER_FORMAT, DATASET);	
			}	
			// restriction to non-negative numbers
			if (j < 0) {
				throwException(INVALID_NUMBER_FORMAT, SEGMENT_MARKER);
			}					
		}
		
		String [] result = new String [labelsSize];
				
		// get the positions, if any
		Vector positions = getAttributeFromElements(dataDocument, DATASET, POSITION);
				
		// use document order
		if (positions == null) {
			for (int i = 0;i < labelsSize; i++) {
				result[i] = (String)labels.get(i);
			}
			return result;
		}
		
		if (positions.size() != labelsSize) {
			throwException(MISSING_PERCENTAGE_OF_DATAPOINT_TOTAL_OR_POSITION, DATASET);				
		}				 		
		
		// use positions		
		for (int i = 0;i < labelsSize; i++) {
			short position = 0;			
			try {
				position = Short.parseShort((String)positions.get(i));
			} catch (Exception exc) {
				throwException(INVALID_NUMBER_FORMAT, DATASET);				
			}					
			result[position] = (String)labels.get(i);
			
		}						
		return result;			
	}	
*/
	
}

