/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;


/**
 * @author amathur
 * This class maintains the information about a given execution result.
 */
public class ExecutionResultData {

	private TPFExecutionResult result = null;
	private TPFInvocationEvent invocationEvent = null;
	private String id = null;

	private TPFTest test = null;
	private ResourceSet			resourceSet;
	private URI					testURI;
	
	/**
	 * Returns the test to which this execution result belongs.
	 * @return TPFTest
	 */
	public TPFTest getTest() {
		if(test.eResource().getResourceSet() == null)
		{
			EObject eObject = resourceSet.getEObject(testURI, true);
			if(eObject instanceof TPFTest)
				test = (TPFTest)eObject;
		}
			
		return test;
	}
	
	/**
	 * Get the ID of this execution result
	 * @return
	 */
	public String getId() {
		return id;
	}

	/**
	 * Set the ID of this execution result
	 * @param id
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * Get the invocation event that created this execution result
	 * @return
	 */
	public TPFInvocationEvent getInvocationEvent() {
		return invocationEvent;
	}

	/**
	 * Set the invocation event that is responsble for creating this execution result
	 * @param invocationEvent
	 */
	public void setInvocationEvent(TPFInvocationEvent invocationEvent) {
		this.invocationEvent = invocationEvent;
		if (result != null) {
			TPFExecutionHistory history = result.getExecutionHistory();
			if (history != null) {
				this.invocationEvent.setExecutionHistory(history);
			}
		}
	}

	/**
	 * Get the TPFExecutionResult.
	 * @return
	 */
	public TPFExecutionResult getResult() {
		if (result == null) {
			
			// Create a result set for the given Id
			result =
				Common_TestprofileFactoryImpl
					.eINSTANCE
					.createTPFExecutionResult();
			result.setId(id);
			result.setTest(getTest());
			if (invocationEvent != null) {
				// This execution result was created by an invocation event
				result.setInvocationExecutionEvent(invocationEvent);
			}
		}

		// Now create an execution history for the result, if it doesnt
		// already exist
		if (result.getExecutionHistory() == null) {
			TPFExecutionHistory history =
				Common_TestprofileFactoryImpl
					.eINSTANCE
					.createTPFExecutionHistory();
			result.setExecutionHistory(history);
		}

		return result;
	}
	
	/**
	 * Set the Execution Result.
	 * @param result
	 */
	public void setResult(TPFExecutionResult result) {
		this.result = result;
	}
	
	
	/**
	 * Get the Execution History
	 * @return
	 */
	public TPFExecutionHistory getHistory() {
		return getResult().getExecutionHistory();
	}

	/**
	 * Create an ExecutionResultData object for the given suite and ID
	 * @param suite
	 * @param id
	 */
	public ExecutionResultData(TPFTest test, String id) {
		this.test = test;
		this.id = id;

		resourceSet = test.eResource().getResourceSet();
		testURI = EcoreUtil.getURI(test);
	}
	
	/**
	 * Find the EObject corresponding to the ID within the test.
	 * @param id
	 * @return
	 */
	public EObject getObjectInTest(String id) {
		if (test == null) {
			return null;
		}
		
		EObject eObject = test.eResource().getEObject(id);
		
		return eObject;
	}

}
