package org.eclipse.hyades.logging.events;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.SimpleTimeZone;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.hyades.logging.core.XmlUtility;

/**
 * The CbeFormatter class defines a utility object used for 
 * converting, serializing and deserializing CommonBaseEvent
 * and related objects.  It is mostly used by other objects
 * in this API to facilitate shared functionality, but can
 * be used by anyone using the API.
 * 
 * @author rduggan
 * @author paules
 * @author jgerken
 * @version 1.0
 * @see CommonBaseEventImpl
 * @see AssociationEngineImpl
 * @since CBE Java API Version 1.0
 */
public final class CbeFormatter {

    private static ISimpleEventFactory factory = SimpleEventFactoryImpl.getInstance();
    public final static String xml_version = "CommonBaseEvent XML version 1.0.0";
    private static String incomingCBEVersion = null;

    /**
     * Converts an IAssociationEngine object into a serialized
     * XML fragment.
     * 
     * @param engine The engine to serialize.
     * 
     * @return An XML fragment String representing the IAssociationEngine object passed.
     * @since V 1.0
     */
    public static synchronized String toCanonicalXMLString(IAssociationEngine engine) {

        StringBuffer buffer = new StringBuffer(256);

        buffer.append("<AssociationEngine");

        String attributeValue = engine.getId();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" id=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = engine.getName();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" name=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = engine.getType();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" type=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append("/>");

        return (buffer.toString());
    }

    public static synchronized String toCanonicalXMLString(ICommonBaseEvent event) {

        /* Start with a 1k buffer to load the XML String into */
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvent");

        String attributeValue = event.getCreationTime();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" creationTime=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = event.getExtensionName();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" extensionName=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = event.getGlobalInstanceId();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" globalInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = event.getLocalInstanceId();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" localInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = event.getMsg();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" msg=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = event.getSituationType();

        if ((attributeValue != null) && (attributeValue.length() > 0)) {
            buffer.append(" situationType=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        if (event.getElapsedTime() != 0) {
            buffer.append(" elapsedTime=\"");
            buffer.append(event.getElapsedTime());
            buffer.append("\"");
        }

        if (event.getPriority() != 0) {
            buffer.append(" priority=\"");
            buffer.append(event.getPriority());
            buffer.append("\"");
        }

        if (event.getRepeatCount() != 0) {
            buffer.append(" repeatCount=\"");
            buffer.append(event.getRepeatCount());
            buffer.append("\"");
        }

        if (event.getSequenceNumber() != 0) {
            buffer.append(" sequenceNumber=\"");
            buffer.append(event.getSequenceNumber());
            buffer.append("\"");
        }

        if (event.getSeverity() != 0) {
            buffer.append(" severity=\"");
            buffer.append(event.getSeverity());
            buffer.append("\"");
        }

        buffer.append(">");

        Object[] contextDataElements = event.getContextDataElements();

        if (contextDataElements != null) {

            for (int i = 0; i < contextDataElements.length; i++) {

                if (contextDataElements[i] != null) {

                    buffer.append("\n\t<contextDataElements");

                    attributeValue = ((IContextDataElement) (contextDataElements[i])).getName();

                    if ((attributeValue != null) && (attributeValue.length() > 0)) {
                        buffer.append(" name=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((IContextDataElement) (contextDataElements[i])).getType();

                    if ((attributeValue != null) && (attributeValue.length() > 0)) {
                        buffer.append(" type=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    attributeValue = ((IContextDataElement) (contextDataElements[i])).getContextId();

                    if ((attributeValue != null) && (attributeValue.length() > 0)) {
                        buffer.append("\n\t\t<contextId>");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("</contextId>");
                    }
                    else {

                        attributeValue = ((IContextDataElement) (contextDataElements[i])).getContextValue();

                        if ((attributeValue != null) && (attributeValue.length() > 0)) {
                            buffer.append("\n\t\t<contextValue>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</contextValue>");
                        }
                    }

                    buffer.append("\n\t</contextDataElements>");
                }
            }
        }

        Object[] extendedDataElements = event.getExtendedDataElements();

        if (extendedDataElements != null) {

            for (int i = 0; i < extendedDataElements.length; i++) {

                buffer.append(getIExtendedDataElementXML(((IExtendedDataElement) (extendedDataElements[i])), "extendedDataElements", 1));
            }
        }

        IAssociatedEvent[] associatedEvents = event.getAssociatedEvents();
        if (associatedEvents != null) {
            for (int i = 0; i < associatedEvents.length; i++) {
                if (associatedEvents[i] != null) {
                    buffer.append("\n\t<associatedEvents");
                    if (associatedEvents[i].getAssociationEngine() != null) {
                        attributeValue = associatedEvents[i].getAssociationEngine().getId();

                        if ((attributeValue != null) && (attributeValue.length() > 0)) {
                            buffer.append(" associationEngine=\"");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("\"");
                        }
                    }

                    String[] refs = associatedEvents[i].getResolvedEvents();
                    if ((refs != null) && (refs.length > 0)) {
                        buffer.append(" resolvedEvents=\"");
                        boolean addSpace = false;
                        for (int j = 0; j < refs.length; j++) {
                            if (refs[j] != null) {
                                if (addSpace) {
                                    buffer.append(" ");
                                }
                                else {
                                    addSpace = true;
                                }
                                buffer.append(refs[j]);
                            }
                        }
                        buffer.append("\"");
                    }
                    buffer.append("/>");
                }
            }
        }

        IComponentIdentification componentId = event.getReporterComponentId();

        if (componentId != null) {

            buffer.append("\n\t<reporterComponentId");

            attributeValue = componentId.getApplication();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        componentId = event.getSourceComponentId();

        if (componentId != null) {

            buffer.append("\n\t<sourceComponentId");

            attributeValue = componentId.getApplication();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        IMsgDataElement msgDataElement = event.getMsgDataElement();

        if (msgDataElement != null) {

            buffer.append("\n\t<msgDataElement");

            attributeValue = msgDataElement.getMsgLocale();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" msgLocale=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            String[] tokens = msgDataElement.getMsgCatalogTokens();

            if (tokens != null) {

                for (int i = 0; i < tokens.length; i++) {

                    if ((tokens[i] != null) && (tokens[i].length() > 0)) {

                        buffer.append("\n\t\t<msgCatalogTokens value=\"");
                        buffer.append(XmlUtility.normalize(tokens[i]));
                        buffer.append("\"/>");
                    }
                }
            }

            attributeValue = msgDataElement.getMsgId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append("\n\t\t<msgId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgId>");
            }

            attributeValue = msgDataElement.getMsgIdType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {

                buffer.append("\n\t\t<msgIdType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgIdType>");
            }

            attributeValue = msgDataElement.getMsgCatalogId();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {

                buffer.append("\n\t\t<msgCatalogId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogId>");
            }

            attributeValue = msgDataElement.getMsgCatalogType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append("\n\t\t<msgCatalogType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogType>");
            }

            attributeValue = msgDataElement.getMsgCatalog();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {

                buffer.append("\n\t\t<msgCatalog>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalog>");
            }

            buffer.append("\n\t</msgDataElement>");
        }

        buffer.append("\n</CommonBaseEvent>");

        return (buffer.toString());
    }

    private static String getIExtendedDataElementXML(IExtendedDataElement extendedDataElement, String tagName, int indent) {

        if (extendedDataElement != null) {

            StringBuffer buffer = new StringBuffer(512);

            buffer.append("\n");

            for (int counter = 0; counter < indent; counter++)
                buffer.append("\t");

            buffer.append("<");
            buffer.append(tagName);

            String attributeValue = extendedDataElement.getName();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" name=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = extendedDataElement.getType();

            if ((attributeValue != null) && (attributeValue.length() > 0)) {
                buffer.append(" type=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            Object[] childDataElements = extendedDataElement.getChildDataElements();

            if (childDataElements != null) {

                for (int counter = 0; counter < childDataElements.length; counter++) {

                    buffer.append(getIExtendedDataElementXML(((IExtendedDataElement) childDataElements[counter]), "children", (indent + 1)));
                }
            }

            if (extendedDataElement.getValues() != null) {

                String[] values = extendedDataElement.getValues();

                for (int k = 0; k < values.length; k++) {

                    if ((values[k] != null) && (values[k].length() > 0)) {

                        buffer.append("\n");

                        for (int counter = 0; counter < (indent + 1); counter++)
                            buffer.append("\t");

                        buffer.append("<values>");
                        buffer.append(XmlUtility.normalize(values[k]));
                        buffer.append("</values>");
                    }
                }
            }
            else if ((extendedDataElement.getHexValue() != null) && (extendedDataElement.getHexValue().length > 0)) {

                buffer.append("\n");

                for (int counter = 0; counter < (indent + 1); counter++)
                    buffer.append("\t");

                buffer.append("<hexValue>");

                byte[] values = extendedDataElement.getHexValue();
                String hexValue = null;

                for (int k = 0; k < values.length; k++) {
                    hexValue = Integer.toHexString((int) values[k]).toUpperCase();

                    if (hexValue.length() == 1)
                        buffer.append('0');
                    buffer.append(hexValue);
                }

                buffer.append("</hexValue>");
            }

            buffer.append("\n");

            for (int counter = 0; counter < indent; counter++)
                buffer.append("\t");

            buffer.append("</");
            buffer.append(tagName);
            buffer.append(">");

            return (buffer.toString());
        }
        else {
            return "";
        }

    }

    /**
     * Converts an ICommonBaseEvent object to an XML document String.
     * 
     * @param event  The ICommonBaseEvent to convert.
     * 
     * @return An XML document String representing the Common Base Event object passed.
     * @since V 1.0
     */
    public static synchronized String toCanonicalXMLDocString(ICommonBaseEvent event) {

        // Create a buffer containing the top level element and namespace info...
        StringBuffer buffer = new StringBuffer(1280);

        buffer.append("<CommonBaseEvents xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" ");
        buffer.append("xsi:noNamespaceSchemaLocation=\"commonbaseevent1_0.xsd\">\n");

        // If associatedEvents exist, get the serialized association engine...
        if (event.getAssociatedEvents().length != 0) {

            IAssociatedEvent[] AEArray = event.getAssociatedEvents();

            int i = 0;
            boolean found = false;
            int test = AEArray.length;
            while ((i < AEArray.length) && (found == false)) {
                if (AEArray[i].getAssociationEngine() != null) {
                    AssociationEngineImpl anAEI = (AssociationEngineImpl) AEArray[i].getAssociationEngine();

                    buffer.append("\t" + anAEI.externalizeCanonicalXmlString() + "\n");
                    found = true;
                }
                i++;
            }

        }

        // Add the CommonBaseEvent element fragment
        buffer.append(toCanonicalXMLString(event));

        // Close and return the top level element...
        buffer.append("\n</CommonBaseEvents>");
        return buffer.toString();
    }

    /**
     * Converts an AssociationEngineImpl object to an XML document String.
     * 
     * @param engine the AssociationEngineImpl object to convert.
     * 
     * @return An XML document String representing the AssociationEngineImpl object passed.
     * @since V 1.0
     */
    public static synchronized String toCanonicalXMLDocString(AssociationEngineImpl engine) {

        StringBuffer buffer = new StringBuffer(256);
        buffer.append("<CommonBaseEvents xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" ");
        buffer.append("xsi:noNamespaceSchemaLocation=\"commonbaseevent1_0.xsd\">\n\t");

        buffer.append(engine.externalizeCanonicalXmlString());

        // Close and return the top level element...
        buffer.append("\n</CommonBaseEvents>");
        return buffer.toString();
    }

    /**
     * Internalizes a passed XML fragment String into a passed CommonBaseEventImpl object.
     * 
     * @param event      The CommonBaseEventImpl object to populate.
     * @param aXMLString The XML fragment containing the information used to populate the Common Base Event object.
     * @since V 1.0
     */
    public static synchronized void fromCanonicalXMLString(ICommonBaseEvent event, String aXMLString) {

        /* create a factory to build the objects we need... */
        factory = SimpleEventFactoryImpl.getInstance();

        /* setup the pointers and our initial work area */
        int masterIndex = aXMLString.indexOf("<CommonBaseEvent "); // tracks location in passed string
        int index = 0; // tracks location in temp work area
        String tempString = aXMLString.substring(masterIndex, aXMLString.indexOf(">", masterIndex + 1));
        masterIndex = aXMLString.indexOf(">", masterIndex + 1) + 3; // push pointer after element

        /* populate the attributes for the root element of this fragment */
        index = tempString.indexOf("creationTime=\"");
        if (index != -1) {
            event.setCreationTime(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)));
        }

        index = tempString.indexOf("extensionName=\"");
        if (index != -1) {
            event.setExtensionName(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)));
        }

        index = tempString.indexOf("globalInstanceId=\"");
        if (index != -1) {
            event.setGlobalInstanceId(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)));
        }

        index = tempString.indexOf("localInstanceId=\"");
        if (index != -1) {
            event.setLocalInstanceId(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)));
        }

        index = tempString.indexOf("msg=\"");
        if (index != -1) {
            event.setMsg(tempString.substring(index + 5, tempString.indexOf("\"", index + 5)));
        }

        index = tempString.indexOf("situationType=\"");
        if (index != -1) {
            event.setSituationType(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)));
        }

        index = tempString.indexOf("elapsedTime=\"");
        if (index != -1) {
            event.setElapsedTime(Long.valueOf(tempString.substring(index + 13, tempString.indexOf("\"", index + 13))).longValue());
        }

        index = tempString.indexOf("priority=\"");
        if (index != -1) {
            event.setPriority(Short.valueOf(tempString.substring(index + 10, tempString.indexOf("\"", index + 10))).shortValue());
        }

        index = tempString.indexOf("repeatCount=\"");
        if (index != -1) {
            event.setRepeatCount(Short.valueOf(tempString.substring(index + 13, tempString.indexOf("\"", index + 13))).shortValue());
        }

        index = tempString.indexOf("sequenceNumber=\"");
        if (index != -1) {
            event.setSequenceNumber(Long.valueOf(tempString.substring(index + 16, tempString.indexOf("\"", index + 16))).longValue());
        }

        index = tempString.indexOf("severity=\"");
        if (index != -1) {
            event.setSeverity(Short.valueOf(tempString.substring(index + 10, tempString.indexOf("\"", index + 10))).shortValue());
        }

        /* Begin processing of ContextDataEements */
        while (aXMLString.indexOf("<contextDataElements ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = aXMLString.indexOf("<contextDataElements ", masterIndex);
            tempString = aXMLString.substring(masterIndex, aXMLString.indexOf("</contextDataElements>", masterIndex + 1));
            masterIndex = aXMLString.indexOf("</contextDataElements>", masterIndex + 1) + 22;

            IContextDataElement cde = factory.createContextDataElement();
            /* process the attributes for this element */
            index = tempString.indexOf("name=\"");
            if (index != -1) {
                cde.setName(tempString.substring(index + 6, tempString.indexOf("\"", index + 6)));
            }

            index = tempString.indexOf("type=\"");
            if (index != -1) {
                cde.setType(tempString.substring(index + 6, tempString.indexOf("\"", index + 6)));
            }

            /**
             * contextValue and contextId are mutually exclusive.  Therefore, only 
             * one of these two properties can be defined.  contextValue takes 
             * precidence, such that if contextValue is set then contextId is ignored. 
             */
            index = tempString.indexOf("<contextValue>");
            if (index != -1) {
                cde.setContextValue(tempString.substring(index + 14, tempString.indexOf("</contextValue>", index + 14)));

            }
            else if (tempString.indexOf("<contextId>") != -1) {
                index = tempString.indexOf("<contextId>");
                cde.setContextId(tempString.substring(index + 11, tempString.indexOf("</contextId>", index + 11)));
            }

            event.addContextDataElement(cde);
            masterIndex = aXMLString.indexOf("</contextDataElements>", masterIndex + 1) + 22;
        }

        /* Begin processing of ExtendedDataEements */
        while (aXMLString.indexOf("<extendedDataElements ", masterIndex) != -1) {
            event.addExtendedDataElement(loadIExtendedDataElements(aXMLString.substring(aXMLString.indexOf("<extendedDataElements ", masterIndex), aXMLString.indexOf("</extendedDataElements>", masterIndex + 1))));
            masterIndex = aXMLString.indexOf("</extendedDataElements>", masterIndex + 1) + 23;
        }

        /* Begin processing of AssociatedEvents */
        while (aXMLString.indexOf("<associatedEvents ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = aXMLString.indexOf("<associatedEvents ", masterIndex);
            tempString = aXMLString.substring(masterIndex, aXMLString.indexOf("/>", masterIndex + 1));
            masterIndex = aXMLString.indexOf("/>", masterIndex + 1) + 2;

            IAssociatedEvent ae = factory.createAssociatedEvent();

            // If there is an associationEngine
            index = tempString.indexOf("associationEngine=\"");
            if (index != -1) {
                IAssociationEngine anEngine = factory.createAssociationEngine();
                anEngine.setId(tempString.substring(index + 19, tempString.indexOf("\"", index + 19)));
                ae.setAssociationEngine(anEngine);
            }

            index = tempString.indexOf("resolvedEvents=\"");
            if (index != -1) {
                String eventList = new String(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)));
                StringTokenizer st = new StringTokenizer(eventList, " ");
                while (st.hasMoreTokens()) {
                    ICommonBaseEvent aCBE = factory.createCommonBaseEvent();
                    aCBE.setGlobalInstanceId(st.nextToken());
                    ae.addResolvedEvent(aCBE.getGlobalInstanceId());
                }

                index = index + eventList.length() + 16;
            }

            event.addAssociatedEvent(ae);
            masterIndex = aXMLString.indexOf("/>", masterIndex + 1) + 2;
        }

        /* if reporterComponentId is included... */
        if (aXMLString.indexOf("<reporterComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = aXMLString.indexOf("<reporterComponentId");
            tempString = aXMLString.substring(masterIndex, aXMLString.indexOf("/>", masterIndex + 1));
            masterIndex = aXMLString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification rci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf("application=\"");
            if (index != -1) {
                rci.setApplication(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)));
            }

            index = tempString.indexOf("component=\"");
            if (index != -1) {
                rci.setComponent(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)));
            }

            index = tempString.indexOf("componentIdType=\"");
            if (index != -1) {
                rci.setComponentIdType(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)));
            }

            index = tempString.indexOf("executionEnvironment=\"");
            if (index != -1) {
                rci.setExecutionEnvironment(tempString.substring(index + 22, tempString.indexOf("\"", index + 22)));
            }

            index = tempString.indexOf("instanceId=\"");
            if (index != -1) {
                rci.setInstanceId(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)));
            }

            index = tempString.indexOf("location=\"");
            if (index != -1) {
                rci.setLocation(tempString.substring(index + 10, tempString.indexOf("\"", index + 10)));
            }

            index = tempString.indexOf("locationType=\"");
            if (index != -1) {
                rci.setLocationType(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)));
            }

            index = tempString.indexOf("processId=\"");
            if (index != -1) {
                rci.setProcessId(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)));
            }

            index = tempString.indexOf("subComponent=\"");
            if (index != -1) {
                rci.setSubComponent(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)));
            }

            index = tempString.indexOf("threadId=\"");
            if (index != -1) {
                rci.setThreadId(tempString.substring(index + 10, tempString.indexOf("\"", index + 10)));
            }

            event.setReporterComponentId(rci);
        }

        /* if sourceComponentId is included... */
        if (aXMLString.indexOf("<sourceComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = aXMLString.indexOf("<sourceComponentId");
            tempString = aXMLString.substring(masterIndex, aXMLString.indexOf("/>", masterIndex + 1));
            masterIndex = aXMLString.indexOf("/>", masterIndex + 1) + 2;

            IComponentIdentification sci = factory.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf("application=\"");
            if (index != -1) {
                sci.setApplication(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)));
            }

            index = tempString.indexOf("component=\"");
            if (index != -1) {
                sci.setComponent(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)));
            }

            index = tempString.indexOf("componentIdType=\"");
            if (index != -1) {
                sci.setComponentIdType(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)));
            }

            index = tempString.indexOf("executionEnvironment=\"");
            if (index != -1) {
                sci.setExecutionEnvironment(tempString.substring(index + 22, tempString.indexOf("\"", index + 22)));
            }

            index = tempString.indexOf("instanceId=\"");
            if (index != -1) {
                sci.setInstanceId(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)));
            }

            index = tempString.indexOf("location=\"");
            if (index != -1) {
                sci.setLocation(tempString.substring(index + 10, tempString.indexOf("\"", index + 10)));
            }

            index = tempString.indexOf("locationType=\"");
            if (index != -1) {
                sci.setLocationType(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)));
            }

            index = tempString.indexOf("processId=\"");
            if (index != -1) {
                sci.setProcessId(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)));
            }

            index = tempString.indexOf("subComponent=\"");
            if (index != -1) {
                sci.setSubComponent(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)));
            }

            index = tempString.indexOf("threadId=\"");
            if (index != -1) {
                sci.setThreadId(tempString.substring(index + 10, tempString.indexOf("\"", index + 10)));
            }

            event.setSourceComponentId(sci);
        }

        /* if msgDataElement is included... */
        if (aXMLString.indexOf("<msgDataElement") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = aXMLString.indexOf("<msgDataElement");
            tempString = aXMLString.substring(masterIndex, aXMLString.indexOf("</msgDataElement>", masterIndex + 1));
            masterIndex = aXMLString.indexOf("</msgDataElement>", masterIndex + 1) + 17;

            IMsgDataElement mde = factory.createMsgDataElement();

            /* populate the attributes for this element... */
            index = tempString.indexOf("msgLocale=\"");
            if (index != -1) {
                mde.setMsgLocale(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)));
            }

            // If there are tokens to process...
            if (tempString.indexOf("<msgCatalogTokens ", index) != -1) {
                Vector mct = new Vector();

                // For each token...
                while (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                    /* process the attribute for this subelement */
                    index = tempString.indexOf("value=\"", index);
                    if (index != -1) {
                        mct.addElement(new String(tempString.substring(index + 7, tempString.indexOf("\"", index + 7))));
                    }

                    index = tempString.indexOf("/>", index + 1) + 2;
                }
                String[] tokenArray = new String[mct.size()];
                mct.toArray(tokenArray);
                mde.setMsgCatalogTokens(tokenArray);
            }

            index = tempString.indexOf("<msgCatalogType>");
            if (index != -1) {
                mde.setMsgCatalogType(tempString.substring(index + 16, tempString.indexOf("</msgCatalogType>", index + 16)));
            }

            index = tempString.indexOf("<msgId>");
            if (index != -1) {
                mde.setMsgId(tempString.substring(index + 7, tempString.indexOf("</msgId>", index + 7)));
            }

            index = tempString.indexOf("<msgIdType>");
            if (index != -1) {
                mde.setMsgIdType(tempString.substring(index + 11, tempString.indexOf("</msgIdType>", index + 11)));
            }

            index = tempString.indexOf("<msgCatalogId>");
            if (index != -1) {
                mde.setMsgCatalogId(tempString.substring(index + 14, tempString.indexOf("</msgCatalogId>", index + 14)));
            }

            index = tempString.indexOf("<msgCatalog>");
            if (index != -1) {
                mde.setMsgCatalog(tempString.substring(index + 12, tempString.indexOf("</msgCatalog>", index + 12)));
            }

            event.setMsgDataElement(mde);
        }

        return;
    }

    /**
     * Internalizes a passed XML document String into a passed CommonBaseEventImpl object.
     * 
     * @param event  The CommonBaseEventImpl object to populate.
     * @param aXMLDocString
     *               The XML document containing the information used to populate the Common Base Event object.
     * @since V 1.0
     */
    public static synchronized void fromCanonicalXMLDocString(ICommonBaseEvent event, String aXMLDocString) {

        int index = 0;
        index = aXMLDocString.indexOf("xsi:noNamespaceSchemaLocation=\"");
        if (index != -1) {
            incomingCBEVersion = new String(aXMLDocString.substring(index + 31, aXMLDocString.indexOf("\"", index + 31)));
        }

        fromCanonicalXMLString(event, aXMLDocString);
    }

    /**
     * Internalizes a passed XML fragment String into a passed IAssociationEngine object.
     * 
     * @param engine     The IAssociationEngine object to populate.
     * @param aXMLString
     *                   The XML fragment containing the information used to populate the Association Engine object.
     * @since V 1.0
     */
    public static synchronized void fromCanonicalXMLString(IAssociationEngine engine, String aXMLString) {

        int index = 0;
        index = aXMLString.indexOf("id=\"");
        if (index != -1) {
            engine.setId(aXMLString.substring(index + 4, aXMLString.indexOf("\"", index + 4)));
        }

        index = aXMLString.indexOf("name=\"");
        if (index != -1) {
            engine.setName(aXMLString.substring(index + 6, aXMLString.indexOf("\"", index + 6)));
        }

        index = aXMLString.indexOf("type=\"");
        if (index != -1) {
            engine.setType(aXMLString.substring(index + 6, aXMLString.indexOf("\"", index + 6)));
        }
    }

    /**
     * Internalizes a passed XML document String into a passed IAssociationEngine object.
     * 
     * @param engine     The IAssociationEngine object to populate.
     * @param aXMLDocString
     *                   The XML document containing the information used to populate the Association Engine object.
     * @since V 1.0
     */
    public static synchronized void fromCanonicalXMLDocString(IAssociationEngine engine, String aXMLDocString) {

        int index = 0;
        index = aXMLDocString.indexOf("xsi:noNamespaceSchemaLocation=\"");
        if (index != -1) {
            incomingCBEVersion = aXMLDocString.substring(index + 31, aXMLDocString.indexOf("\"", index + 31));
        }

        fromCanonicalXMLString(engine, aXMLDocString);
    }

    /**
     * 3.2.7.1 Lexical representation
     *
     * A single lexical representation, which is a subset of the lexical representations allowed by
     * [ISO 8601], is allowed for dateTime. This lexical representation is the [ISO 8601] extended
     * format CCYY-MM-DDThh:mm:ss where "CC" represents the century, "YY" the year, "MM" the month
     * and "DD" the day, preceded by an optional leading "-" sign to indicate a negative number. If
     * the sign is omitted, "+" is assumed. The letter "T" is the date/time separator and "hh",
     * "mm", "ss" represent hour, minute and second respectively. Additional digits can be used
     * to increase the precision of fractional seconds if desired i.e the format ss.ss... with any
     * number of digits after the decimal point is supported. The fractional seconds part is optional;
     * other parts of the lexical form are not optional. To accommodate year values greater than 9999
     * additional digits can be added to the left of this representation. Leading zeros are required
     * if the year value would otherwise have fewer than four digits; otherwise they are forbidden.
     * The year 0000 is prohibited.
     *
     * The CCYY field must have at least four digits, the MM, DD, SS, hh, mm and ss fields exactly
     * two digits each (not counting fractional seconds); leading zeroes must be used if the field
     * would otherwise have too few digits.
     *
     * This representation may be immediately followed by a "Z" to indicate Coordinated Universal
     * Time (UTC) or, to indicate the time zone, i.e. the difference between the local time and
     * Coordinated Universal Time, immediately followed by a sign, + or -, followed by the difference
     * from UTC represented as hh:mm (note: the minutes part is required). See ISO 8601 Date and
     * Time Formats (D) for details about legal values in the various fields. If the time zone is
     * included, both hours and minutes must be present.
     *
     * For example, to indicate 1:20 pm on May the 31st, 1999 for Eastern Standard Time which is
     * 5 hours behind Coordinated Universal Time (UTC), one would write: 1999-05-31T13:20:00-05:00.
     */
    public static long convertXmlSchemaDateTimeToDate(String dateTime) {
        String dateString = null;
        String subSecondsString = null;
        String utcOffsetString = null;
        char utcOffsetDirection = '\0';
        Date date = null;
        long milliseconds = 0;
        int utcOffsetStringIndex = -1;
        int subSecondsStringIndex;

        // define a simple date format object
        String dateFormat = "yyyy-MM-dd'T'HH:mm:ss";
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(dateFormat);
        int dateFormatLength = dateFormat.length() - 2;

        try {

            if (dateTime.length() < dateFormatLength) {
                milliseconds = 0;
            }
            else {

                if (dateTime.indexOf('Z') > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('Z');

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    utcOffsetDirection = 'Z';
                }
                else if (dateTime.indexOf('+') > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('+');

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    // get the offset string
                    utcOffsetDirection = dateTime.charAt(utcOffsetStringIndex);
                    utcOffsetString = dateTime.substring(utcOffsetStringIndex + 1);
                }
                else if (dateTime.indexOf('-', dateFormatLength) > dateFormatLength) {
                    utcOffsetStringIndex = dateTime.indexOf('-', dateFormatLength);

                    // adjust dateTime to remove timezone information
                    dateString = dateTime.substring(0, utcOffsetStringIndex);

                    // get the offset string
                    utcOffsetDirection = dateTime.charAt(utcOffsetStringIndex);
                    utcOffsetString = dateTime.substring(utcOffsetStringIndex + 1);
                }
                else {
                    dateString = dateTime;
                }

                subSecondsStringIndex = dateString.indexOf('.');

                if (subSecondsStringIndex > 0) {
                    // adjust dateTime to remove timezone information
                    if (utcOffsetStringIndex != -1) {
                        dateString = dateTime.substring(0, utcOffsetStringIndex);
                    }

                    // get the subSeconds string
                    // make sure to add 1 to the offset so we can skip the .
                    subSecondsString = dateString.substring(subSecondsStringIndex + 1);
                }

                if (dateString != null) {
                    String year = dateString.substring(0, 4);

                    if (Integer.parseInt(year) > 1969) {
                        // create a the GMT timezone 
                        SimpleTimeZone gmt = new SimpleTimeZone(0, "UTC");

                        // create a GregorianCalendar with the GMT time zone
                        Calendar calendar = new GregorianCalendar(gmt);
                        ParsePosition pos = new ParsePosition(0);
                        simpleDateFormat.setCalendar(calendar);
                        date = simpleDateFormat.parse(dateString, pos);
                    }

                }

                if (date != null) {
                    // now get the milliseconds from the date
                    milliseconds = date.getTime();

                    // now convert microseconds string into a long and add it to the milliseconds in the date
                    if ((subSecondsString != null) && (subSecondsString.length() > 0)) {
                        long subSeconds = Long.parseLong(subSecondsString);
                        int subSecondsLength = subSecondsString.length();
                        if (subSecondsLength < 4) {

                            for (int i = 1; i < (4 - subSecondsLength); i++) {
                                subSeconds = subSeconds * 10;
                            }
                        }
                        else {
                            for (int i = 1; i < (4 - subSecondsLength); i++) {
                                subSeconds = subSeconds / 10;
                            }
                        }

                        milliseconds = milliseconds + subSeconds;
                    }

                    // now depending on the offset adjust the time to UTC   
                    if ((utcOffsetDirection != '\0') && (utcOffsetString != null) && (utcOffsetString.length() == 5)) {
                        long offsetMinutes = Long.parseLong(utcOffsetString.substring(0, 2));
                        long offsetSeconds = Long.parseLong(utcOffsetString.substring(3));

                        if (utcOffsetDirection == '+') {
                            milliseconds = milliseconds + (offsetMinutes * 60 * 1000) + (offsetSeconds * 1000);
                        }
                        else if (utcOffsetDirection == '-') {
                            milliseconds = milliseconds - (offsetMinutes * 60 * 1000) - (offsetSeconds * 1000);
                        }
                    }
                }
            }
        }
        catch (Exception exp) {
            //            exp.printStackTrace();
            milliseconds = 0;
        }

        return milliseconds;
    }

    /**
     * Converts a long representing UTC in milliseconds to the XML Schema datetime format
     *
     * @param date the date in milliseconds
     *
     * @return String the date in the XML Schema datetime format
     */
    public static String convertDateToXmlSchemaDateTime(long date) {
        String dateTime = null;

        // create a the GMT timezone 
        SimpleTimeZone gmt = new SimpleTimeZone(0, "UTC");

        // create a GregorianCalendar with the GMT time zone
        Calendar calendar = new GregorianCalendar(gmt);

        // set the date in the calendar
        calendar.setTime(new Date(date));

        // get the interger representation for all the fields
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);

        // java has January as month 0 so need add 1 to the month
        month++;

        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int minute = calendar.get(Calendar.MINUTE);
        int second = calendar.get(Calendar.SECOND);
        int milliseconds = calendar.get(Calendar.MILLISECOND);

        // now create a string buffer to build the string
        // if the fields are not the correct size they must be padded with 0's
        StringBuffer stringBuffer = new StringBuffer(35);
        stringBuffer.append(year);
        stringBuffer.append('-');

        if (month < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(month);
        stringBuffer.append('-');

        if (day < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(day);
        stringBuffer.append('T');

        if (hour < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(hour);
        stringBuffer.append(':');

        if (minute < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(minute);
        stringBuffer.append(':');

        if (second < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(second);

        stringBuffer.append(".");

        stringBuffer.append(milliseconds);

        // are times are always UTC so append a Z
        stringBuffer.append("Z");
        dateTime = stringBuffer.toString();

        return dateTime;
    }

    /**
     * Returns the CommonBaseEvent XML schema version
     * @return String
     */
    public final static String getEventFormatterVersion() {
        return xml_version;
    }

    private static IExtendedDataElement loadIExtendedDataElements(String xml) {

        IExtendedDataElement extendedDataElement = factory.createExtendedDataElement();

        int startIndex = xml.indexOf(" name=\"");
        int endIndex = -1;

        //Retrieve and set the 'name' property of the IExtendedDataElement:
        if ((startIndex != -1) && ((endIndex = xml.indexOf('"', (startIndex + 7))) != -1)) //7 is the length of " name="".
            extendedDataElement.setName(xml.substring((startIndex + 7), endIndex)); //7 is the length of " name="".

        startIndex = xml.indexOf(" type=\"", endIndex);

        //Retrieve and set the 'type' property of the IExtendedDataElement:
        if ((startIndex != -1) && ((endIndex = xml.indexOf('"', (startIndex + 7))) != -1)) //7 is the length of " type="".
            extendedDataElement.setType(xml.substring((startIndex + 7), endIndex)); //7 is the length of " type="".

        //ASSUMPTION:  Based on the serialization API (e.g. getIExtendedDataElementXML()) for IExtendedDataElements, the <children> elements are serialized before the <values> or <hexValue> elements.

        //Counter of nested <children ...>...</children> elements:
        int nestedLevels = 0;
        char xmlMarkupChar;

        //Retrieve and set the 'children' properties of the IExtendedDataElement:
        while ((startIndex = xml.indexOf("<children ", endIndex)) != -1) {

            //Ensure there is a </children> end tag:
            if (xml.indexOf("</children>", startIndex) == -1)
                break;

            //Advance the end index to the start of the <children> tag:
            endIndex = startIndex;

            //Reset the nested level counter:
            nestedLevels = 0;

            //Find the matching </children> end tag (e.g. endIndex), keeping in mind the possibility of nested <children> and </children> tags:
            while ((nestedLevels >= 0) && ((endIndex = xml.indexOf("children", (endIndex + 8))) != -1)) { //8 is the length of "children".

                xmlMarkupChar = xml.charAt(endIndex - 1);

                if (xmlMarkupChar == '/')
                    nestedLevels--;

                else if (xmlMarkupChar == '<')
                    nestedLevels++;
            }

            //Add any found childDataElements to the IExtendedDataElement:
            if (nestedLevels < 0) {

                //Back up to the beginning of the </children> end tag (e.g. "</"):
                endIndex -= 2;

                extendedDataElement.addChildDataElement(loadIExtendedDataElements(xml.substring(startIndex, endIndex)));
            }
        }

        //Retrieve and set the 'values' properties of the IExtendedDataElement:
        if (xml.indexOf("<values>", endIndex) != -1) {

            ArrayList values = new ArrayList();

            //Process each <values> element:
            while ((startIndex = xml.indexOf("<values>", endIndex)) != -1) {

                if ((startIndex != -1) && ((endIndex = xml.indexOf("</values>", startIndex)) != -1))
                    values.add(xml.substring((startIndex + 8), endIndex)); //8 is the length of "<values>".
            }

            if (values.size() > 0)
                extendedDataElement.setValues(((String[]) (values.toArray(new String[values.size()]))));
        }
        //Otherwise, retrieve and set the 'hexValue' property of the IExtendedDataElement:
        else if (((startIndex = xml.indexOf("<hexValue>", endIndex)) != -1) && ((endIndex = xml.indexOf("</hexValue>", startIndex)) != -1)) {

            //The hex string is a contiguous series of zero padded hex tuples (e.g. 0F):
            String hexString = new String(xml.substring((startIndex + 10), endIndex)); //10 is the length of "<hexValue>".

            //Each byte is one zero padded (if necessary) hex tuple (e.g. 0F or A4):
            int numberOfBytes = ((int) (hexString.length() / 2));
            byte[] byteArray = new byte[numberOfBytes];

            for (int i = 0; i < numberOfBytes; i++)
                byteArray[i] = Byte.parseByte(hexString.substring((i * 2), ((i * 2) + 2)), 16);

            extendedDataElement.setHexBinary(byteArray);
        }

        return extendedDataElement;
    }
}
