/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.events;

import org.eclipse.hyades.logging.events.exceptions.MissingValueException;
import org.eclipse.hyades.logging.events.exceptions.ValueTooLongException;

/**
 * Utility class to create event items.
 */
public final class EventItemsFactory
{

    /** Create a new IComponentIdentification item.
     * 
     * @param componentIdType	The format of the component. The currently defined
     * 							keywords for this property are:
     * 
     * 							<ul>
     * 							<li>ProductName</li>
     *							<li>DeviceName</li>
     *							<li>SystemName</li>
     * 							<li>ServiceName</li>
     * 							<li>Process</li>
     * 							<li>Application</li>
     * 							<li>Unknown</li>
     * 							</ul>
     * 
     *							<p>The value MUST NOT exceed 32 bytes in length.
     *							   The default value is 'Unknown'.</p>
     * 
     * @param component	The logical name of a component. This property MUST
     * 						contain the name of a particular application, product,
     * 						or subsystem (e.g., IBM DB2# V7.1), and SHOULD also
     *						be unique within the scope specified by the location.
     *						Component version information may be appended to the
     *						end of this value, separated by a '#' character. The
     *						value MUST NOT exceed 255 bytes in length.
     * 
     * @param subComponent	The logical name of a subcomponent within a component.
     *						This property can be one of the various parts of an
     *						application or OS resource e.g., a module name, class
     * 						and method name, etc. It should be the most granular
     *						definition specified in the event. Subcomponent version
     *						information may be appended to the end of this value,
     *						separated by a '#' character. The value MUST NOT exceed
     *						255 bytes in length.
     *
     * @param locationType	The format of the location. The currently defined keywords
     * 						for this property are:
     * 
     * 						<ul>
     * 						<li>Unknown</li>
     * 						<li>IPV4</li>
     * 						<li>IPV6</li>
     * 						<li>NWA</li>
     * 						<li>ISDN</li>
     * 						<li>ICD</li>
     * 						<li>OID/OSI</li>
     * 						<li>Dial</li>
     * 						<li>HWA</li>
     * 						<li>HID</li>
     * 						<li>X25</li>
     * 						<li>DCC</li>
     * 						<li>SNA</li>
     * 						<li>IPX</li>
     * 						<li>E.164</li>
     * 						<li>Hostname</li>
     * 						<li>FQHostname</li>
     * 						<li>Devicename</li>
     * 						</ul>
     * 
     *						<p>The value MUST NOT exceed 32 bytes in length.
     *						   The default value is 'Unknown'.</p>
     * 
     * @param location		The physical address where an appliction is running, such
     * 						as a hostname, IP address, or VTAM LU. The format should
     * 						match the type specified by <code>locationType</code>.
     * 						The value MUST NOT exceed 255 bytes in length.
     * 
     * @param application	The optional name of the application (e.g, "myWebApp").
     * 						Application version information may be appended to the
     *						end of this value, separated by a '#' character. The
     *						value MUST NOT exceed 255 bytes in length.
     * 
     * @param executionEnv	The optional name of the environment an application is
     * 						running in. For example, a WebSphere Application Server
     *						might use "name:cell:node:server". Execution environment
     *						version information may be appended to the end of this
     *						value, separated by a '#' character. The value MUST NOT
     *						exceed 255 bytes in length.
     *
     * @param instanceId	An optional identifier for the currently running instance
     * 						(such as an EJB handle or Grid Service Handle). The value
     * 						MUST NOT exceed 128 bytes in length.
     * 
     * @param processId	An optional identifier for the currently running process.
     * 						The value MUST NOT exceed 64 bytes in length.
     * 
     * @param threadId		An optional identifier for the current thread within the
     * 						process. The value MUST NOT exceed 64 bytes in length.
     * 
     *  @return	The new <code>IComponentIdentification</code> object.
     * 
     * @throws MissingValueException
     * 
     * Thrown if any of the required component, subComponent, componentIdType,
     * location, or locationType parameters are not specified.
     *
     * @throws ValueTooLongException
     * 
     * Thrown if any parameter's value exceeds the maximum length specified above. 
     **/

    public static IComponentIdentification createIComponentIdentification(
        String componentIdType,
        String component,
        String subComponent,
        String locationType,
        String location,
        String application,
        String executionEnv,
        String instanceId,
        String processId,
        String threadId)
        throws MissingValueException, ValueTooLongException
    {
        final int COMP_ID_TYPE_MAX_LEN = 32;
        final int COMP_MAX_LEN = 255;
        final int SUBCOMP_MAX_LEN = 255;
        final int LOC_TYPE_MAX_LEN = 32;
        final int LOCATION_MAX_LEN = 255;
        final int APPLICATION_MAX_LEN = 255;
        final int EXEC_ENV_MAX_LEN = 255;
        final int INSTANCE_ID_MAX_LEN = 128;
        final int PROCESS_ID_MAX_LEN = 64;
        final int THREAD_ID_MAX_LEN = 64;

        // Make sure required attributes are specified.

        assertRequiredAttributeSpecified("componentIdType", componentIdType);
        assertRequiredAttributeSpecified("component", component);
        assertRequiredAttributeSpecified("subComponent", subComponent);
        assertRequiredAttributeSpecified("locationType", locationType);
        assertRequiredAttributeSpecified("location", location);

        // Make sure attribute values do not exceed maximum allowed lengths

        assertStringValueNotLongerThan(
            "componentIdType",
            componentIdType,
            COMP_ID_TYPE_MAX_LEN);
        assertStringValueNotLongerThan("component", component, COMP_MAX_LEN);
        assertStringValueNotLongerThan(
            "subComponent",
            subComponent,
            SUBCOMP_MAX_LEN);
        assertStringValueNotLongerThan(
            "locationType",
            locationType,
            LOC_TYPE_MAX_LEN);
        assertStringValueNotLongerThan("location", location, LOCATION_MAX_LEN);
        assertStringValueNotLongerThan(
            "application",
            application,
            APPLICATION_MAX_LEN);
        assertStringValueNotLongerThan(
            "executionEnv",
            executionEnv,
            EXEC_ENV_MAX_LEN);
        assertStringValueNotLongerThan(
            "instanceId",
            instanceId,
            INSTANCE_ID_MAX_LEN);
        assertStringValueNotLongerThan(
            "processId",
            processId,
            PROCESS_ID_MAX_LEN);
        assertStringValueNotLongerThan("threadId", threadId, THREAD_ID_MAX_LEN);

        IComponentIdentification compId = new ComponentIdentificationImpl();

        compId.setComponentIdType(componentIdType);
        compId.setComponent(component);
        compId.setSubComponent(subComponent);
        compId.setLocationType(locationType);
        compId.setLocation(location);

        if (isOptionalAttributeSpecified(executionEnv))
        {
            compId.setExecutionEnvironment(executionEnv);
        }

        if (isOptionalAttributeSpecified(instanceId))
        {
            compId.setInstanceId(instanceId);
        }

        if (isOptionalAttributeSpecified(processId))
        {
            compId.setProcessId(processId);
        }

        if (isOptionalAttributeSpecified(threadId))
        {
            compId.setThreadId(threadId);
        }

        return compId;

    } /* end method createIComponentIdentification() */

    /** Create a new IContextDataElement item (using a context Id).
       * 
       * @param name		The Name of the application that is creating
       *					this context data element.
       *
       * @param type		The data type of the contextValue property.
       * 
       * @param id			A reference to the element that contains the context.
       * 
       * @return			A new IContextDataElement with the specified settings.
       * 
       * @throws MissingValueException
       * 
       * Thrown if the element name, type, or id is not specified.
       **/

    public static IContextDataElement createIContextDataElementById(
        String name,
        String type,
        String id)
        throws MissingValueException
    {
        assertRequiredAttributeSpecified("name", name);
        assertRequiredAttributeSpecified("type", type);
        assertRequiredAttributeSpecified("id", id);

        IContextDataElement ctxData = new ContextDataElementImpl();

        ctxData.setName(name);
        ctxData.setType(type);
        ctxData.setContextId(id);

        return ctxData;

    } /* end method createIContextDataElementById() */

    /** Create a new IContextDataElement item (using a context value).
     *
     * @param name		The Name of the application that is creating
     *					this context data element.
     *
     * @param type		The data type of the contextValue property.
     * 
     * @param value	The value of the context with respect to the
     * 					context	implementation.
     * 
     * @return		A new IContextDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name, type, or value is not specified.
     **/

    public static IContextDataElement createIContextDataElementByValue(
        String name,
        String type,
        String value)
        throws MissingValueException
    {
        assertRequiredAttributeSpecified("name", name);
        assertRequiredAttributeSpecified("type", type);
        assertRequiredAttributeSpecified("value", value);

        IContextDataElement ctxData = new ContextDataElementImpl();

        ctxData.setName(name);
        ctxData.setType(type);
        ctxData.setContextValue(value);

        return ctxData;

    } /* end method createIContextDataElementByValue() */

    /** Create a new IAssociatedEvent item. Also starts the
     *  specified association engine (if not already running).
     * 
     *	@param associationEngine	The association engine.
     *
     *	@param resolvedEvents 		An array of events associated with this one.
     * 
     *  @return	The new <code>IAssociatedEvent</code> object.
     * 
     *  @throws MissingValueException
     * 
     * Thrown if the association engine or resolved events are not specified.
     **/

    public static IAssociatedEvent createIAssociatedEvent(
        IAssociationEngine associationEngine,
        ICommonBaseEvent resolvedEvents[])
        throws MissingValueException
    {
        if (associationEngine == null)
        {
            throw new MissingValueException("associationEngine");
        }
        else if (resolvedEvents == null)
        {
            throw new MissingValueException("resolvedEvents");
        }

        AssociatedEventImpl associatedEvent = new AssociatedEventImpl();

        for (int i = 0; i < resolvedEvents.length; i++)
        {
            if (resolvedEvents[i] != null)
            {
                associatedEvent.addResolvedEvent(resolvedEvents[i].getGlobalInstanceId());
            }
        }

        associatedEvent.setAssociationEngine(associationEngine);

        return associatedEvent;
    }

    /** Create a new IMsgDataElement item.
     * 
     *  @param msgIdType	The format type used by msgId, such as "IBM", "JMX",
     *						"DottedName", etc. The type MUST NOT exceed 32 bytes.
     *
     *	@param msgId		A unique message identifier, such as "DBT1234E",
     * 						etc. This identifier MUST NOT exceed 255 bytes.
     *
     *	@param msgLocale	An optional locale code used by the message. This
     * 						should be an ISO/IEC 639.1 language code, followed by
     * 						an optional underscore and ISO/IEC 3166 country code.
     * 						The locale MUST NOT exceed 5 bytes.
     *
     *  @param msgCatalog	An optional name of a catalog containing translations
     * 						of the message text. The name MUST NOT exceed 128 bytes.
     * 
     *  @param msgCatalogType	The type of message catalog, such as "Java" or "XPG".
     * 							This is required if a message catalog is specified.
     *  
     *  @param msgCatalogId	The index or identifier for the message within a
     * 							message catalog. This is required if a message
     * 							catalog is specified.
     * 
     *  @param msgCatalogTokens	An optional array of string values to be substituted
     * 								into the formatted message text at display time.
     * 
     *  @return	The new <code>IMsgDataElement</code> object.
     * 
     *  @throws MissingValueException
     * 
     * Thrown if the msgId or msgIdType are not specified; or if a message catalog
     * is specified but not the catalog type or catalog id; or if a catalog type or
     * catalog id is specified but not a message catalog.
     * 
     * @throws ValueTooLongException
     * 
     * Thrown if any parameter's value exceeds the maximum length specified above. 
     **/
    public static IMsgDataElement createIMsgDataElement(
        String msgIdType,
        String msgId,
        String msgLocale,
        String msgCatalog,
        String msgCatalogType,
        String msgCatalogId,
        String msgCatalogTokens[])
        throws MissingValueException, ValueTooLongException
    {
        final int MSG_ID_TYPE_MAX_LEN = 32;
        final int MSG_ID_MAX_LEN = 255;
        final int MSG_LOCALE_MAX_LEN = 5;
        final int MSG_CATALOG_MAX_LEN = 128;

        // Make sure required msgIdType and msgId are specified.

        assertRequiredAttributeSpecified("msgIdType", msgIdType);
        assertRequiredAttributeSpecified("msgId", msgId);

        // Make sure optional catalog/catalog type/catalog id
        // are specified together or not at all.

        assertAttributesSpecifiedTogether(
            "msgCatalog",
            msgCatalog,
            "msgCatalogType",
            msgCatalogType);
        assertAttributesSpecifiedTogether(
            "msgCatalog",
            msgCatalog,
            "msgCatalogId",
            msgCatalogId);

        // Make sure attribute values do not exceed maximum allowed lengths

        assertStringValueNotLongerThan(
            "msgIdType",
            msgIdType,
            MSG_ID_TYPE_MAX_LEN);
        assertStringValueNotLongerThan("msgId", msgId, MSG_ID_MAX_LEN);
        assertStringValueNotLongerThan(
            "msgLocale",
            msgLocale,
            MSG_LOCALE_MAX_LEN);
        assertStringValueNotLongerThan(
            "msgCatalog",
            msgCatalog,
            MSG_CATALOG_MAX_LEN);

        IMsgDataElement msgData = new MsgDataElementImpl();

        msgData.setMsgIdType(msgIdType);
        msgData.setMsgId(msgId);

        if (isOptionalAttributeSpecified(msgLocale))
        {
            msgData.setMsgLocale(msgLocale);
        }

        if (isOptionalAttributeSpecified(msgCatalog))
        {
            msgData.setMsgCatalog(msgCatalog);
            msgData.setMsgCatalogType(msgCatalogType);
            msgData.setMsgCatalogId(msgCatalogId);
        }

        if (msgCatalogTokens != null)
        {
            msgData.setMsgCatalogTokens(msgCatalogTokens);
        }

        return msgData;

    } /* end method IMsgDataElement */

    /** Create a new IExtendedDataElement item from a single boolean value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param boolValue	The boolean value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        boolean boolValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.BOOLEAN_DATA_TYPE,
                children);

        String stringVals[] = new String[] { new Boolean(boolValue).toString()};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single byte value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param byteValue	The byte value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        byte byteValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.BYTE_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Byte.toString(byteValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single short integer value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param shortValue	The short integer value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        short shortValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.SHORT_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Short.toString(shortValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single integer value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param intValue		The integer value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        int intValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.INTEGER_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Integer.toString(intValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single long integer value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param longValue	The long integer value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        long longValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.LONG_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Long.toString(longValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single floating-point value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param floatValue	The floating-point value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        float floatValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.FLOAT_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Float.toString(floatValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a double-precision value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param doubleValue	The double-precision value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        double doubleValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.DOUBLE_DATA_TYPE,
                children);

        String stringVals[] = new String[] { Double.toString(doubleValue)};

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single string value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param stringValue	The string value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the string value is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        String stringValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        assertRequiredAttributeSpecified("stringValue", stringValue);

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.STRING_DATA_TYPE,
                children);

        String stringVals[] = new String[] { stringValue };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from a single date/time value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param dateTimeValue	The date/time value for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the date/time value is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        java.sql.Date dateTimeValue,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (dateTimeValue == null)
        {
            throw new MissingValueException("dateTimeValue");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.DATETIME_DATA_TYPE,
                children);

        String dateTimeString =
            CbeFormatter.convertDateToXmlSchemaDateTime(
                dateTimeValue.getTime());

        String stringVals[] = new String[] { dateTimeString };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of boolean values.
      * 
      * @param name			The name of the extended data element. This name
      * 						MUST be	unique with respect to all other fields
      *							in the event.
      * 
      * @param boolValues		An array of boolean values for the element.
      * 
      * @param children		An optional list of child elements for this item.
      * 
      * @return		A new IExtendedDataElement with the specified settings.
      * 
      * @throws MissingValueException
      * 
      * Thrown if the element name or the array of values is not specified.  
      **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        boolean boolValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (boolValues == null)
        {
            throw new MissingValueException("boolValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.BOOLEAN_ARRAY_DATA_TYPE,
                children);

        final int nValues = boolValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = new Boolean(boolValues[i]).toString();
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of byte values.
      * 
      * @param name			The name of the extended data element. This name
      * 						MUST be	unique with respect to all other fields
      *							in the event.
      * 
      * @param isHexBinary		A flag to indicate whether this element should be
      * 						of type IExtendedDataElement.HEX_BINARY_TYPE, or
      * 						type IExtendedDataElement.BYTE_ARRAY_TYPE.
      * 
      * @param byteValues		An array of byte values for the element.
      * 
      * @param children		An optional list of child elements for this item.
      * 
      * @return		A new IExtendedDataElement with the specified settings.
      * 
      * @throws MissingValueException
      * 
      * Thrown if the element name or the array of values is not specified.  
      **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        boolean isHexBinary,
        byte byteValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (byteValues == null)
        {
            throw new MissingValueException("byteValues");
        }

        if (isHexBinary)
        {
            IExtendedDataElement element =
                getEmptyIExtendedDataElement(
                    name,
                    IExtendedDataElement.HEXBINARY_DATA_TYPE,
                    children);

            element.setHexValue(byteValues);

            return element;
        }
        else
        {
            IExtendedDataElement element =
                getEmptyIExtendedDataElement(
                    name,
                    IExtendedDataElement.BYTE_ARRAY_DATA_TYPE,
                    children);

            final int nValues = byteValues.length;

            String stringVals[] = new String[nValues];

            for (int i = 0; i < nValues; i++)
            {
                stringVals[i] = Byte.toString(byteValues[i]);
            };

            element.setValues(stringVals);

            return element;
        }

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of short integer values.
      * 
      * @param name			The name of the extended data element. This name
      * 						MUST be	unique with respect to all other fields
      *							in the event.
      * 
      * @param shortValues		An array of short integer values for the element.
      * 
      * @param children		An optional list of child elements for this item.
      * 
      * @return		A new IExtendedDataElement with the specified settings.
      * 
      * @throws MissingValueException
      * 
      * Thrown if the element name or the array of values is not specified.  
      **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        short shortValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (shortValues == null)
        {
            throw new MissingValueException("shortValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.SHORT_ARRAY_DATA_TYPE,
                children);

        final int nValues = shortValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = Short.toString(shortValues[i]);
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of integer values.
      * 
      * @param name			The name of the extended data element. This name
      * 						MUST be	unique with respect to all other fields
      *							in the event.
      * 
      * @param intValues		An array of integer values for the element.
      * 
      * @param children		An optional list of child elements for this item.
      * 
      * @return		A new IExtendedDataElement with the specified settings.
      * 
      * @throws MissingValueException
      * 
      * Thrown if the element name or the array of values is not specified.  
      **/
    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        int intValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (intValues == null)
        {
            throw new MissingValueException("intValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.INTEGER_ARRAY_DATA_TYPE,
                children);

        final int nValues = intValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = Integer.toString(intValues[i]);
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of long integer values.
      * 
      * @param name			The name of the extended data element. This name
      * 						MUST be	unique with respect to all other fields
      *							in the event.
      * 
      * @param longValues		An array of long integer values for the element.
      * 
      * @param children		An optional list of child elements for this item.
      * 
      * @return		A new IExtendedDataElement with the specified settings.
      * 
      * @throws MissingValueException
      * 
      * Thrown if the element name or the array of values is not specified.  
      **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        long longValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (longValues == null)
        {
            throw new MissingValueException("longValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.LONG_ARRAY_DATA_TYPE,
                children);

        final int nValues = longValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = Long.toString(longValues[i]);
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of floating-point values.
     * 
     * @param name				The name of the extended data element. This name
     * 							MUST be	unique with respect to all other fields
     *							in the event.
     * 
     * @param floatValues		An array of floating-point values for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the array of values is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        float floatValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (floatValues == null)
        {
            throw new MissingValueException("floatValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.FLOAT_ARRAY_DATA_TYPE,
                children);

        final int nValues = floatValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = Float.toString(floatValues[i]);
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of double-precision values.
     * 
     * @param name				The name of the extended data element. This name
     * 							MUST be	unique with respect to all other fields
     *							in the event.
     * 
     * @param doubleValues		An array of double-precision values for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the array of values is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        double doubleValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (doubleValues == null)
        {
            throw new MissingValueException("doubleValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.DOUBLE_ARRAY_DATA_TYPE,
                children);

        final int nValues = doubleValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            stringVals[i] = Double.toString(doubleValues[i]);
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of string values.
     * 
     * @param name				The name of the extended data element. This name
     * 							MUST be	unique with respect to all other fields
     *							in the event.
     * 
     * @param floatValues		An array of string values for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the array of values is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        String stringValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (stringValues == null)
        {
            throw new MissingValueException("stringValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.STRING_ARRAY_DATA_TYPE,
                children);

        element.setValues(stringValues);

        return element;

    } /* end method createIExtendedDataElement() */

    /** Create a new IExtendedDataElement item from an array of date/time values.
     * 
     * @param name				The name of the extended data element. This name
     * 							MUST be	unique with respect to all other fields
     *							in the event.
     * 
     * @param floatValues		An array of date/time values for the element.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or the array of values is not specified.  
     **/

    public static IExtendedDataElement createIExtendedDataElement(
        String name,
        java.sql.Date dateTimeValues[],
        IExtendedDataElement children[])
        throws MissingValueException
    {
        if (dateTimeValues == null)
        {
            throw new MissingValueException("dateTimeValues");
        }

        IExtendedDataElement element =
            getEmptyIExtendedDataElement(
                name,
                IExtendedDataElement.DATETIME_ARRAY_DATA_TYPE,
                children);

        final int nValues = dateTimeValues.length;

        String stringVals[] = new String[nValues];

        for (int i = 0; i < nValues; i++)
        {
            if (dateTimeValues[i] != null)
            {
                stringVals[i] =
                    CbeFormatter.convertDateToXmlSchemaDateTime(
                        dateTimeValues[i].getTime());
            }
            else
            {
                throw new MissingValueException("dateTimeValues[" + i + "]");
            }
        };

        element.setValues(stringVals);

        return element;

    } /* end method createIExtendedDataElement() */

    // ----------------------------------------------------------------------
    // Protected and private utility methods
    // ----------------------------------------------------------------------

    /** Get an new IExtendedDataElement item, initialized with the specified
     *  name, type, and children, but with an empty value.
     * 
     * @param name			The name of the extended data element. This name
     * 						MUST be	unique with respect to all other fields
     *						in the event.
     * 
     * @param type			The type of the element's value. This MUST be one
     * 						of the types in IExtendedDataElement.KNOWN_TYPES.
     * 
     * @param children		An optional list of child elements for this item.
     * 
     * @return		A new IExtendedDataElement with the specified settings.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the element name or type, or wrapped value is not specified.  
     **/

    private static IExtendedDataElement getEmptyIExtendedDataElement(
        String name,
        byte type,
        IExtendedDataElement children[])
        throws MissingValueException
    {
        assertRequiredAttributeSpecified("name", name);

        IExtendedDataElement element = new ExtendedDataElementImpl();

        element.setName(name);
        element.setType(type);

        if (children != null)
        {
            for (int i = 0; i < children.length; i++)
            {
                if (children[i] != null)
                {
                    element.addChildDataElement(children[i]);
                }
            }
        }

        return element;

    } /* end method getEmptyIExtendedDataElement() */

    /** Check whether an optional attribute value is specified.
     * 
     * @param value	The attribute value.
     * 
     * @return			<code>true</code> if the value string
     *					is not null	and longer than zero length,
     *					<code>false</code> otherwise.
     **/

    private static boolean isOptionalAttributeSpecified(String value)
    {
        return ((value != null) && (value.length() > 0));

    } /* end method isOptionalAttributeSpecified() */

    /** Throw an exception if the specified attribute value
     *  is null or empty.
     * 
     * @param name		The attribute name.
     * @param value	The attribute value.
     * 
     * @throws MissingValueException
     * 
     * Thrown if the attribute value is null or empty.
     **/
    private static void assertRequiredAttributeSpecified(
        String name,
        String value)
        throws MissingValueException
    {
        if ((value == null) || (value.length() == 0))
        {
            throw new MissingValueException(name);
        }
    } /* end method assertRequiredAttributeSpecified() */

    /** Throw an exception if, for a pair of related attributes,
     *  they aren't both specified (or both left unspecified).
     * 
     * @param nameA	The first attribute name.
     * @param valueA	The first attribute value.
     * @param nameB	The second attribute name.
     * @param valueB	The second attribute value.
     *
     * @throws MissingValueException
     * 
     * Thrown if one of the attribute values is specified but not both.
     **/
    private static void assertAttributesSpecifiedTogether(
        String nameA,
        String valueA,
        String nameB,
        String valueB)
        throws MissingValueException
    {
        boolean specifiedA = ((valueA != null) && (valueA.length() > 0));
        boolean specifiedB = ((valueB != null) && (valueB.length() > 0));

        if ((!specifiedA) && (specifiedB))
        {
            throw new MissingValueException(nameA);
        }
        else if ((specifiedA) && (!specifiedB))
        {
            throw new MissingValueException(nameB);
        }

    } /* end method assertAttributesSpecifiedTogether() */

    /** Throw an exception if an attribute string value exceeds
     *  the allowed length.
     * 
     * @param name		The attribute name.
     * @param value	The attribute value.
     * @param len		The maximum acceptable length.
     *
     * @throws ValueTooLongException
     * 
     * Thrown if the attribute value exceeds the allowed length.
     **/
    private static void assertStringValueNotLongerThan(
        String name,
        String value,
        int len)
        throws ValueTooLongException
    {
        if ((value != null) && (value.length() > len))
        {
            throw new ValueTooLongException(name, len);
        }

    } /* end method assertStringValueNotLongerThan() */

} /* end class EventItemsFactory */
