package org.eclipse.hyades.logging.events;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * @author rduggan
 *
 * This is the interface that embodies the Common Base Event as described in 
 * 'Autonomic Computing Common Base Event and Situations Data ACAB.BO0301.1.0'.
 * 
 * @version 1.0
 */
public interface ICommonBaseEvent {

    /**
     * Add a context that this event is referencing. See the
     * ContextDataElement definition (described on page 28) for details.
     * This is an OPTIONAL property.
     * 
     */
    public void addContextDataElement(IContextDataElement element);

    /**
     * An array of product specific extensions for extensibility where you can
     * provide any other attributes not accounted for in the CommonBaseEventImpl.  
     * Information placed here is assumed to be product specific data.  This is
     * an OPTIONAL property.
     * 
     */
    public void addExtendedDataElement(IExtendedDataElement element);

    /**
     * 
     * This property allows for the grouping of events. This property is a
     * complex type that is made up of globalInstanceIds identifying the associated
     * events along with a type field describing the type of association that is
     * represented by the name of the association. Reserved keyword values include:
     * 1. Contain 
     * 2. CausedBy
     * 3. Cleared
     * 4. MultiPart
     * 5. Correlated
     * This is an OPTIONAL property.
     */
    public void addAssociatedEvent(IAssociatedEvent event);

    /**
     * Returns the ContextDataElements
     * 
     * An array of contexts that this event is referencing. See the
     * ContextDataElement definition (described on page 28) for details.
     * This is an OPTIONAL property.
     * 
     * @return IContextDataElement[]
     */
    public IContextDataElement[] getContextDataElements();

    /**
     * Returns the ExtendedDataElements associated with this event.
     * 
     * An array of product specific extensions for extensibility where you can
     * provide any other attributes not accounted for in the CommonBaseEventImpl.  
     * Information placed here is assumed to be product specific data.  This is
     * an OPTIONAL property.
     * 
     * @return IExtendedDataElement[]
     */
    public IExtendedDataElement[] getExtendedDataElements();

    /**
     * Returns the Associated events.
     * 
     * This property allows for the grouping of events. This property is a
     * complex type that is made up of globalInstanceIds identifying the associated
     * events along with a type field describing the type of association that is
     * represented by the name of the association. Reserved keyword values include:
     * 1. Contain 
     * 2. CausedBy
     * 3. Cleared
     * 4. MultiPart
     * 5. Correlated
     * This is an OPTIONAL property.
     * 
     * @return IAssociatedEvent[]
     */
    public IAssociatedEvent[] getAssociatedEvents();

    /**
     * Get a named a context that this event is referencing. See the
     * ContextDataElement definition (described on page 28) for details.
     * 
     * @param name the name of the element to be found
     * 
     * @return IContextDataElement if the named element exists otherwise null is returned
     */
    public IContextDataElement getContextDataElement(String name);

    /**
     * Get a named extended data element that this event is referencing
     * 
     * @param name the name of the element to be found
     * 
     * @return IExtendedDataElement if the named element exists otherwise null is returned
     */
    public IExtendedDataElement getExtendedDataElement(String name);

    /**
     * Removes a named a context that this event is referencing. See the
     * ContextDataElement definition (described on page 28) for details.
     * 
     * @param name the name of the element to be removed
     */
    public void removeContextDataElement(String name);

    /**
     * Remove a named extended data element that this event is referencing 
     * 
     * @param name the name of the element to be removed
     */
    public void removeExtendedDataElement(String name);

    /**
     * Removes all IAssociatedEvents that this event is referencing 
     */
    public void clearAssociatedEvents();

    /**
     * Removes all IContextDataElements that this event is referencing 
     */
    public void clearContextDataElements();

    /**
     * Removes all IExtenedDataElements that this event is referencing 
     */
    public void clearExtendedDataElements();

    /**
     * Returns the creationTime.
     * 
     * The date-time when the event was issued. The value MUST be as defined by 
     * the XML Schema dateTime data type. The value of the creationTime MUST provide
     * granularity as precisely as the generating platform allows. This is a REQUIRED
     *  property.
     * 
     * @return String
     */
    public String getCreationTime();
    
    /**
     * Returns the creationTime as a long holding the milliseconds since Jan 1, 1970.
     * 
     * @return long
     */
    public long getCreationTimeAsLong();    

    /**
     * Returns the elapsedTime.
     * 
     * * This is the time interval or the elapsed time for the number of occurrences
     * of a given event type that is specified by the repeatCount property. This
     * value indicates the duration of the time within which the repeated events 
     * were observed. This is an OPTIONAL property with no default value.  However, 
     * if repeatCount is specified you MUST specify a value for elapsedTime.  
     * 
     * @return long
     */
    public long getElapsedTime();

    /**
     * Returns the extensionName.
     * 
     * The name of an event class (or element in XML) that this event represents
     * (e.g., CommonBaseEventImpl). The name is indicative of any additional elements
     * expected to be present within the event. This is a OPTIONAL property.  If 
     * the value specified is null, then its value is assumed to be 'CommonBaseEventImpl'.
     * The maximum string length for extensionName MUST not exceed 64 bytes.
     * 
     * @return String
     */
    public String getExtensionName();

    /**
     * Returns the gloablInstanceId.
     * 
     * The primary identifier for the event. This property MUST be globally unique
     * and MAY be used as the primary key for the event. This is an OPTIONAL property.
     * However, once this value is set it MUST never be changed. The recommend value 
     * is either a 128 bit or 256 bit Globally Unique Id (represented as hex string).
     * 
     * @return String
     */
    public String getGlobalInstanceId();

    /**
     * Returns the localInstanceId.
     * 
     * A source supplied event identifier. There is no guarantee that this value 
     * is globally unique. This is an OPTIONAL property. The maximum string length
     * for localInstanceId MUST not exceed 64 bytes.
     * 
     * @return String
     */
    public String getLocalInstanceId();

    /**
     * Returns the msg.
     * 
     * he text accompanying the event. This is typically the resolved message 
     * string in human readable format rendered for a specific locale. This is 
     * an OPTIONAL property.  The maximum string length for msg MUST not exceed
     * 3072 bytes.
     *  
     * @return String
     */
    public String getMsg();

    /**
     * Returns the priority.
     * 
     * Defines the importance of the event. The predefined priorities are:
     * 1. Low
     * 2. Medium
     * 3. High
     * The values are 0 to 100. The reserved value for Low is 10, for Medium is 50,
     * and for High is 70.  Other priorities MAY be added but MUST not exceed 100.
     * This is an OPTIONAL property.  
     * 
     * @return short
     */
    public short getPriority();

    /**
     * Returns the repeatCount.
     * 
     * The number of occurrences of a given event for a specific time interval.
     * This is an OPTIONAL property with no default. A value of 0 or no value
     * is indicative of no repeat of the event detected.
     *
     * @return short
     */
    public short getRepeatCount();

    /**
     * Returns the sequenceNumber.
     * 
     * A source defined number that allows for multiple messages to be sent
     * and processed in logical order different from the order in which they 
     * arrived at consumer location (e.g., an event server or management tools).
     * The sequence number helps consumers to sort arrived messages. This is 
     * with respect to the creation time and to the particular reporter of the 
     * messages. This is an OPTIONAL property.  There is no default value for
     * this property.
     * 
     * @return long
     */
    public long getSequenceNumber();

    /**
     * Returns the severity.
     * 
     * The severity of the status the event is describing with respect to the 
     * application that reports the event.  The predefined severity levels, in
     * order of increasing severity, are as follows: 
     * 1. Unknown
     * 2. Information SHOULD be used for cases when the event only contains general
     *    information and is not reporting an error.
     * 3. Harmless SHOULD be used for cases when the error event has no effect on the
     *    normal operation of the resource.
     * 4. Warning SHOULD be used when it is appropriate to let the user decide if an
     *    action is needed in response to the event.
     * 5. Minor SHOULD be used to indicate that action is needed, but the situation
     *    is not serious at this time.
     * 6. Critical SHOULD be used to indicate that an immediate action is needed and
     *    the scope is broad (perhaps an imminent outage to a critical resource will 
     *    result).
     * 7. Fatal SHOULD be used to indicate that an error occurred, but it is too late
     *    to take remedial action.
     *   
     * The values are 0 to 100. The reserved values start at 0 for Unknown and increase
     * by increments of 10 to 60 for Fatal.  Other severities MAY be added but MUST not
     * exceed 100.  This is an OPTIONAL property.
     * 
     * @return short
     */
    public short getSeverity();

    /**
     * Returns the situationType.
     * 
     * The situationType specifies the type of the situation that caused the event 
     * to be reported. This is an extensible string value. Proposed reserved keywords
     *  are:
     * 1. START
     * 2. STOP
     * 3. FEATURE
     * 4. DEPENDENCY
     * 5. REQUEST
     * 6. CONFIGURE
     * 7. CONNECT
     * 8. CREATE
     * 9. UNKNOWN
     * This is an OPTIONAL property. The maximum string length for situationType MUST
     * not exceed 64 bytes.
     * NOTE: The situations type is being defined in parallel to the CommonBaseEventImpl
     * effort in a separate document, Canonical Situations ' ACAB.CT0304.0.1. ' The type
     * and the values of situationType are subject to change.
     * 
     * @return String
     */
    public String getSituationType();

    /**
     * Returns the sourceComponentId.
     * 
     * Identification of the component that is 'affected' or was 'impacted'
     * by the event or the situation.  This is a REQUIRED property for the 
     * component that is affected by the situation. 
     * 
     * @return ComponentIdentificationImpl
     */
    public IComponentIdentification getSourceComponentId();

    /**
     * Returns the msgDataElement.
     * 
     * Identification of the message that this event holds.  See the
     * MsgDataElementImpl definition (described on page 30). This is an OPTIONAL
     * property.
     * 
     * @return MsgDataElementImpl
     */
    public IMsgDataElement getMsgDataElement();

    /**
     * Returns the reporterComponentId.
     * 
     * Identification of the component that is the 'reporter' of the event or the
     * situation. It is a REQUIRED property if the reporting component is different
     * than the source component.  Otherwise this field MUST be omitted. 
     * 
     * @return ComponentIdentificationImpl
     */
    public IComponentIdentification getReporterComponentId();

    /**
     * Sets the creationTime.
     * 
     * The date-time when the event was issued. The value MUST be as defined by 
     * the XML Schema dateTime data type. The value of the creationTime MUST provide
     * granularity as precisely as the generating platform allows. This is a REQUIRED
     * property.
     * 
     * @param creationTime The creationTime to set
     */
    public void setCreationTime(String creationTime);
    
    /**
     * Sets the creationTime to a long that contains the milliseconds since Jan 1, 1970.
     * 
     * @param creationTime The creationTime to set
     */
    public void setCreationTime(long creationTime);    

    /**
     * Sets the elapsedTime.
     * 
     * This is the time interval or the elapsed time for the number of occurrences
     * of a given event type that is specified by the repeatCount property. This
     * value indicates the duration of the time within which the repeated events 
     * were observed. This is an OPTIONAL property with no default value.  However, 
     * if repeatCount is specified you MUST specify a value for elapsedTime.  
     * 
     * @param elapsedTime The elapsedTime to set
     */
    public void setElapsedTime(long elapsedTime);

    /**
     * Sets the extensionName.
     * 
     * The name of an event class (or element in XML) that this event represents
     * (e.g., CommonBaseEventImpl). The name is indicative of any additional elements
     * expected to be present within the event. This is a OPTIONAL property.  If 
     * the value specified is null, then its value is assumed to be 'CommonBaseEventImpl'.
     * The maximum string length for extensionName MUST not exceed 64 bytes.
     * 
     * @param extensionName The extensionName to set
     */
    public void setExtensionName(String extensionName);

    /**
     * Sets the gloablInstanceId.
     * 
     * The primary identifier for the event. This property MUST be globally unique
     * and MAY be used as the primary key for the event. This is an OPTIONAL property.
     * However, once this value is set it MUST never be changed. The recommend value 
     * is either a 128 bit or 256 bit Globally Unique Id (represented as hex string).
     * 
     * @param gloablInstanceId The gloablInstanceId to set
     */
    public void setGlobalInstanceId(String globalInstanceId);

    /**
     * Sets the localInstanceId.
     * 
     * A source supplied event identifier. There is no guarantee that this value 
     * is globally unique. This is an OPTIONAL property. The maximum string length
     * for localInstanceId MUST not exceed 64 bytes.
     * 	
     * @param localInstanceId The localInstanceId to set
     */
    public void setLocalInstanceId(String localInstanceId);

    /**
     * Sets the msg.
     * 
     * The text accompanying the event. This is typically the resolved message 
     * string in human readable format rendered for a specific locale. This is 
     * an OPTIONAL property.  The maximum string length for msg MUST not exceed
     * 3072 bytes. 
     * 
     * @param msg The msg to set
     */
    public void setMsg(String msg);

    /**
     * Sets the priority.
     * 
     * Defines the importance of the event. The predefined priorities are:
     * 1. Low
     * 2. Medium
     * 3. High
     * The values are 0 to 100. The reserved value for Low is 10, for Medium is 50,
     * and for High is 70.  Other priorities MAY be added but MUST not exceed 100.
     * This is an OPTIONAL property.  
     * 
     * @param priority The priority to set
     */
    public void setPriority(short priority);

    /**
     * Sets the repeatCount.
     * 
     * The number of occurrences of a given event for a specific time interval.
     * This is an OPTIONAL property with no default. A value of 0 or no value
     * is indicative of no repeat of the event detected.
     * 
     * @param repeatCount The repeatCount to set
     */
    public void setRepeatCount(short repeatCount);

    /**
     * Sets the sequenceNumber.
     * 
     * A source defined number that allows for multiple messages to be sent
     * and processed in logical order different from the order in which they 
     * arrived at consumer location (e.g., an event server or management tools).
     * The sequence number helps consumers to sort arrived messages. This is 
     * with respect to the creation time and to the particular reporter of the 
     * messages. This is an OPTIONAL property.  There is no default value for
     * this property.
     * 
     * @param sequenceNumber The sequenceNumber to set
     */
    public void setSequenceNumber(long sequenceNumber);

    /**
     * Sets the severity.
     * 
     * The severity of the status the event is describing with respect to the 
     * application that reports the event.  The predefined severity levels, in
     * order of increasing severity, are as follows: 
     * 1. Unknown
     * 2. Information SHOULD be used for cases when the event only contains general
     *    information and is not reporting an error.
     * 3. Harmless SHOULD be used for cases when the error event has no effect on the
     *    normal operation of the resource.
     * 4. Warning SHOULD be used when it is appropriate to let the user decide if an
     *    action is needed in response to the event.
     * 5. Minor SHOULD be used to indicate that action is needed, but the situation
     *    is not serious at this time.
     * 6. Critical SHOULD be used to indicate that an immediate action is needed and
     *    the scope is broad (perhaps an imminent outage to a critical resource will 
     *    result).
     * 7. Fatal SHOULD be used to indicate that an error occurred, but it is too late
     *    to take remedial action.
     *   
     * The values are 0 to 100. The reserved values start at 0 for Unknown and increase
     * by increments of 10 to 60 for Fatal.  Other severities MAY be added but MUST not
     * exceed 100.  This is an OPTIONAL property.
     * 
     * @param severity The severity to set
     */
    public void setSeverity(short severity);

    /**
     * Sets the situationType.
     * 
     * The situationType specifies the type of the situation that caused the event 
     * to be reported. This is an extensible string value. Proposed reserved keywords
     *  are:
     * 1. START
     * 2. STOP
     * 3. FEATURE
     * 4. DEPENDENCY
     * 5. REQUEST
     * 6. CONFIGURE
     * 7. CONNECT
     * 8. CREATE
     * 9. UNKNOWN
     * This is an OPTIONAL property. The maximum string length for situationType MUST
     * not exceed 64 bytes.
     * NOTE: The situations type is being defined in parallel to the CommonBaseEventImpl
     * effort in a separate document, Canonical Situations ' ACAB.CT0304.0.1. ' The type
     * and the values of situationType are subject to change.
     * 
     * @param situationType The situationType to set
     */
    public void setSituationType(String situationType);

    /**
     * Sets the sourceComponentId.
     * 
     * Identification of the component that is 'affected' or was 'impacted'
     * by the event or the situation.  This is a REQUIRED property for the 
     * component that is affected by the situation. 
     * 
     * @param sourceComponentId The sourceComponentId to set
     */
    public void setSourceComponentId(IComponentIdentification sourceComponentId);

    /**
     * Sets the msgDataElement.
     * 
     * Identification of the message that this event holds.  See the
     * MsgDataElementImpl definition (described on page 30). This is an OPTIONAL
     * property.
     * 
     * @param msgDataElement The msgDataElement to set
     */
    public void setMsgDataElement(IMsgDataElement msgDataElement);

    /**
     * Sets the reporterComponentId.
     * 
     * Identification of the component that is the 'reporter' of the event or the
     * situation. It is a REQUIRED property if the reporting component is different
     * than the source component.  Otherwise this field MUST be omitted. 
     * 
     * @param reporterComponentId The reporterComponentId to set
     */
    public void setReporterComponentId(IComponentIdentification reporterComponentId);

    /**
     * Initialize a CommonBaseEvent so that it has a new guid and that all the 
     * following relationships have been cleared:
     * 1. IAssociatiatedEvent(s)
     * 2. IExtendeddataElement(s)
     * 3. IContextDataElement(s)
     * 
	 * All components are initialized to either zero or null
     */
    public void init();

	/**
	 * Returns the CommonBaseEvent Implementation version
	 * @return String
	 */
	public String getCommonBaseEventImplVersion();
	
    /**
	 * Returns the CommonBaseEvent XML schema version
	 * @return String
	 */
    public String getEventSchemaVersion();
}
