/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.events;

/**
 * Common iterface for all factories which construct Hyades logging event objects.
 */
public interface IEventFactory
{

    /** A special value for event creation time, to indicate that the
     *  current system time is being specified in an API call.
     */
    long CURRENT_TIME = Long.MIN_VALUE + 1;

    /** A special value for event severity, to indicate that no value
     * (as opposed to any default value) is being specified in an API call.
     */
    short UNSPECIFIED_SEVERITY = Short.MIN_VALUE + 1;

    /** A special value for event priority, to indicate that no value
     * (as opposed to any default value) is being specified in an API call.
     */
    short UNSPECIFIED_PRIORITY = Short.MIN_VALUE + 1;

    /** A special value for event repeat count, to indicate that no value
     * (as opposed to any default value) is being specified in an API call.
     */
    short UNSPECIFIED_REPEAT_COUNT = Short.MIN_VALUE + 1;

    /** A special value for event elapsed time, to indicate that no value
     * (as opposed to any default value) is being specified in an API call.
     */
    long UNSPECIFIED_ELAPSED_TIME = Long.MIN_VALUE + 1;

    /** A special value for event sequence number, to indicate that no value
     * (as opposed to any default value) is being specified in an API call.
     */
    long UNSPECIFIED_SEQUENCE_NUM = Long.MIN_VALUE + 1;

    /**	Get the name of this factory instance.
     *	Instance names are immutable once set.
     *
     *	@return	The instance name.
     **/
    String getName();

    /**	Get the fully-qualified interface name for the events
     *  created by this factory. This should either be:
     *	<code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code>
     *	or an interface which extends it.
     *
     *	@return	The interface name.
     **/
    String getEventInterfaceClassName();

    /**	Get the optional event settings for this factory
     *	instance. The values of this object (if any) will
     *  be used as defaults when creating new events.
     * 
     *	@return	The current event settings, or
     *				<code>null</code>	if no settings
     *				have been defined.
     **/
    ICommonBaseEvent getEventSettings();

    /**	Set the optional event settings for this factory
     *	instance. The values of this object (if any) will
     *  be used as defaults when creating new events.
     * 
     *  <p>The settings object should be an event which was
     *  created by this factory's <code>createEmptyEvent()</code>
     *  method, then populated with values.</p>
     *
     *	@param	The new event settings, or <code>null</code>
     *			if no event settings should be used.
     **/
    void setEventSettings(ICommonBaseEvent settings);

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event,
     *							or <code>null</code> to use the
     *							current system time.
     *
     *	@param sourceCompId	The source component ID for the event,
     *							or <code>null</code> to copy the value
     *							from the event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    ICommonBaseEvent createEvent(
        java.util.Date creationTime,
        IComponentIdentification sourceCompId);

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event, or
     * 							<code>CURRENT_TIME</code> to use the
     *							current system time.
     *
     *	@param sourceCompId	The source component ID for the event,
     *							or <code>null</code> to copy the value
     *							from the event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    ICommonBaseEvent createEvent(
        long creationTime,
        IComponentIdentification sourceCompId);

    /**	Create a new event.
      * 
      *	@param creationTime	The creation time for the event,
      *							or <code>null</code> to use the
      *							current system time.
      *
      *	@param localInstId		An optional local-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param globalInstId	An optional global-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param severity		The optional event severity, or
      *							<code>UNSPECIFIED_SEVERITY</code> to copy
      *							the value (if any) from the event settings.
      *
      *	@param priority		The optional event priority, or
      *							<code>UNSPECIFIED_PRIORITY</code> to copy 
      *							the value (if any) from the event settings
      *
      *	@param reporterCompId	The optional reporter component ID for
      *							the event, or <code>null</code> to copy
      *							the value (if any) from event settings.
      *
      *	@param sourceCompId	The required source component ID for
      *							the event, or <code>null</code> to copy
      *							the value from the event settings.
      *
      *	<p>NOTE: If the source component ID is not specified and is
      *	not found in the factory event settings, the event will not
      *	be created. Use <code>createEmptyEvent()</code> to create an
      *	event with no creation time or source component ID.</p>
      *
      *	@param situationType	An optional situation-type string,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param contextData		An optional array of context data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param msgData			Optional data about the message, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param msg				The optional message text, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param repeatCount		The optional repeat count, or
      *							<code>UNSPECIFIED_REPEAT_COUNT</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param elapsedTime		The optional time (in microseconds)
      *							for the number of events specified
      *							in the repeat count, or 
      *							<code>UNSPECIFIED_ELAPSED_TIME</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param assocEvents		An optional array of associated events,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extensionName	An optional extension name, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extendedData	An optional array of extended data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param sequenceNum		An optional sequence number for the event,
      *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
      *							to copy the value (if any) from the
      *							event settings.
      *
      *	<p>If event settings are defined for this factory instance,
      *	they will be used to set any fields not specified above.</p>
      *
      *	@return	The new event, or <code>null</code> if no value
      *				was found for the required source component ID.
      *
      *	<p>The new event will implement the interface reported by
      *	   <code>getEventInterfaceClassName()</code>.</p>
      **/
    ICommonBaseEvent createEvent(
        java.util.Date creationTime,
        String localInstId,
        String globalInstId,
        short severity,
        short priority,
        IComponentIdentification reporterCompId,
        IComponentIdentification sourceCompId,
        String situationType,
        IContextDataElement contextData[],
        IMsgDataElement msgData,
        String msg,
        short repeatCount,
        long elapsedTime,
        IAssociatedEvent assocEvents[],
        String extensionName,
        IExtendedDataElement extendedData[],
        long sequenceNum);

    /**	Create a new event.
      * 
      *	@param creationTime	The creation time for the event, or
      * 						<code>CURRENT_TIME</code> to use the
      *							current system time.
      *
      *	@param localInstId		An optional local-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param globalInstId	An optional global-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param severity		The optional event severity, or
      *							<code>UNSPECIFIED_SEVERITY</code> to copy
      *							the value (if any) from the event settings.
      *
      *	@param priority		The optional event priority, or
      *							<code>UNSPECIFIED_PRIORITY</code> to copy 
      *							the value (if any) from the event settings
      *
      *	@param reporterCompId	The optional reporter component ID for
      *							the event, or <code>null</code> to copy
      *							the value (if any) from event settings.
      *
      *	@param sourceCompId	The required source component ID for
      *							the event, or <code>null</code> to copy
      *							the value from the event settings.
      *
      *	<p>NOTE: If the source component ID is not specified and is
      *	not found in the factory event settings, the event will not
      *	be created. Use <code>createEmptyEvent()</code> to create an
      *	event with no creation time or source component ID.</p>
      *
      *	@param situationType	An optional situation-type string,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param contextData		An optional array of context data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param msgData			Optional data about the message, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param msg				The optional message text, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param repeatCount		The optional repeat count, or
      *							<code>UNSPECIFIED_REPEAT_COUNT</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param elapsedTime		The optional time (in microseconds)
      *							for the number of events specified
      *							in the repeat count, or 
      *							<code>UNSPECIFIED_ELAPSED_TIME</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param assocEvents		An optional array of associated events,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extensionName	An optional extension name, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extendedData	An optional array of extended data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param sequenceNum		An optional sequence number for the event,
      *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
      *							to copy the value (if any) from the
      *							event settings.
      *
      *	<p>If event settings are defined for this factory instance,
      *	they will be used to set any fields not specified above.</p>
      *
      *	@return	The new event, or <code>null</code> if no value
      *				was found for the required source component ID.
      *
      *	<p>The new event will implement the interface reported by
      *	   <code>getEventInterfaceClassName()</code>.</p>
      **/
    ICommonBaseEvent createEvent(
        long creationTime,
        String localInstId,
        String globalInstId,
        short severity,
        short priority,
        IComponentIdentification reporterCompId,
        IComponentIdentification sourceCompId,
        String situationType,
        IContextDataElement contextData[],
        IMsgDataElement msgData,
        String msg,
        short repeatCount,
        long elapsedTime,
        IAssociatedEvent assocEvents[],
        String extensionName,
        IExtendedDataElement extendedData[],
        long sequenceNum);

    /**	Create an empty event.
     *
     *	<p>No fields will be set for this event. Any event settings
     *	for this factory instance will be ignored.</p>
     *
     *	@return	The empty new event.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    ICommonBaseEvent createEmptyEvent();

    /**	Initialization method. This should not be called by
     *	applications, and has no effect after this factory
     *	instance is initialized.
     *
     *	@param	The new name of this factory instance.
     **/
    void init(String name);

} /* end interface IEventFactory */
