package org.eclipse.hyades.logging.commons;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import org.apache.commons.logging.Log;
import org.eclipse.hyades.internal.logging.core.XmlGenerator;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.core.LoggingAgent;

/** 
 * Implementation of the Apache Commons <code>org.apache.commons.logging.Log</code>
 * interface which represents a logger for logging messages to a Logging Agent
 * with the same name as the logger.
 * <p>
 * The logging level is used to filter which severity of logged messages are sent 
 * to the Logging Agent with the same name as the logger.  The logging level may be 
 * set to one of the following logging levels:
 * <p>
 * -TRACE (least serious and most verbose)
 * -DEBUG
 * -INFO
 * -WARN
 * -ERROR
 * -FATAL (most serious and least verbose)
 * <p>
 * An instance of this class will be returned from the <code>getLog()</code> and/or 
 * <code>getInstance()</code> APIs on the default <code>org.apache.commons.logging.LogFactory</code>
 * implementation.  The default <code>org.apache.commons.logging.LogFactory</code> implementation
 * uses the following discovery process:
 * <p> 
 * 1) Look for a system property named <code>org.apache.commons.logging.Log</code> or
 *    <code>org.apache.commons.logging.log</code>.  For example,
 * 
 *        <code>...-Dorg.apache.commons.logging.Log=org.eclipse.hyades.logging.commons.Logger...</code>
 * 
 *        - or -
 * 
 *        <code>System.setProperty("org.apache.commons.logging.Log","org.eclipse.hyades.logging.commons.Logger");</code>
 * 
 * 2) Look for a configuration attribute of the default <code>org.apache.commons.logging.LogFactory</code> 
 *    implementation named <code>org.apache.commons.logging.Log</code>.  For example,
 * 
 *    <code>org.apache.commons.logging.LogFactory.getFactory().setAttribute("org.apache.commons.logging.Log","org.eclipse.hyades.logging.commons.Logger");</code>
 * <p>
 * A logger instance is created via the constructor and configured (e.g. logging level).  The logger is generally 
 * consulted on its configuration before every message if logged for performance reasons.  For example,
 * <p>
 * <code>
 * Logger logger = new Logger("My Logger");
 * <p>
 * logger.setLevel(Logger.WARN_LEVEL);
 * <p>
 * if(logger.isDebugEnabled()){
 *     logger.debug("My DEBUG log message.");
 * }
 * </code>
 * <p>
 * NOTE:  The Apache Commons logging classes must be on the CLASSPATH at runtime to utilize this logging class
 *  
 * 
 * @author		Paul Slauenwhite
 * @version	March 7, 2003
 * @see		org.apache.commons.logging.Log
 * @see		org.apache.commons.logging.LogFactory
 * @see		org.eclipse.hyades.logging.core.LoggingAgent
 */

public class Logger implements Log {

    /**
     * Trace logging level value.
     */
    public static final int TRACE_LEVEL = 0;

    /**
     * Debug logging level value.
     */
    public static final int DEBUG_LEVEL = 1;

    /**
     * Information logging level value.
     */
    public static final int INFO_LEVEL = 2;

    /**
     * Warning logging level value.
     */
    public static final int WARN_LEVEL = 3;

    /**
     * Error logging level value.
     */
    public static final int ERROR_LEVEL = 4;

    /**
     * Fatal logging level value.
     */
    public static final int FATAL_LEVEL = 5;

    /**
     * Array of String representations of the logging levels.
     */
    private static final String[] LEVEL_NAMES = { "TRACE", "DEBUG", "INFO", "WARN", "ERROR", "FATAL" };

    /**
     * The Logging Agent with the same name as the logger to which successfully logged messages are sent.
     */
    private LoggingAgent loggingAgent = null;

    /**
     * The logger's current logging level value.
     */
    private int loggingLevel;

    /**
     * Utility XML generator to convert logged messages (e.g. <code>java.lang.Object</code> and/or <code>java.lang.Throwable</code>) to XML.
     */
    private XmlGenerator xmlGenerator = null;

    /**
     * Constructor to create a logger instance with a Logging Agent using 
     * the parameter name.
     * 
     * NOTE: The default logging level is set to WARN until explicitly set.
     * 
     * @param name The name of the newly created logger.
     */
    public Logger(String name) {

        loggingAgent = new LoggingAgent(name);
        loggingLevel = WARN_LEVEL;

		xmlGenerator=new XmlGenerator(name);
    }

    /**
     * Checks if TRACE logging is currently enabled.  TRACE logging infers the logging level is currently 
     * set at TRACE or lower.  
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if TRACE logging is currently enabled, otherwise false.
     */
    public boolean isTraceEnabled() {
        return (loggingLevel <= TRACE_LEVEL);
    }

    /**
     * Checks if DEBUG logging is currently enabled.  DEBUG logging infers the logging level is currently 
     * set at DEBUG or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if DEBUG logging is currently enabled, otherwise false.
     */
    public boolean isDebugEnabled() {
        return (loggingLevel <= DEBUG_LEVEL);
    }

    /**
     * Checks if INFO logging is currently enabled.  INFO logging infers the logging level is currently 
     * set at INFO or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if INFO logging is currently enabled, otherwise false.
     */
    public boolean isInfoEnabled() {
        return (loggingLevel <= INFO_LEVEL);
    }

    /**
     * Checks if WARN logging is currently enabled.  WARN logging infers the logging level is currently 
     * set at WARN or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if WARN logging is currently enabled, otherwise false.
     */
    public boolean isWarnEnabled() {
        return (loggingLevel <= WARN_LEVEL);
    }

    /**
     * Checks if ERROR logging is currently enabled.  ERROR logging infers the logging level is currently 
     * set at ERROR or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if ERROR logging is currently enabled, otherwise false.
     */
    public boolean isErrorEnabled() {
        return (loggingLevel <= ERROR_LEVEL);
    }

    /**
     * Checks if FATAL logging is currently enabled.  FATAL logging infers the logging level is currently 
     * set at FATAL or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if FATAL logging is currently enabled, otherwise false.
     */
    public boolean isFatalEnabled() {
        return (loggingLevel <= FATAL_LEVEL);
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if TRACE logging is currently enabled.  
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void trace(Object record) {

        if (isTraceEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if TRACE logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void trace(Object record, Throwable throwable) {

        if (isTraceEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if DEBUG logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void debug(Object record) {

        if (isDebugEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if DEBUG logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void debug(Object record, Throwable throwable) {

        if (isDebugEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if INFO logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void info(Object record) {

        if (isInfoEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if INFO logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void info(Object record, Throwable throwable) {

        if (isInfoEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if WARN logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void warn(Object record) {

        if (isWarnEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if WARN logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void warn(Object record, Throwable throwable) {

        if (isWarnEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if ERROR logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void error(Object record) {

        if (isErrorEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if ERROR logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void error(Object record, Throwable throwable) {

        if (isErrorEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the logger if FATAL logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void fatal(Object record) {

        if (isFatalEnabled())
            loggingAgent.write(convertToXML(record));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and <code>java.lang.Throwable</code> 
     * exception to the Logging Agent with the same name as the logger if FATAL logging is currently enabled.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * The parameter <code>java.lang.Object</code> log record is first logged and then the <code>java.lang.Throwable</code> 
     * exception is logged to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     * @param throwable The exception to be logged to the Logging Agent.
     */
    public void fatal(Object record, Throwable throwable) {

        if (isFatalEnabled()) {
            loggingAgent.write(convertToXML(record));
            loggingAgent.write(convertToXML(throwable));
        }
    }

    /**
     * Retrieves the name of this logger.
     *
     * @return Name of this logger.
     */
    public String getName() {
        return (loggingAgent.getName());
    }

    /**
     * Retrieves the level of this logger.
     *
     * @return Logging level of this logger (i.e. TRACE_LEVEL - FATAL_LEVEL).
     */
    public int getLevel() {
        return loggingLevel;
    }

    /**
     * Sets the logging level to one of the logging levels (i.e. TRACE_LEVEL - FATAL_LEVEL).
     *
     * @param newLevel The new logging level (i.e. TRACE_LEVEL - FATAL_LEVEL).
     * @exception java.lang.IllegalArgumentException Attempting to set an illegal logging level.
     */
    public synchronized void setLevel(int level) throws IllegalArgumentException {

        if ((level < 0) || (level >= LEVEL_NAMES.length))
            throw new IllegalArgumentException("Invalid logging level");
        else
            loggingLevel = level;
    }

    /**
     * Internal utility API for converting the parameter <code>java.lang.Object</code> to XML using 
     * the <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
     *
     * @param object The object to be converted to XML.
     * @return The XML fragment of the converted parameter object.
     */
    private String convertToXML(Object object) {

        if (object instanceof IExternalizableToXml) {
            return ((IExternalizableToXml) object).externalizeCanonicalXmlString();
        }
        else {
            xmlGenerator.reset(LEVEL_NAMES[loggingLevel], false, 4);
            return (xmlGenerator.objectToXML(object));
        }
    }
}
