/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.ILibrary;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ISetSelectionTarget;

import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * Contains utility methods to manipulate the Eclipse resources.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ResourceUtil
{
	/**
	 * Interface to define a visitor for the <code>getFile(...)</code> methods.
	 */
	public static interface IGetFilesVisitor
	{
		/**
		 * This method is invoked before adding a file to a map.
		 * 
		 * <p>The object returned by this mapping is added to the map.  If the
		 * map returns <code>null</code> then the map is not changed.
		 * 
		 * @param file
		 * @return Object
		 */
		public Object visit(IFile file);
	}
	
	/**
	 * Creates the given container and it's parent if necessary.
	 * @param container
	 * @param progressMonitor
	 */
	public static void createContainer(IContainer container, IProgressMonitor progressMonitor)
	throws CoreException
	{
		if((container == null) || container.exists())
			return;	
		createContainer(container.getParent(), progressMonitor);
		
		if(container instanceof IProject)
			((IProject)container).create(progressMonitor);
			
		else if(container instanceof IFolder)
			((IFolder)container).create(true, true, progressMonitor);		
	}
	
	/**
	 * Attempts to select and reveal the specified resource in all
	 * parts within the supplied workbench window's active page.
	 * 
	 * <p>Checks all parts in the active page to see if they implement 
	 * <code>ISetSelectionTarget</code>, either directly or as an adapter. 
	 * If so, tells the part to select and reveal the specified resource.
	 *
	 * @param resource the resource to be selected and revealed
	 * @param window the workbench window to select and reveal the resource
	 * @see org.eclipse.ui.wizards.newresource.BasicNewResourceWizard
	 * @see ISetSelectionTarget
	 */
	public static void selectAndReveal(IResource resource, IWorkbenchWindow window)
	{
		// validate the input
		if (window == null || resource == null)
			return;
		IWorkbenchPage page = window.getActivePage();
		if (page == null)
			return;

		// get all the view and editor parts
		List parts = new ArrayList();
		IWorkbenchPartReference refs[] = page.getViewReferences();
		for (int i = 0; i < refs.length; i++)
		{
			IWorkbenchPart part = refs[i].getPart(false);
			if(part != null)
				parts.add(part);
		}	
		refs = page.getEditorReferences();
		for (int i = 0; i < refs.length; i++)
		{
			if(refs[i].getPart(false) != null)
				parts.add(refs[i].getPart(false));
		}
	
		final ISelection selection = new StructuredSelection(resource);
		Iterator enum = parts.iterator();
		while (enum.hasNext())
		{
			IWorkbenchPart part = (IWorkbenchPart) enum.next();
		
			// get the part's ISetSelectionTarget implementation
			ISetSelectionTarget target = null;
			if (part instanceof ISetSelectionTarget)
				target = (ISetSelectionTarget) part;
			else
				target = (ISetSelectionTarget) part.getAdapter(ISetSelectionTarget.class);
			
			if (target != null)
			{
				// select and reveal resource
				final ISetSelectionTarget finalTarget = target;
				window.getShell().getDisplay().asyncExec(new Runnable()
				{
					public void run()
					{
						finalTarget.selectReveal(selection);
					}
				});
			}
		}
	}
	
	/**
	 * Returns a {@link File} handle to the directory of a plugin or 
	 * <code>null</code> if the directory was not found.
	 * @param pluginId
	 * @return File
	 */
	public static File getPluginDirectory(String pluginId)
	{
		File pluginDir = null;
		
		try
		{
			URL pluginURL = new URL(Platform.getPlugin(pluginId).getDescriptor().getInstallURL(), "plugin.xml");
			File manifestFile = new File(Platform.asLocalURL(pluginURL).getFile()).getAbsoluteFile();
			if(manifestFile.exists())
				pluginDir = manifestFile.getParentFile();
		}
		catch(Exception e)
		{
		}
		
		return pluginDir;
	}
	
	/**
	 * Returna all the libraries of the specified plugin.
	 * @param pluginId
	 * @return String[]
	 */
	public static String[] getPluginLibraries(String pluginId)
	{
		if(pluginId == null)
			return new String[0];

		Plugin plugin = Platform.getPlugin(pluginId);
		if(plugin == null)
			return new String[0];
		
		File pluginDir = getPluginDirectory(pluginId);
		ILibrary[] pluginLibraries = plugin.getDescriptor().getRuntimeLibraries();
		List libraries = new ArrayList(pluginLibraries.length);
		for(int i = 0, maxi = pluginLibraries.length; i < maxi; i++)
		{
			File lib = new File(pluginDir, pluginLibraries[i].getPath().toString());
			try
			{
				libraries.add(lib.getCanonicalPath().replace('\\','/'));
			}
			catch(IOException e)
			{
				libraries.add(lib.getAbsolutePath().replace('\\','/'));
			}			
		}
		
		return (String[])libraries.toArray(new String[libraries.size()]);
	}	
	
	/**
	 * Returns a map with all the "files" from all the projects in a given 
	 * workspace root that have their file extension listed in the file extensions 
	 * array.
	 * 
	 * <p>This method looks for files recursively if the resource is a project 
	 * or a folder.
	 * 
	 * <p>In order to be added to the map the file must exists and must belong to
	 * an open project.
	 *  
	 * <p>Clients can specify a visitor to be used by this method.  The visitor
	 * is able to defined the object that added to the map - that's the reason
	 * the word file is between quotes in the paragraphs above. If the visitor 
	 * is <code>null</code> then the file is added to the map
	 *  
	 * @param workspaceRoot
	 * @param fileExtensions. 
	 * @param visitor
	 * @return a Map with a <code>List</code> of instances of 
	 * {@link org.eclipse.core.resources.IFile} or with the values defined by the
	 * visitor for each file extension.
	 */
	public static Map getFiles(IWorkspaceRoot workspaceRoot, String[] fileExtensions, IGetFilesVisitor visitor)
	{
		Map filesByFileExtension = new HashMap(fileExtensions.length, 1.1f);
		for(int i = 0, maxi = fileExtensions.length; i < maxi; i++)
			filesByFileExtension.put(fileExtensions[i], new ArrayList());
			
		if(visitor == null)
		{
			visitor = new IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					return file;
				}
			};
		}
			
		IProject[] projects = workspaceRoot.getProjects();
		for(int i=0; i<projects.length; i++)
			getFiles(projects[i], filesByFileExtension, visitor);
			
		return filesByFileExtension;
	}

	/**
	 * Returns a map with all the "files" that are "related" to a given resource
	 * and that have their file extension listed in the file extensions array.
	 * 
	 * <p>This method looks for files recursively if the resource is a project 
	 * or a folder.
	 * 
	 * <p>In order to be added to the map the file must exists and must belong to
	 * an open project.
	 * 
	 * <p>Clients can specify a visitor to be used by this method.  The visitor
	 * is able to defined the object that added to the map - that's the reason
	 * the word file is between quotes in the paragraphs above. If the visitor 
	 * is <code>null</code> then the file is added to the map
	 *  
	 * @param resource
	 * @param fileExtensions. 
	 * @return a Map with a <code>List</code> of instances of 
	 * {@link org.eclipse.core.resources.IFile} or with the values defined by the
	 * visitor for each file extension.
	 */
	public static Map getFiles(IResource resource, String[] fileExtensions, IGetFilesVisitor visitor)
	{
		if((resource == null) || (!resource.exists()))
			return Collections.EMPTY_MAP;
		
		if(visitor == null)
		{
			visitor = new IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					return file;
				}
			};
		}

		Map filesByFileExtension = new HashMap(fileExtensions.length, 1.1f);
		for(int i = 0, maxi = fileExtensions.length; i < maxi; i++)
			filesByFileExtension.put(fileExtensions[i], new ArrayList());
		getFiles(resource, filesByFileExtension, visitor);
		
		return filesByFileExtension;
	}

	/**
	 * Internal implementation of {@link #getFiles(IResource, String[]).  The map
	 * is supposed to have a list for all the required file extensions.
	 * @param resource
	 * @param filesByFileExtension
	 * @param a not <code>null</code> visitor
	 */
	private static void getFiles(IResource resource, Map filesByFileExtension, IGetFilesVisitor visitor)
	{
		if((resource == null) || (!resource.exists()))
			return;
			
		switch(resource.getType())
		{
			case IResource.FILE:
				String fileExtension = resource.getFileExtension();
				if(fileExtension != null)
				{
					List files = (List)filesByFileExtension.get(fileExtension);
					if(files != null)
					{
						Object object = visitor.visit((IFile)resource);
						if(object != null)
							files.add(object);
					}
				}
				break;
				
			case IResource.PROJECT:
				IProject project = (IProject)resource;
				if(!project.isOpen())
					return;
					
			case IResource.FOLDER:
				try
				{
					IResource[] members = ((IContainer)resource).members();
					for(int i=0; i< members.length; i++)
						getFiles(members[i], filesByFileExtension, visitor);
				}
				catch (CoreException e)
				{
					HyadesUIPlugin.logError(e);
				}
				break;
		}
	}
	
	/**
	 * Returns a list with the adapters retrieved for each object in the
	 * specified list.  
	 * 
	 * <p>An object from the <code>objects</code> list that is not an instance of 
	 * <code>IAdaptable</code> but is an instance of the <code>adapterClass</code>,
	 * is added to the returned list.
	 * 
	 * <p>The returned list has no <b>null</b> elements.
	 * 
	 * @param objects
	 * @param adapterClass
	 * @return List
	 */
	public static List getValidAdapters(List objects, Class adapterClass)
	{
		List adapters = new ArrayList(objects.size());
		for (Iterator i = objects.iterator(); i.hasNext();)
		{
			Object object = i.next();
			if(object == null)
				continue;
				
			if(object instanceof IAdaptable)
			{
				Object adapter = ((IAdaptable)object).getAdapter(adapterClass);
				if(adapter != null)
					adapters.add(adapter);
			}
			else if(adapterClass.isInstance(object))
			{
				adapters.add(object);
			}
		}
		return adapters;
	}
	
	public static String getFullPath(IPath path)
	{
		if(path == null)
			return null;
			
		IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path.makeAbsolute());
		if(res != null)
			path = res.getLocation().makeAbsolute();
			
		return path.toString();
	}	
}
