/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;

import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.util.internal.PerftraceUtil;
import org.eclipse.hyades.trace.views.util.internal.SpectrumColorMap;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.widgets.Display;

public class SinglePatternDrawStrategy
	extends GraphDrawStrategy
	implements org.eclipse.swt.events.KeyListener {
	public final static String ZOOM_IN = "+";
	public final static String ZOOM_OUT = "-";
	protected final float MAG_STEP = 2;

	private final int PREF_TREE_WIDTH = 1000;

	private String tmpString;
	private SinglePattern _parent;
	private int _drawMode =
		TraceUIPlugin.getDefault().getPreferenceStore().getInt(
			TraceConstants.TIME_OPTION);
	private int _indexOfInvocation = 0;
	private TRCFullMethodInvocation _viewroot;
	private int _treewidth;
	private long _maxtime = 0;
	private float timescale = 1f;
	private boolean _hidedetails = true;

	protected final static int NRLABELS = 10;
	protected final static int MAXINV = 256;
	private TRCMethodInvocation _drawArray[] = new TRCMethodInvocation[MAXINV];
	private float _x[] = new float[MAXINV],
		_y[] = new float[MAXINV],
		_w[] = new float[MAXINV],
		_h[] = new float[MAXINV];
	private float _canvasVisibleBottom,
		_canvasVisibleTop,
		_canvasVisibleLeft,
		_canvasVisibleRight;
		
	private ArrayList _calleeList = new ArrayList();	

	public SinglePatternDrawStrategy(SinglePattern parent) {
		super();
		_parent = parent;
		setNonPropZoom(true);
	}
	public void bgRedraw(GC gc) {

		try {

			JCanvas c = jcanvas();
			if (c == null || _viewroot == null)
				return;
				
			Display display = Display.getCurrent();
			if (display == null)
				display = Display.getDefault();
			// cache these values to use inside drawTree()
			_canvasVisibleBottom = c.visibleBottom();
			_canvasVisibleTop = c.visibleTop();
			_canvasVisibleLeft = c.visibleLeft();
			_canvasVisibleRight = c.visibleRight();

			_hidedetails = true;
			initDrawArray();
			drawTree(
				gc,
				_viewroot,
				0,
				cellWidth(),
				titleMargin()
					- (int) (TString.getTime(_viewroot.getEntryTime())
						* timescale));
			flushDrawArray(gc);

			_hidedetails = false;
			initDrawArray();
			drawTree(
				gc,
				_viewroot,
				0,
				cellWidth(),
				titleMargin()
					- (int) (TString.getTime(_viewroot.getEntryTime())
						* timescale));
			flushDrawArray(gc);
			drawTimeMarks(gc, 0, 0);
		} catch (StackOverflowError exc) {
		}

	}
	protected int cellHeight() {
		return 20;
	}
	protected int cellWidth() {
		return 20;
	}
	public int current() {

		return _indexOfInvocation + 1;
	}

	/**
	 * 
	 */
	protected void drawInvocation(
		GC gc,
		TRCMethodInvocation inv,
		float x,
		float y,
		float w,
		float h,
		boolean showString) {
		JCanvas c = jcanvas();
		if (c == null)
			return;

		float yscale = c.yscale();
		Display display = Display.getCurrent();
		if (display == null)
			display = Display.getDefault();

		String msg = inv.getMethod().getName();

		Color colmsg = display.getSystemColor(SWT.COLOR_BLACK);
		if ((inv.getMethod().getModifier() & TRCMethodProperties.JAVA_CONSTRUCTOR) != 0)
			colmsg = display.getSystemColor(SWT.COLOR_RED);
		TRCObject o = inv.getOwningObject();
		String clss = PerftraceUtil.getClass(inv).getName();
		Color col = SpectrumColorMap.color(clss);

		double scale = c.xscale();
		if (showString && scale > .5 && !_hidedetails) {
			int maxchars = (int) (w * scale / 5); // font width about 5 pix
			if (msg.length() > maxchars)
				msg = msg.substring(0, maxchars);

			float fontheight = gc.getFontMetrics().getHeight() / yscale;
			c.drawString(gc, msg, x, y + fontheight, colmsg);
		}

		if (scale * w > 20.) {
			c.drawLine(gc, x, y, x + 3 * w / 4, y, colmsg);
			c.fillRect(gc, x + 3 * w / 4, y, w / 4, h, col);

		} else
			c.fillRect(gc, x + w / 4, y, w * 3 / 4, h, col);
	}
	protected void drawNextPattern() {

		if (_viewroot != null && _viewroot.getMethod().getInvocations().size() > 0) {
			_viewroot =
				(TRCFullMethodInvocation) _viewroot.getMethod().getInvocations().get(
					_indexOfInvocation++);
					
					
		}
		else
		{
			_viewroot = null;
			_indexOfInvocation=0;
		}
			
		setTitle();	
		_parent.updateButtons();
		
		jcanvas().zoomToFill(1f, 1000f);
		jcanvas().redraw();
		
	}

	/**
	 * 
	 */
	protected void setTitle() {
		
		StringBuffer buffer = new StringBuffer();
				
		if(_viewroot == null)
		{
		  tmpString = "";
		}  
		else
		{  
			TRCMethod method = _viewroot.getMethod();
			if (method != null) {
				tmpString = TraceUIPlugin.getString("METHOD_A");
				tmpString =
					TString.change(
						tmpString,
						"%1",
						method.getDefiningClass().getName()
							+ "."
							+ method.getSignature());
			}
	
          
			buffer.append(tmpString).append(" : ").append(_indexOfInvocation+1).append("/").append(method.getInvocations().size());
		}
				
		_parent.updateTitle(buffer.toString());
	}

	/**
	 * 
	 */
	protected void drawPattern(int i) {
		
		_viewroot = (TRCFullMethodInvocation) _viewroot.getMethod().getInvocations().get(i);
		_indexOfInvocation = i;
		setTitle();
		_parent.updateButtons();
		
		jcanvas().zoomToFill(1f, 1000f);
		jcanvas().redraw();
		
	}

	/**
	 * 
	 */
	protected void drawPreviousPattern() {
		
		_viewroot =
			(TRCFullMethodInvocation) _viewroot.getMethod().getInvocations().get(
				--_indexOfInvocation);
		setTitle();
		_parent.updateButtons();		
		
		jcanvas().zoomToFill(1f, 1000f);
		jcanvas().redraw();
		
	}

	/**
	 * 
	 */
	protected void drawTimeMarks(GC gc, int offx, int offy) {
		JCanvas c = jcanvas();
		if (isDirty() || c == null)
			return;

		Display display = Display.getCurrent();
		if (display == null)
			display = Display.getDefault();

		float margin = 50 / c.xscale();
		float rightvis = c.visibleRight();
		long toptime = (long) ((c.visibleTop() - titleMargin()) / timescale);
		long heighttime = (long) (c.visibleHeight() / timescale);
		long interval = heighttime / NRLABELS;
		float x = rightvis - margin;

		float fontheight = gc.getFontMetrics().getHeight() / c.yscale();
		if (interval > 1) {
			long niceinterval =
				(long) Math.pow(
					10.,
					((int) (Math.log(interval) / Math.log(10)) + 1))
					/ 2;
			long newtoptime =
				niceinterval * ((long) (toptime / niceinterval + 1));
			for (int i = 0; i < NRLABELS * 2; ++i) {
				long timemark = newtoptime + i * niceinterval;
				float y = timemark * timescale + titleMargin();

				c.drawString(
					gc,
					PerftraceUtil.formatTimeValue((double)timemark/1000000),
					offx + x,
					offy + y,
					display.getSystemColor(SWT.COLOR_BLACK),
					display.getSystemColor(SWT.COLOR_GRAY));

			}
		}

		c
			.drawString(
				gc,
				TraceUIPlugin.getString("STR_GRAPH_UNIT"),
				offx + x - 20,
		// The -20 is needed for Spanish translation of "[seconds]" (i.e. "[segundos]").
		c.visibleTop() + fontheight,
			display.getSystemColor(SWT.COLOR_BLACK),
			display.getSystemColor(SWT.COLOR_GRAY));
	}
	protected void drawTree(
		GC gc,
		TRCMethodInvocation invocation,
		int depth,
		float offx,
		float offy) {

		JCanvas canvas = jcanvas();
		if (invocation == null || !(invocation instanceof TRCFullMethodInvocation)
		   || canvas == null || isDirty())
			return;

        TRCFullMethodInvocation inv = (TRCFullMethodInvocation) invocation;
		long start = PerftraceUtil.getTime(inv.getEntryTime());
		long end = PerftraceUtil.getTime(inv.getExitTime());
		if (end < 0)
			end = maxTime();

		if (offy + start * timescale > _canvasVisibleBottom
			|| offy + end * timescale < _canvasVisibleTop) {
			return;
		}

		float height = (end - start) * timescale;
		if (_hidedetails && canvas.yscale() * height < 10f) {
			return;
		}

		Display display = Display.getCurrent();
		if (display == null)
			display = Display.getDefault();

		ITraceSelection model =
			UIPlugin.getDefault().getSelectionModel(
				_parent.getPage().getMOFObject());
		if (model.contains(inv) || model.contains(inv.getMethod())) {
			float fontheight =
				gc.getFontMetrics().getHeight() / canvas.yscale();
			canvas.fill3DRect(
				gc,
				offx + (4 * depth) * cellWidth(),
				offy + start * timescale,
				treeWidth() - (4 * depth + 2) * cellWidth(),
				Math.max(height, fontheight),
				SpectrumColorMap.getSelectionColor());
		} else if (model.contains(inv.getOwningObject())) {
			canvas.fill3DRect(
				gc,
				offx + (4 * depth + 3) * cellWidth() - 5f,
				offy + start * timescale,
				treeWidth() - ((4 * depth + 5) * cellWidth()) + 5f,
				height,
				SpectrumColorMap.getSelectionColor());
		}

		if (offx + (depth + 1) * cellWidth() * 4 < _canvasVisibleRight) {

			Object[] segments = inv.getInvokes().toArray();
			for (int i = 0; i < segments.length; ++i) {
				TRCFullMethodInvocation kid =
					((TRCFullMethodInvocation) segments[i]);
				if (PerftraceUtil.getTime(kid.getEntryTime()) * timescale * offy
					> _canvasVisibleBottom) {
					break;
				} else {
					long kidend = PerftraceUtil.getTime(kid.getExitTime());
					if (kidend < 0)
						kidend = maxTime();
					if (kidend * timescale + offy < _canvasVisibleTop) {
						continue;
					} else
						drawTree(gc, kid, depth + 1, offx, offy);
				}
			}
		}

		if (depth < MAXINV)
			push(
				gc,
				depth,
				inv,
				offx + depth * cellWidth() * 4,
				offy + start * timescale,
				cellWidth() * 4,
				height);
		else // buffer for optimization is too small
			drawInvocation(
				gc,
				inv,
				offx + depth * cellWidth() * 4,
				offy + start * timescale,
				cellWidth() * 4,
				height,
				true);
	}
	/**
	 * This method should fill (or update) the given menu. It is called just before
	 * the menu opens. 
	 */
	public void fillContextMenu(IMenuManager menu) {
		super.fillContextMenu(menu);
	}
	protected void flushDrawArray(GC gc) {
		for (int i = 0; i < MAXINV; ++i) {
			if (_drawArray[i] != null) {
				drawInvocation(
					gc,
					_drawArray[i],
					_x[i],
					_y[i],
					_w[i],
					_h[i],
					true);
			}
		}
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (6/6/2001 5:53:07 PM)
	 * @return boolean
	 */
	public boolean hasCaller() {
		return (_viewroot != null && _viewroot.getInvokedBy() != null);
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (6/6/2001 5:53:07 PM)
	 * @return boolean
	 */
	public boolean hasCallee() {
		
		return (_calleeList.size() > 0);
	}
	
	/**
	 * 
	 */
	public boolean hasLessElements() {
		return _indexOfInvocation > 0;
	}
	
	/**
	 * 
	 */
	public boolean hasMoreElements() {
		return _viewroot != null && _indexOfInvocation < _viewroot.getMethod().getInvocations().size()-1;
	}
	
	/**
	 * 
	 */
	public float height() {
		if (_viewroot != null) {
			return (maxTime() - PerftraceUtil.getTime(_viewroot.getEntryTime())) * timescale
				+ titleMargin()
				+ cellWidth();
		} else
			return titleMargin() + cellWidth();
	}
	
	/**
	 * 
	 */
	protected void initDrawArray() {
		for (int i = 0; i < MAXINV; ++i) {
			_drawArray[i] = null;
			_x[i] = _y[i] = _w[i] = _h[i] = 0f;
		}
	}
	public boolean isSearchable() {
		return UIPlugin
			.getDefault()
			.getSelectionModel(_parent.getPage().getMOFObject())
			.size()
			> 0;
	}
	public void keyPressed(org.eclipse.swt.events.KeyEvent e) {

		JCanvas canvas = jcanvas();
		if (canvas == null)
			return;

		if (e.keyCode == SWT.CONTROL) {
			_controlDown = true;

		} else if (e.keyCode == SWT.HOME) {
			canvas.zoomToFill(1f, 1000f);
		} else {
			float x = canvas.normX(canvas.getSize().width / 2);
			float y = canvas.normY(canvas.getSize().height / 2);
			if (e.character == ',' || e.character == '<') //||
				//				e.keyCode == SWT.DIVIDE)
				{
				canvas.zoom(1f, 1 / MAG_STEP, x, y);

			} else if (e.character == '.' || e.character == '>') //||
				//					   e.character == SWT.MULTIPLY)
				{
				canvas.zoom(1f, MAG_STEP, x, y);
			}
		}
	}
	protected int maxDepth() {
		if (_viewroot != null)
			return _viewroot.getStackDepth();
		else
			return 0;
	}
	protected long maxTime() {
		
		long largestTime = 0;
		
		if (_viewroot != null) {
			largestTime = PerftraceUtil.getTime(_viewroot.getExitTime());
			if (largestTime == 0) {
				largestTime = PerftraceUtil.getTime(_viewroot.getOwningObject().getProcess().getLastEventTime()); 
			}
		}
		
		return largestTime;
	}
	public void mouseDown(org.eclipse.swt.events.MouseEvent e) {
		if (e.button == 3) //right click buton
			return;

		super.mouseDown(e);

	}
	public void mouseUp(org.eclipse.swt.events.MouseEvent e) {

		if (e.button == 3) //right click buton
			return;

		super.mouseUp(e);
	}
	public void moved(float x, float y) {
		if (_viewroot == null)
			return;

		TRCFullMethodInvocation selinv =
			subtreeContains(
				_viewroot,
				x - cellWidth(),
				y
					+ (int) (PerftraceUtil.getTime(_viewroot.getEntryTime())
						* timescale),
				0);
		if (selinv != null) {
			TRCObject obj = selinv.getOwningObject();

			String implname = selinv.getMethod().getDefiningClass().getName();
			String msgname = selinv.getMethod().getName();
			String objclname = PerftraceUtil.getClass(selinv).getName();
			
			double time = selinv.getExitTime();
			if(time == 0)
			  time = obj.getProcess().getLastEventTime();
			
			String statusMsg = TraceUIPlugin.getString("A_ON_B_AT_C_TIME");
			if (_drawMode == TraceConstants.RAW_TIME)
				statusMsg = TraceUIPlugin.getString("A_ON_B_AT_RAW_TIME");

			statusMsg =
				TString.change(statusMsg, "%1", implname + "." + msgname);
			statusMsg =
				TString.change(statusMsg, "%2", objclname + "." + obj.getId());
			statusMsg =
				TString.change(
					statusMsg,
					"%3",
					TString.formatTimeValue(selinv.getEntryTime()));
			statusMsg =
				TString.change(
					statusMsg,
					"%4",
					TString.formatTimeValue(time - selinv.getEntryTime()));

			status(statusMsg);
		} else {
			long tm = (long) ((y - titleMargin()) / timescale);

			String statusMsg = TraceUIPlugin.getString("TIME_X");
			statusMsg =
				TString.change(
					statusMsg,
					"%1",
					TString.formatTimeValue((double)tm/1000000));
			status(statusMsg);
		}
	}
	protected void push(
		GC gc,
		int i,
		TRCMethodInvocation inv,
		float x,
		float y,
		float w,
		float h) {
		JCanvas canvas = jcanvas();
		if (i >= MAXINV || canvas == null)
			return;

		if (_drawArray[i] != null) {
			float yscale = canvas.yscale();
			float spaceforstring = (y - _y[i]) * yscale;
			float spaceforinvoc = (y + h - _y[i]) * yscale;
			if (spaceforinvoc < 1.)
				return;
			drawInvocation(
				gc,
				_drawArray[i],
				_x[i],
				_y[i],
				_w[i],
				_h[i],
				(spaceforstring > 8.));

		}
		_drawArray[i] = inv;
		_x[i] = x;
		_y[i] = y;
		_w[i] = w;
		_h[i] = h;
	}
	public void redraw() {
		setDirty();
	}
	public void selected(
		float x,
		float y,
		boolean shiftdown,
		boolean controldown,
		boolean metadown) {
		if (_viewroot == null)
			return;

		TRCMethodInvocation selinv =
			subtreeContains(
				_viewroot,
				x - cellWidth(),
				y
					+ (int) (TString.getTime(_viewroot.getEntryTime())
						* timescale),
				0);

		ITraceSelection _selectionmodel =
		UIPlugin.getDefault().getSelectionModel(_parent.getPage().getMOFObject());
		_selectionmodel.add(selinv);

		ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
		event.setSource(_parent.getPage().getMOFObject());
		UIPlugin.getDefault().notifyViewSelectionChangedListener(event);
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (6/6/2001 5:32:09 PM)
	 */
	public void showCallee() {
		if (!hasCallee())
			return;

        _viewroot = (TRCFullMethodInvocation)_calleeList.get(_calleeList.size()-1);
        _calleeList.remove(_viewroot);
		_indexOfInvocation = 0;

        setTitle();
        _parent.updateButtons();
        
		jcanvas().zoomToFill(1f, 1000f);
		jcanvas().redraw();
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (6/6/2001 5:32:09 PM)
	 */
	public void showCaller() {
		if (!hasCaller())
			return;

        _calleeList.add(_viewroot);
		_viewroot = (TRCFullMethodInvocation)_viewroot.getInvokedBy();
		_indexOfInvocation = 0;

        setTitle();
        _parent.updateButtons();

		jcanvas().zoomToFill(1f, 1000f);
		jcanvas().redraw();
	}
	public void shutdown() {
		
		_viewroot = null;
		
		if(_calleeList != null)
			_calleeList.clear();
			
		_drawArray = null;
		_parent = null;
	}
	protected void status(String s) {

		_parent.updateStatus(s);
	}
	protected TRCFullMethodInvocation subtreeContains(
		TRCFullMethodInvocation inv,
		float xs,
		float ys,
		int d) {

		if (inv == null)
			return null;
		int depth = (int) (xs / cellWidth() / 4);
		if (d == depth)
			return inv;
		else if (d > depth)
			return null;
		else {
			
			double maxtime = inv.getOwningObject().getProcess().getLastEventTime();
			
			Object[] segments = inv.getInvokes().toArray();
			for (int i = 0; i < segments.length; ++i) {
				TRCFullMethodInvocation calleeinv =
					((TRCFullMethodInvocation) segments[i]);
					
				double time = calleeinv.getExitTime();
				if(time == 0)
				   time = maxtime;
				   	
				if (ys
					< TString.getTime(calleeinv.getEntryTime()) * timescale
						+ titleMargin())
					return null;
				else if (
					time == 0 || TString.getTime(time) * timescale	+ titleMargin()	>= ys)
					return subtreeContains(calleeinv, xs, ys, d + 1);
			}
			return null;
		}
	}
	protected int titleMargin() {
		return 0;
	}
	protected int treeWidth() {
		return cellWidth() * 4 * (10 + (maxDepth() / 10) * 10);
	}
	public float width() {
		return cellWidth() * 4 * (10 + (maxDepth() / 10) * 10);
	}

	public void setDrawMode(int drawMode) {
		_drawMode = drawMode;
	}

   /**
    * 
    */
	protected void searchPattern()
	{
		_calleeList.clear();
		
		Object obj = UIPlugin.getDefault().getSelectionModel(_parent.getPage().getMOFObject()).getFirstElement();
		if (obj == null) {
			_indexOfInvocation = 0;
			_viewroot = null;
			return;
		}

        if(obj instanceof TRCMethodInvocation)
        {
           _viewroot = (TRCFullMethodInvocation) obj;
           _indexOfInvocation=0;
        }
        else if(obj instanceof TRCMethod)
        {
           if(((TRCMethod)obj).getInvocations().size() == 0)
           {
				_indexOfInvocation = 0;
				_viewroot = null;
				return;           	
           }
           
           _viewroot = (TRCFullMethodInvocation) ((TRCMethod)obj).getInvocations().get(0);
           _indexOfInvocation=0;        	
        }
  
		setTitle();
		_parent.updateButtons();
		resetArea();
           
	}   
}
