/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.test.extension;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.core.resources.ResourcesPlugin;

import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;


/**
 * Constains the methods to test the Hyades UI extension framework.
 * 
 * <p>Features not tested:<OL>
 * <LI>Dispose</LI>
 * <LI>Descriptor retrieval by id</LI>
 * <LI>Descriptor filter</LI>
 * <LI>Descriptor copy</LI>
 * <LI>Avoided descriptor for the mapping</LI>
 * <LI>checkpoint and rollback</LI>
 * <LI>preferences</LI>
 * <LI>commit listeners</LI>
 * </OL>
 *  
 * @author marcelop
 * @since 0.0.1
 */
public class ExtensionTestCase 
extends TestCase
{
	/*
	 * This plugin's identifier.
	 */
	private static final String PLUGIN_ID = "org.eclipse.hyades.ui.test";
	
	/*
	 * Extension points declared for this test.
	 */
	private static final String EP_ONE_TYPE_ASSOCIATION = "oneTypeAssociations";
	private static final String EP_MULTI_TYPE_ASSOCIATION = "multiTypeAssociations";
	private static final String EP_UNTYPE_ASSOCIATION = "untypedAssociations";
	
	/**
	 * Constructor for ExtensionTestCase
	 * @param name
	 */
	public ExtensionTestCase(String name)
	{
		super(name);
	}

	/**
	 * Returns a suite with the tests defined in this class.
	 * @return Test
	 */
	public static Test suite()
	{
		TestSuite testSuite = new TestSuite("Hyades UI Extension Framework");
		testSuite.addTest(new ExtensionTestCase("testBasicAttributes"));
		
		testSuite.addTest(new ExtensionTestCase("testOneTypeAssociation_Load"));
		testSuite.addTest(new ExtensionTestCase("testOneTypeAssociation_Manipulation"));
		testSuite.addTest(new ExtensionTestCase("testOneTypeAssociation_Default"));
		
		testSuite.addTest(new ExtensionTestCase("testMultiTypeAssociation_Load"));
		testSuite.addTest(new ExtensionTestCase("testMultiTypeAssociation_Manipulation"));
		testSuite.addTest(new ExtensionTestCase("testMultiTypeAssociation_Default"));

		testSuite.addTest(new ExtensionTestCase("testUntypeAssociation_Load"));
		testSuite.addTest(new ExtensionTestCase("testUntypeAssociation_Manipulation"));
		testSuite.addTest(new ExtensionTestCase("testUntypeAssociation_Default"));

		return testSuite;
	}
	
	/**
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() throws Exception
	{
		assertNotNull("This test is supposed to run inside an Eclipse context", ResourcesPlugin.getWorkspace());
	}

	public void testBasicAttributes()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		assertEquals("category1", registry.getObjectCategory());

		IAssociationMapping mapping = registry.retrieveAssociationMapping("mapping1", PLUGIN_ID);
		assertEquals(PLUGIN_ID, mapping.getPluginId());
		assertEquals(registry, mapping.getAssociationMappingRegistry());
		assertEquals("mapping1", mapping.getExtensionPoint());
		assertEquals(mapping, registry.retrieveAssociationMapping("mapping1", PLUGIN_ID));
		assertEquals(mapping, registry.getAssociationMapping("mapping1"));
		
		registry.retrieveAssociationMapping("mapping2", PLUGIN_ID);
		assertEquals(2, registry.getAssociationMappings().length);

		IAssociationDescriptor descriptor = mapping.createAssociationDescriptor();
		descriptor.setName("name");
		descriptor.setDescription("description");
		descriptor.setExtension("fileExtension");
		descriptor.setId("id");

		assertEquals(mapping, descriptor.getAssociationMapping());
		assertEquals(1, mapping.associationDescriptors().length);
		assertEquals(descriptor, mapping.associationDescriptors()[0]);
		assertEquals(1, mapping.getAssociationDescriptors("type1").length);
		assertEquals(descriptor, mapping.getAssociationDescriptors("type1")[0]);
				
		assertTrue(descriptor.applyToAllTypes());
		assertTrue(descriptor.applyTo("type2"));
		assertEquals("name", descriptor.getName());
		assertEquals("description", descriptor.getDescription());
		assertEquals("fileExtension", descriptor.getExtension());
		assertEquals("id", descriptor.getId());
		assertNull(descriptor.getImplementationClass());
		assertNull(descriptor.getImage());
		assertNull(descriptor.getImageDescriptor());
		
		Exception exception = null;
		try
		{
			mapping.setDefaultAssociationDescriptor(null, descriptor);
		}
		catch (IllegalArgumentException e)
		{
			exception = e;
		}
		assertNotNull(exception);
		assertTrue(exception instanceof IllegalArgumentException);
		
		exception = null;
		try
		{
			mapping.getAssociationDescriptors(null);
		}
		catch (IllegalArgumentException e)
		{
			exception = e;
		}
		assertNotNull(exception);
		assertTrue(exception instanceof IllegalArgumentException);		
		
		exception = null;
		try
		{
			mapping.getDefaultAssociationDescriptor(null);
		}
		catch (IllegalArgumentException e)
		{
			exception = e;
		}
		assertNotNull(exception);
		assertTrue(exception instanceof IllegalArgumentException);
		
		assertEquals(descriptor, mapping.getAssociationDescriptor("type2", "id"));
		assertNull(mapping.getAssociationDescriptor("type2", "fakeId"));		
		assertNull(mapping.getAssociationDescriptor("fakeId"));
		
		IAssociationDescriptor newDescriptor = mapping.createAssociationDescriptor();
		newDescriptor.setId("aNewId");
		
		assertEquals(newDescriptor, mapping.getAssociationDescriptor("aNewId"));
		assertEquals(newDescriptor, mapping.getAssociationDescriptor("type2", "aNewId"));
		assertEquals(newDescriptor, mapping.getAssociationDescriptor("type3", "aNewId"));
		
	}
	
	public void testOneTypeAssociation_Load()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_ONE_TYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(registry, mapping.getAssociationMappingRegistry());
		assertEquals(EP_ONE_TYPE_ASSOCIATION, mapping.getExtensionPoint());
		
		assertEquals(2, mapping.getAssociationDescriptors("type1").length);
		assertEquals(1, mapping.getAssociationDescriptors("type2").length);
		assertEquals(4, mapping.associationDescriptors().length);
		assertEquals(1, mapping.getAssociationDescriptors("type").length);
		assertEquals(0, mapping.getAssociationDescriptors().length);
		
		assertNotNull(mapping.getDefaultAssociationDescriptor("type1"));
		assertNotNull(mapping.getDefaultAssociationDescriptor("type2"));
		assertNotNull(mapping.getDefaultAssociationDescriptor("type"));
		assertNull(mapping.getDefaultAssociationDescriptor("type3"));
		assertNull(mapping.getDefaultAssociationDescriptor());
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors("type1");

		assertTrue(descriptors[0].applyTo("type1"));
		assertFalse(descriptors[0].applyToAllTypes());
		assertNull(descriptors[0].getImage());
		assertNull(descriptors[0].getImageDescriptor());
		assertNull(descriptors[0].getExtension());
		assertNull(descriptors[0].getImplementationClass());
		assertTrue(descriptors[1].applyTo("type1"));
		assertFalse(descriptors[1].applyToAllTypes());
		assertNull(descriptors[1].getImage());
		assertNull(descriptors[1].getImageDescriptor());
		assertNull(descriptors[1].getExtension());
		assertNull(descriptors[1].getImplementationClass());

		if(descriptors[0].getName().indexOf("First") >= 0)
		{
			assertNull(descriptors[0].getId());
			assertEquals("Description for: First One type association - Type1", descriptors[0].getDescription());
			assertEquals("First One type association - Type1", descriptors[0].getName());
			assertEquals("SOTA-Type1", descriptors[1].getId());
			assertEquals("Description for: Second One type association - Type1", descriptors[1].getDescription());
			assertEquals("Second One type association - Type1", descriptors[1].getName());			
		}
		else
		{
			assertNull(descriptors[1].getId());
			assertEquals("Description for: First One type association - Type1", descriptors[1].getDescription());
			assertEquals("First One type association - Type1", descriptors[1].getName());
			assertEquals("SOTA-Type1", descriptors[0].getId());
			assertEquals("Description for: Second One type association - Type1", descriptors[0].getDescription());
			assertEquals("Second One type association - Type1", descriptors[0].getName());			
		}
		
		descriptors = mapping.getAssociationDescriptors("type2");
		assertTrue(descriptors[0].applyTo("type2"));
		assertFalse(descriptors[0].applyToAllTypes());
		assertNull(descriptors[0].getImage());
		assertNull(descriptors[0].getImageDescriptor());
		assertNull(descriptors[0].getExtension());
		assertNull(descriptors[0].getId());
		assertNull(descriptors[0].getImplementationClass());
		assertEquals("Description for: Third One type association - Type2", descriptors[0].getDescription());
		assertEquals("Third One type association - Type2", descriptors[0].getName());

		descriptors = mapping.getAssociationDescriptors("type");
		assertTrue(descriptors[0].applyTo("type"));
		assertFalse(descriptors[0].applyToAllTypes());
		assertNull(descriptors[0].getImage());
		assertNull(descriptors[0].getImageDescriptor());
		assertNull(descriptors[0].getImplementationClass());
		assertEquals("extension", descriptors[0].getExtension());
		assertEquals("id", descriptors[0].getId());
		assertEquals("description", descriptors[0].getDescription());
		assertEquals("name", descriptors[0].getName());		
	}
	
	public void testOneTypeAssociation_Manipulation()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_ONE_TYPE_ASSOCIATION, PLUGIN_ID);
		
		assertEquals(0, mapping.getAssociationDescriptors().length);
		
		IAssociationDescriptor descriptor = mapping.createAssociationDescriptor();
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(3, mapping.getAssociationDescriptors("type1").length);
		assertEquals(2, mapping.getAssociationDescriptors("type2").length);
		assertEquals(2, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors("type1");
		IAssociationDescriptor pluginDescriptor = null;
		if(descriptors[0].applyToAllTypes())
			pluginDescriptor = descriptors[1];
		else
			pluginDescriptor = descriptors[0];
			
		pluginDescriptor.removeType("type1");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(3, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(2, mapping.getAssociationDescriptors().length);
		
		descriptor.addType("type");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(2, mapping.getAssociationDescriptors("type1").length);
		assertEquals(2, mapping.getAssociationDescriptors("type2").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);
		
		descriptor.addType("type2");		

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(2, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);
		
		descriptor.removeType("type1");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(2, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);		
	}
	
	public void testOneTypeAssociation_Default()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mappingFake = registry.retrieveAssociationMapping("fake", PLUGIN_ID);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_ONE_TYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(0, mapping.getAssociationDescriptors().length);
		
		IAssociationDescriptor fakeDescriptor = mappingFake.createAssociationDescriptor();
		fakeDescriptor.setId("fake1");
		Exception exception = null;
		try
		{
			mapping.setDefaultAssociationDescriptor("category1", fakeDescriptor);
		}
		catch (IllegalArgumentException e)
		{
			exception = e;
		}
		assertNotNull(exception);
		assertTrue(exception instanceof IllegalArgumentException);
		
		IAssociationDescriptor descriptor = mapping.getDefaultAssociationDescriptor("type1");
		assertNotNull(descriptor);
		assertTrue(descriptor.applyTo("type1"));
		assertEquals(descriptor, mapping.getAssociationDescriptors("type1")[0]);
		
		assertEquals(0, mapping.getAssociationDescriptors().length);
		IAssociationDescriptor newDescriptor = mapping.createAssociationDescriptor();
		assertEquals(1, mapping.getAssociationDescriptors().length);
		assertEquals(descriptor, mapping.getAssociationDescriptors("type1")[0]);
		mapping.setDefaultAssociationDescriptor("type1", newDescriptor);
		assertEquals(newDescriptor, mapping.getAssociationDescriptors("type1")[0]);
		assertEquals(newDescriptor, mapping.getDefaultAssociationDescriptor("type1"));
		newDescriptor.addType("type5");
		assertEquals(0, mapping.getAssociationDescriptors().length);
		assertTrue(newDescriptor.applyTo("type5"));
		assertEquals(1, mapping.getAssociationDescriptors("type5").length);
		assertEquals(descriptor, mapping.getAssociationDescriptors("type1")[0]);
		assertEquals(newDescriptor, mapping.getAssociationDescriptors("type5")[0]);
		assertEquals(newDescriptor, mapping.getDefaultAssociationDescriptor("type5"));
		
		assertEquals(1, mappingFake.associationDescriptors().length);
		assertEquals(1, mappingFake.getAssociationDescriptors("fakeType1").length);
		assertEquals(fakeDescriptor, mappingFake.associationDescriptors()[0]);
		assertEquals(fakeDescriptor, mappingFake.getAssociationDescriptors("fakeType1")[0]);
		assertEquals(fakeDescriptor, mappingFake.getDefaultAssociationDescriptor("fakeType1"));
		assertEquals(fakeDescriptor, mappingFake.getDefaultAssociationDescriptor("fakeType2"));
		
		IAssociationDescriptor fakeDescriptor2 = mappingFake.createAssociationDescriptor();
		fakeDescriptor2.setId("fake2");
		
		assertEquals(2, mappingFake.associationDescriptors().length);
		assertEquals(2, mappingFake.getAssociationDescriptors("fakeType1").length);
		assertEquals(fakeDescriptor, mappingFake.getAssociationDescriptors("fakeType1")[0]);
		assertEquals(fakeDescriptor, mappingFake.getDefaultAssociationDescriptor("fakeType1"));
		
		mappingFake.setDefaultAssociationDescriptor("fakeType1", fakeDescriptor2);
		assertEquals(fakeDescriptor2, mappingFake.getAssociationDescriptors("fakeType1")[0]);
		assertEquals(fakeDescriptor2, mappingFake.getDefaultAssociationDescriptor("fakeType1"));
	}
	
	public void testMultiTypeAssociation_Load()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_MULTI_TYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(registry, mapping.getAssociationMappingRegistry());
		assertEquals(EP_MULTI_TYPE_ASSOCIATION, mapping.getExtensionPoint());
		
		assertEquals(2, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(1, mapping.getAssociationDescriptors("type3").length);
		assertEquals(4, mapping.associationDescriptors().length);
		assertEquals(1, mapping.getAssociationDescriptors("type").length);
		assertEquals(0, mapping.getAssociationDescriptors().length);
		
		assertNotNull(mapping.getDefaultAssociationDescriptor("type1"));
		assertNotNull(mapping.getDefaultAssociationDescriptor("type2"));
		assertNotNull(mapping.getDefaultAssociationDescriptor("type3"));		
		assertNotNull(mapping.getDefaultAssociationDescriptor("type"));
		assertNull(mapping.getDefaultAssociationDescriptor());
		assertNull(mapping.getDefaultAssociationDescriptor("type4"));
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors("type2");
		assertEquals(descriptors[0], mapping.getDefaultAssociationDescriptor("type2"));
		for (int i = 0, maxi = descriptors.length; i < maxi; i++)
		{
			if(descriptors[i].getName().indexOf("First") >= 0)
			{
				assertTrue(descriptors[i].applyTo("type1"));
				assertTrue(descriptors[i].applyTo("type2"));
				assertEquals("Description for: First One type association - Type1, Type2", descriptors[i].getDescription());
			}
			else if(descriptors[i].getName().indexOf("Second") >= 0)
			{
				assertTrue(descriptors[i].applyTo("type1"));
				assertTrue(descriptors[i].applyTo("type2"));
				assertEquals("SMTA-Type1,Type2", descriptors[i].getId());
			}
			else if(descriptors[i].getName().indexOf("Third") >= 0)
			{
				assertTrue(descriptors[i].applyTo("type2"));
				assertTrue(descriptors[i].applyTo("type3"));
				assertEquals("Description for: Third Multi type association - Type2,Type3", descriptors[i].getDescription());
			}
		}	
	}
	
	public void testMultiTypeAssociation_Manipulation()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_MULTI_TYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(0, mapping.getAssociationDescriptors().length);
		
		IAssociationDescriptor newDescriptor = mapping.createAssociationDescriptor();
		newDescriptor.setId("newDescriptor");
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(3, mapping.getAssociationDescriptors("type1").length);
		assertEquals(4, mapping.getAssociationDescriptors("type2").length);
		assertEquals(2, mapping.getAssociationDescriptors("type3").length);
		assertEquals(2, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);
		
		IAssociationDescriptor pluginDescriptor = null;
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors("type2");
		for(int i = 0, maxi = descriptors.length; i < maxi; i++)
		{
			if(descriptors[i].getName().indexOf("Third") >= 0)
			{
				pluginDescriptor = descriptors[i];
				break;
			} 
		}
		assertNotNull(pluginDescriptor);

		assertFalse(pluginDescriptor.applyToAllTypes());
		assertTrue(pluginDescriptor.applyTo("type2"));
		assertTrue(pluginDescriptor.applyTo("type3"));
		assertFalse(pluginDescriptor.applyTo("type"));
			
		pluginDescriptor.removeType("type2");
		
		assertFalse(pluginDescriptor.applyToAllTypes());
		assertTrue(pluginDescriptor.applyTo("type3"));
		assertFalse(pluginDescriptor.applyTo("type2"));
		assertFalse(pluginDescriptor.applyTo("type"));	
		
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(3, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(2, mapping.getAssociationDescriptors("type3").length);
		assertEquals(2, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);

		pluginDescriptor.removeType("type3");
		
		assertTrue(pluginDescriptor.applyToAllTypes());
		assertTrue(pluginDescriptor.applyTo("type3"));
		assertTrue(pluginDescriptor.applyTo("type2"));
		assertTrue(pluginDescriptor.applyTo("type"));
		
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors("type1").length);
		assertEquals(4, mapping.getAssociationDescriptors("type2").length);
		assertEquals(2, mapping.getAssociationDescriptors("type3").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(2, mapping.getAssociationDescriptors().length);

		pluginDescriptor.addType("type");		

		assertFalse(pluginDescriptor.applyToAllTypes());
		assertFalse(pluginDescriptor.applyTo("type3"));
		assertFalse(pluginDescriptor.applyTo("type2"));
		assertTrue(pluginDescriptor.applyTo("type"));
		
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(3, mapping.getAssociationDescriptors("type1").length);
		assertEquals(3, mapping.getAssociationDescriptors("type2").length);
		assertEquals(1, mapping.getAssociationDescriptors("type3").length);
		assertEquals(3, mapping.getAssociationDescriptors("type").length);
		assertEquals(1, mapping.getAssociationDescriptors().length);
	}
	
	public void testMultiTypeAssociation_Default()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_MULTI_TYPE_ASSOCIATION, PLUGIN_ID);
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors("type2");
		assertEquals(descriptors[0], mapping.getDefaultAssociationDescriptor("type2"));
		
		IAssociationDescriptor newDescriptor = mapping.createAssociationDescriptor();
		
		assertTrue(newDescriptor.applyToAllTypes());
		assertTrue(newDescriptor.applyTo("type2"));
		assertEquals(descriptors[0], mapping.getDefaultAssociationDescriptor("type2"));
		assertEquals(descriptors[0], mapping.getAssociationDescriptors("type2")[0]);
		
		assertTrue(mapping.setDefaultAssociationDescriptor("type2", newDescriptor));

		assertTrue(newDescriptor.applyToAllTypes());
		assertTrue(newDescriptor.applyTo("type2"));
		assertEquals(newDescriptor, mapping.getDefaultAssociationDescriptor("type2"));
		assertEquals(newDescriptor, mapping.getAssociationDescriptors("type2")[0]);
		
		newDescriptor.addType("type2");
		
		assertFalse(newDescriptor.applyToAllTypes());
		assertTrue(newDescriptor.applyTo("type2"));
		assertEquals(newDescriptor, mapping.getDefaultAssociationDescriptor("type2"));
		assertEquals(newDescriptor, mapping.getAssociationDescriptors("type2")[0]);
		
		newDescriptor.removeType("type2");

		assertTrue(newDescriptor.applyToAllTypes());
		assertTrue(newDescriptor.applyTo("type2"));
		assertEquals(mapping.getDefaultAssociationDescriptor("type2"), mapping.getAssociationDescriptors("type2")[0]);

		newDescriptor.addType("type3");
		
		assertFalse(newDescriptor.applyToAllTypes());
		assertFalse(newDescriptor.applyTo("type2"));
		assertFalse(newDescriptor == mapping.getDefaultAssociationDescriptor("type2"));
		assertFalse(newDescriptor == mapping.getAssociationDescriptors("type2")[0]);		
	}
	
	public void testUntypeAssociation_Load()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_UNTYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(registry, mapping.getAssociationMappingRegistry());
		assertEquals(EP_UNTYPE_ASSOCIATION, mapping.getExtensionPoint());
		
		assertEquals(4, mapping.getAssociationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors("type1").length);
		assertEquals(4, mapping.associationDescriptors().length);
		
		assertNotNull(mapping.getDefaultAssociationDescriptor());
		assertNotNull(mapping.getDefaultAssociationDescriptor("type1"));
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors();
		for(int i = 0, maxi = descriptors.length; i < maxi; i++)
		{
			assertTrue(descriptors[i].applyToAllTypes());
		}
	}

	public void testUntypeAssociation_Manipulation()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_UNTYPE_ASSOCIATION, PLUGIN_ID);
		assertEquals(4, mapping.associationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors("type1").length);
		IAssociationDescriptor descriptor = mapping.getAssociationDescriptors()[0];

		IAssociationDescriptor newDescriptor = mapping.createAssociationDescriptor();
		newDescriptor.setId("newDescriptor");
		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors("type1").length);
		
		descriptor.removeType("type2");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors("type1").length);
		
		descriptor.addType("type2");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors("type1").length);
		assertEquals(5, mapping.getAssociationDescriptors("type2").length);
		
		descriptor.removeType("type2");
		descriptor.addType("type1");		

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(4, mapping.getAssociationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors("type1").length);
		assertEquals(4, mapping.getAssociationDescriptors("type2").length);
		
		descriptor.removeType("type1");

		assertEquals(5, mapping.associationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors().length);
		assertEquals(5, mapping.getAssociationDescriptors("type1").length);
		assertEquals(5, mapping.getAssociationDescriptors("type2").length);
	}

	public void testUntypeAssociation_Default()
	{
		IAssociationMappingRegistry registry = HyadesUI.createAssociationMappingRegistry();
		registry.initialize("category1", null, null);
		IAssociationMapping mapping = registry.retrieveAssociationMapping(EP_UNTYPE_ASSOCIATION, PLUGIN_ID);
		
		IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors();
		assertEquals(4, descriptors.length);
		assertEquals(descriptors[0], mapping.getDefaultAssociationDescriptor());
		
		mapping.setDefaultAssociationDescriptor(descriptors[2]);
		assertEquals(descriptors[2], mapping.getDefaultAssociationDescriptor());
		assertEquals(descriptors[2], mapping.getAssociationDescriptors()[0]);
		
		mapping.getAssociationDescriptors()[0].addType("type");
		assertEquals(3, mapping.getAssociationDescriptors().length);
		assertEquals(4, mapping.associationDescriptors().length);
		assertEquals(descriptors[0], mapping.getDefaultAssociationDescriptor());
	}
}