/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.net.UnknownHostException;
import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.Platform;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;


/**
 * This class provides the mechanism for launching Hyades tests.  The 
 * flow in the launchTest method is depicted in the execution use case
 * realization: <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>
 */
public class TestExecutionHarness {
	
	private Object agentLock = new Object();
	private boolean bAgentReady = false;
	private static HashMap testTypeMap = null;
	private static boolean bFactoryInitialized = false;

	/**
	 * 
	 */
	public TestExecutionHarness() {
		super();
	}

	protected void releaseAgentLock() {
		synchronized(agentLock) {
			bAgentReady=true;
			agentLock.notify();
		}
	}


	public void initializeRegisteredExecutionEnvironments(IExecutionComponentFactory factory, boolean bHeadless )
		throws ClassNotFoundException
	{
		if ( !bHeadless )
		{
			IPluginRegistry reg = Platform.getPluginRegistry();
	
			IConfigurationElement[] execComps =
				Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl");
				
			testTypeMap = new HashMap();
				
			for ( int i=0; i<execComps.length; i++)
			{
				String type = execComps[i].getAttribute("type");
				
				if ( type.equalsIgnoreCase("ENVIRONMENT"))
				{
					
					String executionComponentName = execComps[i].getAttribute("name");
					factory.addExecutionComponent(executionComponentName, execComps[i].getAttribute("implClass"));
					factory.addStub(executionComponentName, execComps[i].getAttribute("stubClass"));
					factory.addSkeleton(executionComponentName, execComps[i].getAttribute("skeletonClass"));
					
					Object map = testTypeMap.get(type);
					if ( map == null )
					{
						map = new HashMap();
						testTypeMap.put(type, map);
					}
					HashMap execCompMap = (HashMap)map;
					
					IConfigurationElement[] supportedTestTypes = execComps[i].getChildren("SupportedTestType");
					for ( int j=0; j<supportedTestTypes.length; j++ )
					{
						execCompMap.put(supportedTestTypes[j].getAttribute("name"), executionComponentName);
					}
				}
			}
			
			
			IConfigurationElement[] execEnvAdapters =
				Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter");
				
			for ( int i=0; i<execEnvAdapters.length; i++)
			{
				String adapterClass = execEnvAdapters[i].getAttribute("class");
				String type = "EXECUTION_ENVIRONMENT_ADAPTER";

				Object map = testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execEnvAdapters[i].getChildren("SupportedTestType");
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), adapterClass);
				}
			}
		}
	}

	
	public void initializeRegisteredExecutionComponents(IExecutionComponentFactory factory, boolean bHeadless, boolean bIgnoreExecutionEnvironments )
		throws ClassNotFoundException
	{
		if ( !bHeadless )
		{
			IPluginRegistry reg = Platform.getPluginRegistry();
	
			IConfigurationElement[] execComps =
				Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl");
				
			testTypeMap = new HashMap();
				
			for ( int i=0; i<execComps.length; i++)
			{
				String type = execComps[i].getAttribute("type");
				
				if ( bIgnoreExecutionEnvironments && type.equalsIgnoreCase("ENVIRONMENT"))
					continue;

				String executionComponentName = execComps[i].getAttribute("name");
				factory.addExecutionComponent(executionComponentName, execComps[i].getAttribute("implClass"));
				factory.addStub(executionComponentName, execComps[i].getAttribute("stubClass"));
				factory.addSkeleton(executionComponentName, execComps[i].getAttribute("skeletonClass"));
				
				Object map = testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execComps[i].getChildren("SupportedTestType");
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), executionComponentName);
				}
			}
			
			if ( !bIgnoreExecutionEnvironments )
			{			
			
				IConfigurationElement[] execEnvAdapters =
					Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter");
					
				for ( int i=0; i<execEnvAdapters.length; i++)
				{
					String adapterClass = execEnvAdapters[i].getAttribute("class");
					String type = "EXECUTION_ENVIRONMENT_ADAPTER";
	
					Object map = testTypeMap.get(type);
					if ( map == null )
					{
						map = new HashMap();
						testTypeMap.put(type, map);
					}
					HashMap execCompMap = (HashMap)map;
					
					IConfigurationElement[] supportedTestTypes = execEnvAdapters[i].getChildren("SupportedTestType");
					for ( int j=0; j<supportedTestTypes.length; j++ )
					{
						execCompMap.put(supportedTestTypes[j].getAttribute("name"), adapterClass);
					}
				}
			}
			
			IConfigurationElement[] execObjAdapters =
				Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.ExecutableObjectAdapter");
				
			for ( int i=0; i<execObjAdapters.length; i++)
			{
				String adapterClass = execObjAdapters[i].getAttribute("class");
				String type = "EXECUTABLE_OBJECT_ADAPTER";

				Object map = testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execObjAdapters[i].getChildren("SupportedTestType");
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), adapterClass);
				}
			}

		}
		else
		{
			// TODO: fill in logic for headless determination
		}
	}
	
	public String getExecutionComponentForTestType( String executionComponentType, String testType, boolean bHeadless )
		throws ClassNotFoundException
	{
		if ( !bHeadless )
		{
			Object temp = testTypeMap.get(executionComponentType);
			if ( temp != null && temp instanceof HashMap )
			{
				HashMap execCompMap = (HashMap) temp;
				temp = execCompMap.get(testType);
				if ( temp != null && temp instanceof String )
				{
					return (String) temp;
				}
			}
		}
		
		String msg = ExecutionHarnessPlugin.getString("EXEC_NOT_FOUND_ERR_");
		msg = StringUtil.change(msg, "%1", executionComponentType);
		msg = StringUtil.change(msg, "%2", testType);
		
		// We couldn't find the right execution component
		throw new ClassNotFoundException(msg);
		  		
	}
	
	public String launchTest( ITestSuite suite, ITest theTest , String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bHeadless )
	{
			
		IExecutionComponentFactory factory;
		ISession session;
		INode node;
		boolean bAgentStarted = false;
		String testType = "";
				
		testType = theTest.getType();
		String machine = machineName;
				
		try {
			node = new NodeImpl(machine);
			session = node.connect(port, null);
			
		}
		catch (UnknownHostException e) {
			
			return StringUtil.change(ExecutionHarnessPlugin.getString("TIMEOUT_NODE_ERROR_"), "%1", machine);
		}
		catch (AgentControllerUnavailableException e) {
			
			return StringUtil.change(ExecutionHarnessPlugin.getString("AGENT_UNAV_ERROR_"), "%1", machine);
		}
		catch (UnknownDaemonException e) {
			
			return StringUtil.change(ExecutionHarnessPlugin.getString("AGENT_UNAV_ERROR_"), "%1", machine);
		}
		catch(Exception e)
		{
			ExecutionHarnessPlugin.getDefault().logError(e);			
			return StringUtil.change(ExecutionHarnessPlugin.getString("RUN_FAILED_ERR_"), "%1", machine);
		}
		
		try {
			
			// Get the ExecutionComponentFactory for this session 		
			factory = ExecutionComponentFactoryImpl.getInstance(session);
			if ( !bFactoryInitialized )
				initializeRegisteredExecutionEnvironments(factory, bHeadless);
				
			// Create the appropriate type of execution environment
			String envType = getExecutionComponentForTestType("ENVIRONMENT", testType, bHeadless);
			
			IExecutionEnvironment exeEnvironment =(IExecutionEnvironment)factory.createExecutionComponentByType(envType);
			session.addChild(exeEnvironment);

			// Call out into an adapter class to configure this execution 
			// environment for this test
			IExecutionEnvironmentAdapter envAdapter = null;
			try {
				String envAdapterClass = getExecutionComponentForTestType("EXECUTION_ENVIRONMENT_ADAPTER", testType, bHeadless);
				envAdapter = (IExecutionEnvironmentAdapter)(Class.forName(envAdapterClass).newInstance());
			}
			catch ( Exception accessExc )
			{
				ExecutionHarnessPlugin.getDefault().logError(accessExc);			
				return StringUtil.change(ExecutionHarnessPlugin.getString("EXEC_ENV_ERR_"), "%1", testType);				
			}
			if ( envAdapter != null )
				envAdapter.setupExecutionEnvironment(exeEnvironment, (TPFTestSuite)suite, (TPFTest)theTest);
			
			// Now initialize the rest of the factory
			if ( !bFactoryInitialized )
			{
				initializeRegisteredExecutionComponents(factory, bHeadless, true);
				// TODO: investigate this -- is the session not being reused??
				//bFactoryInitialized = true;
			}
			
			// Create the appropriate type of executor
			String executorType = getExecutionComponentForTestType("EXECUTOR", testType, bHeadless);
			IExecutor executor =(IExecutor)factory.createExecutionComponentByType(executorType);
			exeEnvironment.addChild(executor);

			if ( bMonitor )
			{
				// TODO: Add support for this in next release
			}
			
			// Create the controlling agent
						
			/* Add our hyades component to the executor */
			String agentType = getExecutionComponentForTestType("AGENT", testType, bHeadless);
			IRemoteHyadesComponent agent=(IRemoteHyadesComponent)factory.createExecutionComponentByType(agentType);
			executor.addChild(agent);
			
			/* Start monitoring our hyades component */
			agent.startMonitoring( new XMLExecutionDataProcessor(null, (TPFTestSuite)suite, executionResultLocation, executionResultName));
			
			// Create a state change listener on the Agent.
			agent.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
				public void stateChanged(ExecutionComponentStateChangeEvent event) {
					// When the agent is ready, release the agent lock
					if(event.getState()==IExecutionComponent.READY) {
						releaseAgentLock();
					}	
				}
			});	
			
			
			/* create our executable object */
			String execObjType = getExecutionComponentForTestType("EXECUTABLEOBJECT", testType, bHeadless);
			IExecutableObject executableObject=executor.getCompatibleExecutableObject(execObjType);

			if(executableObject!=null) {

				// Call out into an adapter class to configure this execution 
				// environment for this test
				IExecutableObjectAdapter execObjAdapter = null;
				try {
					String execObjAdapterClass = getExecutionComponentForTestType("EXECUTABLE_OBJECT_ADAPTER", testType, bHeadless);
					execObjAdapter = (IExecutableObjectAdapter)(Class.forName(execObjAdapterClass).newInstance());
				}
				catch ( Exception accessExc )
				{
					ExecutionHarnessPlugin.getDefault().logError(accessExc);			
					return StringUtil.change(ExecutionHarnessPlugin.getString("EXEC_ENV_ERR_"), "%1", testType);					
				}
				if ( execObjAdapter != null )
					execObjAdapter.setupExecutableObject(executableObject, (TPFTestSuite)suite, (TPFTest)theTest);

				
				/* set our executable object, this will move it to the remote side */	
				executor.setExecutableObject(executableObject);
				
				int temp = executor.getState();
				
				if ( agent.getState() == IExecutionComponent.SUSPENDED)
				{
					// Block and wait for the agent to come up.
					while(!bAgentReady) {
						synchronized(agentLock) {
							try {
								agentLock.wait();
							}
							catch(InterruptedException e) {
								/* We can ignore this */
							}
						}
					}
				}
				
				/* start the test */
				executor.launch();
			}
			

		}			
		catch(ClassNotFoundException e) {
			
			ExecutionHarnessPlugin.getDefault().logError(e);			
			return StringUtil.change(ExecutionHarnessPlugin.getString("CLASS_NOT_FOUND_ERR_"), "%1", e.getMessage());			
		}
		catch(Throwable t) {
			
			ExecutionHarnessPlugin.getDefault().logError(t);			
			return t.getMessage();			
			
		}
		finally {
			session.release();
		}
		
		return null;
	}
	
	
	/**
	 * Launches the specified test on the specified machine.
	 * @param rootSuitePath the path to the EMF resource containing the test suite
	 * @param testID the ID of the test being launched
	 * @param machineName the machine name of the machine on which the test will be lanuched
	 * @param port the prt on which the machine is listening for requests
	 * @param bMonitor should we monitor the running test
	 * @param bHeadless are we running headless (without an eclipse shell)
	 */
	public String launchTest( String rootSuitePath, String testID, String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bHeadless )
	{
		ITest theTest = null;
		ITestSuite suite = null;

		// Load the test suite.
		try {
			suite = HyadesFactory.INSTANCE.loadTestSuite(rootSuitePath);
		}
		catch ( Exception exc )
		{
			ExecutionHarnessPlugin.getDefault().logError(exc);
			return StringUtil.change(ExecutionHarnessPlugin.getString("LOAD_SUITE_ERR_"), "%1", rootSuitePath);
		}
		
		// If we're executing a test case and not a suite, get the test case.
		if ( testID != null && testID.length() != 0 )
		{
			theTest = (ITest) HyadesFactory.INSTANCE.getObjectByID(suite, testID);
		}
		else
		{
			theTest = suite;
		}
		
		return launchTest(suite, theTest, machineName, port, executionResultLocation, executionResultName, bMonitor, bHeadless);
		
	}
	
}
