/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.net.UnknownHostException;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.wizard.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

/*
* CONTEXT_ID hstp0001 for Hosts list in Hosts preference page
*/
/*
* CONTEXT_ID hstp0002 for Host name/IP address entry field in Hosts preference page
*/
/*
* CONTEXT_ID hstp0003 for Delete button in host selection page
*/
/*
* CONTEXT_ID hstp0004 for Add button in host selction page
*/
/*
* CONTEXT_ID hstp0005 for Test Connection button in host selection page
*/
/*
* CONTEXT_ID hstp0006 for port number entry field in host selction page
*/

public class TraceHostUI implements SelectionListener
									, ModifyListener
{
	private Button _delete;
	private Button _add;
	private Button _testConnection;
	private Text   _host;
	private Text   _port;
	private Table  _list;
	private WizardPage _wizardPage=null;

/**
 * TraceOptionsUI constructor comment.
 */
public TraceHostUI(WizardPage wp) {
	super();
	setWizardPage(wp);
}

public TraceHostUI() {
	super();
}

private boolean checkPortNumber(String port)
{
	try
	{
		Integer.parseInt(port);
		return true;
	}
	catch (NumberFormatException nfe)
	{
		String text = "Invalid port number "+port;

		Status err =
			new Status(
				Status.WARNING,
				ResourcesPlugin.PI_RESOURCES,
				IResourceStatus.INTERNAL_ERROR,
				text,
				null);
		ErrorDialog
			.openError(
				UIPlugin.getActiveWorkbenchShell(),
				UIPlugin.getResourceString("STR_HYADES_MSG"),
				"",
				err);
		return false;
	}	
}

/**
 * Insert the method's description here.
 * Creation date: (4/26/2001 4:09:19 PM)
 */
public void addHost()
{
	if (checkPortNumber(_port.getText().trim()))
	{
		addHost(_host.getText().trim() + ":" + _port.getText().trim());
		_host.setText("");
		_port.setText(String.valueOf(Constants.CTL_PORT_NUM_SERVER));
		_host.setFocus();
	}
	else
	{
		_port.setSelection(0,_port.getText().length());
		_port.setFocus();
	}
}

/**
 * Insert the method's description here.
 * Creation date: (4/26/2001 5:17:57 PM)
 * @param text java.lang.String
 */
public void addHost(String text)
{
	for(int idx=0; idx<_list.getItemCount(); idx++)
	{
		String data = _list.getItem(idx).getText();
		int i = data.indexOf(":");
		if(i != -1)
		   data = data.substring(0, i);
		   
		if(data.equals(text))
		  return; //already in the list
	}
	
	TableItem item = new TableItem(_list, SWT.NONE);
	item.setText(text);
	item.setImage(PDPluginImages.getImage(PDPluginImages.IMG_UI_NODE));

	_list.setSelection(new TableItem[] {item});
}
/**
 * Insert the method's description here.
 * Creation date: (4/30/2001 12:10:51 PM)
 * @param hosts java.lang.String
 */
private void addHosts(String hosts)
{
	if(hosts != null && !hosts.equals(""))
	{
	   int idx = 0;

	   while(idx != -1)
	   {
		   idx = hosts.indexOf(",");

		   if(idx != -1)
		   {
			   String host = hosts.substring(0, idx);
			   addHost(host);

			   hosts = hosts.substring(idx+1);
			   
		   }
		   else
		   {
			   addHost(hosts);             
		   }
	   }
	}
	
}
/**
 * Insert the method's description here.
 * Creation date: (4/26/2001 3:45:54 PM)
 * @param parent com.ibm.swt.widgets.Composite
 */
public Composite createControl(Composite parent)
{
	GridLayout layout;
	GridData   data;
	
	Composite result = new Composite(parent, SWT.NULL);
	layout = new GridLayout();
	layout.numColumns = 2;
	layout.verticalSpacing=10;
	result.setLayout(layout);
	data = GridUtil.createFill();
	result.setLayoutData(data);

	Label desc = new Label(result, SWT.NULL);
	data = GridUtil.createHorizontalFill();
	data.horizontalSpan = 2;    
	desc.setLayoutData(data);   
	desc.setText(UIPlugin.getResourceString("STR_PREF_HOST_NODE"));

    Composite dataGroup = new Composite(result, SWT.NULL);
	layout = new GridLayout();
	layout.numColumns = 1;
	layout.verticalSpacing=10;
	dataGroup.setLayout(layout);
	data = GridUtil.createFill();
	dataGroup.setLayoutData(data);
    
	_list = new Table(dataGroup, SWT.BORDER|SWT.H_SCROLL);
	data = GridUtil.createFill();
	_list.setLayoutData(data);

	Composite bottomWidget = new Composite (dataGroup, SWT.NULL);
	data = new GridData();
	bottomWidget.setLayoutData(GridUtil.createHorizontalFill());     
	layout = new GridLayout();
	layout.numColumns = 2;
	bottomWidget.setLayout(layout);  

	Label hostLabel = new Label(bottomWidget, SWT.NULL);
	hostLabel.setText(UIPlugin.getResourceString("STR_PREF_HOST_IP"));

	_host = new Text(bottomWidget, SWT.BORDER);
	_host.setLayoutData(GridUtil.createHorizontalFill());

	Label portLabel = new Label(bottomWidget, SWT.NULL);
	portLabel.setText(UIPlugin.getResourceString("RAC_PORT_TEXT"));

	_port = new Text(bottomWidget, SWT.BORDER);
	_port.setLayoutData(GridUtil.createHorizontalFill());
	_port.setText(String.valueOf(Constants.CTL_PORT_NUM_SERVER));

	Composite buttonWidget = new Composite (result, SWT.NULL);
	data = GridUtil.createVerticalFill();
	buttonWidget.setLayoutData(data);    
	layout = new GridLayout();
	layout.numColumns = 1;
	buttonWidget.setLayout(layout);     

	Composite buttonWidget1 = new Composite (buttonWidget, SWT.NULL);
	data = GridUtil.createFill();
	buttonWidget1.setLayoutData(data);    
	layout = new GridLayout();
	layout.numColumns = 1;
	buttonWidget1.setLayout(layout);     
	
	_delete = new Button(buttonWidget1, SWT.NULL);
	_delete.setLayoutData(GridUtil.createHorizontalFill());
	_delete.setText(UIPlugin.getResourceString("DELETE"));

	_testConnection = new Button(buttonWidget1, SWT.NULL);
	_testConnection.setLayoutData(GridUtil.createHorizontalFill());
	_testConnection.setText(UIPlugin.getResourceString("TEST_CONNECTION_TEXT"));
	
	Label l = new Label(buttonWidget1, SWT.NULL);
	l.setLayoutData(GridUtil.createVerticalFill());

	Composite buttonWidget2 = new Composite (buttonWidget, SWT.NULL);
	data = GridUtil.createHorizontalFill();
	buttonWidget2.setLayoutData(data);    
	layout = new GridLayout();
	layout.numColumns = 1;
	buttonWidget2.setLayout(layout);     

	_add = new Button(buttonWidget2, SWT.NULL);
	_add.setLayoutData(GridUtil.createHorizontalFill());    
	_add.setText(UIPlugin.getResourceString("STR_PREF_ADD"));

	_add.addSelectionListener(this);
	_delete.addSelectionListener(this);
	_testConnection.addSelectionListener(this);
	_list.addSelectionListener(this);

	_host.addModifyListener(this);
	_port.addModifyListener(this);

	_add.setEnabled(false);
	_delete.setEnabled(false);
	_testConnection.setEnabled(false);

	initializeValues();
	
	_host.setFocus();
	
	_host.addTraverseListener(new TraverseListener()
		{
				public void keyTraversed(TraverseEvent e)
				{
						if (e.detail == SWT.TRAVERSE_RETURN)
						{
								if (_host.getText().trim()=="" || _port.getText().trim()=="")
									return;
									
			 				    addHost();	                        
								e.doit = false;

							    if (_wizardPage != null)
							   		_wizardPage.setPageComplete(_list.getSelectionIndex() != -1);
								
						}
				}
		});
	
	
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_list,
		UIPlugin.getPluginId()+".hstp0001");
	
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_host,
		UIPlugin.getPluginId()+".hstp0002");
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_delete,
		UIPlugin.getPluginId()+".hstp0003");
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_add,
		UIPlugin.getPluginId()+".hstp0004");
			
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_testConnection,
		UIPlugin.getPluginId()+".hstp0005");
	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		_port,
		UIPlugin.getPluginId()+".hstp0006");	
	
	return result;
}

public Table getListUI()
{
	return _list;
}

public Text getHostUI()
{
	return _host;
}

public Text getPortUI()
{
	return _port;
}

public Button getDeleteUI()
{
	return _delete;
}
public Button getAddUI()
{
	return _add;
}
public void setWizardPage(WizardPage wp)
{
	_wizardPage = wp;
}

/**
 * Insert the method's description here.
 * Creation date: (4/26/2001 4:09:39 PM)
 */
private void deleteHost()
{
   int i = _list.getSelectionIndex();
   if(i != -1)
	 _list.remove(i);

}

private void testConnection()
{
   org.eclipse.swt.custom.BusyIndicator.showWhile(null, new Runnable() {
			public void run() {
				int i = _list.getSelectionIndex();
				if (i != -1) {
					TableItem tblItem = _list.getItem(i);
					String hosts = tblItem.getText();
					String port, host;
				
					if (hosts != null && !hosts.equals("")) {
						int idx = 0;
				
						idx = hosts.indexOf(":");
				
						if (idx != -1) {
							host = hosts.substring(0, idx);
							port = hosts.substring(idx + 1);
				
							HyadesUtil.testConnection(host, port, true);
						}
				
					}
				}
			}
		});
   
}

/**
 * Insert the method's description here.
 * Creation date: (5/8/2001 12:30:45 PM)
 * @return java.lang.String
 */
public String getHost()
{
	String name = null;
	int idx = _list.getSelectionIndex();

	if (idx != -1)
		name = _list.getItem(idx).getText();
	else
		return name;

	int id = name.indexOf(":");
	if (id != -1) {
		name = name.substring(0, id);
	}

	try {
		Node fnode = NodeFactory.createNode(name);
		name = fnode.getInetAddress().getHostName();
	} catch (UnknownHostException exc) {
	}

	return name;
		
}

public int getPort()
{
	String port = String.valueOf(Constants.CTL_PORT_NUM_SERVER);

	int idx = _list.getSelectionIndex();
	if(idx != -1)
		port = _list.getItem(idx).getText();
		
		int id = port.indexOf(":");
		if (id != -1) 
			port = port.substring(id + 1);
		else
			port = String.valueOf(Constants.CTL_PORT_NUM_SERVER);
	
	return Integer.parseInt(port);
}
/*
 * 
 */
public boolean isLocalHost(String hostname)
{
	String name = "localhost";
	
	try {
		  Node fnode = NodeFactory.createNode(name);
		  name = fnode.getInetAddress().getHostName();
	}
	catch(UnknownHostException exc)
	{
		return false;
	}

    if(hostname.equals(name))
      return true;
	
	return false;
}
/**
 * Insert the method's description here.
 * Creation date: (4/30/2001 11:21:50 AM)
 * @return com.ibm.swt.widgets.TableItem[]
 */
public TableItem[] getHosts() {
	return _list.getItems();
}
/**
 * Initializes states of the controls from the preference store.
 */
private void initializeValues()
{
	IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();    

	String hosts = store.getString(HyadesConstants.HOST_KEY);
	addHosts(hosts);
	enableButtons();
}
/**
 * Sent when the text is modified.
 *
 * @param e an event containing information about the modify
 */
public void modifyText(ModifyEvent e)
{
	if(e.widget == _host)
	   _add.setEnabled((!_host.getText().trim().equals("")) && (!_port.getText().trim().equals("")));
	else if (e.widget == _port)
	{
	   try {
			int value = Integer.parseInt(_port.getText().trim());
			if (value<0)
			{
				_add.setEnabled(false);
				return;
			}
			    _add.setEnabled((!_host.getText().trim().equals("")) && (!_port.getText().trim().equals("")));
		} catch (NumberFormatException nfe) {
			_add.setEnabled(false);
		}
	  
	}
}
/**
 * Insert the method's description here.
 * Creation date: (4/30/2001 12:16:48 PM)
 */
public void reset()
{
	_list.removeAll();
}
/**
 * Sent when default selection occurs in the control.
 * <p>
 * For example, on some platforms default selection occurs
 * in a List when the user double-clicks an item or types
 * return in a Text.
 * </p>
 *
 * @param e an event containing information about the default selection
 */
public void widgetDefaultSelected(SelectionEvent e)
{
	// User double-clicked on one of the hosts. Attempt to perform
	// 'Next' if and only if this page is in a wizard.
	if (e.widget == _list && _wizardPage != null && _wizardPage.getNextPage() != null)
		_wizardPage.getWizard().getContainer().showPage(_wizardPage.getNextPage());
}

private void enableButtons()
{
	_delete.setEnabled(_list.getSelectionIndex() != -1);
    _testConnection.setEnabled(_list.getSelectionIndex() != -1 && _list.getSelectionCount()==1);
}
/**
 * Sent when selection occurs in the control.
 * <p>
 * For example, on some platforms selection occurs in
 * a List when the user selects an item or items.
 * </p>
 *
 * @param e an event containing information about the selection
 */
public void widgetSelected(SelectionEvent e)
{
	if(e.widget == _add)
	{
	  addHost();
	  enableButtons();
	 
	  if (_wizardPage != null)
	  	  _wizardPage.setPageComplete(_list.getSelectionIndex() != -1);
	}
	else if(e.widget == _delete)
	{
	  deleteHost();
	  enableButtons();
	 
	  if (_wizardPage != null)
	  	  _wizardPage.setPageComplete(_list.getSelectionIndex() != -1);
	}
	else if(e.widget == _list)
	{  
	   TableItem[] selected = _list.getSelection();
	   enableButtons();
	   _list.setSelection(selected);
	 
	   if (_wizardPage != null)
	  	   _wizardPage.setPageComplete(_list.getSelectionIndex() != -1);
	}   
	else if(e.widget == _testConnection)
	{
	  testConnection();
	  enableButtons();
	  _testConnection.setFocus();
	}
}


public void selectHost(String hostname)
{
	for(int idx=0; idx<_list.getItemCount(); idx++)
	{
		String data = _list.getItem(idx).getText();
		int i = data.indexOf(":");
		if(i != -1)
		   data = data.substring(0, i);
		   
		if(data.equals(hostname))
		{
		   _list.setSelection(idx);
		   return; //already in the list
		}
	}
	
}
}
