/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.piclient;

import java.io.*;
import java.util.Date;
import java.util.Iterator;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.internal.execution.local.common.DataServerListener;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.loaders.util.*;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;


public class XMLTraceDataProcessor extends XMLLoader implements DataServerListener {
    //~ Instance fields ----------------------------------------------------------------------------

    private boolean firstEvent=true;
	private File profilingFile = null;
    private FileWriter fileWriter = null;
    private String profileFileName = null;

    //	private TRCAgentProxy _agent;
    private TRCMonitor _monitor;
    private boolean _changed = false;
    private int timeout = 0;
    
    private String bufStr;
    private final String START_TAG = "<TRACE>";
    private final String END_TAG = "</TRACE>";
    private final String XML_VERSION_TAG = "<?xml version=\"1.0\"?>";
    private final String START_TRACE = START_TAG+"\n";
    private final String END_TRACE = END_TAG+"\n";
    private final String XML_VERSION = XML_VERSION_TAG+"\n";
    
    //~ Constructors -------------------------------------------------------------------------------

    /**
     * XMLDataProcessor constructor comment.
     */
    public XMLTraceDataProcessor(TRCAgentProxy agent) {
        super(agent);
        
        setCollectionMode(agent.getCollectionMode().getValue());

        (new RefreshUI()).start();
    }

    /**
     * XMLDataProcessor constructor comment.
     */
    public XMLTraceDataProcessor(TRCMonitor monitor) {
        super(monitor);
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * Sets the fw.
     * @param fw The fw to set
     */
    public void setFileWriter(FileWriter fw) {
        this.fileWriter = fw;
    }

    /**
     * Returns the fw.
     * @return FileWriter
     */
    public FileWriter getFileWriter() {
        return fileWriter;
    }

    /**
     * Sets the profileFileName.
     * @param profileFileName The profileFileName to set
     */
    public void setProfileFileName(String profileFileName) {
        this.profileFileName = profileFileName;
    }

    /**
     * Returns the profileFileName.
     * @return String
     */
    public String getProfileFileName() {
        return profileFileName;
    }

    /**
     * Sets the pFile.
     * @param pFile The pFile to set
     */
    public void setProfilingFile(File pFile) {
        this.profilingFile = pFile;
    }

    /**
     * Returns the pFile.
     * @return File
     */
    public File getProfilingFile() {
        return profilingFile;
    }

    public File createFileWriter() {
        File file = null;

        try {
            if (profileFileName != null) {
                file = new File(profileFileName);
                fileWriter = new FileWriter(profileFileName, true);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return file;
    }

    public void dataServerExited() {
        try {
            if (getAgentProxy() != null) {
                getAgentProxy().setCollectionData(false);
                endTrace(getFileWriter());
                setProfileFileName(null);
                setProfilingFile(null);
                setFileWriter(null);
                
                cleanUp();

                Display d = Display.getDefault();

                d.asyncExec(new Runnable() {
                        public void run() {
                            ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                            event.setSource(getAgentProxy());
                            event.setType(ProfileEvent.STOP_COLLECTING);
							UIPlugin.getDefault().notifyProfileEventListener(event);
                        }
                    });
            }
        } catch (Exception exc) {
            //do nothing
        }
    }

    public void dispose() {
        super.cleanUp();

        fileWriter = null;
    }

    public void endTrace(FileWriter randomFile) {
	    try {
	        if (randomFile != null) {
//	            randomFile.write(END_TRACE);
//	            randomFile.flush();
	            randomFile.close();
	            randomFile = null;
	        }
	    } catch (IOException e) {
	    }
	}

	private boolean isValidTag(String buf){
		if  (buf.startsWith(START_TAG) 
			 || buf.startsWith(XML_VERSION_TAG))
			return false;
		else 
			return true;
	
	}

    /**
     * Insert the method's description here.
     * Creation date: (3/1/01 12:06:07 PM)
     * @param buffer byte[]
     * @param length int
     * @param peer java.net.InetAddress
     */
    public void incommingData(byte[] buffer, int length, java.net.InetAddress peer) {
        _changed = true;


        try {
            if ((getAgentProxy() != null) && getAgentProxy().isToProfileFile()) {
            	bufStr=new String(buffer, 0, length, "UTF8");
            	if (isValidTag(bufStr))
				{
					writeByte(bufStr);
				}
            } else {
            	if(firstEvent)
            	{
            		if(length>1 && !(buffer[0]=='<' && buffer[1]=='?'))
            		{
            			LoadersUtils.loadRootEvent(this);
            		}
            		firstEvent = false;
            	}
                super.loadEvent(buffer, length);
            }
        } catch (InvalidXMLException e) {
        } catch (java.lang.OutOfMemoryError e) {
            handleOutOfMemoryError();
            stopAgent();
            throw e;
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (3/1/01 12:06:07 PM)
     * @param buffer char[]
     * @param length int
     * @param peer java.net.InetAddress
     */
    public void incommingData(char[] buffer, int length, java.net.InetAddress peer) {
        byte[] newBuffer = new byte[length];

        for (int i = 0; i < length; i++) {
            newBuffer[i] = (byte) buffer[i];
        }

        try {
            if ((getAgentProxy() != null) && getAgentProxy().isToProfileFile()) {
				bufStr=new String(newBuffer, "UTF8");
				if (isValidTag(bufStr))
				{
					writeByte(bufStr);
				}
                

                //new emf			
                //                super.loadEvent(newBuffer, length, false, true);
            } else {
				if(firstEvent)
				{
					if(length>1 && !(buffer[0]=='<' && buffer[1]=='?'))
					{
						LoadersUtils.loadRootEvent(this);
					}
					firstEvent = false;
				}
                super.loadEvent(newBuffer, length);
            }
        } catch (InvalidXMLException e) {
        } catch (java.lang.OutOfMemoryError e) {
            handleOutOfMemoryError();
            stopAgent();
            throw e;
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (3/1/01 12:06:07 PM)
     * @param data byte[]
     * @param length int
     * @param peer java.net.InetAddress
     */
    public void invalidDataType(byte[] data, int length, java.net.InetAddress peer) {
    }

    /**
     * Insert the method's description here.
     * Creation date: (3/1/01 12:06:07 PM)
     */
    public void waitingForData() {
    }

	public void startTrace(FileWriter randomFile) {
		try {
			 if (getAgentProxy() != null) {
				 randomFile.write(START_TRACE);
				 randomFile.flush();
			 }
			 } catch (IOException e) {}		
	}
	
    public void writeHeader(FileWriter randomFile) {
// //       try {
//            if (getAgentProxy() != null) {
//                startTrace(randomFile);
////                writeNode(randomFile);
////                writeProcessCreate(randomFile);
////                writeAgentCreate(randomFile);
////                writeTraceStart(randomFile);
////                writeFilter(randomFile);
////                writeOption(randomFile);
//            }
////        } catch (IOException e) {
////        }
    }

    /*
     * Write the necessary trace event to the head to the head of the file
     */
    public void writeXMLVersion(FileWriter randomFile) {
        try {
            randomFile.write(XML_VERSION);
            randomFile.flush();
        } catch (IOException e) {
        }
    }

    /**
     *
     */
    private TRCAgentProxy getAgentProxy() {
        return getContext().getAgentProxy();
    }

    private String getApplicationExecutableString(TRCProcessProxy process) {
        String exeStr = "";

        if (process.getName() == null) {
            return "";
        }

        if (!process.getName().equals("")) {
            exeStr = "java.exe -XrunpiAgent:server=controlled ";

            if ((process.getClasspath() != null) && !process.getClasspath().equals("")) {
                exeStr += "-cp ";
                exeStr += process.getClasspath().trim();
                exeStr += " ";
            }

            if ((process.getVmArguments() != null) && !process.getVmArguments().equals("")) {
                exeStr += process.getVmArguments().trim();
                exeStr += " ";
            }

            exeStr += process.getName();

            if ((process.getParameters() != null) && !process.getParameters().equals("")) {
                exeStr += process.getParameters();
            }
        }

        return exeStr;
    }

    private void handleOutOfMemoryError() {
        /* Inform the user we are running out of memory */
        String text = UIPlugin.getResourceString("PROFILE_OUT_OF_MEMORY_ERROR_");
        final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

        Display.getDefault().asyncExec(new Runnable() {
                public void run() {
                    ErrorDialog.openError(new Shell(), UIPlugin.getResourceString("TRACE_MSG"), UIPlugin.getResourceString("OUT_OF_MEMORY_ERROR_TITLE"), err);
                }
            });
    }

    private synchronized void notifyListener(Agent a, TRCAgentProxy agent) {
        final TRCAgentProxy ag = agent;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(ag);
                    event.setType(ProfileEvent.STOP_MONITOR);
					UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }

    private void stopAgent() {
        if (getAgentProxy() != null) {

			try {
				Agent ai = (Agent)LoadersUtils.locateAgentInstance(getAgentProxy());
					if (ai!=null)
						 ai.stopMonitoring();
			} catch (InactiveAgentException e) {
					e.printStackTrace();
			}
            getAgentProxy().setMonitored(false);
            getAgentProxy().setAttached(false);
            getAgentProxy().setActive(false);

            LoadersUtils.deregisterDataProcessor(getAgentProxy());

            if (fileWriter != null) {
                endTrace(fileWriter);
            }

            notifyListener((Agent) LoadersUtils.locateAgentInstance(getAgentProxy()), getAgentProxy());
        }
    }

    private void writeAgentCreate(FileWriter randomFile) throws IOException {
        if (getAgentProxy() != null) {
            randomFile.write(new StringBuffer("<agentCreate agentId=\"").append(getAgentProxy().getRuntimeId()).append("\" version=\"1.000").append("\" processIdRef=\"").append(getAgentProxy().getProcessProxy().getRuntimeId()).append("\" agentName=\"").append(getAgentProxy().getName()).append("\" agentType=\"").append(getAgentProxy().getType()).append("\" agentParameters=\"").append((getAgentProxy().getProcessProxy().getLaunchMode() == 0) ? "server=controlled" : "").append("\" time=\"").append(new Date().getTime() / 1000).append("\"/>\n").toString());
            randomFile.flush();
        }
    }

    private void writeByte(String data) {
    	
        try {
            if (fileWriter != null) {
                fileWriter.write(data + "\n");
                fileWriter.flush();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void writeConfigFilter(EList filters, FileWriter randomFile) throws IOException {
        if (filters != null) {
            Iterator i = filters.iterator();

            while (i.hasNext()) {
                TRCFilter filter = (TRCFilter) i.next();
                String method = filter.getOperation();

                randomFile.write(new StringBuffer("<filter pattern=\"").append(filter.getPattern()).append("\" mode=\"").append(filter.getMode()).append("\" genericPattern=\"SUFFIX\"/>\n").toString());
                randomFile.flush();
            }
        }
    }

    private void writeConfigOption(EList options, FileWriter randomFile) throws IOException {
        if (options != null) {
            Iterator it = options.iterator();

            while (it.hasNext()) {
                TRCOption option = (TRCOption) it.next();

                randomFile.write(new StringBuffer("<option key=\"").append(option.getKey()).append("\" value=\"").append(option.getValue()).append("\"/>\n").toString());
                randomFile.flush();
            }
        }
    }

    private void writeFilter(FileWriter randomFile) throws IOException {
        if (getAgentProxy() != null) {
            EList filters = null;

            EList configs = getAgentProxy().getConfigurations();
            Iterator i = configs.iterator();

            if (i.hasNext()) {
                TRCConfiguration config = (TRCConfiguration) i.next();

                filters = config.getFilters();

                writeConfigFilter(filters, randomFile);

                return;
            }

            writeConfigFilter(filters, randomFile);
        }
    }

    private void writeNode(FileWriter randomFile) throws IOException {
        if (getAgentProxy() != null) {
            TRCNode node = getAgentProxy().getProcessProxy().getNode();

            if (node == null) {
                return;
            }

            randomFile.write(new StringBuffer("<node nodeId=\"").append(getAgentProxy().getProcessProxy().getNode().getRuntimeId()).append("\" hostname=\"").append(node.getName()).append("\" ipaddress=\"").append((node.getIpAddress() != null) ? node.getIpAddress() : "-Unavailable-").append("\" timezone=\"").append(node.getTimezone()).append("\" time=\"").append(new Date().getTime() / 1000).append("\"/>\n").toString());
            randomFile.flush();
        }
    }

    private void writeOption(FileWriter randomFile) throws IOException {
        EList options = null;

        EList configs = getAgentProxy().getConfigurations();
        Iterator i = configs.iterator();

        if (i.hasNext()) {
            TRCConfiguration config = (TRCConfiguration) i.next();

            options = config.getOptions();

            writeConfigOption(options, randomFile);
        }
    }

    private void writeProcessCreate(FileWriter randomFile) throws IOException {
        if (getAgentProxy() != null) {
            TRCProcessProxy processCreate = getAgentProxy().getProcessProxy();

            if (processCreate == null) {
                return;
            }

            randomFile.write(new StringBuffer("<processCreate processId=\"").append(processCreate.getRuntimeId()).append("\" pid=\"" + processCreate.getPid()).append("\" nodeIdRef=\"").append(processCreate.getNode().getRuntimeId()).append("\" time=\"").append(new Date().getTime() / 1000).append("\"").append(" application_executable=\"").append(getApplicationExecutableString(processCreate)).append("\"/>\n").toString());
            randomFile.flush();
        }
    }

    private void writeTraceStart(FileWriter randomFile) throws IOException {
        if (getAgentProxy() != null) {
            randomFile.write(new StringBuffer("<traceStart traceId=\"").append("\" agentIdRef=\"").append(getAgentProxy().getRuntimeId()).append("\" time=\"\"/>\n").toString());
            randomFile.flush();
        }
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    class RefreshUI extends Thread {
        public RefreshUI() {
            super("Profile UI");
        }

        public void run() {
            while ((getAgentProxy() != null) && getAgentProxy().isActive()) {
                try {
                    if (_changed) {
                        _changed = false;

                        if (!getAgentProxy().isCollectionData()) {
                            getAgentProxy().setCollectionData(true);

                            Display d = Display.getDefault();

                            d.asyncExec(new Runnable() {
                                    public void run() {
                                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                                        event.setSource(getAgentProxy());
                                        event.setType(ProfileEvent.COLLECTING);
										UIPlugin.getDefault().notifyProfileEventListener(event);
                                    }
                                });
                        }
                    } else {
                        if (getAgentProxy().isCollectionData()) {
                            getAgentProxy().setCollectionData(false);

                            Display d = Display.getDefault();

                            d.asyncExec(new Runnable() {
                                    public void run() {
                                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                                        event.setSource(getAgentProxy());
                                        event.setType(ProfileEvent.STOP_COLLECTING);
										UIPlugin.getDefault().notifyProfileEventListener(event);
                                    }
                                });
                        }
                    }

                    sleep(1000);
                } catch (InterruptedException exc) {
                }
            }
        }
    }
}
