/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.internal.codegen;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 1.0.1
 */
abstract public class JavaGenerator
extends Generator implements IDisposable
{
	private Collection requiredLibraries;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		requiredLibraries.clear();
	}
	
	/**
	 * Adds all the libraries from a given plugin to the list of
	 * required libraries that must be in the project's classpath.
	 * 
	 * <p>This method must be invoked before the 
	 * {@link #generate(ITestSuite, IProgressMonitor)}.
	 *  
	 * @param pluginId
	 */
	public void addAllLibraries(String pluginId)
	{
		if(requiredLibraries == null)
			requiredLibraries = new HashSet();
		requiredLibraries.addAll(Arrays.asList(ResourceUtil.getPluginLibraries(pluginId)));
	}

	/**
	 * Adds a library that is required to be in the project's classpath.
	 * 
	 * <p>This method must be invoked before the 
	 * {@link #generate(ITestSuite, IProgressMonitor)}.
	 *  
	 * @param pluginId
	 * @param library
	 */
	public void addRequiredLibrary(String pluginId, String library)
	{
		String lib = getLibrary(pluginId, library);
		if(lib == null)
			return;
			
		if(requiredLibraries == null)
			requiredLibraries = new HashSet();
		requiredLibraries.add(lib);
	}

	protected String getLibrary(String pluginId, String library)
	{
		File lib = ResourceUtil.getPluginDirectory(pluginId);
		if(lib == null)
			return null;
			
		lib = new File(lib, library);
		try
		{
			return lib.getCanonicalPath().replace('\\','/');
		}
		catch (IOException e)
		{
			return lib.getAbsolutePath().replace('\\','/');
		}
	}

	protected void doGenerateFile(ITestSuite testSuite, IContainer sourceContainer, IFile file, IProgressMonitor monitor)
	throws Exception
	{
		String cls = testSuite.getImplementor().getResource();
		int index = cls.lastIndexOf('.');		
		adjustSourceContainer(sourceContainer, cls.substring(0, index), monitor);

		adjustProjectDependencies(testSuite, file.getProject(), requiredLibraries, monitor);
		generateFile(testSuite, file, monitor);
	}

	public IFile getFileHandle(ITestSuite testSuite)
	{
		IContainer sourceContainer = getSourceContainerHandle(testSuite);
		return sourceContainer.getFile(new Path(testSuite.getImplementor().getResource().replace('.','/') + ".java"));
	}
			
	protected void adjustSourceContainer(IContainer sourceContainer, String packageName, IProgressMonitor monitor)
	throws Exception
	{
		IJavaProject javaProject = JavaCore.create(sourceContainer.getProject());
		if(sourceContainer.getType() == IResource.FOLDER)
		{
			if(!javaProject.isOnClasspath(sourceContainer))
			{
				IClasspathEntry entry = JavaCore.newSourceEntry(sourceContainer.getFullPath());
				List entries = new ArrayList(Arrays.asList(javaProject.getRawClasspath()));
				entries.add(1, entry);
				javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), monitor);
			}
		}
		
		javaProject.getPackageFragmentRoot(sourceContainer).createPackageFragment(packageName, true, monitor);
	}
	
	protected void adjustProjectDependencies(ITestSuite testSuite, IProject project, Collection libraries, IProgressMonitor monitor)
	{
		if((libraries == null) || (libraries.isEmpty()))
			return;
			
		IJavaProject javaProject = JavaCore.create(project);
		List rawClasspathEntries;
		try
		{
			rawClasspathEntries = new ArrayList(Arrays.asList(javaProject.getRawClasspath()));
		}
		catch (JavaModelException e)
		{
			TestCommonPlugin.logError(e);
			return;
		}
		
		int pos = rawClasspathEntries.size()-2;
		if(pos < 0)
			pos = 0;
			
		IClasspathEntry[] rawEntries = getLibrariesEntries(libraries);
		for(int i = 0, maxi = rawEntries.length; i < maxi; i++)
		{
			if((!rawClasspathEntries.contains(rawEntries[i])) && (!rawClasspathEntries.contains(JavaCore.getResolvedClasspathEntry(rawEntries[i]))))
				rawClasspathEntries.add(pos, rawEntries[i]);
		}
		
		rawEntries = getProjectEntries(testSuite, project);
		for(int i = 0, maxi = rawEntries.length; i < maxi; i++)
		{
			if((!rawClasspathEntries.contains(rawEntries[i])) && (!rawClasspathEntries.contains(JavaCore.getResolvedClasspathEntry(rawEntries[i]))))
				rawClasspathEntries.add(pos, rawEntries[i]);
		}
		
		try
		{
			javaProject.setRawClasspath((IClasspathEntry[])rawClasspathEntries.toArray(new IClasspathEntry[rawClasspathEntries.size()]), null);
		}
		catch(JavaModelException e)
		{
			TestCommonPlugin.logError(e);
			return;
		}
	}
	
	protected IClasspathEntry[] getLibrariesEntries(Collection libraries)
	{
		List entries = new ArrayList(libraries.size());
		for(Iterator i = libraries.iterator(); i.hasNext();)
			entries.add(JavaCore.newLibraryEntry(new Path(i.next().toString()), null, null));
				
		return (IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]);
	}
	
	protected IClasspathEntry[] getProjectEntries(ITestSuite testSuite, IProject project)
	{
		if((testSuite.getIReferencedSuites() == null) || (testSuite.getIReferencedSuites().isEmpty()))
			return new IClasspathEntry[0];
			
		Set entries = new HashSet(testSuite.getIReferencedSuites().size()+1);
		for (Iterator i = testSuite.getIReferencedSuites().iterator(); i.hasNext();)
		{
			ITestSuite referenced = (ITestSuite)i.next();
			if(referenced.getImplementor() == null)
				continue;
				
			String location = referenced.getImplementor().getLocation();
			if(location == null)
				continue;
				
			IPath path = new Path(location);
			IProject referencedProject = ResourcesPlugin.getWorkspace().getRoot().getProject(path.segment(0));
			if((referencedProject != null) && referencedProject.exists() && (!project.equals(referencedProject)))
				entries.add(JavaCore.newProjectEntry(referencedProject.getFullPath()));
		}		
		return (IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]);
	}
}