/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.remote;

import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;

/**
 * In addition to delegating method invocations to the associated
 * implementation objects, instances this class create the remote portion
 * of the "test agent". The test agent is used for monitoring and in a
 * handshaking scheme that attempts to insure that the test does not begin
 * running before the client on the local side is ready for it to do so.
 */
public class JavaTaskExecutorSkeleton extends ExecutorSkeleton {
	
	private RemoteComponentSkeleton agent;
	private boolean isOKToStart;

	public JavaTaskExecutorSkeleton() {
		super();
	}

	public JavaTaskExecutorSkeleton(IExecutionComponent delegate, Integer id) {
		super(delegate, id);
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor#setExecutableObject(org.eclipse.hyades.execution.core.IExecutableObject)
	 */
	public void setExecutableObject(IExecutableObject theExecutableObject) throws ExecutionComponentStateException {
		((IExecutor)delegate).setExecutableObject(theExecutableObject);
	}
	
	/**
	 * Get the remote end of the test monitoring and control agent for the test that
	 * this executor can run. The agent is guaranteed to have been properly initialized.
	 * <p>
	 * This method is only available in the skeleton layer.
	 * 
	 * @return a <code>RemoteComponentSkeleton</code> instance
	 */
	public RemoteComponentSkeleton getAgent() {
		return agent;
	}
	
	/**
	 * Setup the remote test agent.
	 * 
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {
		super.init();
		
		// Create the remote side of the test agent.
		agent = new RemoteComponentSkeleton("Executor$"+getUniqueId(), "tester");
		
		// Add a command listener that can okay the starting of the test.
		agent.addCommandListener(new CustomCommandHandler() {
			public void handleCommand(CustomCommand command) {
				if(command.getData().equals(IControlMessage.START))
					setOKToStart(true);
			}
		});
	
		/* Handshake with the Agent controller */
		try {
			agent.initialize();
		}
		catch(Throwable e) {
			throw new RemoteInvocationException(e);
		}
	}
	
	/**
	 * Is it okay to continue starting the test?
	 * 
	 * @return <code>true</code> if it is okay to start the test,
	 *         <code>false</code> otherwise
	 */
	public synchronized boolean isOKToStart() {
		return isOKToStart;
	}
	
	// Set the "okay to start" flag and notify any threads that are waiting
	// on this object's monitor.
	private synchronized void setOKToStart(boolean isOK) {
		isOKToStart = isOK;
		notifyAll();
	}
	
	/**
	 * Block the calling thread until it is okay to start the test.
	 * Calls to <code>setOKToStart(boolean)</code> result in notifications
	 * being issued to all threads blocked waiting on this object's
	 * monitor.
	 */
	public void waitForOKToStart() {
		while (!isOKToStart()) {
			synchronized(this){
				try {
					wait();
				}
				catch ( InterruptedException e ) {}
			}
			if ( isOKToStart() )
				break;
		}
	}

}
