/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.editor.EditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.hyades.ui.internal.util.UIUtil;

/**
 * @author marcelop
 * @since 0.2.0
 */
public abstract class BaseEditorExtension 
extends EditorExtension implements ISynchronizedEditorAdapter
{
	private ListenerList writeAccessChangeListeners;
	
	private boolean dirty = false;
	private Clipboard clipboard;
	 
	public BaseEditorExtension()
	{
		writeAccessChangeListeners = new ListenerList();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if(clipboard != null)
		{
			clipboard.dispose();
			clipboard = null;
		}
		writeAccessChangeListeners.clear();
		
		super.dispose();
	}

	/**
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter)
	{
		if(ISynchronizedEditorAdapter.class == adapter)
			return this;
			
		return super.getAdapter(adapter);
	}

	/**
	 * Returns this editor extension's resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return ((EObject)getHyadesEditorPart().getEditorObject()).eResource().getResourceSet();	
	}
	
	/**
	 * Returns the extension editor's clipboard.
	 * @return Clipboard
	 */
	public Clipboard getClipboard()
	{
		if(clipboard == null)
			clipboard = new Clipboard(getHyadesEditorPart().getEditorPart().getEditorSite().getShell().getDisplay());
		return clipboard;
	}
	
	public void addListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.add(listener);
	}
	
	public void removeListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.remove(listener);
	}
	
	protected void notifyWriteAccessChangeListeners(IFile file, boolean isReadOnly)
	{
		if(file == null)
			return;
			
		Object[] listeners = writeAccessChangeListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
			((IWriteAccessChangeListener)listeners[i]).writeAccessChanged(file, isReadOnly);
	}
	
	/**
	 * Marks the editor as dirty.
	 */
	public synchronized void markDirty()
	{
		if(!dirty)
		{
			dirty = true;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);
		}
	}
	
	/**
	 * Clears the dirty flag without refreshing the editor.
	 */
	protected void clearDirty()
	{
		dirty = false;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#isDirty()
	 */
	public boolean isDirty()
	{
		return dirty;
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public final void doSave(IProgressMonitor monitor)
	{
		WorkspaceModifyOperation operation = new WorkspaceModifyOperation()
		{
			protected void execute(IProgressMonitor monitor) 
			throws CoreException, InvocationTargetException, InterruptedException
			{
				try
				{
					save(monitor);
					clearDirty();
				}
				catch(Exception e)
				{
					handleException(e);
				}
			}
		};
		
		
		IActionBars actionBars = getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
		if((actionBars != null) && (actionBars.getStatusLineManager() != null))
		{
			actionBars.getStatusLineManager().setErrorMessage(null);
			actionBars.getStatusLineManager().setMessage(null);
			monitor = actionBars.getStatusLineManager().getProgressMonitor();
			actionBars.getStatusLineManager().setCancelEnabled(false);
		}
		else if(monitor == null)
		{
			monitor = new NullProgressMonitor();
		}
		
		try
		{
			ModalContext.run(operation, false, monitor, Display.getCurrent());
		}
		catch (Exception e)
		{
			handleException(e);
		}
	}
	
	/**
	 * Saves the contents of this editor extension.
	 * @param monitor the progress monitor to report the work
	 * @throws Exception this method is supposed to throw any exception.
	 */
	public void save(IProgressMonitor monitor)
	throws Exception
	{
		Resource[] resources = (Resource[])getResourceSet().getResources().toArray(new Resource[getResourceSet().getResources().size()]);
		int length = resources.length;
		int counter = 0;
		monitor.beginTask("", length);
		
		for(int i = 0, maxi = resources.length; i < maxi; i++)
		{
			if(resources[i].isModified())
			{
				logSaveMessage(monitor, resources[i], length, counter++);
				EMFUtil.save(resources[i]);
			}
			monitor.worked(1);
		}
		monitor.done();
	}
	
	protected void logSaveMessage(IProgressMonitor monitor, Resource resource, int total, int current)
	{
		String resourceName = "";
		IFile file = EMFUtil.getWorkspaceFile(resource);
		if(file != null)
			resourceName = file.getFullPath().toString();
			
		monitor.subTask(TestUIPlugin.getString("PRG_SAVE_RSRS", new String[]{Integer.toString(current), Integer.toString(total), resourceName}));
	}
	
	/**
	 * Handles any exception thrown during the 
	 * {@link #save(IProgressMonitor)} execution.
	 * @param exception
	 */
	protected void handleException(Exception exception)
	{
		TestUIPlugin.logError(exception);
		
		String filePath = "";
		IFile file = EMFUtil.getWorkspaceFile((EObject)getHyadesEditorPart().getEditorObject());
		if(file != null)
			filePath = file.getFullPath().toString();
			
		UIUtil.openSaveFileErrorDialog(getHyadesEditorPart().getEditorPart().getSite().getShell(), filePath, exception);
	}
	
	/**
	 * Reloades the editor's object.  This method is invoked by the synchronizer
	 * adapter and should return <code>true</code> if the synchronizer is not
	 * supposed to continue with the default treatment.
	 * @return boolean
	 */
	protected boolean reloadEditorObject()
	{
		Resource resource = ((EObject)getHyadesEditorPart().getEditorObject()).eResource(); 
		
		resource.unload();
		try
		{
			resource.load(EMFUtil.RESOURCE_OPTIONS);
		}
		catch(Exception e)
		{
			TestUIPlugin.logError(e);
		}
		
		if(!resource.getContents().isEmpty())
			getHyadesEditorPart().setEditorObject(resource.getContents().get(0));
		else
			getHyadesEditorPart().setEditorObject(null);

		return false;
	}

	/**
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(part instanceof TestNavigator)
		{
			if(!(selection instanceof IStructuredSelection))
				return;

			IStructuredSelection structuredSelection = (IStructuredSelection)selection;
			if(structuredSelection.size() != 1)
				return;
				
			URI uri = getObjectURI(structuredSelection.getFirstElement());
			if(uri == null)
				return;
				
			EObject eObject = getResourceSet().getEObject(uri, false);
			if(eObject == null)
				return;
			
			setSelection(new StructuredSelection(eObject));
		}
	}
	
	/**
	 * Returns the complete EMF URI which includes the resource information.
	 * @param object
	 * @return
	 */
	protected URI getObjectURI(Object object)
	{
		if(object != null)
		{
			if(object instanceof EObject)
			{
				EObject eObject = (EObject)object;
				if(eObject.eResource() != null)
					return eObject.eResource().getURI().appendFragment(eObject.eResource().getURIFragment(eObject));
			}
		}
		
		return null;
	}
		
	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileChanged()
	 */
	public boolean editorFileChanged()
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileDeleted()
	 */
	public boolean editorFileDeleted()
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileWriteAccessChanged(boolean)
	 */
	public boolean editorFileWriteAccessChanged(boolean isReadOnly)
	{
		notifyWriteAccessChangeListeners(EMFUtil.getWorkspaceFile((EObject)getHyadesEditorPart().getEditorObject()), isReadOnly);
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#handle(java.util.List, java.util.List)
	 */
	public void handle(List changedFiles, List removedFiles)
	{

	}

	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#reload()
	 */
	public boolean reload()
	{
		String fragment = null;
		IStructuredSelection structuredSelection = getSelection();
		if(structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof EObject)
				fragment = ((EObject)selection).eResource().getURIFragment((EObject)selection);
		}
		
		if(!reloadEditorObject())
		{
			Object editorObject = getHyadesEditorPart().getEditorObject();
			refreshContent(editorObject);
			
			if((editorObject instanceof EObject) && (fragment != null))
			{
				Object selection = ((EObject)(editorObject)).eResource().getEObject(fragment);
				if(selection != null)
					structuredSelection = new StructuredSelection(selection);
				else
					structuredSelection = StructuredSelection.EMPTY;
					
				setSelection(structuredSelection);
			}
			
			dirty = false;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);			
		}
		
		return true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#doSaveEditorFile(boolean)
	 */
	public boolean doSaveEditorFile(boolean wasDeleted)
	{
		getHyadesEditorPart().getEditorPart().doSave(new NullProgressMonitor());
		return false;
	}
}
