/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPersistableElement;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.common.CommonPackage;
import org.eclipse.hyades.ui.adapter.NamedElementPropertySource;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;

/**
 * @author marcelop
 * @since 0.2.0
 */
public class CMNNamedElementUtil
{
	public static class PersistableElement
	implements IPersistableElement, IDisposable
	{
		private CMNNamedElement namedElement;
		
		/**
		 * Constructor for PersistableElement
		 * @param namedElement
		 */
		public PersistableElement(CMNNamedElement namedElement)
		{
			this.namedElement = namedElement;
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			namedElement = null;
		}

		/**
		 * @see org.eclipse.ui.IPersistableElement#getFactoryId()
		 */
		public String getFactoryId()
		{
			return ElementFactory.class.getName();
		}

		/**
		 * @see org.eclipse.ui.IPersistableElement#saveState(org.eclipse.ui.IMemento)
		 */
		public void saveState(IMemento memento)
		{
			ElementFactory.save(memento, namedElement);
		}
	}
	
	
	public static class ElementFactory
	implements IElementFactory
	{
		/*
		 * TAG* constants are used in the IMemento
		 */
		public static final String TAG_FILE = "FILE";

		/**
		 * Saves information of the logical folder in the memento. 
		 * @param memento
		 * @param logicalFolder
		 */
		public static void save(IMemento memento, CMNNamedElement namedElement)
		{
			IFile file = EMFUtil.getWorkspaceFile(namedElement);
			if(file != null)
				memento.putString(TAG_FILE, file.getFullPath().toString());
		}	

		/**
		 * @see org.eclipse.ui.IElementFactory#createElement(org.eclipse.ui.IMemento)
		 */
		public IAdaptable createElement(IMemento memento)
		{
			String filePath = memento.getString(TAG_FILE);
			if(filePath == null)
				return null;

			IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(filePath);
			if((resource == null) || (resource.getType() != IResource.FILE))
				return null;
			
			EObject[] eObjects = EMFUtil.load(null, (IFile)resource);
			if((eObjects.length > 0) && (eObjects[0] instanceof IAdaptable))
				return (IAdaptable)eObjects[0];
				
			return null;		
		}
	}
	
	public static class UINamedElement
	implements INamedElement, IDisposable
	{
		private CMNNamedElement namedElement;
		
		/**
		 * Constructor for UINamedElement
		 * @param namedElement
		 */
		public UINamedElement(CMNNamedElement namedElement)
		{
			this.namedElement = namedElement;
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			namedElement = null;
		}

		/**
		 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
		 */
		public Object getAdapter(Class adapter)
		{
			if(INamedElement.class == adapter)
				return this;
			return namedElement.getAdapter(adapter);
		}

		/**
		 * @see org.eclipse.hyades.ui.util.INamedElement#getDescription()
		 */
		public String getDescription()
		{
			if(namedElement.getDescription() == null)
				return "";
			return namedElement.getDescription();
		}

		/**
		 * @see org.eclipse.hyades.ui.util.INamedElement#getName()
		 */
		public String getName()
		{
			if(namedElement.getName() == null)
				return namedElement.eClass().getName();
			return namedElement.getName();
		}

		/**
		 * @see org.eclipse.hyades.ui.util.INamedElement#setDescription(java.lang.String)
		 */
		public void setDescription(String description)
		{
			namedElement.setDescription(description);
		}

		/**
		 * @see org.eclipse.hyades.ui.util.INamedElement#setName(java.lang.String)
		 */
		public void setName(String name)
		{
			namedElement.setName(name);
		}
	}
	
	public static class PropertySource
	extends NamedElementPropertySource
	{
		public PropertySource(CMNNamedElement namedElement)
		{
			super(new UINamedElement(namedElement));		
		}
	}
	
	public static interface NameChangeObserver
	{
		public void update(CMNNamedElement namedElement, String oldName);
	}
	
	public static class NameChangeAdapter
	extends AdapterImpl implements IDisposable
	{
		private NameChangeObserver observer;
		
		public NameChangeAdapter(NameChangeObserver observer)
		{
			this.observer = observer;
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			observer = null;
		}

		/**
		 * @see org.eclipse.emf.common.notify.Adapter#isAdapterForType(java.lang.Object)
		 */
		public boolean isAdapterForType(Object type)
		{
			return CMNNamedElement.class.isInstance(type);
		}

		/**
		 * @see org.eclipse.emf.common.notify.Adapter#notifyChanged(org.eclipse.emf.common.notify.Notification)
		 */
		public void notifyChanged(Notification msg)
		{
			if((msg.getNotifier() instanceof CMNNamedElement) && (CommonPackage.CMN_NAMED_ELEMENT__NAME == msg.getFeatureID(CMNNamedElement.class)))
			{
				switch(msg.getEventType())
				{
					case Notification.SET:
					case Notification.UNSET:
						observer.update((CMNNamedElement)msg.getNotifier(), msg.getOldStringValue());
						break;
				}	
			}
		}

	}	
}
