/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.http.internal.junit.editor;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

import org.eclipse.hyades.models.common.facades.behavioral.IProperty;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.interactions.Common_Behavior_InteractionsFactory;
import org.eclipse.hyades.test.http.TestHttpPlugin;
import org.eclipse.hyades.test.http.internal.util.HttpConstants;
import org.eclipse.hyades.test.http.internal.util.RequestHelper;
import org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;

public class HttpHeaderViewer 
implements SelectionListener, IDisposable
{
	protected static class HeaderContentProvider 
	implements IStructuredContentProvider
	{
		/**
		 * @see IContentProvider#dispose()
		 */
		public void dispose()
		{
		}

		/**
		 * @see IStructuredContentProvider#getElements(Object)
		 */
		public Object[] getElements(Object element)
		{
			if(element instanceof RequestHelper)
				return ((RequestHelper)element).getHeaders();

			return new Object[0];
		}

		/**
		 * @see IContentProvider#inputChanged(Viewer, Object, Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}
	}
	
	protected static class HeaderLabelProvider 
	extends LabelProvider implements ITableLabelProvider
	{
		/**
		 * @see ITableLabelProvider#getColumnImage(Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex)
		{
			return null;
		}
		
		/**
		 * @see ITableLabelProvider#getColumnText(Object, int)
		 */
		public String getColumnText(Object element, int columnIndex)
		{
			if (element instanceof BVRProperty)
			{
				BVRProperty myTableElement = (BVRProperty)element;

				if (columnIndex == 0)
					return myTableElement.getName();

				if (columnIndex == 1)
					return myTableElement.getValue();

			}
			return "";
		}
	}
	
	protected static class HeaderDialog 
	extends Dialog implements Listener
	{
		private BVRProperty property;
		private HeaderUI propertyUI;

		public HeaderDialog(Shell shell, BVRProperty element)
		{
			super(shell);
			this.property = element;
		}

		protected Control createDialogArea(Composite parent)
		{
			getShell().setText(TestHttpPlugin.getString("Headers_Label"));
				
			Composite result = new Composite(parent, SWT.NULL);
			GridData gridData = GridDataUtil.createFill();
			gridData.heightHint = 250;
			gridData.widthHint = 400;
			result.setLayoutData(gridData);
			result.setLayout(new GridLayout());

			propertyUI = new HeaderUI(getShell());
			Composite content = propertyUI.createControl(result);

			if (property != null)
			{
				propertyUI.setName(property.getName());
				propertyUI.setValue(property.getValue());
			}

			return content;
		}

		protected Control createButtonBar(Composite parent)
		{
			Control control = super.createButtonBar(parent);
			propertyUI.registerListener(this);

			if (property == null)
				getButton(IDialogConstants.OK_ID).setEnabled(false);

			return control;
		}

		protected void okPressed()
		{
			if (property == null)
				property = Common_Behavior_InteractionsFactory.eINSTANCE.createBVRProperty();
			property.setName(propertyUI.getName());
			property.setValue(propertyUI.getValue());
			
			super.okPressed();
		}

		public BVRProperty getProperty()
		{
			return property;
		}

		public void handleEvent(Event e)
		{
			getButton(IDialogConstants.OK_ID).setEnabled(!propertyUI.getName().equals(""));
		}
	}
	
	protected static class HeaderUI 
	implements ModifyListener
	{
		private CCombo nameCombo;
		private Text valueText;
		private Shell shell;
		private Listener listener;

		public HeaderUI(Shell shell)
		{
			super();
			this.shell = shell;
		}

		public Composite createControl(Composite parent)
		{
			Label label;

			Composite content = new Composite(parent, SWT.NULL);
			GridLayout layout = new GridLayout();
			layout.numColumns = 1;
			layout.verticalSpacing = 10;
			layout.horizontalSpacing = 5;
			content.setLayout(layout);
			GridData gd = GridDataUtil.createFill();
			gd.horizontalIndent = 5;
			content.setLayoutData(gd);

			label = new Label(content, SWT.NULL);
			label.setText(TestHttpPlugin.getString("ENV_NAME"));
			nameCombo = new CCombo(content, SWT.FLAT | SWT.BORDER);
			nameCombo.setLayoutData(GridDataUtil.createHorizontalFill());
			nameCombo.setItems(HttpConstants.getAllHeaders());
			nameCombo.setFocus();
			nameCombo.addModifyListener(this);

			label = new Label(content, SWT.NULL);
			label.setText(TestHttpPlugin.getString("ENV_VALUE"));

			valueText = new Text(content, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
			valueText.setLayoutData(GridDataUtil.createFill());
			valueText.addModifyListener(this);

			return content;
		}

		public String getValue()
		{
			return valueText.getText().trim();
		}

		public String getName()
		{
			return nameCombo.getText().trim();
		}

		/**
		 * Sent when the text is modified.
		 * @param e an event containing information about the modify
		 */
		public void modifyText(ModifyEvent e)
		{
			notifyListener();
		}

		/**
		 * @param text
		 */
		public void setValue(String text)
		{
			if(text != null)
				valueText.setText(text);
			else
				valueText.setText("");
		}

		/**
		 * @param text
		 */
		public void setName(String text)
		{
			if(text != null)
				nameCombo.setText(text);
			else
				nameCombo.setText("");
		}

		public void registerListener(Listener listener)
		{
			this.listener = listener;
		}

		public void notifyListener()
		{
			if (listener != null)
				listener.handleEvent(null);
		}
	}

	private final static String[] COLUMNS = { "STRING", "STRING" };

	private TableViewer tableViewer;
	private Button btnAdd;
	private Button btnRename;
	private Button btnRemove;

	private Button btnUp;
	private Button btnDown;

	private RequestHelper requestHelper;
	
	private boolean readOnly;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		requestHelper = null;
	}

	public void createControl(Composite parent, WidgetFactory widgetFactory)
	{
		Composite composite = widgetFactory.createComposite(parent, SWT.NONE);
		composite.setLayoutData(GridDataUtil.createFill());
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginHeight = 5;
		gridLayout.marginWidth = 0;
		composite.setLayout(gridLayout);

		widgetFactory.createLabel(composite, TestHttpPlugin.getString("Headers_Label"));
		createTable(composite, widgetFactory);	
	}

	private void createButtons(Composite parent, WidgetFactory widgetFactory)
	{
		btnAdd = widgetFactory.createButton(parent, TestHttpPlugin.getString("label.Add"), SWT.PUSH);
		btnAdd.setLayoutData(GridDataUtil.createHorizontalFill());
		btnAdd.setEnabled(false);

		btnRename = widgetFactory.createButton(parent, TestHttpPlugin.getString("label.Update"), SWT.PUSH);
		btnRename.setLayoutData(GridDataUtil.createHorizontalFill());
		btnRename.setEnabled(false);

		btnRemove = widgetFactory.createButton(parent, TestHttpPlugin.getString("label.Remove"), SWT.PUSH);
		btnRemove.setLayoutData(GridDataUtil.createHorizontalFill());
		btnRemove.setEnabled(false);

		btnUp = widgetFactory.createButton(parent, TestHttpPlugin.getString("LBL_UP"), SWT.PUSH);
		GridData data = GridDataUtil.createHorizontalFill();
		data.verticalAlignment = GridData.VERTICAL_ALIGN_END;
		btnUp.setLayoutData(data);
		btnUp.setEnabled(false);

		btnDown = widgetFactory.createButton(parent, TestHttpPlugin.getString("LBL_DOWN"), SWT.PUSH);
		data = GridDataUtil.createHorizontalFill();
		data.verticalAlignment = GridData.VERTICAL_ALIGN_END;
		btnDown.setLayoutData(data);
		btnDown.setEnabled(false);
	}

	private Control createTable(Composite parent, WidgetFactory widgetFactory)
	{
		Composite composite = widgetFactory.createComposite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.verticalSpacing = 10;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(GridDataUtil.createFill());

		createTableViewer(composite, widgetFactory);

		Composite buttons = widgetFactory.createComposite(composite, SWT.NONE);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		data = new GridData(GridData.FILL_VERTICAL);
		buttons.setLayoutData(data);
		layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.numColumns = 1;
		buttons.setLayout(layout);

		createButtons(buttons, widgetFactory);

		btnAdd.addSelectionListener(this);
		btnRename.addSelectionListener(this);
		btnRemove.addSelectionListener(this);
		btnUp.addSelectionListener(this);
		btnDown.addSelectionListener(this);

		return composite;
	}

	private void createTableViewer(Composite parent, WidgetFactory widgetFactory)
	{
		// Create the table viewer.
		Table table = widgetFactory.createTable(parent, SWT.SINGLE | SWT.FULL_SELECTION);		
		tableViewer = new TableViewer(table);

		// Create the table control.
		table.setHeaderVisible(false);
		table.setLinesVisible(true);
		table.setLayoutData(GridDataUtil.createFill());
		table.computeSize(SWT.DEFAULT, SWT.DEFAULT);

		TableLayout tableLayout = new TableLayout();
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];

		TableColumn nameColumn = new TableColumn(table, SWT.LEFT | SWT.FLAT);
		nameColumn.setText(TestHttpPlugin.getString("ENV_NAME"));
		tableLayout.addColumnData(new ColumnWeightData(40));
		cellEditors[0] = new TextCellEditor(table);

		TableColumn valueColumn = new TableColumn(table, SWT.LEFT);
		valueColumn.setText(TestHttpPlugin.getString("ENV_VALUE"));
		tableLayout.addColumnData(new ColumnWeightData(60));
		cellEditors[1] = new TextCellEditor(table);

		table.setLayout(tableLayout);

		// Adjust the table viewer.
		tableViewer.setColumnProperties(COLUMNS);
		tableViewer.setContentProvider(new HeaderContentProvider());
		tableViewer.setLabelProvider(new HeaderLabelProvider());
		tableViewer.getTable().addSelectionListener(this);
	}
	
	public void widgetDefaultSelected(SelectionEvent e)
	{
		widgetSelected(e);
	}
	
	public void widgetSelected(SelectionEvent event)
	{
		if(!isReadOnly() && (requestHelper != null))
		{
			if (event.widget == btnAdd)
			{
				HeaderDialog dlg = new HeaderDialog(btnAdd.getShell(), null);
				if(dlg.open() == Window.OK)
				{
					BVRProperty property = dlg.getProperty();
					requestHelper.addHeader((IProperty)property);
					tableViewer.add(property);
					tableViewer.setSelection(new StructuredSelection(property));
				}
			}
	
			else if (event.widget == btnRemove)
			{
				int index = tableViewer.getTable().getSelectionIndex();
				if(index >= 0)
				{
					IProperty property = (IProperty)tableViewer.getTable().getItem(index).getData();
					requestHelper.removeHeader(property);
					tableViewer.getTable().remove(index);
					index = (index<tableViewer.getTable().getItemCount())?index:(tableViewer.getTable().getItemCount()-1);
					if(index >= 0)
						tableViewer.getTable().select(index);
				}
			}
	
			else if (event.widget == btnRename)
			{
				IStructuredSelection structuredSelection = (IStructuredSelection)tableViewer.getSelection();
				if (structuredSelection.size() > 0)
				{
					Object obj = structuredSelection.getFirstElement();
					if (obj instanceof BVRProperty)
					{
						HeaderDialog dlg = new HeaderDialog(btnRename.getShell(), (BVRProperty)obj);
						dlg.open();
						if (dlg.getReturnCode() == Window.OK)
							tableViewer.update(obj, null);
					}
				}
			}
	
			else if (event.widget == btnUp)
			{
				int index = tableViewer.getTable().getSelectionIndex();
				if(index > 0)
				{
					IProperty property1 = (IProperty)tableViewer.getTable().getItem(index).getData();
					IProperty property2 = (IProperty)tableViewer.getTable().getItem(index-1).getData();
					requestHelper.swapHeaders(property1, property2);
					tableViewer.refresh();
					tableViewer.getTable().select(index-1);
				}
			}
	
			else if (event.widget == btnDown)
			{
				int index = tableViewer.getTable().getSelectionIndex();
				if(index < (tableViewer.getTable().getItemCount()-1))
				{
					IProperty property1 = (IProperty)tableViewer.getTable().getItem(index).getData();
					IProperty property2 = (IProperty)tableViewer.getTable().getItem(index+1).getData();
					requestHelper.swapHeaders(property1, property2);
					tableViewer.refresh();
					tableViewer.getTable().select(index+1);
				}
			}
		}

		btnAdd.setEnabled(requestHelper != null);
		
		boolean enable = 	(requestHelper != null) 
							&& (!tableViewer.getSelection().isEmpty());
		btnRemove.setEnabled(enable);
		btnRename.setEnabled(enable);
		btnUp.setEnabled(enable && (tableViewer.getTable().getSelectionIndex() > 0));
		btnDown.setEnabled(enable && (tableViewer.getTable().getSelectionIndex() < (tableViewer.getTable().getItemCount()-1)));
	}

	public void setInput(RequestHelper requestHelper)
	{
		this.requestHelper = requestHelper;
		setEnabled(requestHelper != null);
		
		tableViewer.setInput(requestHelper);
		
		btnAdd.setEnabled(requestHelper != null);
		boolean enable = 	(requestHelper != null) 
							&& (!tableViewer.getSelection().isEmpty());
		btnDown.setEnabled(enable);
		btnRemove.setEnabled(enable);
		btnRename.setEnabled(enable);
		btnUp.setEnabled(enable);
	}
	
	public boolean isReadOnly()
	{
		return readOnly;
	}
	
	public void setReadOnly(boolean readOnly)
	{
		this.readOnly = readOnly;
	}
	
	public void setEnabled(boolean enabled)
	{
		btnAdd.setEnabled(enabled);
		btnDown.setEnabled(enabled);
		btnRemove.setEnabled(enabled);
		btnRename.setEnabled(enabled);
		btnUp.setEnabled(enabled);
	}
	
	public void clearTable()
	{
		tableViewer.getTable().removeAll();
	}
}