/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.swt.widgets.Display;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;
import org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 1.1.0
 */
public class LogNavigatorSynchronizer
extends AdapterImpl implements IDisposable, IResourceChangeUpdater
{
	private LogNavigator navigator;
	
	private boolean active;
	private Collection addedResources;
	private Collection removedResources;
	private Collection changedResources;
	
	public LogNavigatorSynchronizer(LogNavigator navigator)
	{
		this.navigator = navigator;
		active = true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		navigator = null;
	}
	
	public void register(ResourceChangeUpdaterProvider updaterProvider, ResourceSet resourceSet)
	{
		if(updaterProvider != null)
			updaterProvider.setResourceChangeUpdater(this);
			
		if(resourceSet != null)
		{
			if(resourceSet.eAdapters().add(this))
			{
				for(Iterator i = resourceSet.getResources().iterator(); i.hasNext();)
					handleAdd((Resource)i.next());
			}
		}
	}
	
	/**
	 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse.emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg)
	{
		if(!isActive())
			return;
			
		switch(msg.getEventType())
		{
			case Notification.ADD:
			{
				Object newValue = msg.getNewValue();
				if(newValue instanceof Resource)
					handleAdd((Resource)newValue);
				else if(newValue instanceof TRCProcessProxy)
					handleAdd((TRCProcessProxy)newValue);
				else if(newValue instanceof TRCAgentProxy)
					handleAdd((TRCAgentProxy)newValue);
				break;
			}
				
			case Notification.ADD_MANY:
			{
				Collection newValues = (Collection)msg.getNewValue();
				for (Iterator i = newValues.iterator(); i.hasNext();)
				{
					Object newValue = i.next();
					if(newValue instanceof Resource)
						handleAdd((Resource)newValue);
					else if(newValue instanceof TRCProcessProxy)
						handleAdd((TRCProcessProxy)newValue);
					else if(newValue instanceof TRCAgentProxy)
						handleAdd((TRCAgentProxy)newValue);
				}
				break;
			}

			case Notification.REMOVE:
			{
				Object oldValue = msg.getOldValue();
				if(oldValue instanceof Resource)
					handleRemove((Resource)oldValue);
				else if(oldValue instanceof TRCProcessProxy)
					handleRemove((TRCProcessProxy)oldValue);
				else if(oldValue instanceof TRCAgentProxy)
					handleRemove((TRCAgentProxy)oldValue);
				break;
			}

			case Notification.REMOVE_MANY:
				Collection oldValues = (Collection)msg.getOldValue();
				for (Iterator i = oldValues.iterator(); i.hasNext();)
				{
					Object oldValue = i.next();
					if(oldValue instanceof Resource)
						handleRemove((Resource)oldValue);
					else if(oldValue instanceof TRCProcessProxy)
						handleRemove((TRCProcessProxy)oldValue);
					else if(oldValue instanceof TRCAgentProxy)
						handleRemove((TRCAgentProxy)oldValue);
				}
				break;
				
			case Notification.SET:
				if(msg.getNotifier() instanceof TRCAgentProxy)
					handleSet((TRCAgentProxy)msg.getNotifier());
				break;
		}
	}
	
	protected void handleAdd(Resource resource)
	{
		if(TraceConstants.PROCESS_EXT.equals(resource.getURI().fileExtension()))
		{
			if(resource.eAdapters().add(this))
			{
				for (Iterator i = resource.getContents().iterator(); i.hasNext();)
				{
					EObject eObject = (EObject)i.next();
					if(eObject instanceof TRCProcessProxy)
						handleAdd((TRCProcessProxy)eObject);
				}
			}
		}
	}
	
	protected void handleAdd(TRCProcessProxy processProxy)
	{
		if(processProxy.eAdapters().add(this))
		{
			for(Iterator i = processProxy.getAgentProxies().iterator(); i.hasNext();)
				handleAdd((TRCAgentProxy)i.next());
		}
	}

	protected void handleAdd(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
		{
			if(agentProxy.eAdapters().add(this))
			{
				LogicalFolder logLogicalFolder = navigator.getContentProvider().getLogFolder();
				if(!logLogicalFolder.getChildren().contains(agentProxy))
				{
					logLogicalFolder.getChildren().add(agentProxy);
					addToTree(navigator.getContentProvider().getLogFolder(), agentProxy);
				}
			}
		}
	}

	protected void handleRemove(Resource resource)
	{
		if(TraceConstants.PROCESS_EXT.equals(resource.getURI().fileExtension()))
		{
			resource.eAdapters().remove(this);
			for (Iterator i = resource.getContents().iterator(); i.hasNext();)
			{
				EObject eObject = (EObject)i.next();
				if(eObject instanceof TRCProcessProxy)
					handleRemove((TRCProcessProxy)eObject);
			}
		}
	}
	
	protected void handleRemove(TRCProcessProxy processProxy)
	{
		processProxy.eAdapters().remove(this);
		for(Iterator i = processProxy.getAgentProxies().iterator(); i.hasNext();)
			handleRemove((TRCAgentProxy)i.next());
	}

	protected void handleRemove(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
		{
			agentProxy.eAdapters().remove(this);
			LogicalFolder logLogicalFolder = navigator.getContentProvider().getLogFolder();
			if(logLogicalFolder.getChildren().remove(agentProxy))
				removeFromTree(agentProxy);
		}
	}
	
	protected void handleSet(TRCAgentProxy agentProxy)
	{
		if(HyadesConstants.LOG_AGENT_TYPE.equals(agentProxy.getType()))
			refreshTree(agentProxy);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#setActive(boolean)
	 */
	public void setActive(boolean enable)
	{
		active = enable;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#isActive()
	 */
	public boolean isActive()
	{
		return active;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#started()
	 */
	public void started()
	{
		addedResources = new ArrayList();
		removedResources = new ArrayList();
		changedResources = new ArrayList();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#ended()
	 */
	public void ended()
	{
		LogicalFolder sdFolder = navigator.getContentProvider().getSDFolder();
		boolean doRefresh = false;

		if(!removedResources.isEmpty())
		{
			int initialSize = sdFolder.getChildren().size();
			for(Iterator i = removedResources.iterator(); i.hasNext();)
			{
				IResource resource = (IResource)i.next();
				if(resource.getType() == IResource.FILE)
					continue;
				
				IPath fullPath = resource.getFullPath();
				IResource[] resources = (IResource[])sdFolder.getChildren().toArray(new IResource[sdFolder.getChildren().size()]);
				for(int j = 0, maxj = resources.length; j < maxj; j++)
				{
					if(fullPath.isPrefixOf(resources[j].getFullPath()))
						sdFolder.getChildren().remove(j);
				}
			}
			
			sdFolder.getChildren().removeAll(removedResources);
			doRefresh = (initialSize != sdFolder.getChildren().size());
		}

		if(!addedResources.isEmpty())
		{
			sdFolder.getChildren().addAll(addedResources);
			if(!doRefresh)
				addToTree(sdFolder, addedResources.toArray());
		}

		if(!changedResources.isEmpty())
		{
			if(!doRefresh)
				refreshTree(changedResources.toArray());
		}
		
		if(doRefresh)
			refreshTree(sdFolder);
			
		
		addedResources.clear();
		addedResources = null;
		removedResources.clear();
		removedResources = null;
		changedResources.clear();
		changedResources = null;		
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#add(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean add(IResource parent, IResource[] affectedResources)
	{
		for (int i = 0, maxi = affectedResources.length; i < maxi; i++)
		{
			if(LogContentProvider.SD_FILE_EXTENSION.equals(affectedResources[i].getFileExtension()))
			{
				if(navigator.getContentProvider().getSDFolder().getChildren().add(affectedResources[i]))
					addedResources.add(affectedResources[i]);
			}
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#remove(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean remove(IResource parent, IResource[] affectedResources)
	{
		for (int i = 0, maxi = affectedResources.length; i < maxi; i++)
		{
			if(affectedResources[i].getType() != IResource.FILE)
			{
				removedResources.add(affectedResources[i]);
			}
			else if(LogContentProvider.SD_FILE_EXTENSION.equals(affectedResources[i].getFileExtension()))
			{
				if(navigator.getContentProvider().getSDFolder().getChildren().remove(affectedResources[i]))
					removedResources.add(affectedResources[i]);
			}
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#replaced(org.eclipse.core.resources.IResource)
	 */
	public boolean replaced(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateChildrenType(org.eclipse.core.resources.IResource)
	 */
	public boolean updateChildrenType(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateContent(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource)
	 */
	public boolean updateContent(IResource parent, IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateProperties(org.eclipse.core.resources.IResource)
	 */
	public boolean updateProperties(IResource affectedResource)
	{
		if(LogContentProvider.SD_FILE_EXTENSION.equals(affectedResource.getFileExtension()))
			changedResources.add(affectedResource);
		return false;
	}
	
	protected void addToTree(final Object parent, final Object element)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getContentProvider().getTreeViewer().add(parent, element);
			}
		});
	}

	protected void addToTree(final Object parent, final Object[] elements)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getContentProvider().getTreeViewer().add(parent, elements);
			}
		});
	}

	protected void removeFromTree(final Object element)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getContentProvider().getTreeViewer().remove(element);
			}
		});
	}

	protected void removeFromTree(final Object[] elements)
	{
		Display.getCurrent().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getContentProvider().getTreeViewer().remove(elements);
			}
		});
	}
	
	protected void refreshTree(final Object element)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				getNavigator().getContentProvider().getTreeViewer().refresh(element, true);
			}
		});
	}
	
	protected void refreshTree(final Object[] elements)
	{
		Display.getDefault().asyncExec(new Runnable()
		{
			public void run()
			{
				for (int i = 0, maxi = elements.length; i < maxi; i++)
					getNavigator().getContentProvider().getTreeViewer().refresh(elements[i], true);
			}
		});
	}
	
	protected LogNavigator getNavigator()
	{
		return navigator;
	}
}
