/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;

import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.PackageStatisticPage;
import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;

public class PackageStatistic extends StatisticView
{
	private final TRCPackageImpl.TRCPackageSnapshot pdelta1 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCPackageImpl.TRCPackageSnapshot pdelta2 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta1 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta2 = new TRCClassImpl.TRCClassSnapshot();
	
	/**
	 * 
	 *
	 */
	public class PackageStatisticFilter extends StatisticFilter {

		public PackageStatisticFilter() {
			super();
		}

		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			
			if(!(element instanceof TRCPackage))
			  return true;
			  
			String compareText = "";
		   compareText = ((TRCPackage)element).getName();
		   if(compareText.equals(""))
			 compareText = TraceUIPlugin.getString("DEFAULT_PACKAGE");
			
			if (_noPattern)
				return true;
			
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	/**
	 * 
	 */
	public class PackageStatisticSorter extends StatisticSorter {

		public PackageStatisticSorter() {
			super();
		}

		public int compare(Viewer viewer, Object e1, Object e2) {
			
			double d = 0;
 			if(e1 instanceof TRCPackage)	
			{
				TRCPackage p1 = (TRCPackage)e1;
				TRCPackage p2 = (TRCPackage)e2;
				TRCPackageImpl.TRCPackageSnapshot ps1 = (TRCPackageImpl.TRCPackageSnapshot)p1.retrieveSnapshot();
				TRCPackageImpl.TRCPackageSnapshot ps2 = (TRCPackageImpl.TRCPackageSnapshot)p2.retrieveSnapshot();

				switch(_pos)
				{
					case 0://package name
					  return (int) _sortSequence * (int)p1.getName().compareToIgnoreCase(p2.getName());
					  
					case 1: //total inst
					  if(!_info.isDeltaColumn())
					  {	
						  return (int) _sortSequence * (ps1.getTotalInstances() - ps2.getTotalInstances());					  						  					
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						return (int) _sortSequence * (pdelta1.getTotalInstances() - pdelta2.getTotalInstances());					  	
					  }
					  					 
					case 2: //live inst
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalInstances() - ps1.getCollectedInstances()- p2.getTotalInstances() + p2.getCollectedInstances());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						
						return (int) _sortSequence * (pdelta1.getTotalInstances() - pdelta1.getCollectedInstances()
						                       - pdelta2.getTotalInstances() + pdelta2.getCollectedInstances());					  	
					  }
					  
					case 3://collected
					  if(!_info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  int inst = ps1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = ((double)ps1.getCollectedInstances())/inst;
						  	  
						  	  inst = ps2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= ((double)ps2.getCollectedInstances())/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {
							  return (int) _sortSequence * (ps1.getCollectedInstances() - ps2.getCollectedInstances());
						  }
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						return (int) _sortSequence * (pdelta1.getCollectedInstances() - pdelta2.getCollectedInstances());					  	
					  }
					  
					case 4: //total size
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalSize() - ps2.getTotalSize());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						return (int) _sortSequence * (pdelta1.getTotalSize() - pdelta2.getTotalSize());					  	
					  }
					  
					case 5: //active size
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getTotalSize() - ps2.getCollectedSize()- p2.getTotalSize() + p2.getCollectedSize());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						
						return (int) _sortSequence * (pdelta1.getTotalSize() - pdelta1.getCollectedSize()
											   - pdelta2.getTotalSize() + pdelta2.getCollectedSize());					  	
					  }
					  					  
					case 6: //base time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getBaseTime() - ps2.getBaseTime();						  
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__BASE_TIME);
						d = pdelta1.getBaseTime() - pdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					    
			
					case 7: //inherited base time
						
					  if(!_info.isDeltaColumn())
					  {											
						  d = ps1.getInheritedBaseTime() - ps2.getInheritedBaseTime();
					  } 
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						d = pdelta1.getInheritedBaseTime() - pdelta2.getInheritedBaseTime();					  	
					  }
					  	
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					  					  
					case 8: //cumulative time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getCumulativeTime() - ps2.getCumulativeTime();
					  } 
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						d = pdelta1.getCumulativeTime() - pdelta2.getCumulativeTime();					  	
					  } 
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					  
					case 9: //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getInheritedCumulativeTime() - ps2.getInheritedCumulativeTime();
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						d = pdelta1.getInheritedCumulativeTime() - pdelta2.getInheritedCumulativeTime();					  	
					  }  
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					
					case 10: //calls
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getCalls() - ps2.getCalls());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__CALLS);
						return (int) _sortSequence * (pdelta1.getCalls() - pdelta2.getCalls());					  	
					  }
				
					case 11: //inherited calls
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getInheritedCalls() - ps2.getInheritedCalls());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						return (int) _sortSequence * (pdelta1.getInheritedCalls() - pdelta2.getInheritedCalls());					  	
					  }					  
				}				  				  				 				   				  
			}
			else
			{//class objects
				
				TRCClass c1 = (TRCClass) e1;
				TRCClass c2 = (TRCClass) e2;
				TRCClassImpl.TRCClassSnapshot cs1 = (TRCClassImpl.TRCClassSnapshot)c1.retrieveSnapshot();
				TRCClassImpl.TRCClassSnapshot cs2 = (TRCClassImpl.TRCClassSnapshot)c2.retrieveSnapshot();
				
				switch(_pos)
				{
					case 0://class name
					  return (int) _sortSequence * (int)c1.getName().compareToIgnoreCase(c2.getName());
					  
					case 1: //total inst
					  if(!_info.isDeltaColumn())
					  {	
						  return (int) _sortSequence * (cs1.getTotalInstances() - cs2.getTotalInstances());					  						  															
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return (int) _sortSequence * (cdelta1.getTotalInstances() - cdelta2.getTotalInstances());					  	
					  }
					  					 
					case 2: //live inst
					  if(!_info.isDeltaColumn())
					  {															
						  return (int) _sortSequence * (cs1.getTotalInstances() - cs1.getCollectedInstances()- c2.getTotalInstances() + c2.getCollectedInstances());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * ((cdelta1.getTotalInstances() - cdelta1.getCollectedInstances())
							   - (cdelta2.getTotalInstances() - cdelta2.getCollectedInstances()));					  	
					  }
					  
					case 3://collected
					  if(!_info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  int inst = cs1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = (double)cs1.getCollectedInstances()/inst;
						  	  
						  	  inst = cs2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= (double)cs2.getCollectedInstances()/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {					
						  	return (int) _sortSequence * (cs1.getCollectedInstances() - cs2.getCollectedInstances());
						  }
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * (cdelta1.getCollectedInstances() - cdelta2.getCollectedInstances());					  	
					  }
					  
					case 4: //total size
					  if(!_info.isDeltaColumn())
					  {															
						  return (int) _sortSequence * (cs1.getTotalSize() - cs2.getTotalSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return (int) _sortSequence * (cdelta1.getTotalSize() - cdelta2.getTotalSize());					  	
					  }
					  
					case 5: //active size
					  if(!_info.isDeltaColumn())
					  {															
						  return (int) _sortSequence * (cs1.getTotalSize() - cs2.getCollectedSize()- ((TRCClass)e2).getTotalSize() + ((TRCClass)e2).getCollectedSize());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return (int) _sortSequence * ((cdelta1.getTotalSize() - cdelta1.getCollectedSize())
							   - (cdelta2.getTotalSize() - cdelta2.getCollectedSize()));					  	
					  }
					  					  
					case 6: //base time
					  if(!_info.isDeltaColumn())
					  {															
						  d = cs1.getBaseTime() - cs2.getBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__BASE_TIME);
						d = cdelta1.getBaseTime() - cdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 7: //inherited base time
					  if(!_info.isDeltaColumn())
					  {															
						  d = cs1.getInheritedBaseTime() - cs2.getInheritedBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						d = cdelta1.getInheritedBaseTime() - cdelta2.getInheritedBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					  					  
					case 8: //cumulative time
					  if(!_info.isDeltaColumn())
					  {															
						  d = cs1.getCumulativeTime() - cs2.getCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						d = cdelta1.getCumulativeTime() - cdelta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 9: //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {															
						  d = cs1.getInheritedCumulativeTime() - cs2.getInheritedCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						d = cdelta1.getInheritedCumulativeTime() - cdelta2.getInheritedCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					  
					case 10: //calls
					  if(!_info.isDeltaColumn())
					  {															
						  return (int) _sortSequence * (cs1.getCalls() - cs2.getCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CALLS);
						return (int) _sortSequence * (cdelta1.getCalls() - cdelta2.getCalls());					  	
					  }
					
					case 11: //inherited calls
					  if(!_info.isDeltaColumn())
					  {															
						  return (int) _sortSequence * (cs1.getInheritedCalls() - cs2.getInheritedCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return (int) _sortSequence * (cdelta1.getInheritedCalls() - cdelta2.getInheritedCalls());					  	
					  }					  
				}				  				  				 				   				  
			}
			
			return 0;
		}
		
	}
	
    /**
     * 
     */
	public class PackageStatisticContentProvider implements ITreeContentProvider {
		
		public void dispose() {
		}
		
		public Object getParent(Object element) {
			
			if(element instanceof TRCClass)
				return ((TRCClass)element).getPackage();
				
			return null;	
		}
		
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		
		public Object[] getElements(Object inputElement)
		{
			Object[] packages = PerftraceUtil.getAllPackages(_page.getMOFObject());
			ArrayList list = new ArrayList();
			for(int idx=0; idx<packages.length; idx++)
			{
				TRCPackage pack = (TRCPackage)packages[idx];
				if(pack.getClasses().size() > 0)
				   list.add(pack);
			}
			
			return list.toArray();
		}
				
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCPackage)
				return ((TRCPackage)element).getClasses().toArray();
			
			return tmpList.toArray(); 	
		}
	
		public boolean hasChildren(Object element) {
			if (element instanceof TRCPackage)
				return true;

			return false;
		}
	}
	
    /**
     * 
     */
	public class PackageStatisticLabelProvider extends LabelProvider implements ITableLabelProvider
	{
		protected StatisticView _viewer;
		
		public PackageStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}
		
		public Image getColumnImage(Object obj, int col) {
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage)obj;

				if(p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES)<0)
				{
					if(col == 1)
						return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					return null;	
				}
				 
				switch(pos)
				{
					case 1: //total inst
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);					
					  state = pdelta1.getTotalInstances();
					  break;
					  					 
					case 2: //live inst	
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);				
					  state = pdelta1.getTotalInstances() - pdelta1.getCollectedInstances();
					  break;
					  
					case 3://collected
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
					  state = pdelta1.getCollectedInstances();
					  break;
					  
					case 4: //total size
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  state = pdelta1.getTotalSize();
					  break;
					  
					case 5: //active size
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
					  state = pdelta1.getTotalSize() - pdelta1.getCollectedSize();
					  break;
									  					  
					case 6: //base time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);					
					  d = pdelta1.getBaseTime();
					  
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;

					case 7: //inherited base time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
					  d = pdelta1.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  break;
					  					  
					case 8: //cumulative time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
					  d = pdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 9: //inherited cumulative time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
					  d = pdelta1.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 10: //calls
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
					  state = pdelta1.getCalls();
					  break;
					  
					case 11: //inherited calls
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
					  state = pdelta1.getInheritedCalls();
					  break;
					
				}				  				  				 				   				  
			}
			else
			{
	            //TRClass				
				TRCClass clas = (TRCClass)obj;

				if(((TRCClass)obj).computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES)<0)
				{
					if(col == 1)
						return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					return null;	
				}
				
				switch(pos)
				{					
					case 1: //total inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);					
					  state = cdelta1.getTotalInstances();
					   break;
					  					 
					case 2: //live inst
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getTotalInstances() - cdelta1.getCollectedInstances();
					   break;
					  
					case 3://collected
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = cdelta1.getCollectedInstances();
					   break;
					
					case 4: //total size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  state = cdelta1.getTotalSize();
					   break;
					  
					case 5: //active size
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
					  state = cdelta1.getTotalSize() - cdelta1.getCollectedSize();
					   break;
					
					case 6: //base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);					
					  d = cdelta1.getBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 7: //inherited base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
					  d = cdelta1.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  					  
					case 8: //cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
					  d = cdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
						  
					case 9: //inherited cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
					  d = cdelta1.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;						  
					  
					case 10: //calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
					  state = cdelta1.getCalls();
					  break;
					  
					case 11: //inherited calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
					  state = cdelta1.getInheritedCalls();
					  break;
				}
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
		
		}

		public String getColumnText(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage) obj;
				TRCPackageImpl.TRCPackageSnapshot ps = (TRCPackageImpl.TRCPackageSnapshot)p.retrieveSnapshot();
				
				switch(pos)
				{
					case 0://package name
					{
						return PerftraceUtil.getPackageName(p, _page.getMOFObject()).toString();
					}
					  	
					case 1: //total inst
					
					  if(!info.isDeltaColumn())
					  {
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getTotalInstances()/_totalInst);
						  }
						  else
						  {
							  return String.valueOf(ps.getTotalInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						return String.valueOf(pdelta1.getTotalInstances());					  	
					  }
					  					 
					case 2: //live inst
					
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(ps.getTotalInstances() - ps.getCollectedInstances())/_activeInst);
						  }
						  else
						  {					
							  return String.valueOf(ps.getTotalInstances() - ps.getCollectedInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES);
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);						
						return String.valueOf(pdelta1.getTotalInstances() - pdelta1.getCollectedInstances());					  	
					  }
					  
					case 3://collected
					
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  int inst = ps.getTotalInstances();
						  	  if(inst == 0) inst = 1;
						  	  
						  	  return TString.formatAsPercentage((double)ps.getCollectedInstances()/inst);
						  }
						  else
						  {					
						  		return String.valueOf(ps.getCollectedInstances());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_INSTANCES);
						return String.valueOf(pdelta1.getCollectedInstances());					  	
					  }
					  
					case 4: //total size
					
					  if(!info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getTotalSize()/_totalSize);
						  }
						  else
						  {					
							  return String.valueOf(ps.getTotalSize());
						  }
					  }
					  else
					  {
					  	p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
					  	return String.valueOf(pdelta1.getTotalSize());					  	
					  }
					  
					  
					case 5: //active size
					  if(!info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(ps.getTotalSize() - ps.getCollectedSize())/_activeSize);
						  }
						  else
						  {					
						  	return String.valueOf(ps.getTotalSize() - ps.getCollectedSize());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_SIZE);
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__COLLECTED_SIZE);
						
						return String.valueOf(pdelta1.getTotalSize() - pdelta1.getCollectedSize());					  	
					  }
					  					  
					case 6: //base time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getBaseTime()/_maxTime);
						  }
						  else
						  {				
							  return TString.formatTimeValue(ps.getBaseTime());						  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);
						 return TString.formatTimeValue(pdelta1.getBaseTime());						  					  	
					  }

					case 7: //inherited base time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {				
							  return TString.formatTimeValue(ps.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						 return TString.formatTimeValue(pdelta1.getInheritedBaseTime());						  					  	
					  }
					  
					  					  
					case 8: //cumulative time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getCumulativeTime()/_maxTime);
						  }
						  else
						  {									
							  return TString.formatTimeValue(ps.getCumulativeTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						 return TString.formatTimeValue(pdelta1.getCumulativeTime());						  					  	
					  }
					  
					  
					case 9: //inherited cumulative time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getInheritedCumulativeTime()/_maxTime);
						  }
						  else
						  {									
							  return TString.formatTimeValue(ps.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						 return TString.formatTimeValue(pdelta1.getInheritedCumulativeTime());						  					  	
					  }
					  
					  
					case 10: //calls
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getCalls()/_totalCalls);
						  }
						  else
						  {					
							  return String.valueOf(ps.getCalls());					  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
						return String.valueOf(pdelta1.getCalls());					  	
					  }
					  
					  
					case 11: //inherited calls
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getInheritedCalls()/_totalCalls);
						  }
						  else
						  {					
							  return String.valueOf(ps.getInheritedCalls());					  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						return String.valueOf(pdelta1.getInheritedCalls());					  	
					  }					  
					  
				}				  				  				 				   				  
			}
			else
			{
                TRCClass c = (TRCClass) obj;
				TRCClassImpl.TRCClassSnapshot cs = (TRCClassImpl.TRCClassSnapshot)c.retrieveSnapshot();
                
	            //TRClass
				switch(pos)
				{
					case 0://class name
					  return c.getName();
					  	
					case 1: //total inst
					  if(!info.isDeltaColumn())
					  {																								
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getTotalInstances()/_totalInst);
						  }
						  else
						  {				
							  return String.valueOf(cs.getTotalInstances());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances());					  	
					  }
					  					 
					case 2: //live inst
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(cs.getTotalInstances() - cs.getCollectedInstances())/_totalInst);
						  }
						  else
						  {				
							  return String.valueOf(cs.getTotalInstances() - cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getTotalInstances() - cdelta1.getCollectedInstances());					  	
					  }
					  
					  
					case 3://collected
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  int inst = cs.getTotalInstances();
						  	  if(inst == 0) inst = 1;
						  	  
						  	  return TString.formatAsPercentage((double)cs.getCollectedInstances()/inst);
						  }
						  else
						  {				
							  return String.valueOf(cs.getCollectedInstances());
						   }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return String.valueOf(cdelta1.getCollectedInstances());					  	
					  }
					  					  
					case 4: //total size
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getTotalSize()/_totalSize);
						  }
						  else
						  {								
							  return String.valueOf(cs.getTotalSize());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return String.valueOf(cdelta1.getTotalSize());					  	
					  }
					  
					case 5: //active size
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)(cs.getTotalSize() - cs.getCollectedSize())/_activeSize);
						  }
						  else
						  {								
							  return String.valueOf(cs.getTotalSize() - cs.getCollectedSize());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return String.valueOf(cdelta1.getTotalSize() - cdelta1.getCollectedSize());					  	
					  }
					  					  
					case 6: //base time
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getBaseTime()/_maxTime);
						  }
						  else
						  {													
					  		return TString.formatTimeValue(cs.getBaseTime());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						return TString.formatTimeValue(cdelta1.getBaseTime());					  	
					  }
	
					case 7: //inherited base time
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {													
					  		return TString.formatTimeValue(cs.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedBaseTime());					  	
					  }
					  					  
					case 8: //cumulative time
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getCumulativeTime()/_maxTime);
						  }
						  else
						  {									
					
							  return TString.formatTimeValue(cs.getCumulativeTime());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getCumulativeTime());					  	
					  }
						  
					case 9: //inherited cumulative time
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedCumulativeTime()/_maxTime);
						  }
						  else
						  {									
					
							  return TString.formatTimeValue(cs.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedCumulativeTime());					  	
					  }
						  
					  
					case 10: //calls
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getCalls()/_totalCalls);
						  }
						  else
						  {				
							  return String.valueOf(cs.getCalls());
						  }
					  }
					  else
					  {				
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						return String.valueOf(cdelta1.getCalls());					  	
					  }
					  
					case 11: //inherited calls
					  if(!info.isDeltaColumn())
					  {																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getInheritedCalls()/_totalCalls);
						  }
						  else
						  {				
							  return String.valueOf(cs.getInheritedCalls());
						  }
					  }
					  else
					  {				
						c.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return String.valueOf(cdelta1.getInheritedCalls());					  	
					  }					  
				}
			}
            							
			return "";
		}
	}
	
	class PackageTreeViewer extends TableTreeViewer {
		public PackageTreeViewer(Composite parent) {
			super(parent);
		}

		public PackageTreeViewer(TableTree table) {
			super(table);
		}

		public void expandItem(TableTreeItem item) {
			item.setExpanded(true);
			createChildren(item);

		}

	}
	protected Composite createTable(Composite parent, int options) {
		return new TableTree(parent, options);
	}

	public PackageStatistic(Composite parent, TraceViewerPage page) {
		super(parent, page);
		_viewerFilter = new PackageStatisticFilter();		
	}

	/**
	 * getClassColumnsPerferencesKey returns the string key used to find the
	 * column preferences information in the preferences store.
	 * @return java.lang.String key name
	 */
	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
		return "Pack60"; 
		
	}
	
	public String getDefaultColumnsTemplate() {
		//Class Columns Data
		String classColumn =
			TraceUIPlugin.getString("STR_ST_PACKAGE") 	+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
			+ TraceUIPlugin.getString("STR_ST_INSTANCES")	+ ":1:" 
			    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_LIVE_INSTANCES")+ ":2:"
			    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_COLLECTED")	+ ":3:"
			    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"			    
			+ TraceUIPlugin.getString("STR_ST_SIZE") 			+ ":4:"
			    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"						    
			+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE") 	+ ":5:"
			    + String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":6:"
			  + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_CPU_INHTIME")		+ ":7:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_STACK_TIME") 	+ ":8:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_STACK_INHTIME") 	+ ":9:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_CALLS") 		+ ":10:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:50,"
			+ TraceUIPlugin.getString("STR_ST_INHCALLS") 		+ ":11:"
			  + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100"			
		;
		return classColumn;
	}

	protected IContentProvider getContentProvider() {
		return new PackageStatisticContentProvider();
	}

	public LabelProvider getTableLabelProvider()
	{
		return new PackageStatisticLabelProvider(this);
	}

	public Table getTable() {
		return ((TableTree) getTableViewer().getControl()).getTable();
	}

	protected StructuredViewer createTableViewer(Composite table) {
		PackageTreeViewer tv = new PackageTreeViewer((TableTree) table);
		return (StructuredViewer) tv;
	}

	protected void handleSelectionEvent() {

		ITraceSelection model = UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {

			Object sel = model.getFirstElement();              

			// If the selection is an message then turn the selection into the class of the object
			if (sel instanceof TRCMethod) {
				sel = ((TRCMethod) sel).getDefiningClass();
			}
			else if (sel instanceof TRCMethodInvocation) {
				sel = ((TRCMethodInvocation) sel).getMethod().getDefiningClass();
			}			
			if (sel instanceof TRCObject) {
				
				select(PerftraceUtil.getClass((TRCObject)sel));
			}
			if (sel instanceof TRCClass) {
				select((TRCClass)sel);
			}
		}
	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {

		menu.add(getUpdateAction());				
		menu.add(getChooseColumnsAction(getColumnDataList(), getColumnsPreferencesKey()));
	}

	private void select(Object obj) {
		
		TableTree table = (TableTree) getTableViewer().getControl();

        TRCClass cls = null;
        TRCPackage pack = null;
        
        if(obj instanceof TRCPackage)
        {
        	pack = (TRCPackage)obj;
        }
        else if(obj instanceof TRCClass)
        {
        	cls = (TRCClass)obj;
        	pack = cls.getPackage();
        }
		
		TableTreeItem[] items = table.getItems();
		for (int idx = 0; idx < items.length; idx++) {
			TableTreeItem item = items[idx];
			if(((TRCPackage)item.getData()).getName().equals(pack.getName()))
			{
				if(cls == null)
				{
					table.setSelection(new TableTreeItem[] { item });
					return;					
				}
				
				((PackageTreeViewer) getTableViewer()).expandItem(item);
				TableTreeItem[] childItems = item.getItems();
	
				for (int i = 0; i < childItems.length; i++) {
					TableTreeItem childItem = childItems[i];
	  				if (((TRCClass)childItem.getData()).getName().equals(cls.getName())) {
							((PackageTreeViewer) getTableViewer()).expandItem(item);
							table.setSelection(new TableTreeItem[] { childItem });
							return;
					}
				}
			}
		}
		
		table.deselectAll();		
	}
	

	public void updateButtons() {
		((PackageStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());
		((PackageStatisticPage) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
	}

	protected void updateDetailsPane() {
		int selIndex = getTable().getSelectionIndex();
		if (selIndex < 0) {
			return;
		}

		Item item = getTable().getItem(selIndex);
		Object itemData = item.getData();

		if (itemData == null) {
			return;
		}
		if (itemData != null && itemData instanceof TableTreeItem) {
			itemData = ((TableTreeItem) itemData).getData();
		}
	}

	public void updateModelSelection() {
		ISelection selection = getTableViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			UIPlugin.getDefault().getSelectionModel(
				_page.getMOFObject()).add(
				sel);
	
			ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
			event.setSource(_page.getMOFObject());
			UIPlugin.getDefault().notifyViewSelectionChangedListener(event);			
		}
	}
	
	public void update() {
		if (_firstTime) {
			getTableViewer().addFilter(getViewerFilter());
			_firstTime = false;
			Table table = getTable();
			TableColumn firstColumn = table.getColumn(0);
			
			_viewerSorter = new PackageStatisticSorter();

			getViewerSorter().setSortedColumn(firstColumn);
			getTableViewer().setSorter(getViewerSorter());
			
		}
		// set the input of the viewer
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());	
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());	
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());	
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());						
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
		}
			
		getTableViewer().setInput(_page.getMOFObject());

        getTable().setRedraw(false);
		getTableViewer().refresh();
        getTable().setRedraw(true);
        
        handleSelectionEvent();
		
	}
	
	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event)
	{
		if(!getTraceViewerPage().getTraceViewer().isProcessRefreshEvents()		
		|| getTable().isFocusControl())
		  return;
		  
		handleSelectionEvent();  
	}
	
}
