/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;
import java.util.ArrayList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.trace.*;

/**
 * @author hind.hasnaoui@fr.ibm.com  /                           
 * 
 * Coverage objects are assosciated to TRC* objects (_traceObj)
 * They store coverage information:
 * number of methods (_methods)
 * number of methods hit (_methods hit)
 * number of method calls (_calls)
 * and the method coverage rate (_coveragePercentage)
 * 
 */
public class Coverage {

	private int _methods;
	private int _methodsHit;
	private ArrayList _coverageList;
	private double _covPercentage;
	private Object _traceObj;
	private int _calls;

	public Coverage(Object obj) {
		//coverage object initialization

		setMethods(0);
		setMethodsHit(0);
		setCalls(0);
		setCovPercentage(0);
		setCoverageList(new ArrayList());
		setTraceObj(obj);

		if (obj instanceof TRCMethod)
			setMethods(1);
	}

	/**
	*  This method cover the model trace tree :             
	*  TRCPackage/TRCClass/TRCMethods/TRCMethodInvocation  
	*/
	public void cover(EList list) {
		EList subList = null;

		if (_traceObj instanceof TRCMethod) { //Invoqued method 
			setCalls(list.size());
			if (list.size() != 0)
				setMethodsHit(1);
			return;
		} else {
			//list of Packages, Classes or methods
			for (int i = 0; i < list.size(); i++) {
				//create a coverage object associated with a TRC* object and add it to the _coverageList
				Coverage cov = create(list.get(i));

				if (cov.getTraceObj() instanceof TRCMethod)
					subList = ((TRCMethod) cov.getTraceObj()).getInvocations();
				else if (cov.getTraceObj() instanceof TRCClass)
					subList = ((TRCClass) cov.getTraceObj()).getMethods();
				else if (cov.getTraceObj() instanceof TRCPackage)
					subList = ((TRCPackage) cov.getTraceObj()).getClasses();

				// subList coverage
				if (subList != null) {
					cov.cover(subList);

					//test if the cov object is associated with a TRCClass     
					if (cov.getTraceObj() instanceof TRCClass) {
						//Test if subClass :the name of subClasses in the package =ClassParentName$subClassName
						if ((((TRCClass) cov.getTraceObj()).getName())
							.indexOf("$")
							!= -1) {
							String className =
								((TRCClass) cov.getTraceObj()).getName();

							//Search for parent class in _coverageList
							String parentClassName =
								className.substring(0, className.indexOf("$"));
							Coverage covParent =
								findParentClass(parentClassName);

							if (covParent != null) {
								//Report subClass information in parent class	
								covParent.sum(
									cov.getMethods(),
									cov.getMethodsHit(),
									cov.getCalls());
								//Update parent class percentage
								covParent.computePercentage();

								//Remove this coverage object from the TRCPackage list
								getCoverageList().remove(cov);

								//Add this coverage object to the list of the covParent
								(covParent.getCoverageList()).add(cov);
							}
						}
					}
				}

				//Report coverage information in the current object
				sum(cov.getMethods(), cov.getMethodsHit(), cov.getCalls());
			}

			//Compute the method coverage percentage
			computePercentage();
			return;
		}
	}

	/**
	 * This method create and initialize a coverage object                             
	 * and add it to the _coverageList         
	*/
	protected Coverage create(Object obj) {
		Coverage cov = new Coverage(obj);
		getCoverageList().add(cov);
		return cov;
	}

	/**
	 * Sum method           						                                                   
	*/
	protected void sum(int methods, int methodsHit, int calls) {

		setMethods(getMethods() + methods);
		setMethodsHit(getMethodsHit() + methodsHit);
		setCalls(getCalls() + calls);
	}

	/**
	 * Compute the percentage of method coverage			
	*/
	protected void computePercentage() {

		if (getMethods() != 0)
			setCovPercentage(((double) getMethodsHit()) / getMethods());
	}

	/**
	 * Search for the parent class to report in it the number of 
	 * methods and methodsHit of the subclass                        
	*/
	protected Coverage findParentClass(String parentClassName) {

		ArrayList list = getCoverageList();

		for (int i = 0; i < list.size(); i++) {
			Object obj = ((Coverage) list.get(i)).getTraceObj();
			if ((((TRCClass) obj).getName()).equals(parentClassName))
				return ((Coverage) list.get(i));

		}
		return null;
	}

	/**
	 * @return the number of methods not called during the execution
	 */
	public int getMethodsMissed() {
		return (getMethods() - getMethodsHit());

	}

	/**
	 * @return the TRC* Object referenced by this Coverage object
	 */
	public Object getTraceObj() {
		return _traceObj;
	}

	/**
	* @param number of method calls
	*/
	public void setCalls(int i) {
		_calls = i;
	}

	/**
	 * @param number of methods
	 */
	public void setMethods(int i) {
		_methods = i;
	}

	/**
	 * @param number of methods hit
	 */
	public void setMethodsHit(int i) {
		_methodsHit = i;
	}

	/**
	 * @return the number of method calls
	 */
	public int getCalls() {
		return _calls;
	}

	/**
	 * @return the number of methods
	 */
	public int getMethods() {
		return _methods;
	}

	/**
	 * @return the number of methods called durin the execution
	 */
	public int getMethodsHit() {
		return _methodsHit;
	}

	/**
	 * @param TRC* object associated
	 */
	public void setTraceObj(Object object) {
		_traceObj = object;
	}

	/**
	 * @return the list of coverage objects
	 */
	public ArrayList getCoverageList() {
		return _coverageList;
	}

	/**
	 * @param the coverage method percentage
	 */
	public void setCovPercentage(double percentage) {
		_covPercentage = percentage;
	}

	/**
	 * @return the coverage method percentage
	 */
	public double getCovPercentage() {
		return _covPercentage;
	}

	/**
	 * @param the list of coverage objects
	 */
	public void setCoverageList(ArrayList list) {
		_coverageList = list;
	}

}
